/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */
using namespace std;

#include "app.h"
#include "menus.h"
#include "gnome-apt.h"


///// Declarations for non-menu callbacks

static gint delete_event_cb(GtkWidget* w, GdkEvent* event, gpointer data);

///// config load/save stuff


// Eventually the --geometry command-line argument should override
//  this stuff.

static void
load_geometry(GtkWidget* app, const gchar* override)
{
  const gchar* s = 0;
  gchar* alloced = 0;
  if (override == 0)
    {
      s = alloced = gnome_config_get_string("/gnome-apt/App/geometry");
    }
  else 
    {
      s = override;
    }

  bool gotset = false;

  if (s != 0)
    {
      gint xpos, ypos, width, height;

      if (gnome_parse_geometry(s, &xpos, &ypos, &width, &height))
        {
          gtk_window_set_default_size(GTK_WINDOW(app), width, height);
          gtk_widget_set_uposition(app, xpos, ypos);
          gotset = true;
        }
      
      if (alloced) g_free(alloced);
    }

  if (gotset == false)
    {
      gtk_window_set_default_size(GTK_WINDOW(app), 400, 500);
    }
}

static void
save_geometry(GtkWidget* app)
{
  if (!GTK_WIDGET_REALIZED(app)) return; // no window on screen

  gchar* s = gnome_geometry_string(app->window);

  if (s)
    {
      gnome_config_set_string("/gnome-apt/App/geometry", s);
      gnome_config_sync();
      g_free(s);
    }
}


////////////////////

GAptApp::GAptApp()
  : app_(0)
{
  init(0);
}

GAptApp::GAptApp(const gchar* geometry)
  : app_(0)
{
  init(geometry);
}

void
GAptApp::init(const gchar* geometry)
{
  g_return_if_fail(app_ == 0);

  app_    = gnome_app_new(APPNAME, _("GNOME Apt: A Package Tool"));

  load_geometry(app_, geometry);

  // we export a progress interface but for now there's no progress
  // bar shown
  appbar_ = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_NEVER);

  gnome_app_set_statusbar(GNOME_APP(app_),
                          appbar_);

  pkgtree_ = new GAptPkgTree;
  pkglist_ = new GAptPkgList(pkgtree_, GNOME_APPBAR(appbar_));

  gnome_app_set_contents(GNOME_APP(app_), pkglist_->widget());

  gtk_signal_connect(GTK_OBJECT(app_),
                     "delete_event",
                     GTK_SIGNAL_FUNC(delete_event_cb),
                     this);

  gtk_signal_connect(GTK_OBJECT(app_),
                     "key_press_event",
                     GTK_SIGNAL_FUNC(keypress_cb),
                     this);

  pkgtree_->set_status(this);

  progress_ = new Progress(appbar_);

  // do this last so it reflects the final state.
  gnome_apt_create_main_menu(GNOME_APP(app_), GNOME_APPBAR(appbar_), pkglist_);

  set_selection(0); // get all initial state set up

  gtk_widget_show (app_);
}

GAptApp::~GAptApp()
{
  // Order is very important here, since some of these things
  //  hold references to others.
  delete pkgtree_; // Deletes the DrawTree for us.
  pkgtree_ = 0;
  delete pkglist_;
  pkglist_ = 0;
  delete progress_;
  progress_ = 0;
  
  if (app_ != 0)
    {
      save_geometry(app_);

      gtk_widget_destroy(app_);
    }
}

void
GAptApp::set_selection(pkgCache::Package* p)
{
  pkglist_->set_selection(p);
  gnome_apt_menus_selection_changed(pkglist_);
}

//////////////////////////////////////////////
// General callbacks

static gint 
delete_event_cb(GtkWidget* w, GdkEvent* event, gpointer data)
{
  gnome_apt_quit();
  return TRUE;      // if we don't really quit, block default action.
}

// We have to process any "accelerators" here.

gint
GAptApp::keypress_cb(GtkWidget* w, GdkEventKey* event, gpointer data)
{
  GAptApp* app = static_cast<GAptApp*>(data);

  return app->key(event);
}

gint
GAptApp::key(GdkEventKey* event)
{
  if (pkgtree_)
    {
      DrawTree* dt = pkgtree_->tree();
      if (dt)
        {
          bool handled = dt->consider_key_event(event);
          if (handled)
            gtk_signal_emit_stop_by_name(GTK_OBJECT(app_), "key_press_event");
          
          return handled;
        }
    }

  return FALSE;
}

//////////////////// Progress

void 
GAptApp::Progress::Update()
{
  if (CheckChange() == false)
    return;
  
  if (MajorChange == true)
    {
      gnome_appbar_set_status(GNOME_APPBAR(appbar_), Op.c_str());
    }
  
  gnome_appbar_set_progress(GNOME_APPBAR(appbar_), Percent/100.0);
  
  // flush
  while (gtk_events_pending())
    gtk_main_iteration();
}


void 
GAptApp::Progress::Done()
{
  gnome_appbar_refresh(GNOME_APPBAR(appbar_));
  gnome_appbar_set_progress(GNOME_APPBAR(appbar_), 0.0);
}

GAptApp::Progress::Progress(GtkWidget* appbar)
  : appbar_(appbar)
{

}

GAptApp::Progress::~Progress()
{

}
