/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include "gnome-cmd-includes.h"

#ifdef HAVE_LIBFAM

FAMConnection *fam = NULL;
gint fam_fd = -1;


static void on_fam_event (FAMConnection *fam, gint fd, GdkInputCondition condition)
{
	/* Handle all pending events
	 */
	while (FAMPending(fam))
	{
		FAMEvent ev;
		GnomeCmdDir *dir;
		
		if (FAMNextEvent(fam, &ev) != 1)
		{
			/* Disable FAM for the rest of the session
			 */
			perror ("FAMNextEvent error");
			return;
		}

		dir = (GnomeCmdDir*)ev.userdata;

		if (!dir_pool_exists (dir))
			return;
		
		switch (ev.code) {
			case FAMChanged:
				/* A file has been altered
				 */
				gnome_cmd_dir_file_changed (dir, ev.filename);
				break;
				
			case FAMDeleted:
				/* A file has been deleted
				 */
				gnome_cmd_dir_file_deleted (dir, ev.filename);
				break;
				
			case FAMStartExecuting:
				/* Dont know
				 */
				break;
				
			case FAMStopExecuting:
				/* Dont know
				 */
				break;
				
			case FAMCreated:
				/* A file has been created
				 */
				gnome_cmd_dir_file_created (dir, ev.filename);
				break;
				
			case FAMMoved:
				/* Is this used at all?
				 */
				debug_print ("Unexpected FAMMoved in on_fam_event\n");
				break;
				
			case FAMAcknowledge:
				/* Dont know
				 */
				break;
				
			case FAMExists:
				/* We use gnome-vfs to list files so this is not useful
				 */
				break;
				
			case FAMEndExist:
				/* We use gnome-vfs to list files so this is not useful
				 */
				break;

			default:
				debug_print ("Unknown FAMCode in on_fam_event\n");
		}
	}
}


void fam_init (void)
{
	if (fam != NULL) {
		debug_print ("FAM already initialized\n");
		return;
	}
	
	fam = g_new (FAMConnection, 1);
	
	if (FAMOpen2 (fam, "gnome-commander") != 0) {
		perror ("Failed to FAMOpen2 in fam_init");
		g_free (fam);
		fam = NULL;
		warn_print (_("Failed to initialize FAM\n"));
		return;
	}

	fam_fd = FAMCONNECTION_GETFD(fam);

	gdk_input_add (fam_fd, GDK_INPUT_READ, (GdkInputFunction)on_fam_event, fam);
}


void fam_shutdown (void)
{
	if (fam != NULL) {
		while (FAMPending(fam)) {
			FAMEvent ev;
			FAMNextEvent(fam, &ev);
		}
		
		gdk_input_remove (fam_fd);
		FAMClose (fam);
		g_free (fam);
		fam = NULL;
	}
}


void fam_monitor_start (GnomeCmdDir *dir)
{
	g_return_if_fail (fam != NULL);
	
	if (dir->request == NULL) {
		dir->request = g_new (FAMRequest, 1);
	
		if (FAMMonitorDirectory (fam, dir->path, dir->request, dir) != 0) {
			perror ("Failed to FAMMonitorDirectory in fam_monitor_start");
			return;
		}
	}
	else
		debug_print ("The directory %s is already monitored\n", dir->path);
}


void fam_monitor_cancel (GnomeCmdDir *dir)
{
	g_return_if_fail (fam != NULL);	
	g_return_if_fail (dir != NULL);
	g_return_if_fail (dir->request != NULL);
	
	FAMCancelMonitor (fam, dir->request);
	g_free (dir->request);
	dir->request = NULL;
}


#endif //HAVE_LIBFAM
