/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include "gnome-cmd-includes.h"
#include <zvt/zvtterm.h>


#undef ZVT_FONT
#define ZVT_FONT "-adobe-courier-medium-r-normal-*-*-120-*-*-m-*-iso8859-1"
#undef ZVT_SCROLLSIZE
#define ZVT_SCROLLSIZE 200

struct _GnomeCmdZvtTermPrivate
{
	GtkWidget *zvt_term;
};

static GtkFrameClass *parent_class = NULL;

gushort linux_red[] = { 0x0000, 0xaaaa, 0x0000, 0xaaaa, 0x0000, 0xaaaa, 0x0000, 0xaaaa,
						0x5555, 0xffff, 0x5555, 0xffff, 0x5555, 0xffff, 0x5555, 0xffff,
						0x0,    0x0 };
gushort linux_grn[] = { 0x0000, 0x0000, 0xaaaa, 0x5555, 0x0000, 0x0000, 0xaaaa, 0xaaaa,
						0x5555, 0x5555, 0xffff, 0xffff, 0x5555, 0x5555, 0xffff, 0xffff,
						0x0,    0x0 };
gushort linux_blu[] = { 0x0000, 0x0000, 0x0000, 0x0000, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
						0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0xffff, 0xffff, 0xffff,
						0x0,    0x0 };


static void
update_color_scheme ()
{
	GnomeCmdLayoutColors *c = gnome_cmd_data_get_layout_colors ();

	linux_red[16] = c->term_fg->red;
	linux_grn[16] = c->term_fg->green;
	linux_blu[16] = c->term_fg->blue;

	linux_red[17] = c->term_bg->red;
	linux_grn[17] = c->term_bg->green;
	linux_blu[17] = c->term_bg->blue;
}

static gboolean
zvterm_mouse_clicked (GtkWidget* widget, GdkEvent* event, GnomeCmdZvtTerm* term)
{
	g_return_val_if_fail (term != NULL, FALSE);
	g_return_val_if_fail (term->priv != NULL, FALSE);
	
	gtk_widget_grab_focus (term->priv->zvt_term);
	
	return TRUE;
}


static void
zvterm_reinit_child (GtkWidget *widget, GnomeCmdZvtTerm* term)
{
	gint ret;
	
	g_return_if_fail (term != NULL);
	g_return_if_fail (term->priv != NULL);

	gnome_cmd_main_win_hide_embterm (main_win);
	
	/* Fork the pty and start a shell
	 * 
	 */
	ret = zvt_term_forkpty(
		ZVT_TERM (term->priv->zvt_term),
		ZVT_TERM_DO_UTMP_LOG |  ZVT_TERM_DO_WTMP_LOG);
		
	if (ret == 0) {
		struct passwd *pw;
		static gchar *shell = NULL;
		static gchar *name = NULL;
			
		pw = getpwuid (getuid ());
		if (pw) {
			shell = g_strdup (pw->pw_shell);
			name = g_strdup ("-");
		}
		else {
			shell = g_strdup ("/bin/sh");
			name = g_strdup ("-sh");
		}
		execl (shell, name, NULL);
	}
}


static void
zvterm_terminate (GtkWidget *widget, GnomeCmdZvtTerm* term)
{
	g_return_if_fail (term != NULL);
	g_return_if_fail (term->priv != NULL);
	
	gtk_signal_disconnect_by_func (GTK_OBJECT(term->priv->zvt_term),
								   GTK_SIGNAL_FUNC(zvterm_reinit_child),
								   term);
	zvt_term_closepty (ZVT_TERM (term->priv->zvt_term));

	g_free (term->priv);
}


static int
zvterm_focus_in (GtkWidget *widget, GdkEventFocus* event, GnomeCmdZvtTerm* term)
{
	static gboolean need_init = TRUE;
	
	g_return_val_if_fail (term != NULL, FALSE);
	g_return_val_if_fail (term->priv != NULL, FALSE);

	if (need_init) {
		zvterm_reinit_child (widget, term);
		need_init = FALSE;
	}
	return TRUE;
}



/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdZvtTermClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gtk_frame_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}


static void
init (GnomeCmdZvtTerm *term)
{
	GtkFrame *frame = GTK_FRAME (term);
	term->priv = g_new (GnomeCmdZvtTermPrivate, 1);
	
    putenv("TERM=xterm");

	/* Create the widget
	 *
	 */
	gtk_frame_set_shadow_type (frame, GTK_SHADOW_IN);
	
	term->priv->zvt_term = zvt_term_new ();
	zvt_term_set_font_name(ZVT_TERM(term->priv->zvt_term), ZVT_FONT);
	zvt_term_set_blink(ZVT_TERM(term->priv->zvt_term), TRUE);
	zvt_term_set_bell(ZVT_TERM(term->priv->zvt_term), TRUE);
	zvt_term_set_scrollback(ZVT_TERM(term->priv->zvt_term), ZVT_SCROLLSIZE);
	zvt_term_set_scroll_on_keystroke(ZVT_TERM(term->priv->zvt_term), TRUE);
	zvt_term_set_scroll_on_output(ZVT_TERM(term->priv->zvt_term), FALSE);
	zvt_term_set_background(ZVT_TERM(term->priv->zvt_term), NULL, 0, 0);
	zvt_term_set_wordclass(ZVT_TERM(term->priv->zvt_term),
						   (unsigned char*) "-A-Za-z0-9/_:.,?+%=");
	zvt_term_set_size (ZVT_TERM (term->priv->zvt_term),
					   -1, gnome_cmd_data_get_embterm_height ());
	gtk_widget_ref (term->priv->zvt_term);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "zvt_term", term->priv->zvt_term,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (term->priv->zvt_term);
	gtk_container_add (GTK_CONTAINER(frame), term->priv->zvt_term);

	/* Setup color scheme
	 *
	 */
	update_color_scheme ();
	zvt_term_set_color_scheme (ZVT_TERM (term->priv->zvt_term),
							   linux_red, linux_grn, linux_blu);
	
	gtk_signal_connect(
		GTK_OBJECT(term->priv->zvt_term), "button_press_event",
		GTK_SIGNAL_FUNC(zvterm_mouse_clicked), term);
	gtk_signal_connect (
		GTK_OBJECT(term->priv->zvt_term), "child_died",
		GTK_SIGNAL_FUNC (zvterm_reinit_child), term);
	gtk_signal_connect (
		GTK_OBJECT (term->priv->zvt_term),"destroy",
		GTK_SIGNAL_FUNC (zvterm_terminate), term);
	gtk_signal_connect (
		GTK_OBJECT (term->priv->zvt_term), "focus_in_event",
		GTK_SIGNAL_FUNC (zvterm_focus_in), term);
}



/***********************************
 * Public functions
 ***********************************/

GtkType
gnome_cmd_zvt_term_get_type         (void)
{
	static GtkType type = 0;

	if (type == 0)
	{
		GtkTypeInfo info =
		{
			"GnomeCmdZvtTerm",
			sizeof (GnomeCmdZvtTerm),
			sizeof (GnomeCmdZvtTermClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gtk_frame_get_type (), &info);
	}
	return type;
}



GtkWidget *
gnome_cmd_zvt_term_new ()
{
	return GTK_WIDGET (gtk_type_new (gnome_cmd_zvt_term_get_type ()));
}


void
gnome_cmd_zvt_term_update (GnomeCmdZvtTerm *term)
{
	g_return_if_fail (GNOME_CMD_IS_ZVT_TERM (term));
	g_return_if_fail (term->priv != NULL);
	
	zvt_term_set_size (ZVT_TERM (term->priv->zvt_term),
					   -1, gnome_cmd_data_get_embterm_height ());

	update_color_scheme ();
	zvt_term_set_color_scheme (ZVT_TERM (term->priv->zvt_term),
							   linux_red, linux_grn, linux_blu);
}


void
gnome_cmd_zvt_term_set_focus (GnomeCmdZvtTerm *term)
{
	g_return_if_fail (GNOME_CMD_IS_ZVT_TERM (term));
	g_return_if_fail (term->priv != NULL);

	gtk_widget_grab_focus (GTK_WIDGET (term->priv->zvt_term));
}

