/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-device.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gdm-device.h"

struct _GdmDevicePrivate
{
        LibHalContext *hal_ctx;
        LibHalPropertySet *properties;
        char *udi;
        char *parent_udi;
};

enum {
        HAL_PROPERTY_CHANGED,
        LAST_SIGNAL
};

static GObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (GdmDevice, gdm_device, G_TYPE_OBJECT);

static void
gdm_device_finalize (GdmDevice *device)
{
        if (device->priv->properties != NULL)
                libhal_free_property_set (device->priv->properties);
        g_free (device->priv->udi);
        g_free (device->priv->parent_udi);
        device->priv->hal_ctx = NULL;
        device->priv->properties = NULL;
        device->priv->udi = NULL;
        device->priv->parent_udi = NULL;

        if (G_OBJECT_CLASS (parent_class)->finalize)
                (* G_OBJECT_CLASS (parent_class)->finalize) (G_OBJECT (device));
}

static void
gdm_device_class_init (GdmDeviceClass *klass)
{
        GObjectClass *obj_class = (GObjectClass *) klass;

        parent_class = g_type_class_peek_parent (klass);

        obj_class->finalize = (GObjectFinalizeFunc) gdm_device_finalize;

        signals[HAL_PROPERTY_CHANGED] =
                g_signal_new ("hal_property_changed",
                              G_TYPE_FROM_CLASS (klass),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmDeviceClass, hal_property_changed),
                              NULL, NULL,
                              g_cclosure_marshal_VOID__STRING,
                              G_TYPE_NONE, 1,
                              G_TYPE_STRING);
}

static void
gdm_device_init (GdmDevice *device)
{
        device->priv = g_new0 (GdmDevicePrivate, 1);
}

const char *
gdm_device_get_property_string (GdmDevice *device, const char *key)
{
        const char *ret;
        LibHalPropertySetIterator it;

        ret = NULL;
        if (device->priv->properties == NULL)
                goto out;

        libhal_psi_init (&it, device->priv->properties);
        while (libhal_psi_has_more (&it)) {
                if (libhal_psi_get_type (&it) == LIBHAL_PROPERTY_TYPE_STRING) {
                        char *pkey;
                        pkey = libhal_psi_get_key (&it);
                        if (pkey != NULL && g_ascii_strcasecmp (pkey, key) == 0) {
                                ret = libhal_psi_get_string (&it);
                                break;
                        }
                }
                libhal_psi_next (&it);
        }

out:
        return ret;
}

int
gdm_device_get_property_int (GdmDevice *device, const char *key)
{
        int ret;
        LibHalPropertySetIterator it;

        ret = -1;
        if (device->priv->properties == NULL)
                goto out;

        libhal_psi_init (&it, device->priv->properties);
        while (libhal_psi_has_more (&it)) {
                if (libhal_psi_get_type (&it) == LIBHAL_PROPERTY_TYPE_INT32) {
                        char *pkey;
                        pkey = libhal_psi_get_key (&it);
                        if (pkey != NULL && g_ascii_strcasecmp (pkey, key) == 0) {
                                ret = libhal_psi_get_int (&it);
                                break;
                        }
                }
                libhal_psi_next (&it);
        }

out:
        return ret;
}

double
gdm_device_get_property_double (GdmDevice *device, const char *key)
{
        double ret;
        LibHalPropertySetIterator it;

        ret = -1.0;
        if (device->priv->properties == NULL)
                goto out;

        libhal_psi_init (&it, device->priv->properties);
        while (libhal_psi_has_more (&it)) {
                if (libhal_psi_get_type (&it) == LIBHAL_PROPERTY_TYPE_DOUBLE) {
                        char *pkey;
                        pkey = libhal_psi_get_key (&it);
                        if (pkey != NULL && g_ascii_strcasecmp (pkey, key) == 0) {
                                ret = libhal_psi_get_double (&it);
                                break;
                        }
                }
                libhal_psi_next (&it);
        }

out:
        return ret;
}

guint64
gdm_device_get_property_uint64 (GdmDevice *device, const char *key)
{
        guint64 ret;
        LibHalPropertySetIterator it;

        ret = -1;
        if (device->priv->properties == NULL)
                goto out;

        libhal_psi_init (&it, device->priv->properties);
        while (libhal_psi_has_more (&it)) {
                if (libhal_psi_get_type (&it) == LIBHAL_PROPERTY_TYPE_UINT64) {
                        char *pkey;
                        pkey = libhal_psi_get_key (&it);
                        if (pkey != NULL && g_ascii_strcasecmp (pkey, key) == 0) {
                                ret = libhal_psi_get_uint64 (&it);
                                break;
                        }
                }
                libhal_psi_next (&it);
        }

out:
        return ret;
}

gboolean
gdm_device_get_property_bool (GdmDevice *device, const char *key)
{
        gboolean ret;
        LibHalPropertySetIterator it;

        ret = FALSE;
        if (device->priv->properties == NULL)
                goto out;

        libhal_psi_init (&it, device->priv->properties);
        while (libhal_psi_has_more (&it)) {
                if (libhal_psi_get_type (&it) == LIBHAL_PROPERTY_TYPE_BOOLEAN) {
                        char *pkey;
                        pkey = libhal_psi_get_key (&it);
                        if (pkey != NULL && g_ascii_strcasecmp (pkey, key) == 0) {
                                ret = libhal_psi_get_bool (&it);
                                break;
                        }
                }
                libhal_psi_next (&it);
        }

out:
        return ret;
}

char **
gdm_device_get_property_strlist (GdmDevice *device, const char *key)
{
        char **ret;
        LibHalPropertySetIterator it;

        ret = NULL;
        if (device->priv->properties == NULL)
                goto out;

        libhal_psi_init (&it, device->priv->properties);
        while (libhal_psi_has_more (&it)) {
                if (libhal_psi_get_type (&it) == LIBHAL_PROPERTY_TYPE_STRLIST) {
                        char *pkey;
                        pkey = libhal_psi_get_key (&it);
                        if (pkey != NULL && g_ascii_strcasecmp (pkey, key) == 0) {
                                ret = libhal_psi_get_strlist (&it);
                                break;
                        }
                }
                libhal_psi_next (&it);
        }

out:
        return ret;
}

gboolean
gdm_device_test_capability (GdmDevice *device, const char *capability)
{
        int n;
        char **caps;
        gboolean ret;

        ret = FALSE;
        caps = gdm_device_get_property_strlist (device, "info.capabilities");
        if (caps == NULL)
                goto out;
        for (n = 0; caps[n] != NULL; n++) {
                if (g_ascii_strcasecmp (caps[n], capability) == 0) {
                        ret = TRUE;
                        break;
                }
        }

out:
        return ret;
}

GdmDevice *
gdm_device_new_from_udi (LibHalContext *hal_ctx, const char *udi)
{
        GdmDevice *device;
        const char *parent_udi;

        device = GDM_DEVICE (g_object_new (GDM_TYPE_DEVICE, NULL));
        device->priv->udi = g_strdup (udi);
        device->priv->hal_ctx = hal_ctx;
        device->priv->properties = libhal_device_get_all_properties (hal_ctx, udi, NULL);
        if ((parent_udi = gdm_device_get_property_string (device, "info.parent")) != NULL) {
                device->priv->parent_udi = g_strdup (parent_udi);
        }
        return device;
}

void
gdm_device_hal_property_changed (GdmDevice *device, const char *key)
{
        libhal_free_property_set (device->priv->properties);
        device->priv->properties = libhal_device_get_all_properties (device->priv->hal_ctx, device->priv->udi, NULL);
        g_signal_emit (device, signals[HAL_PROPERTY_CHANGED], 0, key);
}

const char *
gdm_device_get_udi (GdmDevice *device)
{
        return device->priv->udi;
}

const char *
gdm_device_get_parent_udi (GdmDevice *device)
{
        return device->priv->parent_udi;
}

LibHalPropertySet *
gdm_device_get_properties (GdmDevice *device)
{
        return device->priv->properties;
}
