/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* chaninfo-list.cpp
 *
 * Copyright (C) 2005 Takuo KITAME.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * Author: Takuo KITAME.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdarg.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>

#include "peercast.h"

#include "utils.h"
#include "callbacks.h"
#include "chaninfo-dialog.h"
#include "channel-list.h"


#define DELAY_UPDATE 1000 /* 1 second */

enum
{
	CHANNEL_ID,  /* Channel ID */
	CHANNEL_KEEP, /* KEEP */
	CHANNEL_NAME,  /* Channel Name */
	CHANNEL_GENRE,  /* Channel Genre */
	CHANNEL_DESC,  /* Channel Description */
	CHANNEL_TYPE,  /* Channel Type */
	CHANNEL_BITRATE,  /* Channel Bitrate */
	CHANNEL_RELAYS,  /* Total Relays */
	CHANNEL_LOCAL,  /* Local Relays */
	CHANNEL_STATUS,  /* Channel STATUS */
	N_COLUMNS
};

static void
keep_toggled_cb (GtkCellRendererToggle *cell,
		 gchar *path_str,
		 gpointer data)
{
	GtkTreeModel *model = GTK_TREE_MODEL (data);
	GtkTreeIter iter;
	GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
	gchar *idStr;
	GnuID *gid;
	Channel *ch;
	gboolean keep;

	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter,
			    CHANNEL_ID, &idStr,
			    CHANNEL_KEEP, &keep,
			    -1);
	keep ^= 1;

	if (idStr) {
		gid = new GnuID;
		gid->fromStr((const gchar *)idStr);
		ch = chanMgr->findChannelByID (*gid);
		delete gid;
		if (ch)
			ch->stayConnected = keep;
	}
	gtk_list_store_set (GTK_LIST_STORE (model),
			    &iter, CHANNEL_KEEP, keep, -1);

	/* clean up */
	gtk_tree_path_free (path);
}

static void
channel_list_do_popup (ChannelList *cl, Channel *ch, gint time)
{
	GtkWidget *menu;
	GtkWidget *item;
	GtkTooltips *tips;
	gchar *tmp;
	gchar id[64];

	if (!ch) return;

	menu = cl->getMenuItem ("channel_popup_menu");

	/* Go to URL */
	item = cl->getMenuItem ("go_to_url");
	gtk_widget_set_sensitive (item, ! ch->info.url.isEmpty ());
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, item, ch->info.url.cstr (), NULL);

	item = cl->getMenuItem ("play_pls_url");
	tmp = gen_local_url (ch->info, TRUE);
	gtk_tooltips_set_tip (tips, item, tmp, NULL);
	g_free (tmp);

	item = cl->getMenuItem ("play_stream_url");
	tmp = gen_local_url (ch->info, FALSE);
	gtk_tooltips_set_tip (tips, item, tmp, NULL);
	gtk_widget_set_sensitive (item, ch->info.contentType!=ChanInfo::T_UNKNOWN);
	g_free (tmp);

	item = cl->getMenuItem ("copy_peercast_url");
	tmp = gen_peercast_url (ch->info);
	gtk_tooltips_set_tip (tips, item, tmp, NULL);
	g_free (tmp);

	item = cl->getMenuItem ("copy_local_stream_url");
	tmp = gen_local_url (ch->info, FALSE);
	gtk_tooltips_set_tip (tips, item, tmp, NULL);
	gtk_widget_set_sensitive (item, ch->info.contentType!=ChanInfo::T_UNKNOWN);
	g_free (tmp);

	item = cl->getMenuItem ("copy_global_stream_url");
	tmp = gen_global_url (ch->info, FALSE);
	gtk_tooltips_set_tip (tips, item, tmp, NULL);
	gtk_widget_set_sensitive (item, ch->info.contentType!=ChanInfo::T_UNKNOWN);
	g_free (tmp);

	item = cl->getMenuItem ("copy_channel_id");
	ch->info.id.toStr(id);
	gtk_tooltips_set_tip (tips, item, id, NULL);

//	g_object_set_data (G_OBJECT (menu), "channel", ch);
	gtk_menu_popup (GTK_MENU (menu),
			NULL, NULL,
			NULL, NULL,
			3, /* Right */
			time);
}

static gboolean
button_press_cb (GtkWidget *w, GdkEventButton *event, ChannelList *cl)
{
	GtkTreePath *path;
	GtkTreeSelection *sel;
	Channel *channel;

	if (! event) return FALSE;

	if (!gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (w),
					    (gint)event->x, (gint)event->y,
					    &path, NULL, NULL, NULL))
                return FALSE;

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (w));
	gtk_tree_selection_select_path (sel, path);
	gtk_tree_path_free (path);

	channel = cl->selectedChannel ();

	if (!channel) return FALSE;

	if (event->button == 3)
		channel_list_do_popup (cl, channel, event->time);

	return FALSE;
}

static gboolean
popup_menu_cb (GtkWidget *w, ChannelList *cl)
{

	channel_list_do_popup (cl, cl->selectedChannel (),
			       gtk_get_current_event_time ());

	return TRUE;
}

static GtkTreeModel *
create_model (void)
{
	GtkListStore * store;

	store = gtk_list_store_new (N_COLUMNS,
				    G_TYPE_STRING, /* ID */
				    G_TYPE_BOOLEAN, /* keep */
				    G_TYPE_STRING, /* NAME */
				    G_TYPE_STRING, /* GENRE */
				    G_TYPE_STRING, /* DESC */
				    G_TYPE_STRING, /* TYPE */
				    G_TYPE_UINT,   /* RATE */
				    G_TYPE_STRING, /* RELAYS */
				    G_TYPE_STRING, /* LOCAL */
				    G_TYPE_STRING  /* STATUS */
		);
	return GTK_TREE_MODEL (store);
}


static void
add_columns (GtkTreeView *treeview)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkTreeModel *model = gtk_tree_view_get_model (treeview);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("ID"), renderer,
							"text", CHANNEL_ID,
							NULL);
	gtk_tree_view_column_set_visible(col, false);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_toggle_new ();
	g_signal_connect (renderer, "toggled",
			  G_CALLBACK (keep_toggled_cb), model);
	col = gtk_tree_view_column_new_with_attributes (_("Keep"), renderer,
							"active", CHANNEL_KEEP,
							NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Name"), renderer,
							"text", CHANNEL_NAME,
							NULL);
	gtk_tree_view_column_set_resizable (col, TRUE);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Genre"), renderer,
							"text", CHANNEL_GENRE,
							NULL);
	gtk_tree_view_column_set_resizable (col, TRUE);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Description"),renderer,
							"text", CHANNEL_DESC,
							NULL);
	gtk_tree_view_column_set_resizable (col, TRUE);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Type"), renderer,
							"text", CHANNEL_TYPE,
							NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Bitrate"), renderer,
							"text", CHANNEL_BITRATE,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Relays"), renderer,
							"text", CHANNEL_RELAYS,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Local Relays"), renderer,
							"text", CHANNEL_LOCAL,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Status"), renderer,
							"text", CHANNEL_STATUS,
							NULL);
	gtk_tree_view_append_column (treeview, col);
}

static gboolean
update_iter_info (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
		  gpointer data)
{
	guint *idStr;
	Channel *ch;
	ChanInfo info;
	gchar ti[1024]; /* track info */
	gchar tr[16]; /* total relays */
	gchar lr[16]; /* local relays */
	GnuID *gid;

	GtkListStore *store = GTK_LIST_STORE (model);

	gtk_tree_model_get (model, iter,
			    CHANNEL_ID, &idStr, -1);
	if (idStr) {
		gid = new GnuID;
		gid->fromStr((const gchar *)idStr);
		ch = chanMgr->findChannelByID (*gid);
		delete gid;
		if (!ch) {
			gtk_list_store_remove (store, iter);
			return TRUE;
		}
		info = ch->info;

		g_snprintf (ti, sizeof(ti), "%s - %s",
			  info.track.title.cstr(), info.track.artist.cstr());
		g_snprintf (tr, sizeof(tr), "%d / %d",
			  ch->totalListeners(), ch->totalRelays());
		g_snprintf (lr, sizeof(tr), "%d / %d",
			  ch->localListeners(), ch->localRelays());
		gtk_list_store_set (store, iter,
				    CHANNEL_ID, idStr,
				    CHANNEL_KEEP, ch->stayConnected,
				    CHANNEL_NAME, info.name.cstr(),
				    CHANNEL_GENRE, info.genre.cstr(),
				    CHANNEL_DESC, info.desc.cstr(),
				    CHANNEL_TYPE, ChanInfo::getTypeStr(info.contentType),
				    CHANNEL_BITRATE, info.bitrate,
				    CHANNEL_RELAYS,  tr,
				    CHANNEL_LOCAL,  lr,
				    CHANNEL_STATUS, ch->getStatusStr(),
				    -1);
	}
	return FALSE;
}

static void
tree_append_info (GtkTreeView *treeview, ChanInfo *info)
{
	GtkListStore *store;
	GtkTreeIter iter;
	gchar ti[1024], tr[16], lr[16];
	Channel *chan;
	gchar id[64];

	info->id.toStr(id);
	store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));

	chan = chanMgr->findChannelByNameID (*info);

	snprintf(ti, sizeof(ti), "%s - %s",
		 info->track.artist.cstr(), info->track.title.cstr());
	snprintf(tr, sizeof(tr), "%d / %d",
		 chan->totalListeners(), chan->totalRelays());
	snprintf(lr, sizeof(tr), "%d / %d",
		 chan->localListeners(), chan->localRelays());

	gtk_list_store_append (store, &iter);

	gtk_list_store_set (store, &iter,
			    CHANNEL_ID, id,
			    CHANNEL_KEEP, chan->stayConnected,
			    CHANNEL_NAME, info->name.cstr(),
			    CHANNEL_GENRE, info->genre.cstr(),
			    CHANNEL_DESC, info->desc.cstr(),
			    CHANNEL_TYPE, ChanInfo::getTypeStr(info->contentType),
			    CHANNEL_BITRATE, info->bitrate,
			    CHANNEL_RELAYS,  tr,
			    CHANNEL_LOCAL,  lr,
			    CHANNEL_STATUS, chan->getStatusStr(),
			    -1);
}

static gboolean
cl_find_row_by_id (GtkTreeView *tv, GtkTreeIter *iter, const gchar *id)
{
	GtkTreeModel *model;
	gchar *idStr;

	model = gtk_tree_view_get_model (tv);
	if (gtk_tree_model_get_iter_first (model, iter)) {
		do {
			gtk_tree_model_get (model, iter,
					    CHANNEL_ID, &idStr, -1);
			if (strcmp(id, idStr) == 0) {
				return TRUE;
			}
		} while (gtk_tree_model_iter_next (model, iter));
	}

	return FALSE; /* not found */
}

static gboolean
update_channel_list (gpointer data)
{
	GtkTreeView *treeview = GTK_TREE_VIEW (data);
	GtkTreeModel *model;
	GtkTreeIter iter;
	Channel *ch;

	ch = chanMgr->channel;
	while (ch) {
		gchar idStr[64];
		ch->info.id.toStr(idStr);
		if (!ch->isActive()) {
			ch = ch->next;
			continue;
		}
		if (! cl_find_row_by_id (treeview, &iter, idStr)) {
			tree_append_info (treeview, &ch->info);
		}
		ch = ch->next;
	}

	model = gtk_tree_view_get_model (treeview);
	gtk_tree_model_foreach (model, update_iter_info, NULL);

	return TRUE;
}


/* static callback handler */
void
ChannelList::play_cb (GtkWidget *w, ChannelList *cl)
{
	Channel *ch;
	ch = cl->selectedChannel ();
	if (ch) play_channel (ch);
}

void
ChannelList::info_cb (GtkWidget *w, ChannelList *cl)
{
	cl->infoSelected ();
}

void
ChannelList::url_cb (GtkWidget *w, ChannelList *cl)
{
	cl->urlSelected ();
}

void
ChannelList::bump_cb (GtkWidget *w, ChannelList *cl)
{
	cl->bumpSelected ();
}

void
ChannelList::stop_cb (GtkWidget *w, ChannelList *cl)
{
	cl->stopSelected ();
}

/* public */

/*
 * Constructor.
 */
ChannelList::ChannelList (GtkWidget *w, GtkWindow *window)
{
	static gint timeout_source = 1;
	GtkTreeModel *model = create_model ();

	debug_print (1, "New object: ChannelList");

	parent = window;
	treeview = GTK_TREE_VIEW (w);
	selection = gtk_tree_view_get_selection (treeview);

	gtk_tree_view_set_model (treeview, model);

	add_columns (treeview);

	popup_xml = load_glade_xml ("gnome-peercast.glade", "channel_popup_menu");
	g_signal_connect (G_OBJECT (treeview), "popup_menu",
			  G_CALLBACK (popup_menu_cb), this);
	g_signal_connect (G_OBJECT (treeview), "button_press_event",
			  G_CALLBACK (button_press_cb), this);

	/* popup menu handler */
	popupSignalConnect ("go_to_url_activate",
			    G_CALLBACK (&ChannelList::url_cb), this);
	popupSignalConnect ("properties_activate",
			    G_CALLBACK (&ChannelList::info_cb), this);
	popupSignalConnect ("bump_activate",
			    G_CALLBACK (&ChannelList::bump_cb), this);
	popupSignalConnect ("stop_activate",
			    G_CALLBACK (&ChannelList::stop_cb), this);

	popupSignalConnect ("play_pls_url_activate",
			    G_CALLBACK(&ChannelList::play_cb), this);
	popupSignalConnect ("play_stream_url_activate",
			    G_CALLBACK(chan_play_stream_url_cb), this);

	popupSignalConnect ("copy_peercast_url_activate",
			    G_CALLBACK(chan_copy_peercast_url_cb), this);
	popupSignalConnect ("copy_local_stream_url_activate",
			    G_CALLBACK(chan_copy_local_url_cb), this);
	popupSignalConnect ("copy_global_stream_url_activate",
			    G_CALLBACK(chan_copy_global_url_cb), this);
	popupSignalConnect ("copy_channel_id_activate",
			    G_CALLBACK(chan_copy_channel_id_cb), this);


	timeout_source = g_timeout_add (DELAY_UPDATE,
					update_channel_list, treeview);
}

Channel *
ChannelList::selectedChannel (void)
{
	Channel *channel = NULL;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *idStr;
	GnuID *gid;

	if (! gtk_tree_selection_get_selected (selection, &model, &iter))
		return NULL;

	gtk_tree_model_get (model, &iter,
			    CHANNEL_ID, &idStr,
			    -1);
	if (idStr) {
		gid = new GnuID;
		gid->fromStr((const gchar *)idStr);
		channel = chanMgr->findChannelByID (*gid);
		delete gid;
	}
	return channel;
}

void
ChannelList::setInfo (ChanInfo *info)
{
	GtkListStore *store;
	GtkTreeIter iter;
	gchar ti[1024], tr[16], lr[16];
	Channel *chan;
	gchar id[64];

	info->id.toStr(id);
	store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));

	chan = chanMgr->findChannelByNameID (*info);

	g_snprintf (ti, sizeof(ti), "%s - %s",
		    info->track.artist.cstr(), info->track.title.cstr());
	g_snprintf (tr, sizeof(tr), "%d / %d",
		    chan->totalListeners(), chan->totalRelays());
	g_snprintf (lr, sizeof(tr), "%d / %d",
		    chan->localListeners(), chan->localRelays());

	if (! cl_find_row_by_id (treeview, &iter, id))
		gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CHANNEL_ID, id,
			    CHANNEL_KEEP, chan->stayConnected,
			    CHANNEL_NAME, info->name.cstr(),
			    CHANNEL_GENRE, info->genre.cstr(),
			    CHANNEL_DESC, info->desc.cstr(),
			    CHANNEL_TYPE, ChanInfo::getTypeStr(info->contentType),
			    CHANNEL_BITRATE, info->bitrate,
			    CHANNEL_RELAYS,  tr,
			    CHANNEL_LOCAL,  lr,
			    CHANNEL_STATUS, chan->getStatusStr(),
			    -1);
}

void
ChannelList::deleteInfo (ChanInfo *info)
{
	GtkListStore *store;
	GtkTreeIter iter;
	gchar id[64];

	info->id.toStr(id);

	store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));
	if (! cl_find_row_by_id (treeview, &iter, id)) {
		/* not found */
		/* FIXME: already removed? */
		return;
	}
	gtk_list_store_remove (store, &iter);
}

void
ChannelList::playSelected (gboolean pls)
{
	Channel *ch;
	gchar *url;
	GError *e = NULL;

	ch = selectedChannel ();

	if (! ch) return;

	url = gen_local_url (ch->info, pls);
	gnome_url_show (url, &e);
	g_free (url);
}

void
ChannelList::infoSelected (void)
{
	ChaninfoDialog *cd;

	Channel *ch = selectedChannel ();

	if (!ch) return;

	cd = new ChaninfoDialog (ch, parent);
}

void
ChannelList::urlSelected (void)
{
	Channel *ch;
	GError *e = NULL;

	ch = selectedChannel();

	if (ch && ! ch->info.url.isEmpty ())
		gnome_url_show (ch->info.url.cstr(), &e);
}

void
ChannelList::bumpSelected (void)
{
	Channel *ch;

	ch = selectedChannel ();

	if (ch) ch->bump = true;
}

void
ChannelList::stopSelected (void)
{
	Channel *ch;

	ch = selectedChannel ();

	if (ch) ch->thread.active = false;
}

Channel *
ChannelList::getPopupChannel (void)
{
	GtkWidget *menu;
	menu = getMenuItem ("channel_popup_menu");
	return (Channel *)g_object_get_data (G_OBJECT (menu), "channel");
}
