/* GnomerMind - A MasterMind(R)-based game for Gnome
 * (C) 2001 Germano Rizzo
 *
 * save.c - routines to control the config file
 * Author: Germano Rizzo
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include "gnomermind.h"
#include "face.h"
#include "prefs.h"
#include "keyboard.h"
#include "themes.h"
#include "sounds.h"

/*
 * the default key bindings
 */

static const guint dkey[] = {
  49, 50, 51, 52, 53, 54, 55, 56, 57, 48,
  113, 119, 101, 114, 116, 121,
  32
};

/**
 * encode
 *
 * `Encrypts' (poorly) the field to guess
 **/
static inline gchar *
encode ()
{
  guint nfield = 0;
  gint i;

  /*
   * just treats the field as it were a 10-digits decimal,
   * and converts it to hexadecimals...
   */

  for (i = 0; i < holez; i++)
    nfield = (nfield + field[i] - 1) * MAXC;

  return g_strdup_printf ("0x%X", nfield);
}

/**
 * decode
 *
 * `Decrypts' the field to guess
 **/
static inline void
decode (gchar * hfield)
{
  glong nfield;
  gint i;

  nfield = strtol (hfield, NULL, 16) / MAXC;

  for (i = holez - 1; i >= 0; i--)
    {
      field[i] = nfield % MAXC + 1;
      nfield = (nfield - field[i] + 1) / MAXC;
    }
}

/**
 * allow_rep_check
 *
 * Checks if the repetitions parameters can be unset
 **/
static inline void
allow_rep_check ()
{
  if (!rep && (holez > colorz))
    {
      g_message ("\n%s\n%s",
		 _
		 ("You can't disable repetitions if the number of tokens to guess is greater than the number of different tokens!"),
		 _("Turning repetitions on..."));
      rep = TRUE;
    }
}

/**
 * current_save
 *
 * Saves the current game
 **/
void
current_save ()
{
  gint i, j;

  g_assert (rep || (holez <= colorz));

  gnome_config_set_bool ("/gnomermind/Suspended/Resume", TRUE);

  gnome_config_set_int ("/gnomermind/Suspended/Holes", holez);
  gnome_config_set_int ("/gnomermind/Suspended/Colors", colorz);
  gnome_config_set_int ("/gnomermind/Suspended/Rounds", roundz);
  gnome_config_set_bool ("/gnomermind/Suspended/Infinite", inf);
  gnome_config_set_bool ("/gnomermind/Suspended/Repetitions", rep);
  gnome_config_set_bool ("/gnomermind/Suspended/Blank_places", bla);
  gnome_config_set_int ("/gnomermind/Suspended/State", state);
  gnome_config_set_int ("/gnomermind/Suspended/Row", act);
  gnome_config_set_int ("/gnomermind/Suspended/Color_sel", colsel);
  if (!strcmp (theme_path, THEMESPATH))
    gnome_config_set_string ("/gnomermind/Suspended/Theme", theme);
  gnome_config_set_bool ("/gnomermind/Suspended/Confirmations", ask);
  gnome_config_set_bool ("/gnomermind/Suspended/Drop", drop);
#ifdef HAVE_ESD
  gnome_config_set_bool ("/gnomermind/Suspended/Sound", sound_enable);
#endif
  for (i = 0; i < KEY; i++)
    gnome_config_set_float (g_strdup_printf
			    ("/gnomermind/Suspended/Key%d", i),
			    (gdouble) key[i]);

  gnome_config_set_string ("/gnomermind/Suspended/Field", encode ());

  for (i = 0; i < MAXH; i++)
    for (j = 0; j < MAXR; j++)
      gnome_config_set_int (g_strdup_printf
			    ("/gnomermind/Suspended/Try%d-%d", j, i),
			    try[j][i]);

  for (i = 0; i < MAXR; i++)
    gnome_config_set_int (g_strdup_printf
			  ("/gnomermind/Suspended/Black%d", i), b[i]);

  for (i = 0; i < MAXR; i++)
    gnome_config_set_int (g_strdup_printf
			  ("/gnomermind/Suspended/White%d", i), w[i]);

  gnome_config_sync ();
}

/**
 * current_load
 *
 * Loads a saved game, returns FALSE if it hasn't to be restored
 **/
static gboolean
current_load ()
{
  if (gnome_config_get_bool ("/gnomermind/Suspended/Resume=FALSE"))
    {
      gint i, j;

      holez =
	gnome_config_get_int (g_strdup_printf
			      ("/gnomermind/Suspended/Holes=%d", DEFH));
      colorz =
	gnome_config_get_int (g_strdup_printf
			      ("/gnomermind/Suspended/Colors=%d", DEFC));
      roundz =
	gnome_config_get_int (g_strdup_printf
			      ("/gnomermind/Suspended/Rounds=%d", DEFR));
      inf = gnome_config_get_bool ("/gnomermind/Suspended/Infinite=FALSE");
      rep = gnome_config_get_bool ("/gnomermind/Suspended/Repetitions=TRUE");
      allow_rep_check ();

      bla =
	gnome_config_get_bool ("/gnomermind/Suspended/Blank_places=FALSE");

      state = gnome_config_get_int ("/gnomermind/Suspended/State=0");
      act = gnome_config_get_int ("/gnomermind/Suspended/Row=0");
      colsel = gnome_config_get_int ("/gnomermind/Suspended/Color_sel=0");
      theme = gnome_config_get_string ("/gnomermind/Suspended/Theme=Classic");
      if (!strcmp (theme, "classic"))
	{
	  theme = "Classic";
	  gnome_config_set_string ("/gnomermind/Suspended/Theme", "Classic");
	  gnome_config_sync ();
	}
      else if (!validate (theme))
	theme = "Classic";
      newtheme = theme;
      ask =
	gnome_config_get_bool ("/gnomermind/Suspended/Confirmations=TRUE");
      drop = gnome_config_get_bool ("/gnomermind/Suspended/Drop=FALSE");
#ifdef HAVE_ESD
      sound_enable =
	gnome_config_get_bool ("/gnomermind/Suspended/Sound=TRUE");
#endif

      for (i = 0; i < KEY; i++)
	key[i] =
	  (guint)
	  gnome_config_get_float (g_strdup_printf
				  ("/gnomermind/Suspended/Key%d=%u", i,
				   dkey[i]));

      decode (gnome_config_get_string ("/gnomermind/Suspended/Field=0x0"));

      for (i = 0; i < MAXH; i++)
	for (j = 0; j < MAXR; j++)
	  try[j][i] =
	    gnome_config_get_int (g_strdup_printf
				  ("/gnomermind/Suspended/Try%d-%d=0", j, i));

      for (i = 0; i < MAXR; i++)
	{
	  b[i] =
	    gnome_config_get_int (g_strdup_printf
				  ("/gnomermind/Suspended/Black%d=0", i));
	  w[i] =
	    gnome_config_get_int (g_strdup_printf
				  ("/gnomermind/Suspended/White%d=0", i));
	}

      gnome_config_set_bool ("/gnomermind/Suspended/Resume", FALSE);
      gnome_config_sync ();

      return TRUE;
    }

  return FALSE;
}

/**
 * save_and_exit
 *
 * Saves and exits, of course! ;)
 **/
inline void
save_and_exit ()
{
  current_save ();
  gtk_main_quit ();
}

/**
 * normalize_popt_int
 * @value: the value to set
 * @lowbound: the lower bound
 * @highbound: the higher bound
 *
 * If the value is different from 0, clamps it between the bounds
 **/
inline gint
normalize_popt_int (gint value, gint lowbound, gint highbound)
{
  if (value == 0)
    return 0;
  if (value < lowbound)
    return lowbound;
  if (value > highbound)
    return highbound;
  return value;
}

/**
 * validate_popt_string
 * @string: the string to parse for truth
 *
 * Validates a boolean options string
 **/
inline gboolean
validate_popt_string (gchar * string)
{
  return (!
	  ((g_strcasecmp (string, "y")) && (g_strcasecmp (string, "yes"))
	   && (g_strcasecmp (string, "n")) && (g_strcasecmp (string, "no"))));
}

/**
 * parse_popt_string
 * @string: the string to parse for truth
 *
 * Parses a boolean options string
 **/
static inline gboolean
parse_popt_string (gchar * string)
{
  return (!((g_strcasecmp (string, "y")) && (g_strcasecmp (string, "yes"))));
}

/**
 * parameter_load
 *
 * Loads the saved game or the defaults, returns TRUE if defaults
 **/
gboolean
parameter_load_with_commandline (gchar * inn, gchar * ree, gchar * bll,
				 gboolean over, gchar * drr, gchar * ass)
{
  gboolean defaults = FALSE;

  if (!over)
    defaults = current_load ();

  if (!defaults)
    {
      gint i;

      if (!holez)
	holez =
	  gnome_config_get_int (g_strdup_printf
				("/gnomermind/Defaults/Holes=%d", DEFH));
      if (!colorz)
	colorz =
	  gnome_config_get_int (g_strdup_printf
				("/gnomermind/Defaults/Colors=%d", DEFC));
      if (!roundz)
	roundz =
	  gnome_config_get_int (g_strdup_printf
				("/gnomermind/Defaults/Rounds=%d", DEFR));
      if (!inn)
	inf =
	  gnome_config_get_bool (g_strdup_printf
				 ("/gnomermind/Defaults/Infinite=FALSE"));
      else
	inf = parse_popt_string (inn);

      if (!ree)
	rep =
	  gnome_config_get_bool (g_strdup_printf
				 ("/gnomermind/Defaults/Repetitions=TRUE"));
      else
	rep = parse_popt_string (ree);

      allow_rep_check ();

      if (!bll)
	bla =
	  gnome_config_get_bool (g_strdup_printf
				 ("/gnomermind/Defaults/Blank_places=FALSE"));
      else
	bla = parse_popt_string (bll);

      if (!drr)
	drop = gnome_config_get_bool ("/gnomermind/Defaults/Drop=FALSE");
      else
	drop = parse_popt_string (drr);

      if (!ass)
	ask =
	  gnome_config_get_bool ("/gnomermind/Defaults/Confirmations=TRUE");
      else
	ask = parse_popt_string (ass);

      if (!theme)
	theme =
	  gnome_config_get_string ("/gnomermind/Defaults/Theme=Classic");
      if (!strcmp (theme, "classic"))
	{
	  theme = "Classic";
	  gnome_config_set_string ("/gnomermind/Defaults/Theme", "Classic");
	  gnome_config_sync ();
	}

      newtheme = theme;
#ifdef HAVE_ESD
      sound_enable =
	gnome_config_get_bool ("/gnomermind/Defaults/Sound=TRUE");
#endif
      for (i = 0; i < KEY; i++)
	key[i] =
	  (guint)
	  gnome_config_get_float (g_strdup_printf
				  ("/gnomermind/Defaults/Key%d=%u", i,
				   dkey[i]));

      return TRUE;
    }

  return FALSE;
}

/**
 * save_defaults
 *
 * Saves the defaults
 **/
void
save_defaults ()
{
  gint i;

  g_assert (rep || (holez <= colorz));

  gnome_config_set_int ("/gnomermind/Defaults/Holes", h);
  gnome_config_set_int ("/gnomermind/Defaults/Colors", c);
  gnome_config_set_int ("/gnomermind/Defaults/Rounds", r);
  gnome_config_set_bool ("/gnomermind/Defaults/Infinite", tempinf);
  gnome_config_set_bool ("/gnomermind/Defaults/Repetitions", temprep);
  gnome_config_set_bool ("/gnomermind/Defaults/Blank_places", tempbla);
  gnome_config_set_bool ("/gnomermind/Defaults/Drop", drop);
  gnome_config_set_bool ("/gnomermind/Defaults/Confirmations", ask);
#ifdef HAVE_ESD
  gnome_config_set_bool ("/gnomermind/Defaults/Sound", sound_enable);
#endif
  if (!strcmp (theme_path, THEMESPATH))
    gnome_config_set_string ("/gnomermind/Defaults/Theme", newtheme);
  for (i = 0; i < KEY; i++)
    gnome_config_set_float (g_strdup_printf ("/gnomermind/Defaults/Key%d", i),
			    (gdouble) key[i]);

  gnome_config_sync ();
}
