/* GnoSamba 0.3.3
 * copyright 1998-9 Perry Piplani
 * redistributable under the terms of the GPL:
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "gtksamba.h"

#define MAX_ENUM 10

/*** from smb.h *** defs for parm structure */

typedef enum
{
  P_BOOL,P_BOOLREV,P_CHAR,P_INTEGER,P_OCTAL,
  P_STRING,P_USTRING,P_GSTRING,P_UGSTRING,P_ENUM,P_SEP
} parm_type;

typedef enum
{
  P_LOCAL,P_GLOBAL,P_SEPARATOR,P_NONE
} parm_class;

struct enum_list {
	int value;
	char *name;
};

/* four members snipped */
struct parm_struct
{
	char *label;
	parm_type type;
	parm_class class;
	struct enum_list *enum_list;
	
};

/* value enums for emum list */

/* protocol types. It assumes that higher protocols include lower protocols
   as subsets */
enum protocol_types {PROTOCOL_NONE,PROTOCOL_CORE,PROTOCOL_COREPLUS,PROTOCOL_LANMAN1,PROTOCOL_LANMAN2,PROTOCOL_NT1};

/* security levels */
enum security_types {SEC_SHARE,SEC_USER,SEC_SERVER,SEC_DOMAIN};

/* printing types */
enum printing_types {PRINT_BSD,PRINT_SYSV,PRINT_AIX,PRINT_HPUX,
		     PRINT_QNX,PRINT_PLP,PRINT_LPRNG,PRINT_SOFTQ};

/* case handling */
enum case_handling {CASE_LOWER,CASE_UPPER};

/* Types of machine we can announce as. */
#define ANNOUNCE_AS_NT 1
#define ANNOUNCE_AS_WIN95 2
#define ANNOUNCE_AS_WFW 3



#define NEVER_MAP_TO_GUEST 0
#define MAP_TO_GUEST_ON_BAD_USER 1
#define MAP_TO_GUEST_ON_BAD_PASSWORD 2

#ifdef WITH_SSL
/* SSL version options */
enum ssl_version_enum {SMB_SSL_V2,SMB_SSL_V3,SMB_SSL_V23,SMB_SSL_TLS1};
#endif /* WITH_SSL */


static struct enum_list enum_protocol[] = {{PROTOCOL_NT1, "NT1"}, {PROTOCOL_LANMAN2, "LANMAN2"}, 
					   {PROTOCOL_LANMAN1, "LANMAN1"}, {PROTOCOL_CORE,"CORE"}, 
					   {PROTOCOL_COREPLUS, "COREPLUS"}, 
					   {PROTOCOL_COREPLUS, "CORE+"}, {-1, NULL}};

static struct enum_list enum_security[] = {{SEC_SHARE, "SHARE"},  {SEC_USER, "USER"}, 
					   {SEC_SERVER, "SERVER"}, {SEC_DOMAIN, "DOMAIN"},
                                           {-1, NULL}};

static struct enum_list enum_printing[] = {{PRINT_SYSV, "sysv"}, {PRINT_AIX, "aix"}, 
					   {PRINT_HPUX, "hpux"}, {PRINT_BSD, "bsd"},
					   {PRINT_QNX, "qnx"},   {PRINT_PLP, "plp"},
					   {PRINT_LPRNG, "lprng"}, {PRINT_SOFTQ, "softq"},
					   {-1, NULL}};

static struct enum_list enum_announce_as[] = {{ANNOUNCE_AS_NT, "NT"}, {ANNOUNCE_AS_WIN95, "win95"},
					      {ANNOUNCE_AS_WFW, "WfW"}, {-1, NULL}};

static struct enum_list enum_case[] = {{CASE_LOWER, "lower"}, {CASE_UPPER, "upper"}, {-1, NULL}};

static struct enum_list enum_lm_announce[] = {{0, "False"}, {1, "True"}, {2, "Auto"}, {-1, NULL}};


static struct enum_list enum_map_to_guest[] = {{NEVER_MAP_TO_GUEST, "Never"}, {MAP_TO_GUEST_ON_BAD_USER, "Bad User"}, {MAP_TO_GUEST_ON_BAD_PASSWORD, "Bad Password"}, {-1, NULL}};




#ifdef WITH_SSL
static struct enum_list enum_ssl_version[] = {{SMB_SSL_V2, "ssl2"}, {SMB_SSL_V3, "ssl3"},
  {SMB_SSL_V23, "ssl2or3"}, {SMB_SSL_TLS1, "tls1"}, {-1, NULL}};
#endif

/* param table modified with awk and some hand editing */
static struct parm_struct parm_table[] =
{
  {"Base Options", P_SEP, P_SEPARATOR},
  {"comment",          P_STRING,  P_LOCAL,   NULL},
  {"path",             P_STRING,  P_LOCAL,   NULL},
  {"directory",        P_STRING,  P_LOCAL,   NULL},
  {"workgroup",        P_USTRING, P_GLOBAL,   NULL},
  {"netbios name",     P_UGSTRING,P_GLOBAL,   NULL},
  {"netbios aliases",  P_STRING,  P_GLOBAL,   NULL},
  {"server string",    P_STRING,  P_GLOBAL,   NULL},
  {"interfaces",       P_STRING,  P_GLOBAL,   NULL},
  {"bind interfaces only", P_BOOL,P_GLOBAL,   NULL},

  {"Security Options", P_SEP, P_SEPARATOR},
  {"security",         P_ENUM,    P_GLOBAL,   enum_security},
  {"encrypt passwords",P_BOOL,    P_GLOBAL,   NULL},
  {"update encrypted", P_BOOL,    P_GLOBAL,   NULL},
  {"use rhosts",       P_BOOL,    P_GLOBAL,   NULL},
  {"map to guest",     P_ENUM,    P_GLOBAL,   enum_map_to_guest},
  {"null passwords",   P_BOOL,    P_GLOBAL,   NULL},
  {"password server",  P_STRING,  P_GLOBAL,   NULL},
  {"smb passwd file",  P_STRING,  P_GLOBAL,   NULL},
  {"hosts equiv",      P_STRING,  P_GLOBAL,   NULL},
  {"root directory",   P_STRING,  P_GLOBAL,   NULL},
  {"root dir",         P_STRING,  P_GLOBAL,   NULL},
  {"root",             P_STRING,  P_GLOBAL,   NULL},
  {"passwd program",   P_STRING,  P_GLOBAL,   NULL},
  {"passwd chat",      P_STRING,  P_GLOBAL,   NULL},
  {"passwd chat debug",P_BOOL,    P_GLOBAL,   NULL},
  {"username map",     P_STRING,  P_GLOBAL,   NULL},
  {"password level",   P_INTEGER, P_GLOBAL,   NULL},
  {"username level",   P_INTEGER, P_GLOBAL,   NULL},
  {"unix password sync", P_BOOL,  P_GLOBAL,   NULL},
  {"alternate permissions",P_BOOL,P_LOCAL,   NULL},
  {"revalidate",       P_BOOL,    P_LOCAL,   NULL},
  {"username",         P_STRING,  P_LOCAL,   NULL},
  {"user",             P_STRING,  P_LOCAL,   NULL},
  {"users",            P_STRING,  P_LOCAL,   NULL},
  {"guest account",    P_STRING,  P_LOCAL,   NULL},
  {"invalid users",    P_STRING,  P_LOCAL,   NULL},
  {"valid users",      P_STRING,  P_LOCAL,   NULL},
  {"admin users",      P_STRING,  P_LOCAL,   NULL},
  {"read list",        P_STRING,  P_LOCAL,   NULL},
  {"write list",       P_STRING,  P_LOCAL,   NULL},
  {"force user",       P_STRING,  P_LOCAL,   NULL},
  {"force group",      P_STRING,  P_LOCAL,   NULL},
  {"group",            P_STRING,  P_LOCAL,   NULL},
  {"read only",        P_BOOL,    P_LOCAL,   NULL},
  {"write ok",         P_BOOLREV, P_LOCAL,   NULL},
  {"writeable",        P_BOOLREV, P_LOCAL,   NULL},
  {"writable",         P_BOOLREV, P_LOCAL,   NULL},
  {"create mask",      P_OCTAL,   P_LOCAL,   NULL},
  {"create mode",      P_OCTAL,   P_LOCAL,   NULL},
  {"force create mode",P_OCTAL,   P_LOCAL,   NULL},
  {"directory mask",   P_OCTAL,   P_LOCAL,   NULL},
  {"directory mode",   P_OCTAL,   P_LOCAL,   NULL},
  {"force directory mode",   P_OCTAL,   P_LOCAL,   NULL},
  {"guest only",       P_BOOL,    P_LOCAL,   NULL},
  {"only guest",       P_BOOL,    P_LOCAL,   NULL},
  {"guest ok",         P_BOOL,    P_LOCAL,   NULL},
  {"public",           P_BOOL,    P_LOCAL,   NULL},
  {"only user",        P_BOOL,    P_LOCAL,   NULL},
  {"hosts allow",      P_STRING,  P_LOCAL,   NULL},
  {"allow hosts",      P_STRING,  P_LOCAL,   NULL},
  {"hosts deny",       P_STRING,  P_LOCAL,   NULL},
  {"deny hosts",       P_STRING,  P_LOCAL,   NULL},

#ifdef WITH_SSL
  {"Secure Socket Layer Options", P_SEP, P_SEPARATOR},
  {"ssl",              P_BOOL,    P_GLOBAL,   NULL},
  {"ssl hosts",        P_STRING,  P_GLOBAL,   NULL},
  {"ssl hosts resign", P_STRING,  P_GLOBAL,   NULL},
  {"ssl CA certDir",   P_STRING,  P_GLOBAL,   NULL},
  {"ssl CA certFile",  P_STRING,  P_GLOBAL,   NULL},
  {"ssl server cert",  P_STRING,  P_GLOBAL,   NULL},
  {"ssl server key",   P_STRING,  P_GLOBAL,   NULL},
  {"ssl client cert",  P_STRING,  P_GLOBAL,   NULL},
  {"ssl client key",   P_STRING,  P_GLOBAL,   NULL},
  {"ssl require clientcert",  P_BOOL,  P_GLOBAL,   NULL },
  {"ssl require servercert",  P_BOOL,  P_GLOBAL,   NULL },
  {"ssl ciphers",      P_STRING,  P_GLOBAL,   NULL},
  {"ssl version",      P_ENUM,    P_GLOBAL,   enum_ssl_version},
  {"ssl compatibility", P_BOOL,    P_GLOBAL,   NULL},
#endif

  {"Logging Options", P_SEP, P_SEPARATOR},
  {"log level",        P_INTEGER, P_GLOBAL,   NULL},
  {"debuglevel",       P_INTEGER, P_GLOBAL,   NULL},
  {"syslog",           P_INTEGER, P_GLOBAL,   NULL},
  {"syslog only",      P_BOOL,    P_GLOBAL,   NULL},
  {"log file",         P_STRING,  P_GLOBAL,   NULL},
  {"max log size",     P_INTEGER, P_GLOBAL,   NULL},
  {"timestamp logs",   P_BOOL,    P_GLOBAL,   NULL},
  {"debug timestamp",  P_BOOL,    P_GLOBAL,   NULL},
  {"status",           P_BOOL,    P_LOCAL,   NULL},

  {"Protocol Options", P_SEP, P_SEPARATOR},
  {"protocol",         P_ENUM,    P_GLOBAL,   enum_protocol},
  {"read bmpx",        P_BOOL,    P_GLOBAL,   NULL},
  {"read raw",         P_BOOL,    P_GLOBAL,   NULL},
  {"write raw",        P_BOOL,    P_GLOBAL,   NULL},
  {"nt smb support",   P_BOOL,    P_GLOBAL,   NULL},
  {"nt pipe support",   P_BOOL,    P_GLOBAL,   NULL},
  {"announce version", P_STRING,  P_GLOBAL,   NULL},
  {"announce as",      P_ENUM,    P_GLOBAL,   enum_announce_as},
  {"max mux",          P_INTEGER, P_GLOBAL,   NULL},
  {"max xmit",         P_INTEGER, P_GLOBAL,   NULL},
  {"name resolve order",  P_STRING,  P_GLOBAL,   NULL},
  {"max packet",       P_INTEGER, P_GLOBAL,   NULL},
  {"packet size",      P_INTEGER, P_GLOBAL,   NULL},
  {"max ttl",          P_INTEGER, P_GLOBAL,   NULL},
  {"max wins ttl",     P_INTEGER, P_GLOBAL,   NULL},
  {"min wins ttl",     P_INTEGER, P_GLOBAL,   NULL},
  {"time server",      P_BOOL,    P_GLOBAL,   NULL},

  {"Tuning Options", P_SEP, P_SEPARATOR},
  {"change notify timeout", P_INTEGER, P_GLOBAL,   NULL},
  {"deadtime",         P_INTEGER, P_GLOBAL,   NULL},
  {"getwd cache",      P_BOOL,    P_GLOBAL,   NULL},
  {"keepalive",        P_INTEGER, P_GLOBAL,   NULL},
  {"lpq cache time",   P_INTEGER, P_GLOBAL,   NULL},
  {"max connections",  P_INTEGER, P_LOCAL,   NULL},
  {"max disk size",    P_INTEGER, P_GLOBAL,   NULL},
  {"max open files",   P_INTEGER, P_GLOBAL,   NULL},
  {"min print space",  P_INTEGER, P_LOCAL,   NULL},
  {"read prediction",  P_BOOL,    P_GLOBAL,   NULL},
  {"read size",        P_INTEGER, P_GLOBAL,   NULL},
  {"shared mem size",  P_INTEGER, P_GLOBAL,   NULL},
  {"socket options",   P_GSTRING, P_GLOBAL,   NULL},
  {"stat cache size",  P_INTEGER, P_GLOBAL,   NULL},
  {"strict sync",      P_BOOL,    P_LOCAL,   NULL},
  {"sync always",      P_BOOL,    P_LOCAL,   NULL},

  {"Printing Options", P_SEP, P_SEPARATOR},
  {"load printers",    P_BOOL,    P_GLOBAL,   NULL},
  {"printcap name",    P_STRING,  P_GLOBAL,   NULL},
  {"printcap",         P_STRING,  P_GLOBAL,   NULL},
  {"printer driver file", P_STRING,  P_GLOBAL,   NULL},
  {"print ok",         P_BOOL,    P_LOCAL,   NULL},
  {"printable",        P_BOOL,    P_LOCAL,   NULL},
  {"postscript",       P_BOOL,    P_LOCAL,   NULL},
  {"printing",         P_ENUM,    P_LOCAL,   enum_printing},
  {"print command",    P_STRING,  P_LOCAL,   NULL},
  {"lpq command",      P_STRING,  P_LOCAL,   NULL},
  {"lprm command",     P_STRING,  P_LOCAL,   NULL},
  {"lppause command",  P_STRING,  P_LOCAL,   NULL},
  {"lpresume command", P_STRING,  P_LOCAL,   NULL},
  {"queuepause command", P_STRING, P_LOCAL, NULL},
  {"queueresume command", P_STRING, P_LOCAL, NULL},
  {"printer name",     P_STRING,  P_LOCAL,   NULL},
  {"printer",          P_STRING,  P_LOCAL,   NULL},
  {"printer driver",   P_STRING,  P_LOCAL,   NULL},
  {"printer driver location",   P_STRING,  P_LOCAL,   NULL},

  {"Filename Handling", P_SEP, P_SEPARATOR},
  {"strip dot",        P_BOOL,    P_GLOBAL,   NULL},
  {"character set",    P_STRING,  P_GLOBAL, NULL},
  {"mangled stack",    P_INTEGER, P_GLOBAL,   NULL},
  {"coding system",    P_STRING,  P_GLOBAL, NULL},
  {"client code page", P_INTEGER, P_GLOBAL,   NULL},
  {"default case",     P_ENUM, P_LOCAL,   enum_case},
  {"case sensitive",   P_BOOL,    P_LOCAL,   NULL},
  {"casesignames",     P_BOOL,    P_LOCAL,   NULL},
  {"preserve case",    P_BOOL,    P_LOCAL,   NULL},
  {"short preserve case",P_BOOL,  P_LOCAL,   NULL},
  {"mangle case",      P_BOOL,    P_LOCAL,   NULL},
  {"mangling char",    P_CHAR,    P_LOCAL,   NULL},
  {"hide dot files",   P_BOOL,    P_LOCAL,   NULL},
  {"delete veto files",P_BOOL,    P_LOCAL,   NULL},
  {"veto files",       P_STRING,  P_LOCAL,   NULL},
  {"hide files",       P_STRING,  P_LOCAL,   NULL},
  {"veto oplock files",P_STRING,  P_LOCAL,   NULL},
  {"map system",       P_BOOL,    P_LOCAL,   NULL},
  {"map hidden",       P_BOOL,    P_LOCAL,   NULL},
  {"map archive",      P_BOOL,    P_LOCAL,   NULL},
  {"mangled names",    P_BOOL,    P_LOCAL,   NULL},
  {"mangled map",      P_STRING,  P_LOCAL,   NULL},
  {"stat cache",       P_BOOL,    P_GLOBAL,   NULL},

  {"Domain Options", P_SEP, P_SEPARATOR},
  {"domain groups",    P_STRING,  P_GLOBAL,   NULL},
  {"domain admin group",P_STRING, P_GLOBAL,   NULL},
  {"domain guest group",P_STRING, P_GLOBAL,   NULL},
  {"domain admin users",P_STRING, P_GLOBAL,   NULL},
  {"domain guest users",P_STRING, P_GLOBAL,   NULL},
#ifdef USING_GROUPNAME_MAP
  {"groupname map",     P_STRING, P_GLOBAL,   NULL},
#endif
  {"machine password timeout", P_INTEGER, P_GLOBAL,   NULL},

  {"Logon Options", P_SEP, P_SEPARATOR},
  {"logon script",     P_STRING,  P_GLOBAL,   NULL},
  {"logon path",       P_STRING,  P_GLOBAL,   NULL},
  {"logon drive",      P_STRING,  P_GLOBAL,   NULL},
  {"logon home",       P_STRING,  P_GLOBAL,   NULL},
  {"domain logons",    P_BOOL,    P_GLOBAL,   NULL},

  {"Browse Options", P_SEP, P_SEPARATOR},
  {"os level",         P_INTEGER, P_GLOBAL,   NULL},
  {"lm announce",      P_ENUM,    P_GLOBAL,   enum_lm_announce},
  {"lm interval",      P_INTEGER, P_GLOBAL,   NULL},
  {"preferred master", P_BOOL,    P_GLOBAL,   NULL},
  {"prefered master",  P_BOOL,    P_GLOBAL,   NULL},
  {"local master",     P_BOOL,    P_GLOBAL,   NULL},
  {"domain master",    P_BOOL,    P_GLOBAL,   NULL},
  {"browse list",      P_BOOL,    P_GLOBAL,   NULL},
  {"browseable",       P_BOOL,    P_LOCAL,   NULL},
  {"browsable",        P_BOOL,    P_LOCAL,   NULL},

  {"WINS Options", P_SEP, P_SEPARATOR},
  {"dns proxy",        P_BOOL,    P_GLOBAL,   NULL},
  {"wins proxy",       P_BOOL,    P_GLOBAL,   NULL},
  {"wins server",      P_STRING,  P_GLOBAL,   NULL},
  {"wins support",     P_BOOL,    P_GLOBAL,   NULL},

  {"Locking Options", P_SEP, P_SEPARATOR},
  {"blocking locks",   P_BOOL,    P_LOCAL,   NULL},
  {"fake oplocks",     P_BOOL,    P_LOCAL,   NULL},
  {"kernel oplocks",   P_BOOL,    P_GLOBAL,   NULL},
  {"locking",          P_BOOL,    P_LOCAL,   NULL},
  {"ole locking compatibility",   P_BOOL,    P_GLOBAL,   NULL},
  {"oplocks",          P_BOOL,    P_LOCAL,   NULL},
  {"strict locking",   P_BOOL,    P_LOCAL,   NULL},
  {"share modes",      P_BOOL,    P_LOCAL,   NULL},

#ifdef WITH_LDAP
  {"Ldap Options", P_SEP, P_SEPARATOR},
  {"ldap server",      P_STRING,  P_GLOBAL,   NULL},
  {"ldap port",        P_INTEGER, P_GLOBAL,   NULL},
  {"ldap suffix",      P_STRING,  P_GLOBAL,   NULL},
  {"ldap filter",      P_STRING,  P_GLOBAL,   NULL},
  {"ldap root",        P_STRING,  P_GLOBAL,   NULL},
  {"ldap root passwd", P_STRING,  P_GLOBAL,   NULL},
#endif

  {"Miscellaneous Options", P_SEP, P_SEPARATOR},
  {"smbrun",           P_STRING,  P_GLOBAL,   NULL},
  {"config file",      P_STRING,  P_GLOBAL,   NULL},
  {"preload",          P_STRING,  P_GLOBAL,   NULL},
  {"auto services",    P_STRING,  P_GLOBAL,   NULL},
  {"lock dir",         P_STRING,  P_GLOBAL,   NULL},
  {"lock directory",   P_STRING,  P_GLOBAL,   NULL},
  {"default service",  P_STRING,  P_GLOBAL,   NULL},
  {"default",          P_STRING,  P_GLOBAL,   NULL},
  {"message command",  P_STRING,  P_GLOBAL,   NULL},
  {"dfree command",    P_STRING,  P_GLOBAL,   NULL},
  {"valid chars",      P_STRING,  P_GLOBAL, NULL},
  {"remote announce",  P_STRING,  P_GLOBAL,   NULL},
  {"remote browse sync",P_STRING, P_GLOBAL,   NULL},
  {"socket address",   P_STRING,  P_GLOBAL,   NULL},
  {"homedir map",      P_STRING,  P_GLOBAL,   NULL},
  {"time offset",      P_INTEGER, P_GLOBAL,   NULL},
  {"unix realname",    P_BOOL,    P_GLOBAL,   NULL},
  {"NIS homedir",      P_BOOL,    P_GLOBAL,   NULL},
  {"-valid",           P_BOOL,    P_LOCAL,   NULL},
  {"copy",             P_STRING,  P_LOCAL, NULL},
  {"include",          P_STRING,  P_LOCAL, NULL},
  {"exec",             P_STRING,  P_LOCAL,   NULL},
  {"preexec",          P_STRING,  P_LOCAL,   NULL},
  {"postexec",         P_STRING,  P_LOCAL,   NULL},
  {"root preexec",     P_STRING,  P_LOCAL,   NULL},
  {"root postexec",    P_STRING,  P_LOCAL,   NULL},
  {"available",        P_BOOL,    P_LOCAL,   NULL},
  {"volume",           P_STRING,  P_LOCAL,   NULL},
  {"fstype",           P_STRING,  P_LOCAL,   NULL},
  {"set directory",    P_BOOLREV, P_LOCAL,   NULL},
  {"wide links",       P_BOOL,    P_LOCAL,   NULL},
  {"follow symlinks",  P_BOOL,    P_LOCAL,   NULL},
  {"dont descend",     P_STRING,  P_LOCAL,   NULL},
  {"magic script",     P_STRING,  P_LOCAL,   NULL},
  {"magic output",     P_STRING,  P_LOCAL,   NULL},
  {"delete readonly",  P_BOOL,    P_LOCAL,   NULL},
  {"dos filetimes",    P_BOOL,    P_LOCAL,   NULL},
  {"dos filetime resolution",P_BOOL,P_LOCAL,  NULL},
  {"fake directory create times", P_BOOL,P_LOCAL,   NULL},
  {"panic action",     P_STRING,  P_GLOBAL,   NULL},

  {NULL,               P_BOOL,    P_NONE,   NULL}

};

int is_oct(int c){
  if(c >= '0' && c <= '7')
    return 1;
  else
    return 0;
}


void c_edit_to_delete(GtkWidget *button, gpointer window){
  GtkWidget *diagwin;

  diagwin=gtk_object_get_data(GTK_OBJECT(window),"dialog");
  if(diagwin){
    gtk_widget_destroy(diagwin);
    gtk_object_set_data(GTK_OBJECT(window),"dialog",NULL);    
  }
  
  make_del_dialog(GTK_WIDGET(window));
  return;
}


void c_entry_diag_ok(GtkWidget *button, gpointer window){
  GtkWidget *diagwin;
  GtkWidget *entry;
  int row;
  int pindex;
  char *text, *p;
  GtkWidget *sublist;

  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");
  diagwin=gtk_object_get_data(GTK_OBJECT(window),"dialog");
  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));
  entry=gtk_object_get_data(GTK_OBJECT(diagwin),"entry"); 


  text = gtk_entry_get_text(GTK_ENTRY(entry));

  text = gtk_entry_get_text(GTK_ENTRY(entry));
  for(; *text && isspace(*text); text++);
  if(!strlen(text)){
    make_user_mesg("Field is empty","Warning");
    return;
  }
  
  switch(parm_table[pindex].type){
  case P_INTEGER:
    for(p = text; *p && isdigit(*p); p++);
    if(*p){
      make_user_mesg("Integer value required","Warning");
      return;
    }
    break;
      
  case P_OCTAL:
    for(p = text; *p && is_oct(*p); p++);
    if(*p){
      make_user_mesg("Octal value required\ne.g. 0755","Warning");
      return;
    }
    break;
    
  default:
    break;
  }
  
  row = (int)(GTK_CLIST(sublist)->selection->data);

  gtk_clist_set_text(GTK_CLIST(sublist),row,1,text);

  resensitize_sw(GTK_WIDGET(window));

  gtk_widget_destroy(diagwin);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",NULL);
  return;
  
}


void make_entry_dialog(GtkWidget *window){
  GtkWidget *diagwin;
  GtkWidget *entry;
  GtkWidget *label;
  GtkWidget *entrybox;
  GtkWidget *vbox;
  int pindex;
  char *text;
  int row;
  GtkRequisition requisition;
  GtkWidget *sublist;
  
  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");


  row = (int)(GTK_CLIST(sublist)->selection->data);

  gtk_clist_get_text(GTK_CLIST(sublist),
			row ,0,&text);


  diagwin = gnome_dialog_new (text,NULL);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",diagwin);

  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));

  vbox=gtk_vbox_new(FALSE,0);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (diagwin)->vbox), 
		      vbox, TRUE, TRUE, 10);
  gtk_container_border_width (GTK_CONTAINER (vbox), 0);
  gtk_widget_show (vbox);

  gtk_signal_connect(GTK_OBJECT(diagwin), "delete_event",
                     GTK_SIGNAL_FUNC(c_param_diag_delete),
                     window);
  
  gtk_window_position (GTK_WINDOW (diagwin),GTK_WIN_POS_CENTER);
  

  label = gtk_label_new(text);
  gtk_box_pack_start (GTK_BOX (vbox), 
		      label, TRUE, TRUE, 10);
  gtk_widget_show (label);


  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_OK);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_HELP);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_CANCEL);
  gnome_dialog_append_button_with_pixmap(GNOME_DIALOG(diagwin),
					 "Delete",
					 GNOME_STOCK_MENU_CUT);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 0,
				     GTK_SIGNAL_FUNC(c_entry_diag_ok),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 1,
				     GTK_SIGNAL_FUNC(c_help_sel_param),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 2,
				     GTK_SIGNAL_FUNC(c_param_diag_cancel),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 3,
				     GTK_SIGNAL_FUNC(c_edit_to_delete),
				     window);

  
  gtk_clist_get_text(GTK_CLIST(sublist),
			row ,1,&text);  

  entrybox = gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX (vbox), 
		     entrybox, TRUE, TRUE, 5);
  gtk_widget_show(entrybox);

  if(parm_table[pindex].type == P_OCTAL)
    label = gtk_label_new("Octal Value e.g. 0755:");
  else if(parm_table[pindex].type == P_CHAR)
    label = gtk_label_new("Enter Character:");
  else if(parm_table[pindex].type == P_INTEGER)
    label = gtk_label_new("Integer Value:");
  else
    label = gtk_label_new("Value:");
  
  gtk_box_pack_start (GTK_BOX (entrybox), 
			label, FALSE, FALSE, 10);
  gtk_widget_show (label);

  if(parm_table[pindex].type == P_CHAR){
    entry = gtk_entry_new_with_max_length(1);
    gtk_widget_size_request(entry,&requisition);
    gtk_widget_set_usize(entry,20,requisition.height);
  }
  else if(parm_table[pindex].type == P_OCTAL){
    entry = gtk_entry_new_with_max_length(4);
    gtk_widget_size_request(entry,&requisition);
    gtk_widget_set_usize(entry,40,requisition.height);
  }
  else if(parm_table[pindex].type == P_INTEGER){
    entry = gtk_entry_new_with_max_length(5);
    gtk_widget_size_request(entry,&requisition);
    gtk_widget_set_usize(entry,20,requisition.height);
  }
  else{
    entry = gtk_entry_new();
    gtk_widget_size_request(entry,&requisition);
    gtk_widget_set_usize(entry,100,requisition.height);
  }
  gtk_object_set_data(GTK_OBJECT(diagwin),"entry",entry); 
  gtk_entry_set_text(GTK_ENTRY(entry),text);
  gtk_entry_select_region (GTK_ENTRY (entry),
			   0, GTK_ENTRY(entry)->text_length);
  gtk_box_pack_start(GTK_BOX (entrybox), 
		     entry, TRUE, TRUE, 10);
  gtk_signal_connect(GTK_OBJECT(entry), "activate",
		     GTK_SIGNAL_FUNC(c_entry_diag_ok),
                     window);
  gtk_widget_show(entry);
  gtk_widget_grab_focus(entry);
  
  gtk_widget_show(diagwin);
  return;
}

void c_enum_diag_ok(GtkWidget *button, gpointer window){
  GtkWidget *diagwin;
  GtkWidget *choicebox;
  int row;
  GtkWidget *sublist;  

  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");  
  row = (int)(GTK_CLIST(sublist)->selection->data);

  diagwin=gtk_object_get_data(GTK_OBJECT(window),"dialog");
  choicebox=gtk_object_get_data(GTK_OBJECT(diagwin),"choicebox"); 
 

  gtk_clist_set_text(GTK_CLIST(sublist),row,1,
		     gnome_choice_box_get_choice(GNOME_CHOICE_BOX(choicebox)));

  resensitize_sw(GTK_WIDGET(window));

  gtk_widget_destroy(diagwin);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",NULL);

  return;
  
}


void make_enum_dialog(GtkWidget *window){
  GtkWidget *diagwin;
  GtkWidget *choicebox;
  gchar *choices[MAX_ENUM+1];  
  GtkWidget *label;
  GtkWidget *vbox;
  char *text;
  int pindex;
  int row;
  struct enum_list *elist;
  int i,j=0;
  GtkWidget *sublist;

  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");
  row = (int)(GTK_CLIST(sublist)->selection->data);

  gtk_clist_get_text(GTK_CLIST(sublist),
			row ,0,&text);

  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));

  diagwin = gnome_dialog_new (text,NULL);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",diagwin);

  vbox=gtk_vbox_new(FALSE,0);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (diagwin)->vbox), 
		      vbox, TRUE, TRUE, 10);
  gtk_container_border_width (GTK_CONTAINER (vbox), 0);
  gtk_widget_show (vbox);

  gtk_signal_connect(GTK_OBJECT(diagwin), "delete_event",
                     GTK_SIGNAL_FUNC(c_param_diag_delete),
                     window);
  
  gtk_window_position (GTK_WINDOW (diagwin),GTK_WIN_POS_CENTER);
  label = gtk_label_new(text);
  gtk_box_pack_start (GTK_BOX (vbox), 
		      label, TRUE, TRUE, 0);
  gtk_widget_show (label);


  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_OK);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_HELP);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_CANCEL);
  gnome_dialog_append_button_with_pixmap(GNOME_DIALOG(diagwin),
					 "Delete",
					 GNOME_STOCK_MENU_CUT);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 0,
				     GTK_SIGNAL_FUNC(c_enum_diag_ok),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 1,
				     GTK_SIGNAL_FUNC(c_help_sel_param),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 2,
				     GTK_SIGNAL_FUNC(c_param_diag_cancel),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 3,
				     GTK_SIGNAL_FUNC(c_edit_to_delete),
				     window);

  gtk_clist_get_text(GTK_CLIST(sublist),
		     row ,1,&text);

  elist = parm_table[pindex].enum_list;
  

  for(i=0; elist[i].value != -1; i++){
    choices[i]= elist[i].name;
    if(!strcasecmp(text,elist[i].name))
      j=i;
  }
  choices[i]=NULL;

  choicebox=gnome_choice_box_newv(CHOICE_BOX_VV,
				  NULL,choices);
  gtk_object_set_data(GTK_OBJECT(diagwin),"choicebox",choicebox);
  gtk_box_pack_start (GTK_BOX(vbox), 
		      choicebox, TRUE, TRUE, 0);
  gnome_choice_box_set_index(GNOME_CHOICE_BOX(choicebox),j);

  gtk_widget_show(choicebox);
  gtk_widget_show(diagwin);
}

int is_true(char *s){
  if(!strcasecmp(s,"true") || 
     !strcasecmp(s,"yes") ||
     !strcasecmp(s,"1"))
    return 1;
  else
    return 0;
}

void c_bool_diag_toggle(GtkWidget *button, gpointer window){
  GtkWidget *diagwin;
  int row;
  char *text;
  GtkWidget *sublist;

  diagwin=gtk_object_get_data(GTK_OBJECT(window),"dialog");  
  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");


  row = (int)(GTK_CLIST(sublist)->selection->data);
  gtk_clist_get_text(GTK_CLIST(sublist),
			row ,1,&text);

  if(is_true(text))
    gtk_clist_set_text(GTK_CLIST(sublist),row,1,"False");
  else
    gtk_clist_set_text(GTK_CLIST(sublist),row,1,"True");

  resensitize_sw(GTK_WIDGET(window));

  gtk_widget_destroy(diagwin);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",NULL);
  return;
}


void make_bool_dialog(GtkWidget *window){
  GtkWidget *diagwin;
  GtkWidget *label;
  GtkWidget *vbox;
  char *text;
  int row;
  GtkWidget *sublist;
 
  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");
  
  row = (int)(GTK_CLIST(sublist)->selection->data);

  gtk_clist_get_text(GTK_CLIST(sublist),
			row ,0,&text);



  diagwin = gnome_dialog_new (text,NULL);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",diagwin);

  vbox=gtk_vbox_new(FALSE,0);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (diagwin)->vbox), 
		      vbox, TRUE, TRUE, 10);
  gtk_container_border_width (GTK_CONTAINER (vbox), 0);
  gtk_widget_show (vbox);

  gtk_signal_connect(GTK_OBJECT(diagwin), "delete_event",
                     GTK_SIGNAL_FUNC(c_param_diag_delete),
                     window);
  
  gtk_window_position (GTK_WINDOW (diagwin),GTK_WIN_POS_CENTER);
  label = gtk_label_new(text);
  gtk_box_pack_start (GTK_BOX (vbox), 
		      label, TRUE, TRUE, 10);
  gtk_widget_show (label);

  gnome_dialog_append_button_with_pixmap(GNOME_DIALOG(diagwin),
					 "Toggle",
					 GNOME_STOCK_MENU_REFRESH);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_HELP);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_CANCEL);
  gnome_dialog_append_button_with_pixmap(GNOME_DIALOG(diagwin),
					 "Delete",
					 GNOME_STOCK_MENU_CUT);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 0,
				     GTK_SIGNAL_FUNC(c_bool_diag_toggle),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 1,
				     GTK_SIGNAL_FUNC(c_help_sel_param),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 2,
				     GTK_SIGNAL_FUNC(c_param_diag_cancel),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 3,
				     GTK_SIGNAL_FUNC(c_edit_to_delete),
				     window);

  gtk_clist_get_text(GTK_CLIST(sublist),
			row ,1,&text);
  
  if(is_true(text))
    label = gtk_label_new("Current value: True");
  else
    label = gtk_label_new("Current value: False");
  gtk_box_pack_start (GTK_BOX (vbox), 
		      label, TRUE, TRUE, 10);
  gtk_widget_show (label);

  gtk_widget_show(diagwin);
}


void make_edit_dialog(GtkWidget *window){
  int i;
  char *text;
  int row;
  GtkWidget *sublist;

  desensitize_sw(window);

  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");
  row = (int)(GTK_CLIST(sublist)->selection->data);

  gtk_clist_get_text(GTK_CLIST(sublist),
			row ,0,&text);
  
  for(i=0; 
      parm_table[i].label && strcasecmp(text,parm_table[i].label) ; 
      i++);

  if(!parm_table[i].label){
    make_entry_dialog(window);
    return;
  }

  gtk_object_set_data(GTK_OBJECT(window),
		      "pindex",
		      GINT_TO_POINTER(i));

  switch (parm_table[i].type) 
    {
    case P_BOOL:
    case P_BOOLREV:
      make_bool_dialog(window);
      return;
	
    case P_CHAR:
      make_entry_dialog(window);
      return;
	
    case P_INTEGER:
      make_entry_dialog(window);
      return;

    case P_OCTAL:
      make_entry_dialog(window);
      return;	
      
    case P_STRING:
    case P_USTRING:
    case P_GSTRING:
    case P_UGSTRING:
      make_entry_dialog(window);
      return;
      
    case P_ENUM:
      make_enum_dialog(window);
      return;;
	
    default:
      make_entry_dialog(window);
      return;
    }
  return;
}

/************************************************************/
/************************************************************/
/************************************************************/
/************************************************************/

void c_entry_insert_help(GtkWidget *button, gpointer data){

  make_param_help((char *)data);  
  return;
}


void c_enum_insert_ok(GtkWidget *button, gpointer window){
  GtkWidget *diagwin;
  GtkWidget *choicebox;
  int pindex;
  int row;
  char *text[2];
  GtkWidget *sublist;

  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");

  diagwin=gtk_object_get_data(GTK_OBJECT(window),"dialog");
  choicebox=gtk_object_get_data(GTK_OBJECT(diagwin),"choicebox");
  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));

  text[0] = parm_table[pindex].label;

  if(GTK_CLIST(sublist)->selection)
    row = (int)(GTK_CLIST(sublist)->selection->data);
  else
    row=GTK_CLIST(sublist)->rows;
  text[1] = gnome_choice_box_get_choice(GNOME_CHOICE_BOX(choicebox));
					
  gtk_clist_insert(GTK_CLIST(sublist),row,text);

  resensitize_sw(GTK_WIDGET(window));

  gtk_widget_destroy(diagwin);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",NULL);
  return;
  
}


void make_enum_insert(GtkWidget *window){
  GtkWidget *diagwin;
  GtkWidget *choicebox;
  gchar *choices[MAX_ENUM+1];
  GtkWidget *label;
  GtkWidget *vbox;
  int pindex;
  struct enum_list *elist;
  int i;

  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));

  diagwin = gnome_dialog_new (parm_table[pindex].label,NULL);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",diagwin);

  vbox=gtk_vbox_new(FALSE,0);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (diagwin)->vbox), 
		      vbox, TRUE, TRUE, 10);
  gtk_container_border_width (GTK_CONTAINER (vbox), 10);
  gtk_widget_show (vbox);

  gtk_signal_connect(GTK_OBJECT(diagwin), "delete_event",
                     GTK_SIGNAL_FUNC(c_param_diag_delete),
                     window);
  

  gtk_window_position (GTK_WINDOW (diagwin),GTK_WIN_POS_CENTER);
  label = gtk_label_new(parm_table[pindex].label);
  gtk_box_pack_start (GTK_BOX(vbox), 
		      label, TRUE, TRUE, 0);
  gtk_widget_show (label);
 
  gnome_dialog_append_button_with_pixmap(GNOME_DIALOG(diagwin),
			     "Insert",
			     GNOME_STOCK_MENU_NEW);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_HELP);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_CANCEL);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 0,
				     GTK_SIGNAL_FUNC(c_enum_insert_ok),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 1,
				     GTK_SIGNAL_FUNC(c_entry_insert_help),
				     parm_table[pindex].label);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 2,
				     GTK_SIGNAL_FUNC(c_param_diag_cancel),
				     window);


  elist = parm_table[pindex].enum_list;
  

  for(i=0; elist[i].value != -1; i++)
    choices[i]= elist[i].name;  
  choices[i]=NULL;

  choicebox=gnome_choice_box_newv(CHOICE_BOX_VV,
				  NULL,choices);
  gtk_object_set_data(GTK_OBJECT(diagwin),"choicebox",choicebox);
  gtk_box_pack_start (GTK_BOX(vbox), 
		      choicebox, TRUE, TRUE, 0);  
  gtk_widget_show(choicebox);
  gtk_widget_show(diagwin);
}



void c_entry_insert_ok(GtkWidget *button, gpointer window){
  GtkWidget *diagwin;
  GtkWidget *entry;
  int pindex;
  int row;
  char *text[2], *p;
  GtkWidget *sublist;

  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");  
  diagwin=gtk_object_get_data(GTK_OBJECT(window),"dialog");
  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));
  entry=gtk_object_get_data(GTK_OBJECT(diagwin),"entry"); 

  
  text[1] = gtk_entry_get_text(GTK_ENTRY(entry));
  for(; *text[1] && isspace(*text[1]); text[1]++);
  if(!strlen(text[1])){
    make_user_mesg("Field is empty","Warning");
    return;
  }
  
    
  switch(parm_table[pindex].type){
  case P_INTEGER:
    for(p = text[1]; *p && isdigit(*p); p++);
    if(*p){
      make_user_mesg("Integer value required","Warning");
      return;
    }
    break;
      
  case P_OCTAL:
    for(p = text[1]; *p && is_oct(*p); p++);
    if(*p){
      make_user_mesg("Octal value required\ne.g. 0755","Warning");
      return;
    }
    break;
    
  default:
    break;
  }
  text[0] = parm_table[pindex].label;

  if(GTK_CLIST(sublist)->selection)
    row = (int)(GTK_CLIST(sublist)->selection->data);
  else
    row=GTK_CLIST(sublist)->rows;

  gtk_clist_insert(GTK_CLIST(sublist),row,text);

  resensitize_sw(GTK_WIDGET(window));

  gtk_widget_destroy(diagwin);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",NULL);
  return;
  
}


void make_entry_insert(GtkWidget *window){
  GtkWidget *diagwin;
  GtkWidget *entry;
  GtkWidget *label;
  GtkWidget *entrybox;
  GtkWidget *vbox;
  int pindex;
  GtkRequisition requisition;

  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));

  diagwin = gnome_dialog_new (parm_table[pindex].label,NULL);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",diagwin);

  vbox=gtk_vbox_new(FALSE,0);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (diagwin)->vbox), 
		      vbox, TRUE, TRUE, 10);
  gtk_container_border_width (GTK_CONTAINER (vbox), 10);
  gtk_widget_show (vbox);


  gtk_signal_connect(GTK_OBJECT(diagwin), "delete_event",
                     GTK_SIGNAL_FUNC(c_param_diag_delete),
                     window);
  
  gtk_window_position (GTK_WINDOW (diagwin),GTK_WIN_POS_CENTER);
  label = gtk_label_new(parm_table[pindex].label);
  gtk_box_pack_start (GTK_BOX (vbox), 
		      label, TRUE, TRUE, 10);
  gtk_widget_show (label);

  gnome_dialog_append_button_with_pixmap(GNOME_DIALOG(diagwin),
			     "Insert",
			     GNOME_STOCK_MENU_NEW);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_HELP);
  gnome_dialog_append_button(GNOME_DIALOG(diagwin),
			     GNOME_STOCK_BUTTON_CANCEL);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 0,
				     GTK_SIGNAL_FUNC(c_entry_insert_ok),
				     window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 1,
				     GTK_SIGNAL_FUNC(c_entry_insert_help),
				     parm_table[pindex].label);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 2,
				     GTK_SIGNAL_FUNC(c_param_diag_cancel),
				     window);


  entrybox = gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX (vbox), 
		     entrybox, TRUE, TRUE, 5);
  gtk_widget_show(entrybox);

  if(parm_table[pindex].type == P_OCTAL)
    label = gtk_label_new("Octal Value e.g. 0755:");
  else if(parm_table[pindex].type == P_CHAR)
    label = gtk_label_new("Enter Character:");
  else if(parm_table[pindex].type == P_INTEGER)
    label = gtk_label_new("Integer Value:");
  else
    label = gtk_label_new("Value:");

  gtk_box_pack_start (GTK_BOX (entrybox), 
		      label, FALSE, FALSE, 10);
  gtk_widget_show (label);

  if(parm_table[pindex].type == P_CHAR){
    entry = gtk_entry_new_with_max_length(1);
    gtk_widget_size_request(entry,&requisition);
    gtk_widget_set_usize(entry,20,requisition.height);
  }
  else if(parm_table[pindex].type == P_OCTAL){
    entry = gtk_entry_new_with_max_length(4);
    gtk_widget_size_request(entry,&requisition);
    gtk_widget_set_usize(entry,40,requisition.height);
  }
  else if(parm_table[pindex].type == P_INTEGER){
    entry = gtk_entry_new_with_max_length(5);
    gtk_widget_size_request(entry,&requisition);
    gtk_widget_set_usize(entry,50,requisition.height);
  }
  else{
    entry = gtk_entry_new();
    gtk_widget_size_request(entry,&requisition);
    gtk_widget_set_usize(entry,150,requisition.height);
  }
  gtk_object_set_data(GTK_OBJECT(diagwin),"entry",entry); 
  gtk_box_pack_start(GTK_BOX (entrybox), 
		     entry, TRUE, TRUE, 10);
  gtk_signal_connect(GTK_OBJECT(entry), "activate",
		     GTK_SIGNAL_FUNC(c_entry_insert_ok),
                     window);
  gtk_widget_show(entry);
  gtk_widget_grab_focus(entry);
  
  gtk_widget_show(diagwin);
  return;
}

void c_bool_insert_yes(GtkWidget *button, gpointer window){
  GtkWidget *diagwin;
  int row;  
  int pindex;
  char *text[2];
  GtkWidget *sublist;


  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");
  diagwin=gtk_object_get_data(GTK_OBJECT(window),"dialog");
  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));
  
  
  text[0] = parm_table[pindex].label;
  text[1] = "yes";

  if(GTK_CLIST(sublist)->selection)
    row = (int)(GTK_CLIST(sublist)->selection->data);
  else
    row=GTK_CLIST(sublist)->rows;

  gtk_clist_insert(GTK_CLIST(sublist),row,text);

  resensitize_sw(GTK_WIDGET(window));

  gtk_widget_destroy(diagwin);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",NULL);
  return;
  
}

void c_bool_insert_no(GtkWidget *button, gpointer window){
  GtkWidget *diagwin;
  int row;
  int pindex;
  char *text[2];
  GtkWidget *sublist;

  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");
  diagwin=gtk_object_get_data(GTK_OBJECT(window),"dialog");  
  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));
   
  text[0] = parm_table[pindex].label;
  text[1] = "no";

  if(GTK_CLIST(sublist)->selection)
    row = (int)(GTK_CLIST(sublist)->selection->data);
  else
    row=GTK_CLIST(sublist)->rows;

  gtk_clist_insert(GTK_CLIST(sublist),row,text);

  resensitize_sw(GTK_WIDGET(window));

  gtk_widget_destroy(diagwin);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",NULL);
  return;
  
}



void make_bool_insert(GtkWidget *window){
  GtkWidget *diagwin;
  GtkWidget *label;
  GtkWidget *vbox;
  int pindex;

  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));

  diagwin = gnome_dialog_new (parm_table[pindex].label,
			      GNOME_STOCK_BUTTON_YES,
			      GNOME_STOCK_BUTTON_NO,
			      GNOME_STOCK_BUTTON_HELP,
			      GNOME_STOCK_BUTTON_CANCEL,
			      NULL);
  gtk_object_set_data(GTK_OBJECT(window),"dialog",diagwin);

  vbox=gtk_vbox_new(FALSE,0);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (diagwin)->vbox), 
		      vbox, TRUE, TRUE, 10);
  gtk_container_border_width (GTK_CONTAINER (vbox), 10);
  gtk_widget_show (vbox);

  gtk_signal_connect(GTK_OBJECT(diagwin), "delete_event",
                     GTK_SIGNAL_FUNC(c_param_diag_delete),
                     window);
  
  gtk_window_position (GTK_WINDOW (diagwin),GTK_WIN_POS_CENTER);
  label = gtk_label_new(parm_table[pindex].label);
  gtk_box_pack_start (GTK_BOX (vbox), 
		      label, TRUE, TRUE, 10);
  gtk_widget_show (label);

  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 0,
			      GTK_SIGNAL_FUNC(c_bool_insert_yes),
			      window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 1,
			      GTK_SIGNAL_FUNC(c_bool_insert_no),
			      window);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 2,
			      GTK_SIGNAL_FUNC(c_entry_insert_help),
			      parm_table[pindex].label);
  gnome_dialog_button_connect(GNOME_DIALOG(diagwin), 3,
			      GTK_SIGNAL_FUNC(c_param_diag_cancel),
			      window);


  gtk_widget_show(diagwin);
}



void inserter_action(GtkWidget *window){
  int j, row;
  char *text;
  int pindex;
  GtkWidget *servicelist;
  GtkWidget *sublist;
  
  sublist=gtk_object_get_data(GTK_OBJECT(window),"sublist");
  servicelist=gtk_object_get_data(GTK_OBJECT(window),"servicelist");

  pindex=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window),
					     "pindex"));

  if(!sublist){
    make_user_mesg("Select or create a service\nto insert into","Message");
    return;
  }

  if(parm_table[pindex].class == P_GLOBAL){
    row = (int)(GTK_CLIST(servicelist)->selection->data);
    gtk_clist_get_text(GTK_CLIST(servicelist),
		       row ,0,&text);
    if(strcasecmp(text,"global")){
      make_user_mesg("This parameter is global in scope only\nswitch service to 'global' to use it",
		     "Warning");
      return;
    }
  }

  desensitize_sw(window);
  
  for(j=0; j < GTK_CLIST(sublist)->rows; j++){
    gtk_clist_get_text(GTK_CLIST(sublist),
		       j,0,&text);
    if(!strcasecmp(text,parm_table[pindex].label)){
      gtk_clist_select_row(GTK_CLIST(sublist),j,0);
      make_edit_dialog(window);
      return;
    }
  }

  switch (parm_table[pindex].type) 
    {
    case P_BOOL:
    case P_BOOLREV:
      make_bool_insert(window);
      return;
	
    case P_CHAR:
      make_entry_insert(window);
      return;
	
    case P_INTEGER:
      make_entry_insert(window);
      return;

    case P_OCTAL:
      make_entry_insert(window);
      return;	
      
    case P_STRING:
    case P_USTRING:
    case P_GSTRING:
    case P_UGSTRING:
      make_entry_insert(window);
      return;
      
    case P_ENUM:
      make_enum_insert(window);
      return;;
	
    default:
      make_entry_insert(window);
      return;
    }


  make_entry_insert(window);
  return;
}

      

void c_insertmenu_insert(GtkWidget *widget, GtkWidget *window){
  int i, row;
  GtkWidget *innerbook;
  GtkWidget *clist;
  GtkWidget *page;
  GList *inslists;
  GtkWidget *outbook;  

  outbook=gtk_object_get_data(GTK_OBJECT(window),
			       "outbook");

  inslists=gtk_object_get_data(GTK_OBJECT(window),
			       "inslists");  



  i = gtk_notebook_current_page(GTK_NOTEBOOK(outbook));
  if(!(innerbook = g_list_nth_data(inslists,i) )){
    make_user_mesg("Cannot get active page1",
		    "Error");
    return;
  }
  i = gtk_notebook_current_page(GTK_NOTEBOOK(innerbook));
  if(!(page = g_list_nth_data(GTK_NOTEBOOK(innerbook)->children,i))){
    make_user_mesg("Cannot get active page2",
		    "Error"); 
    return;
  }
  if(!(clist = ((GtkNotebookPage *)page)->child)){
    make_user_mesg("Cannot get active list",
		    "Error");
    return;
  }
  
  /*** if gtk version !< 1.1.5 clist in scrolled *****/
  if(!(clist = gtk_container_children(GTK_CONTAINER(clist))->data)){
    make_user_mesg("Cannot get active list",
		   "Error");
    return;
  }


  if(!GTK_CLIST(clist)->selection){
    make_user_mesg("Select the parmater\nyou wish to enter","Message");
    return;
  }
  row = (int)(GTK_CLIST(clist)->selection->data);
  i = (int)gtk_clist_get_row_data(GTK_CLIST(clist),row);
  gtk_object_set_data(GTK_OBJECT(window),
		      "pindex",
		      GINT_TO_POINTER(i));
  inserter_action(window);
  return;
}

void c_insertmenu_help(GtkWidget *widget, GtkWidget *window){
  int i, row;
  GtkWidget *innerbook;
  GtkWidget *clist;
  GtkWidget *page;
  GList *inslists;
  GtkWidget *outbook;

  outbook=gtk_object_get_data(GTK_OBJECT(window),
			       "outbook");

  inslists=gtk_object_get_data(GTK_OBJECT(window),
			       "inslists");  


  i = gtk_notebook_current_page(GTK_NOTEBOOK(outbook));
  if(!(innerbook = g_list_nth_data(inslists,i) )){
    make_user_mesg("Cannot get active page",
		   "Error");
    return;
  }
  i = gtk_notebook_current_page(GTK_NOTEBOOK(innerbook));
  if(!(page = g_list_nth_data(GTK_NOTEBOOK(innerbook)->children,i))){
    make_user_mesg("Cannot get active page",
		   "Error"); 
    return;
  }
  if(!(clist = ((GtkNotebookPage *)page)->child)){
    make_user_mesg("Cannot get active list",
		   "Error");
    return;
  }
  

  /*** if gtk version !< 1.1.5 clist in scrolled *****/
  if(!(clist = gtk_container_children(GTK_CONTAINER(clist))->data)){
    make_user_mesg("Cannot get active list",
		   "Error");
    return;
  }


  if(!GTK_CLIST(clist)->selection){
    make_user_mesg("Select the parmater\nyou wish to enter","Message");
    
    return;
  }
  row = (int)(GTK_CLIST(clist)->selection->data);
  i = (int)gtk_clist_get_row_data(GTK_CLIST(clist),row);
  make_param_help(parm_table[i].label);
  return;
}

void c_insertmenu_click(GtkWidget *clist,
			gint row,
			gint column,
			GdkEventButton *event,
			gpointer window){

  int i;  

  if(event && event->type==GDK_2BUTTON_PRESS){
    i = (int)gtk_clist_get_row_data(GTK_CLIST(clist),row);
    gtk_object_set_data(GTK_OBJECT(window),
			"pindex",
			GINT_TO_POINTER(i));
    inserter_action(window);
  }
  return;
}


void c_make_inserter(GtkWidget *widget, gpointer window){
  GtkWidget *inserter;
  GtkWidget *outbook;
  GList *inslists;  
  GtkWidget *button;
  GtkWidget *clist1, *clist2, *clist3;
  GtkWidget *vbox;
  GtkWidget *innervbox;
  GtkWidget *innerbook;

  GtkWidget *scrolled1, *scrolled2, *scrolled3;

  GtkWidget *label;
  int i;
  int row;
  
  inserter=gtk_object_get_data(GTK_OBJECT(window),
			       "inserter");

  if(!inserter){
    inslists = NULL;
    clist1 = clist2 = clist3 = NULL;
    inserter = gtk_dialog_new ();
    gtk_object_set_data(GTK_OBJECT(window),
			"inserter",inserter);
    gtk_window_set_title(GTK_WINDOW(inserter),"Insert Parameters");
    gtk_signal_connect_object (GTK_OBJECT (inserter), 
			       "delete_event",
			       GTK_SIGNAL_FUNC(gtk_widget_hide),
			       GTK_OBJECT (inserter));
    gtk_widget_set_usize(inserter, 420, 300);


    button=gnome_stock_pixmap_widget(inserter,
				     GNOME_STOCK_MENU_NEW);    
    button=gnome_pixmap_button(button,"Insert");

    gtk_box_pack_start(GTK_BOX (GTK_DIALOG (inserter)->action_area), 
		       button, TRUE, TRUE, 10);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(c_insertmenu_insert), window);
    gtk_widget_show(button);



    button = gnome_stock_button(GNOME_STOCK_BUTTON_HELP);

    gtk_box_pack_start(GTK_BOX (GTK_DIALOG (inserter)->action_area), 
		       button, TRUE, TRUE, 10);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(c_insertmenu_help), window);
    gtk_widget_show(button);



    button=gnome_stock_pixmap_widget(inserter,
				     GNOME_STOCK_MENU_CLOSE);    
    button=gnome_pixmap_button(button,"Done");

    gtk_box_pack_start(GTK_BOX (GTK_DIALOG (inserter)->action_area), 
		       button, TRUE, TRUE, 10);
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_hide),
			      GTK_OBJECT(inserter));
    gtk_widget_show(button);

    vbox=gtk_vbox_new(FALSE,0);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (inserter)->vbox), 
			vbox, TRUE, TRUE, 0);
    gtk_container_border_width (GTK_CONTAINER (vbox), 5);
    gtk_widget_show (vbox);
    
    outbook = gtk_notebook_new ();
    gtk_object_set_data(GTK_OBJECT(window),
			"outbook",outbook);
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (outbook), GTK_POS_LEFT);
    gtk_notebook_set_show_border(GTK_NOTEBOOK (outbook),FALSE);
    gtk_notebook_set_scrollable(GTK_NOTEBOOK (outbook),TRUE);
    gtk_box_pack_start (GTK_BOX (vbox), 
            outbook, TRUE, TRUE, 0);
    gtk_widget_show(outbook);

    for(i=0; parm_table[i].label; i++){
      if(parm_table[i].type == P_SEP){

	innervbox=gtk_vbox_new(FALSE,0);
	gtk_widget_show (innervbox);
	label = gtk_label_new(parm_table[i].label);
	gtk_widget_show (label);
	gtk_notebook_append_page (GTK_NOTEBOOK (outbook), innervbox, label);
	

        label = gtk_label_new(" Available Parameters ");
	gtk_box_pack_start (GTK_BOX (innervbox), 
		      label, FALSE, TRUE, 0);
	gtk_widget_show (label);
	
	innerbook = gtk_notebook_new ();
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (innerbook), GTK_POS_TOP);
	gtk_notebook_set_show_border(GTK_NOTEBOOK (innerbook),FALSE);
	gtk_box_pack_start (GTK_BOX (innervbox), 
            innerbook, TRUE, TRUE, 0);
	gtk_widget_show (innerbook);

	clist1 = gtk_clist_new(1);
 	gtk_clist_set_selection_mode(GTK_CLIST(clist1),
			       GTK_SELECTION_BROWSE);
	clist2 = gtk_clist_new(1);
 	gtk_clist_set_selection_mode(GTK_CLIST(clist2),
			       GTK_SELECTION_BROWSE);
	clist3 = gtk_clist_new(1);
 	gtk_clist_set_selection_mode(GTK_CLIST(clist3),
			       GTK_SELECTION_BROWSE);

	gtk_signal_connect(GTK_OBJECT(clist1),
			   "select_row",
			   GTK_SIGNAL_FUNC(c_insertmenu_click),
			   window);
	gtk_signal_connect(GTK_OBJECT(clist2),
			   "select_row",
			   GTK_SIGNAL_FUNC(c_insertmenu_click),
			   window);
	gtk_signal_connect(GTK_OBJECT(clist3),
			   "select_row",
			   GTK_SIGNAL_FUNC(c_insertmenu_click),
			   window);

	scrolled1 = gtk_scrolled_window_new(NULL, NULL);
	scrolled2 = gtk_scrolled_window_new(NULL, NULL);
	scrolled3 = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled1), 
				       GTK_POLICY_NEVER, 
				       GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled2), 
				       GTK_POLICY_NEVER, 
				       GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled3), 
				       GTK_POLICY_NEVER, 
				       GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled1), clist1);
	gtk_container_add (GTK_CONTAINER (scrolled2), clist2);
	gtk_container_add (GTK_CONTAINER (scrolled3), clist3);
	gtk_widget_show (clist1);
	gtk_widget_show (clist2);
	gtk_widget_show (clist3);
	gtk_widget_show (scrolled1);
	gtk_widget_show (scrolled2);
	gtk_widget_show (scrolled3);
	label = gtk_label_new("  Global  ");
	gtk_widget_show (label);
	gtk_notebook_append_page (GTK_NOTEBOOK (innerbook), scrolled1, label);
	label = gtk_label_new("  Local  ");
	gtk_widget_show (label);
	gtk_notebook_append_page (GTK_NOTEBOOK (innerbook), scrolled2, label);
	label = gtk_label_new("  All  ");
	gtk_widget_show (label);
	gtk_notebook_append_page (GTK_NOTEBOOK (innerbook), scrolled3, label);

	inslists = g_list_append(inslists,innerbook);

      }
      else {
	if(parm_table[i].class == P_GLOBAL){
	  row = gtk_clist_append(GTK_CLIST(clist1), &(parm_table[i].label) );
	  gtk_clist_set_row_data(GTK_CLIST(clist1), row, (gpointer)i);
	}
	else {
	  row = gtk_clist_append(GTK_CLIST(clist2), &(parm_table[i].label) );
	  gtk_clist_set_row_data(GTK_CLIST(clist2), row, (gpointer)i);
	}
	row = gtk_clist_append(GTK_CLIST(clist3), &(parm_table[i].label) );
	gtk_clist_set_row_data(GTK_CLIST(clist3), row, (gpointer)i);
      }
    }
    gtk_notebook_set_page(GTK_NOTEBOOK (outbook),0);
    gtk_object_set_data(GTK_OBJECT(window),
			"inslists",inslists);
  }
  gtk_widget_show(inserter);
  return;
}
    
