/* *************************************************************************
                          gdlc.i.g 
the GDL interpreter
interprets the output of the treeparser/compiler
                             -------------------
    begin                : July 22 2002
    copyright            : (C) 2002 by Marc Schellens
    email                : m_schellens@hotmail.com
 ***************************************************************************/

/* *************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

header "pre_include_cpp" {
    // gets inserted before the antlr generated includes in the cpp file
#include "includefirst.hpp"
}

header "post_include_cpp" {
    // gets inserted after the antlr generated includes in the cpp file
#include "dinterpreter.hpp"

#include <cassert>

// tweaking ANTLR
#define ASTNULL          NULLProgNodeP
#define ProgNodeP( xxx ) NULL             /* ProgNodeP(antlr::nullAST) */
#define RefAST( xxx)     ConvertAST( xxx) /* antlr::RefAST( Ref type)  */
#define match( a, b)     /* remove from source */

using namespace std;
}

header {
    // antlr header

    // make sure it gets included before the 'tweak'
#include "GDLParser.hpp" 
#include "GDLTreeParser.hpp" 

#include <map>
#include <iomanip>
//#include <exception>

#include "datatypes.hpp"
#include "objects.hpp"
#include "dpro.hpp"
#include "dnode.hpp"
#include "accessdesc.hpp"
#include "initsysvar.hpp"
#include "gdljournal.hpp"

// tweaking ANTLR
#define RefAST( xxx)     ConvertAST( xxx) /* antlr::RefAST( Ref type)  */
}

options {
	language="Cpp";
	genHashLines = false;
	namespaceStd="std";         // cosmetic option to get rid of long defines
	namespaceAntlr="antlr";     // cosmetic option to get rid of long defines
}	

// the GDL TreeParser  ****************************************
class GDLInterpreter extends TreeParser;

options {
  importVocab = GDL;	// use vocab generated by lexer
  buildAST = false;     // no modifying of AST anymore
    // no AST is created in the interpreter, hence we don't need ref counting
//  ASTLabelType = "RefDNode"; 
  ASTLabelType = "ProgNodeP"; 
//  defaultErrorHandler = true;
  defaultErrorHandler = false;
//  codeGenMakeSwitchThreshold = 2;
//  codeGenBitsetTestThreshold = 32;
}

{
private:
    // ASTNULL replacement
    static ProgNode  NULLProgNode;
    static ProgNodeP NULLProgNodeP;

public: 
    enum RetCode {
        RC_OK=0,
        RC_BREAK,
        RC_CONTINUE,
        RC_RETURN, 
        RC_ABORT, // checked as retCode >= RC_RETURN
    };  

    // code in: dinterpreter.cpp
    static bool SearchCompilePro(const std::string& pro);
    static int GetFunIx( const std::string& subName);
    static int GetProIx( const std::string& subName);
    DStructGDL* ObjectStruct( BaseGDL* self, ProgNodeP mp);
    DStructGDL* ObjectStructCheckAccess( BaseGDL* self, ProgNodeP mp);

private: 
    // code in: dinterpreter.cpp
    static void SetFunIx( ProgNodeP f); // triggers read/compile
    static void SetProIx( ProgNodeP f); // triggers read/compile
    static void AdjustTypes( BaseGDL*&, BaseGDL*&);


protected:
    std::istringstream executeLine; // actual interactive executed line

//     std::vector<BaseGDL*> tmpList;
//     void ClearTmpList()
//     {
//         std::vector<BaseGDL*>::iterator i;
//         for(i = tmpList.begin(); i != tmpList.end(); ++i) 
//             { delete *i;}
//         tmpList.clear();
//     }

    class RetAllException 
    {
        public:
        enum ExCode {
            NONE=0, // normal RETALL
            RUN     // RETALL from .RUN command
        };  

        private:
        ExCode code;

        public:
        RetAllException( ExCode code_=NONE): code( code_) {}

        ExCode Code() { return code;}
    };
    
    // code in: dinterpreter.cpp
//    static bool CompleteFileName(std::string& fn); -> str.cpp
    static bool CompileFile(const std::string& f, const std::string& untilPro=""); 
    BaseGDL*  returnValue;  // holding the return value for functions
    BaseGDL** returnValueL; // holding the return value for l_functions

    bool interruptEnable;

    typedef std::map<SizeT, BaseGDL*> HeapT;
    typedef std::map<SizeT, DStructGDL*> ObjHeapT;

    // the following must be all static because several interpreter might be active
    // the heap for all dynamic variables
    // ease the handling, no memory leaks, gc possible
    static HeapT     heap; 
    static ObjHeapT  objHeap; 

    // index for newly allocated heap variables
    static SizeT objHeapIx;
    static SizeT heapIx;

    static EnvStackT  callStack; 

public:
    // triggers read/compile/interpret
    DStructDesc* GetStruct(const std::string& name, const ProgNodeP cN); 

    // the New... functions 'own' their BaseGDL*
    SizeT NewObjHeap( SizeT n=1, DStructGDL* var=NULL)
    {
        SizeT tmpIx=objHeapIx;
        for( SizeT i=0; i<n; i++)
        objHeap.insert( objHeap.end(),
            std::pair<SizeT, DStructGDL*>( objHeapIx++, var));
        return tmpIx;
    }
    SizeT NewHeap( SizeT n=1, BaseGDL* var=NULL)
    {
        SizeT tmpIx=heapIx;
        for( SizeT i=0; i<n; i++)
        heap.insert( heap.end(),
            std::pair<SizeT, BaseGDL*>( heapIx++, var));
        return tmpIx;
    }
    static void FreeObjHeap( DObj id)
    {
        if( id != 0)
        {       
            ObjHeapT::iterator it=objHeap.find( id);
            if  ( it != objHeap.end()) 
            { 
                delete (*it).second;
                objHeap.erase( id);
            }
        }
    }
    static void FreeHeap( DPtr id)
    {
        if( id != 0)
            {
                HeapT::iterator it=heap.find( id);
                if( it != heap.end()) 
                    { 
                        delete (*it).second;
                        heap.erase( id); 
                    }
            }
    }

   static void FreeHeap( DPtrGDL* p)
    {
        SizeT nEl=p->N_Elements();
        for( SizeT ix=0; ix < nEl; ix++)
        {
            DPtr id= (*p)[ix];
            FreeHeap( id);
       }
    }

    class HeapException {};

    static BaseGDL*& GetHeap( DPtr ID)
    {
        HeapT::iterator it=heap.find( ID);
        if( it == heap.end()) throw HeapException();
        return it->second;
    }
    static DStructGDL*& GetObjHeap( DObj ID)
    {
        ObjHeapT::iterator it=objHeap.find( ID);
        if( it == objHeap.end()) throw HeapException();
        return it->second;
    }

    static bool PtrValid( DPtr ID)
    {
        HeapT::iterator it=heap.find( ID);
        return  (it != heap.end());
    }

    static SizeT HeapSize()
    {
        return heap.size();
    }

    static DPtrGDL* GetAllHeap()
    {
        SizeT nEl = heap.size();
        if( nEl == 0) return new DPtrGDL( 0);
        DPtrGDL* ret = new DPtrGDL( dimension( &nEl, 1), BaseGDL::NOZERO);
        SizeT i=0;
        for( HeapT::iterator it=heap.begin(); it != heap.end(); ++it)
        {
            (*ret)[ i++] = it->first;
        }
        return ret;
    }

    static bool ObjValid( DObj ID)
    {
        ObjHeapT::iterator it=objHeap.find( ID);
        return  (it != objHeap.end());
    }

    static SizeT ObjHeapSize()
    {
        return objHeap.size();
    }

    static DObjGDL* GetAllObjHeap()
    {
        SizeT nEl = objHeap.size();
        if( nEl == 0) return new DObjGDL( 0);
        DObjGDL* ret = new DObjGDL( dimension( &nEl, 1), BaseGDL::NOZERO);
        SizeT i=0;
        for( ObjHeapT::iterator it=objHeap.begin(); it != objHeap.end(); ++it)
        {
            (*ret)[ i++] = it->first;
        }
        return ret;
    }

    // name of data
    static const std::string Name( BaseGDL* p) // const
    {
        return callStack.back()->GetString( p);
    }

    static const std::string Name( BaseGDL** p) // const
    {
        return "<(Find name not implemented yet)>";
    }

    // compiler (lexer, parser, treeparser) def in dinterpreter.cpp
    static void ReportCompileError( GDLException& e, const std::string& file = "");

    // interpreter
    static void ReportError( GDLException& e, const std::string emsg, 
                             bool dumpStack=true)
    {
        DString msgPrefix = SysVar::MsgPrefix();

        std::cout << std::flush;
        if( dumpStack)
        if( e.Prefix())
        {
            std::cerr << msgPrefix << e.toString() << std::endl;
            lib::write_journal_comment(msgPrefix+e.toString());
        }
        else
        {
            std::cerr << e.toString() << std::endl;
            lib::write_journal_comment(e.toString());
        }

        std::cerr << msgPrefix << emsg << " " << 
        std::left << std::setw(16) << callStack.back()->GetProName();
        std::string file=callStack.back()->GetFilename();
        if( file != "")
        {
            SizeT line = e.getLine();
            if( line != 0)
            {       
                std::cerr << std::right << std::setw(6) << line;
            }
            else
            {
                std::cerr << std::right << std::setw(6) << "";
            }
            std::cerr << std::left << " " << file;
        }
        std::cerr << std::endl;
        
        if( dumpStack) DumpStack( emsg.size() + 1);
    }
    
    static void DumpStack( SizeT w)
    {
        DString msgPrefix = SysVar::MsgPrefix();

        EnvStackT::reverse_iterator upEnv = callStack.rbegin();
        EnvStackT::reverse_iterator env = upEnv++;
        for(; 
            upEnv != callStack.rend();
            ++upEnv, ++env)
        {
            std::cerr << msgPrefix << std::right << std::setw( w) << "";
            std::cerr << std::left << std::setw(16) << (*upEnv)->GetProName();

            std::string file = (*upEnv)->GetFilename();
            if( file != "")
            {
                ProgNodeP cNode= (*env)->CallingNode();
                if( cNode != NULL)
                {       
                    std::cerr << std::right << std::setw(6) << cNode->getLine();
                }
                else
                {
                    std::cerr << std::right << std::setw(6) << "";
                }
                std::cerr << std::left << " " << file;
            }
            std::cerr << std::endl;
        }
    }

    static void DebugMsg( ProgNodeP _t, const std::string& msg)
    {    
        DString msgPrefix = SysVar::MsgPrefix();

        std::cout << std::flush;
        std::cerr << msgPrefix << msg
        << std::left << std::setw(16) << callStack.back()->GetProName();
        std::string file=callStack.back()->GetFilename();
        if( file != "")
        {
            ProgNodeP eNode = _t;
            if( eNode != NULL)
            {       
                std::cerr << std::right << std::setw(6) << eNode->getLine();
            }
            else
            {
                std::cerr << std::right << std::setw(6) << "";
            }
            std::cerr << std::left << " " << file;
        }
        std::cerr << std::endl;
    }

    static void RetAll( RetAllException::ExCode c=RetAllException::NONE)    
    {
        throw RetAllException( c);
    }

    static EnvStackT& CallStack() { return callStack;} // the callstack
    static EnvBaseT*  CallStackBack() { return callStack.back();} 
    
    std::string GetClearActualLine()
    {
        std::string ret = executeLine.str();
        executeLine.str("");
        return ret;
    }

    RetCode NewInterpreterInstance(); // code in dinterpreter.cpp

    ~GDLInterpreter()
    {
    }
}

//***********************************************************************
// interpreter functions ************************************************
//***********************************************************************

// intercative usage
interactive returns[ GDLInterpreter::RetCode retCode]
    : retCode=statement_list
    ;

// execute statement
execute returns[ GDLInterpreter::RetCode retCode]
{
//    GDLInterpreter::RetCode retCode;
    ValueGuard<bool> guard( interruptEnable);
    interruptEnable = false;
}
    : retCode=statement_list
    ;

// used to call functions
// same as statement list, but different behaviour for returncodes
call_fun returns[ BaseGDL* res]
{

    res = NULL;
    returnValue = NULL;
    GDLInterpreter::RetCode retCode;
}
    : (retCode=statement
            {
                if( retCode == RC_RETURN) 
                {
                    res=returnValue;
                    returnValue=NULL;
                    
                    break;
                }
            }
        )*
        {
            // default return value if none was set
            if( res == NULL) res = new DIntGDL( 0); 
        }
    ;

call_lfun returns[ BaseGDL** res]
{
    res = NULL;
    returnValueL = NULL;
    GDLInterpreter::RetCode retCode;
}
    : (retCode=statement
            {
                if( retCode == RC_RETURN) 
                {
                    res=returnValueL;
                    returnValueL=NULL;
                    break;
                }
            }
        )*
        {
            // default return value if none was set
            if( res == NULL)
            throw GDLException( call_lfun_AST_in, "Function "+
                callStack.back()->GetProName()+
                " must return a left-value in this context.");
        }
    ;

// used to call procedures
call_pro
{
    GDLInterpreter::RetCode retCode;
}
    : (retCode=statement
            {
                // added RC_ABORT here
                if( retCode >= RC_RETURN) break;
            }
        )*
    ;

// used on many occasions
statement_list returns[ GDLInterpreter::RetCode retCode]
    : (retCode=statement
            {
                if( retCode != RC_OK) break; // break out if non-regular
            }
        )+
    ;

statement returns[ GDLInterpreter::RetCode retCode]
{
    retCode = RC_OK;
    ProgNodeP actPos = _t;
}
	: (
            // note: assignment must take care to update the owner of the lvalue
            // a real copy must be performed (creating a new BaseGDL)  
            assignment
        |   procedure_call
            //        |   lib_procedure_call
        |   decinc_statement
        |   retCode=for_statement 
        |   retCode=repeat_statement
        |   retCode=while_statement
        |   retCode=if_statement
        |   retCode=if_else_statement
        |   retCode=case_statement
        |   retCode=switch_statement
        |   retCode=block
        |   retCode=jump_statement
        |   LABEL
        |   ON_IOERROR_NULL
            {
                static_cast<EnvUDT*>(callStack.back())->SetIOError( -1);
            }
        |   o:ON_IOERROR
            {
                static_cast<EnvUDT*>(callStack.back())->
                    SetIOError( o->targetIx);
            }
        )
        // control c and debugging
        {
           // possible optimization: make sigControlC a debugMode 
           if( interruptEnable && sigControlC)
            {
                DebugMsg( actPos, "Interrupted at: "); 

                sigControlC = false;

                retCode = NewInterpreterInstance();
            }
            else if( debugMode != DEBUG_CLEAR)
            {
                if( debugMode == DEBUG_STOP)
                {
                    DebugMsg( actPos, "Stop encoutered: ");
                    if( !interruptEnable)
                        debugMode = DEBUG_PROCESS_STOP;
                }

                if( interruptEnable)
                {
                    if( debugMode == DEBUG_PROCESS_STOP)
                    {
                        DebugMsg( actPos, "Stepped to: ");
                    }

                    debugMode = DEBUG_CLEAR;
                
                    retCode = NewInterpreterInstance();
                }   
                else
                {
                    retCode = RC_ABORT;
                }
            }
        }
	;
    exception 
    catch [ GDLException& e] 
    { 
        if( dynamic_cast< GDLIOException*>( &e) != NULL)
            {
                // set the jump target - also logs the jump
                ProgNodeP onIOErr = static_cast<EnvUDT*>(callStack.back())->GetIOError();
                if( onIOErr != NULL)
                    {
                        _t = onIOErr;
                        retCode=RC_OK;		

                        _retTree = _t;
                        return retCode;
                    }
            }

        EnvUDT* targetEnv = e.GetTargetEnv();
        if( targetEnv == NULL)
        {
            // initial exception, set target env
            // look if ON_ERROR is set somewhere
            for( EnvStackT::reverse_iterator i = callStack.rbegin();
                i != callStack.rend(); ++i)
            {
                DLong oE = static_cast<EnvUDT*>(*i)->GetOnError();
                
                if( oE != -1) 
                { // oE was set
                    
                    // 0 -> stop here
                    if( oE == 0) 
                    targetEnv = static_cast<EnvUDT*>(callStack.back()); 
                    // 1 -> $MAIN$
                    else if( oE == 1) 
                    {
                        EnvUDT* cS_begin = 
                        static_cast<EnvUDT*>(*callStack.begin());
                        targetEnv = cS_begin;  
                    }
                    // 2 -> caller of routine which called ON_ERROR
                    else if( oE == 2)
                    {
                        ++i; // set to caller
                        if( i == callStack.rend())
                        {
                            EnvUDT* cS_begin = 
                            static_cast<EnvUDT*>(*callStack.begin());
                            targetEnv = cS_begin;
                        }
                        else
                        {
                            EnvUDT* iUDT = static_cast<EnvUDT*>(*i);
                            targetEnv = iUDT;
                        }
                    }   
                    // 3 -> routine which called ON_ERROR
                    else if( oE == 3)
                    {
                        EnvUDT* iUDT = static_cast<EnvUDT*>(*i);
                        targetEnv = iUDT;
                    }
                    
                    // remeber where to stop
                    e.SetTargetEnv( targetEnv);
                    
                    // State where error occured
                    if( e.getLine() == 0 && _t != NULL)
                        e.SetLine( _t->getLine());
                    
                    ReportError(e, "Error occurred at:");
                    
                    // break on first occurence of set oE
                    break;
                }
            }
        }
        
        if( targetEnv != NULL && targetEnv != callStack.back())
        {
            throw e; // rethrow
        }
        lib::write_journal( GetClearActualLine());

        // many low level routines don't have errorNode info
        // set line number here in this case
        if( e.getLine() == 0 && _t != NULL)
        {
            e.SetLine( _t->getLine());
        }

        if( interruptEnable)
            {
                // tell where we are
                ReportError(e, "Execution halted at:", targetEnv == NULL); 

                retCode = NewInterpreterInstance();
            }    
        else
            {

                DString msgPrefix = SysVar::MsgPrefix();
                if( e.Prefix())
                    {
                        std::cerr << msgPrefix << e.toString() << std::endl;
                        lib::write_journal_comment(msgPrefix+e.toString());
                    }
                else
                    {
                        std::cerr << e.toString() << std::endl;
                        lib::write_journal_comment(e.toString());
                    }

                retCode = RC_ABORT;
            }
    }

block returns[ GDLInterpreter::RetCode retCode]
{
    retCode = RC_OK;
}
	: #(BLOCK (retCode=statement_list)?)
	;

switch_statement returns[ GDLInterpreter::RetCode retCode]
{
    BaseGDL* e;
    retCode = RC_OK; // not set if no branch is executed
}
	: #(s:SWITCH e=expr 
            {
                auto_ptr<BaseGDL> e_guard(e);
                
                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                ProgNodeP b=_t; // remeber block begin (block)

                bool hook=false; // switch executes everything after 1st match
                for( int i=0; i<s->numBranch; i++)
                {
                    if( b->getType() == ELSEBLK)
                    {
                        hook=true;

                        ProgNodeP sL = b->GetFirstChild(); // statement_list

                        if( sL != NULL )
                        {
                            // statement there
                            retCode=statement_list( sL);
                            if( retCode == RC_BREAK) 
                            {
                                retCode = RC_OK;    
                                break;          // break
                            }
                            if( retCode >= RC_RETURN) break; // goto

                            if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                                !s->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                            {
                                // a jump (goto) occured out of this loop
                                return retCode;
                            }

                        }
                    }
                    else
                    {
                        ProgNodeP ex = b->GetFirstChild();  // EXPR
                        ProgNodeP bb = ex->GetNextSibling(); // statement_list

                        if( !hook)
                        {
//                            RefDNode ee_ = _t->GetFirstChild(); // expr

                            BaseGDL* ee=expr(ex);
                            // auto_ptr<BaseGDL> ee_guard(ee);

                            hook=e->Equal(ee); // Equal deletes ee
                        }
                            
                        if(bb != NULL && hook)
                        {
                            // statement there
                            retCode=statement_list(bb);
                            if( retCode == RC_BREAK) 
                            {
                                retCode = RC_OK;    
                                break;          // break
                            }
                            if( retCode >= RC_RETURN) break; // goto

                            if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                                !s->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                            {
                                // a jump (goto) occured out of this loop
                                return retCode;
                            }
                        }
                        
                    }
                    b=b->GetNextSibling(); // next block
                }
                // finish or break
//                retCode=RC_OK; // clear RC_BREAK retCode
            }
        )
    ;

case_statement returns[ GDLInterpreter::RetCode retCode]
{
    BaseGDL* e;
    retCode = RC_OK; // not set if no branch is executed
}
	: #(c:CASE e=expr
            {
                auto_ptr<BaseGDL> e_guard(e);

                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                if( !e->Scalar())
                throw GDLException( _t, "Expression must be a"
                    " scalar in this context: "+Name(e));

                ProgNodeP b=_t; // remeber block begin

                for( int i=0; i<c->numBranch; i++)
                {
                    if( b->getType() == ELSEBLK)
                    {
                        ProgNodeP sL = b->GetFirstChild(); // statement_list

                        if(sL != NULL )
                        {
                            // statement there
                            retCode=statement_list(sL);
                            //if( retCode == RC_BREAK) break; // break anyway
//                            if( retCode >= RC_RETURN) return retCode; 
                            if( retCode >= RC_RETURN) break;
                            
                            if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                                !c->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                            {
                                // a jump (goto) occured out of this loop
                                return retCode;
                            }

                        }
                        retCode = RC_OK;
                        break;
                    }
                    else
                    {
                        ProgNodeP ex = b->GetFirstChild();  // EXPR
                        ProgNodeP bb = ex->GetNextSibling(); // statement_list

                        BaseGDL* ee=expr(ex);
                        // auto_ptr<BaseGDL> ee_guard(ee);

                        bool equalexpr=e->Equal(ee); // Equal deletes ee

                        if( equalexpr)
                        {
                            if(bb != NULL)
                            {
                                // statement there
                                retCode=statement_list(bb);
                                //if( retCode == RC_BREAK) break; // break anyway
//                                if( retCode >= RC_RETURN) return retCode;
                                if( retCode >= RC_RETURN) break;

                                if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                                    !c->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                                {
                                    // a jump (goto) occured out of this loop
                                    return retCode;
                                }

                            }
                            retCode = RC_OK;
                            break;
                        }
                        
                    }
                    b=b->GetNextSibling(); // next block
                }
                // finish or break
//                retCode=RC_OK; // clear RC_BREAK retCode
            }
        )
	;

repeat_statement returns[ GDLInterpreter::RetCode retCode]
	: #(r:REPEAT // block expr
            {
                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                // remember block and expr nodes
                ProgNodeP e =_t;
                ProgNodeP bb  = e->GetNextSibling();

//                 ProgNodeP bb =_t;
//                 ProgNodeP e  = bb->GetNextSibling();
//                 bb = bb->GetFirstChild();
                
                auto_ptr<BaseGDL> eVal;
                do {
                    if( bb != NULL)
                    {
                    retCode=statement_list(bb);

                    if( retCode == RC_CONTINUE) continue;  
                    if( retCode == RC_BREAK) 
                    {
                        retCode = RC_OK;
                        break;        
                    }
                    if( retCode >= RC_RETURN) break;
                    // if( retCode == RC_BREAK) break;        
                    // if( retCode >= RC_RETURN) return retCode;

                    if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                        !r->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                    {
                        // a jump (goto) occured out of this loop
                        return retCode;
                    }
                    }
                    eVal.reset( expr(e));
                } while( eVal.get()->False());
                
                // retCode=RC_OK; // clear RC_BREAK/RC_CONTINUE retCode
            }
        )
	;

while_statement returns[ GDLInterpreter::RetCode retCode]
{
    retCode = RC_OK;
}
	: #(w:WHILE // statement expr 
            {
                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                ProgNodeP s = _t; //->GetFirstChild();  // statement
                ProgNodeP e =  s->GetNextSibling(); // expr

                auto_ptr< BaseGDL> eVal( expr( e));
                while( eVal.get()->True()) {
                    retCode=statement(s);

                    if( retCode == RC_CONTINUE) continue;  
                    if( retCode == RC_BREAK) 
                    {
                        retCode = RC_OK;
                        break;        
                    }
                    if( retCode >= RC_RETURN) break;
                    
                    if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                        !w->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                    {
                        // a jump (goto) occured out of this loop
                        return retCode;
                    }

                    eVal.reset( expr( e));
                } 

                // retCode=RC_OK; // clear RC_BREAK/RC_CONTINUE retCode
            }
        )
	;

for_statement returns[ GDLInterpreter::RetCode retCode]
{
    BaseGDL** v;
    BaseGDL* s;
    BaseGDL* e;
    BaseGDL* st;
    retCode = RC_OK;
}
    : #(f:FOR // (VAR|VARPTR) expr expr 
            {
                ProgNodeP sv = _t;
            }
            v=l_simple_var
            s=expr e=expr
            {
                auto_ptr<BaseGDL> s_guard(s);
                auto_ptr<BaseGDL> e_guard(e);

                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());
                SizeT nJump = callStack_back->NJump();

                s->ForCheck( &e);
                e_guard.release();
                e_guard.reset(e);

                ProgNodeP b=_t; //->getFirstChild();
                
                // ASSIGNMENT used here also
                delete (*v);

// problem:
// EXECUTE may call DataListT.loc.resize(), as v points to the
// old sequence v might be invalidated -> segfault
// note that the value (*v) is preserved by resize()
                s_guard.release(); // s hold in *v after this
                for((*v)=s; (*v)->ForCondUp( e); 
                    v=l_simple_var( sv), (*v)->ForAdd()) 
                {
//                    retCode=block(b);
                    if( b != NULL)
                    {
                        retCode=statement_list(b);
                    
                        if( retCode != RC_OK) // optimization
                        {
                            if( retCode == RC_CONTINUE) continue;  
                            if( retCode == RC_BREAK) 
                            {
                                retCode = RC_OK;
                                break;        
                            }
                            if( retCode >= RC_RETURN) break;
                        }

                        if( (callStack_back->NJump() != nJump) &&
                            !f->LabelInRange( callStack_back->LastJump()))
                        {
                            // a jump (goto) occured out of this loop
                            return retCode;
                        }
                    }
                }
//                retCode=RC_OK; // clear RC_BREAK/RC_CONTINUE retCode
            }
        )
    | #(fs:FOR_STEP // (VAR|VARPTR) expr expr expr 
            {
                ProgNodeP sv = _t;
            }
            v=l_simple_var
            s=expr e=expr st=expr
            {
                auto_ptr<BaseGDL> s_guard(s);
                auto_ptr<BaseGDL> e_guard(e);
                auto_ptr<BaseGDL> st_guard(st);

                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                s->ForCheck( &e, &st);
                e_guard.release();
                e_guard.reset(e);
                st_guard.release();
                st_guard.reset(st);
                
                ProgNodeP bs=_t;
                
                // ASSIGNMENT used here also
                delete (*v);
                
                if( st->Sgn() == -1) 
                {
                    s_guard.release();
                    for((*v)=s; (*v)->ForCondDown( e); 
                        v=l_simple_var( sv), (*v)->ForAdd(st))
                    {
                        if( bs != NULL)
                        {
                            retCode=statement_list(bs);
                            
                            if( retCode == RC_CONTINUE) continue;  
                            if( retCode == RC_BREAK) 
                            {
                                retCode = RC_OK;
                                break;        
                            }
                            if( retCode >= RC_RETURN) break;
                            
                            if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                                !fs->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                            {
                                // a jump (goto) occured out of this loop
                                return retCode;
                            }
                        }
                    }
                } 
                else
                {
                    s_guard.release();
                    for((*v)=s; (*v)->ForCondUp( e);
                        v=l_simple_var( sv), (*v)->ForAdd(st))
                        {
                        if( bs != NULL)
                        {
                            retCode=statement_list(bs);
                        
                            if( retCode == RC_CONTINUE) continue;  
                            if( retCode == RC_BREAK) 
                            {
                                retCode = RC_OK;
                                break;        
                            }
                            if( retCode >= RC_RETURN) break;
                            
                            if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                                !fs->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                            {
                                // a jump (goto) occured out of this loop
                                return retCode;
                            }
                        }
                    }
                }
            }
        )
	;

if_statement returns[ GDLInterpreter::RetCode retCode]
{
    BaseGDL* e;
    retCode = RC_OK; // not set if not executed
}
	: #(i:IF e=expr
            { 
                auto_ptr<BaseGDL> e_guard(e);

                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                if( e->True())
                {
                    retCode=statement(_t);
//                    if( retCode != RC_OK) return retCode;

                        if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                            !i->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                        {
                            // a jump (goto) occured out of this loop
                            return retCode;
                        }
                }
            }
        )
	;   

if_else_statement returns[ GDLInterpreter::RetCode retCode]
{
    BaseGDL* e;
}
	: #(i:IF_ELSE e=expr
            { 
                auto_ptr<BaseGDL> e_guard(e);

                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                if( e->True())
                {
                    retCode=statement(_t);
//                    if( retCode != RC_OK) return retCode;

                    if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                        !i->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                    {
                        // a jump (goto) occured out of this loop
                        return retCode;
                    }
                }
                else
                {
                    _t=_t->GetNextSibling(); // jump over 1st statement
                    retCode=statement(_t);
//                    if( retCode != RC_OK) return retCode;

                    if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
                        !i->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
                    {
                        // a jump (goto) occured out of this loop
                        return retCode;
                    }
                }
            }
        )
	;   

// to be processed (to make sure it really returns)
jump_statement returns[ GDLInterpreter::RetCode retCode]
{
BaseGDL*  e;
BaseGDL** eL;
}
    : g:GOTO // target in g->gotoTarget; // pointer to an DNode (NO RefDNode)
        {
            // note that this version jumps 'dumb'
            // jumping into loops is legal, even then looping is not done

            // set the jump target - also logs the jump
            _t = static_cast<EnvUDT*>(callStack.back())->GotoTarget( g->targetIx);
            _t = _t->GetNextSibling();
            retCode=RC_OK;
        }
    | #(RETF ( { !static_cast<EnvUDT*>(callStack.back())->LFun()}? e=expr // expr -> r value
                {
                    delete returnValue;
                    returnValue=e;
                    retCode=RC_RETURN;
                    callStack.back()->RemoveLoc( e); // steal e from local list
                }
            | eL=l_ret_expr
                {
                    // returnValueL is otherwise owned
                    returnValueL=eL;
                    retCode=RC_RETURN;
                }
            )
        ) 
    | RETP
        {
            retCode=RC_RETURN;
        }
	| BREAK    // only in loops or switch_statement and case_statement
        {
            retCode=RC_BREAK;
        }
	| CONTINUE // only in loops
        {
            retCode=RC_CONTINUE;
        }
  ;

procedure_call
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
    BaseGDL *self;
    EnvUDT*   newEnv;
}
	: #(PCALL_LIB pl:IDENTIFIER
            {
                EnvT* newEnv=new EnvT( pl, pl->libPro);//libProList[pl->proIx]);
            }
            parameter_def[ newEnv]
            {
                // push environment onto call stack
                callStack.push_back(newEnv);

                // make the call
                static_cast<DLibPro*>(newEnv->GetPro())->Pro()(newEnv);
            }   
        )
    |
        (
        ( #(MPCALL 
                self=expr mp:IDENTIFIER
                {  
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( mp, self);

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )
        | #(MPCALL_PARENT 
                self=expr parent:IDENTIFIER pp:IDENTIFIER
                {
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv = new EnvUDT( pp, self, parent->getText());

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )
        | #(PCALL p:IDENTIFIER
            {
                SetProIx( p);
            
                newEnv = new EnvUDT( p, proList[p->proIx]);
            }
            parameter_def[ newEnv]
            )
        )
        {
            // push environment onto call stack
            callStack.push_back(newEnv);
            
            // make the call
            call_pro(static_cast<DSubUD*>(newEnv->GetPro())->GetTree());
        }   
        )
	;	

assignment
{
    BaseGDL*  r;
    BaseGDL** l;
    auto_ptr<BaseGDL> r_guard;
}
    : #(ASSIGN 
//             ( r=tmp_expr
//                 {
//                     r_guard.reset( r);
//                 }
//             | r=check_expr
//                 {
//                     if( !callStack.back()->Contains( r)) 
//                         r_guard.reset( r);
//                 }
//             )
            ( r=indexable_expr
            | r=indexable_tmp_expr { r_guard.reset( r);}
            | r=check_expr
                {
                    if( !callStack.back()->Contains( r)) 
                        r_guard.reset( r); // guard if no global data
                }
            )

            l=l_expr[ r]
//             {
//                 // no delete if assigned to itself
//                 // only possible from lib function
//                 if( (*l) == r || callStack.back()->Contains( r)) 
//                     r_guard.release();
//             }
        )
//     | #(ASSIGN_INPLACE // ;=, *=, ...
//             {
//                  ProgNodeP op = _t;
//                 _t = _t->getNextSibling();
//             }
//             ( r=indexable_expr
//             | r=indexable_tmp_expr { r_guard.reset( r);}
//             | r=check_expr
//                 {
//                     if( !callStack.back()->Contains( r)) 
//                         r_guard.reset( r); // guard if no global data
//                 }
//             )
// // don't forget ASSIGN_EXPR!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//             l=l_inplace_expr[ op, r]
//         )
    | #(ASSIGN_REPLACE 
            ( r=tmp_expr
                {
                    r_guard.reset( r);
                }
            | r=check_expr
                {

                    if( !callStack.back()->Contains( r)) 
                        r_guard.reset( r);
                }
            )
            (
              l=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
            | l=l_deref           // DEREF
            | l=l_simple_var      // VAR, VARPTR
            )
        {
            if( r != (*l))
            {
                delete *l;

                if( r_guard.get() == r)
                  *l = r_guard.release();
                else  
                  *l = r->Dup();
            }
        }
        )
    ;

decinc_statement
{
    BaseGDL* l;
}
    : #(DEC l=l_decinc_expr[ DECSTATEMENT])
    | #(INC l=l_decinc_expr[ INCSTATEMENT])
    ;

// ***************************************************************************
// the expressions ***********************************************************
// ***************************************************************************

l_deref returns [BaseGDL** res]
{
    BaseGDL*       e1;
}
    : #(DEREF e1=expr 
            {
                auto_ptr<BaseGDL> e1_guard(e1);
                
                DPtrGDL* ptr=dynamic_cast<DPtrGDL*>(e1);
                if( ptr == NULL)
                throw GDLException( _t, "Pointer type required"
                    " in this context: "+Name(e1));
                DPtr sc; 
                if( !ptr->Scalar(sc))
                throw GDLException( _t, "Expression must be a "
                    "scalar in this context: "+Name(e1));
                if( sc == 0)
                throw GDLException( _t, "Unable to dereference"
                    " NULL pointer: "+Name(e1));
                
                try{
                    res = &GetHeap(sc);
                }
                catch( HeapException)
                {
                    throw GDLException( _t, "Invalid pointer: "+Name(e1));
                }
            }
        )
    ;

// return value from functions when used as l var
// used only from jump_statement and within itself
l_ret_expr returns [BaseGDL** res]
{
    BaseGDL*       e1;
}
    : res=l_deref
    | #(QUESTION e1=expr
            { 
                auto_ptr<BaseGDL> e1_guard(e1);
                if( e1->True())
                {
                    res=l_ret_expr(_t);
                }
                else
                {
                    _t=_t->GetNextSibling(); // jump over 1st expression
                    res=l_ret_expr(_t);
                }
            }
        ) // trinary operator
//    | #(EXPR res=l_ret_expr) // does not exist anymore
    | res=l_function_call 
        { // here a local to the actual environment could be returned
            if( callStack.back()->IsLocalKW( res))
            throw GDLException( _t, 
                "Attempt to return indirectly a local variable "
                "from left-function.");
        }
    | varPtr:VARPTR // DNode.var   is ptr to common block variable
        {
            res=&varPtr->var->Data(); // returns BaseGDL* of var (DVar*) 
        }
    | var:VAR // DNode.varIx is index into functions/procedures environment
        {     // check if variable is non-local 
              // (because it will be invalid after return otherwise)
            if( !callStack.back()->GlobalKW(var->varIx))
            throw GDLException( _t, 
                "Attempt to return a non-global variable from left-function.");
            
            res=&callStack.back()->GetKW(var->varIx); 
        }
    | // here ASSIGN and ASSIGN_REPLACE are identical
      #(ASSIGN // can it occur at all?
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( e1=tmp_expr
                {
                    r_guard.reset( e1);
                }
            | e1=check_expr
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1);
                }
            )
            res=l_ret_expr
            {
                if( e1 != (*res))
                    {
                    delete *res;
                    *res = e1;
                    }
                r_guard.release();
            }
        )
    | #(ASSIGN_REPLACE 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( e1=tmp_expr
                {
                    r_guard.reset( e1);
                }
            | e1=check_expr
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1);
                }
            )
            res=l_ret_expr
            {
                if( e1 != (*res))
                    {
                    delete *res;
                    *res = e1;
                    }
                r_guard.release();
            }
        )

    // the following are forbiden    
    | #(ARRAYEXPR
            {
                throw GDLException( _t, 
                    "Indexed expression not allowed as left-function"
                    " return value.");
            }
        )
    | #(DOT 
            {
                throw GDLException( _t, 
                    "Struct expression not allowed as left-function"
                    " return value.");
            }
        )
    | SYSVAR
        {
            throw GDLException( _t, 
                "System variable not allowed as left-function"
                " return value.");
        }
    | e1=r_expr
        {
            delete e1;
            throw GDLException( _t, 
                "Expression not allowed as left-function return value.");
        }
    | e1=constant_nocopy
        {
            throw GDLException( _t, 
                "Constant not allowed as left-function return value.");
        }
    ;

// l expressions for DEC/INC ********************************
// called from l_decinc_array_expr
l_decinc_indexable_expr [int dec_inc] returns [BaseGDL* res]
{
    BaseGDL** e;
}
//     : #(EXPR e = l_expr[ NULL])                       
//         {
//             res = *e;
//             if( res == NULL)
//             throw GDLException( _t, "Variable is undefined: "+Name(e));
//         }
//     | e=l_function_call
    : e=l_function_call
        {
            res = *e;
            if( res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(e));
        }
    | e=l_deref 
        {
            res = *e;
            if( res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(e));
        }
    | e=l_defined_simple_var { res = *e; } // no Dup here
    | e=l_sys_var { res = *e; }            // no Dup here
    ;

// called from l_decinc_expr
l_decinc_array_expr [int dec_inc] returns [BaseGDL* res]
{
    ArrayIndexListT* aL;
    BaseGDL*         e;
    ArrayIndexListGuard guard;

}
    : #(ARRAYEXPR 
            e=l_decinc_indexable_expr[ dec_inc]   
            aL=arrayindex_list)
        {
            guard.reset( aL); 
            aL->SetVariable( e);

            if( dec_inc == DECSTATEMENT) 
            {
                e->DecAt( aL); 
                res = NULL;
                break;
            }
            if( dec_inc == INCSTATEMENT)
            {
                e->IncAt( aL);
                res = NULL;
                break;
            }

            if( dec_inc == DEC) e->DecAt( aL); 
            else if( dec_inc == INC) e->IncAt( aL);
//
            res=e->Index( aL);

            if( dec_inc == POSTDEC) e->DecAt( aL);
            else if( dec_inc == POSTINC) e->IncAt( aL);
        }
    | e=l_decinc_indexable_expr[ dec_inc]
        {
            if( dec_inc == DECSTATEMENT) 
            {
                e->Dec(); 
                res = NULL;
                break;
            }
            if( dec_inc == INCSTATEMENT)
            {
                e->Inc();
                res = NULL;
                break;
            }

            if( dec_inc == DEC) e->Dec();
            else if( dec_inc == INC) e->Inc();
  //          
            res = e->Dup();
            
            if( dec_inc == POSTDEC) e->Dec();
            else if( dec_inc == POSTINC) e->Inc();
        }
    ;

// struct assignment
// MAIN function: called from l_decinc_expr
l_decinc_dot_expr [int dec_inc] returns [BaseGDL* res]
    : #(dot:DOT 
            { 
                SizeT nDot=dot->nDot;
                auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
            } 
            l_dot_array_expr[ aD.get()] 
            (tag_array_expr[ aD.get()] /* nDot times*/ )+ 
        )         
        {
            if( dec_inc == DECSTATEMENT) 
            {
                aD->Dec(); 
                res = NULL;
            }
            else if( dec_inc == INCSTATEMENT)
            {
                aD->Inc();
                res = NULL;
            }
            else
            {
                if( dec_inc == DEC) aD->Dec(); //*** aD->Assign( dec_inc);
                else if( dec_inc == INC) aD->Inc();
//                
                res=aD->Resolve();
                
                if( dec_inc == POSTDEC) aD->Dec();
                else if( dec_inc == POSTINC) aD->Inc();
            }
        }
    ;

// l_decinc_expr is only used in dec/inc statements and within itself
l_decinc_expr [int dec_inc] returns [BaseGDL* res]
{
    BaseGDL*       e1;
}
    : #(QUESTION e1=expr
            { 
                auto_ptr<BaseGDL> e1_guard(e1);

                if( e1->True())
                {
                    res=l_decinc_expr(_t, dec_inc);
                }
                else
                {
                    _t=_t->GetNextSibling(); // jump over 1st expression
                    res=l_decinc_expr(_t, dec_inc);
                }
            }
        ) // trinary operator
    | #(ASSIGN 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
//             ( e1=tmp_expr
//                 {
//                     r_guard.reset( e1);
//                 }
//             | e1=check_expr
//                 {
//                     if( !callStack.back()->Contains( e1)) 
//                         r_guard.reset( e1);
//                 }
//             )
            ( e1=indexable_expr
            | e1=indexable_tmp_expr { r_guard.reset( e1);}
            | e1=check_expr
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1); // guard if no global data
                }
            )
            { 
                ProgNodeP l = _t;

                BaseGDL** tmp;
            } 
            tmp=l_expr[ e1] // assign
            {
                _t = l;
            }
            res=l_decinc_expr[ dec_inc]
        )
    | #(ASSIGN_REPLACE 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( e1=tmp_expr
                {
                    r_guard.reset( e1);
                }
            | e1=check_expr
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1);
                }
            )
            { 
                ProgNodeP l = _t;

                BaseGDL** tmp;
            } 
//            tmp=l_expr[ e1] // assign
            (
              tmp=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
            | tmp=l_deref           // DEREF
            | tmp=l_simple_var      // VAR, VARPTR
            )
        {
            if( e1 != (*tmp))
            {
                delete *tmp;

                if( r_guard.get() == e1)
                  *tmp = r_guard.release();
                else  
                  *tmp = e1->Dup();
            }
        }
            {
                _t = l;
            }
            res=l_decinc_expr[ dec_inc]
        )
    | res=l_decinc_array_expr[ dec_inc]
    | res=l_decinc_dot_expr[ dec_inc]
    | e1=r_expr
        {
            delete e1;
            throw GDLException( _t, 
                "Expression not allowed with decrement/increment operator.");
        }
    | e1=constant_nocopy
        {
            throw GDLException( _t, 
                "Constant not allowed with decrement/increment operator.");
        }
    ;

// l expressions for assignment *************************

// an indexable expression must be defined
l_indexable_expr returns [BaseGDL** res]
    : #(EXPR res=l_expr[ NULL]) // for l_dot_array_expr
        {
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(res));
        }
    | res=l_function_call
        {
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(res));
        }
    | res=l_deref
        {
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(res));
        }
    | res=l_defined_simple_var                         
    | res=l_sys_var 
    ;

// called from l_expr
l_array_expr [BaseGDL* right] returns [BaseGDL** res]
{
    ArrayIndexListT* aL;
    ArrayIndexListGuard guard;
}
    : #(ARRAYEXPR res=l_indexable_expr aL=arrayindex_list { guard.reset(aL);})   
        {
            if( right == NULL)
            throw GDLException( _t, 
                "Indexed expression not allowed in this context.");

            aL->AssignAt( *res, right);

//             aL->SetVariable( *res);
            
//             if( (*res)->EqType( right))
//             {
//                 (*res)->AssignAt( right, aL); // assigns inplace
//             }
//             else
//             {
//                 BaseGDL* rConv = right->Convert2( (*res)->Type(), BaseGDL::COPY);
//                 auto_ptr<BaseGDL> conv_guard( rConv);
                
//                 (*res)->AssignAt( rConv, aL); // assigns inplace
//             }
        }
    ;

l_dot_array_expr [DotAccessDescT* aD] // 1st
{
    ArrayIndexListT* aL;
    BaseGDL**        rP;
    DStructGDL*      structR;
    ArrayIndexListGuard guard;
    bool isObj = callStack.back()->IsObject();
}
    : #(ARRAYEXPR rP=l_indexable_expr aL=arrayindex_list { guard.reset(aL);})   
        {
            // check here for object and get struct
            structR=dynamic_cast<DStructGDL*>(*rP);
            if( structR == NULL)
            {
                if( isObj)
                {
                    DStructGDL* oStruct = ObjectStructCheckAccess( *rP, _t);

                    // oStruct cannot be "Assoc_"
                    aD->Root( oStruct, guard.release()); 
                }
                else
                {
                    throw GDLException( _t, "Expression must be a"
                        " STRUCT in this context: "+Name(*rP));
                }
            }
            else 
            {
                if( (*rP)->IsAssoc())
                throw GDLException( _t, "File expression not allowed "
                    "in this context: "+Name(*rP));
                
                aD->Root( structR, guard.release() /* aL */); 
            }
        }
    | rP=l_indexable_expr
        {
            // check here for object and get struct
            structR = dynamic_cast<DStructGDL*>(*rP);
            if( structR == NULL)
            {
                if( isObj) // member access to object?
                {
                    DStructGDL* oStruct = ObjectStructCheckAccess( *rP, _t);

                    // oStruct cannot be "Assoc_"
                    aD->Root( oStruct); 
                }
                else
                {
                    throw GDLException( _t, "Expression must be a"
                        " STRUCT in this context: "+Name(*rP));
                }
            }
            else
            {
                if( (*rP)->IsAssoc())
                {
                    throw GDLException( _t, "File expression not allowed "
                        "in this context: "+Name(*rP));
                }
                
                aD->Root(structR); 
            }
        }
    ;


// l_expr is only used in assignment and within itself
l_expr [BaseGDL* right] returns [BaseGDL** res]
{
    BaseGDL*       e1;
}
    : #(QUESTION e1=expr
            { 
                auto_ptr<BaseGDL> e1_guard(e1);

                if( e1->True())
                {
                    res=l_expr(_t, right);
                }
                else
                {
                    _t=_t->GetNextSibling(); // jump over 1st expression
                    res=l_expr(_t, right);
                }
            }
        ) // trinary operator
    | #(ASSIGN //???e1=expr
//             { 
//                 auto_ptr<BaseGDL> r_guard;
//             } 
//             ( e1=tmp_expr
//                 {
//                     r_guard.reset( e1);
//                 }
//             | e1=check_expr
//                 {
//                     if( !callStack.back()->Contains( e1)) 
//                         r_guard.reset( e1);
//                 }
//             )
            ( e1=indexable_expr
            | e1=indexable_tmp_expr { delete e1;}
//            | e1=indexable_tmp_expr { r_guard.reset( e1);}
            | e1=check_expr
                {
                    if( !callStack.back()->Contains( e1)) 
                        delete e1; // guard if no global data
//                        r_guard.reset( e1); // guard if no global data
                }
            )

            res=l_expr[ right]
//             {
//                 if( (*res) == e1 || callStack.back()->Contains( e1)) 
//                     r_guard.release();
//             }
        )
    | #(ASSIGN_REPLACE //???e1=expr
//             { 
//                 auto_ptr<BaseGDL> r_guard;
//             } 
            ( e1=tmp_expr
                {
                    delete e1;
//                    r_guard.reset( e1);
                }
            | e1=check_expr
                {
                    if( !callStack.back()->Contains( e1)) 
                        delete e1;
//                        r_guard.reset( e1);
                }
            )
            (
              res=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
            | res=l_deref           // DEREF
            | res=l_simple_var      // VAR, VARPTR
            )
        {
            if( right != (*res))
//            if( e1 != (*res))
            {
                delete *res;
//
//                if( r_guard.get() == e1)
//                  *res = r_guard.release();
//                else  
                  *res = right->Dup();
            }
        }
        )
    | res=l_array_expr[ right]
    | { ProgNodeP sysVar = _t;} // for error reporting
        res=l_sys_var // sysvars cannot change their type
        {
            if( right == NULL)
            throw GDLException( _t, 
                "System variable not allowed in this context.");
            
           auto_ptr<BaseGDL> conv_guard; //( rConv);
           BaseGDL* rConv = right;
           if( !(*res)->EqType( right))
            {
                rConv = right->Convert2( (*res)->Type(), 
                                                  BaseGDL::COPY);
                conv_guard.reset( rConv);
            }
 
            if( right->N_Elements() != 1 && 
                ((*res)->N_Elements() != right->N_Elements()))
            {
                throw GDLException( _t, "Conflicting data structures: <"+
                    right->TypeStr()+" "+right->Dim().ToString()+">,!"+ 
                    sysVar->getText());
            }
            
            (*res)->AssignAt( rConv); // linear copy
        }
//   | res=l_indexoverwriteable_expr 
     | // can be called via QUESTION
       ( res=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
       | res=l_deref           // DEREF
       | res=l_simple_var      // VAR, VARPTR
       )
     {
            if( right != NULL && right != (*res))
            {
                delete *res;
                *res = right->Dup();
            }
     }
//         {
//             if( right != NULL && right != (*res))
//             {
//                 // only here non-inplace copy is done
//                 delete *res;
//                 *res = right->Dup();
//             }
//         }
//    | res=l_dot_expr[ right]
    | #(dot:DOT  // struct assignment
            { 
                SizeT nDot=dot->nDot;
                auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
            } 
            l_dot_array_expr[ aD.get()] 
            (tag_array_expr[ aD.get()] /* nDot times*/ )+ 
        )         
        {
            if( right == NULL)
            throw GDLException( _t, "Struct expression not allowed in this context.");
            
            aD->Assign( right);

            res=NULL;
        }
//    | { right == NULL}? res=l_function_call
    | e1=r_expr
        {
            delete e1;
            throw GDLException( _t, 
                "Expression not allowed as l-value.");
        }
    | e1=constant_nocopy
        {
            throw GDLException( _t, 
                "Constant not allowed as l-value.");
        }
    ;

// used in l_expr but also in parameter_def
l_simple_var returns [BaseGDL** res]
    : var:VAR // DNode.varIx is index into functions/procedures environment
        {

            res=&callStack.back()->GetKW(var->varIx); 
        }
    | varPtr:VARPTR // DNode.var   is ptr to common block variable
        {
            res=&varPtr->var->Data(); // returns BaseGDL* of var (DVar*) 
        }
    ;

l_defined_simple_var returns [BaseGDL** res]
    : var:VAR // DNode.varIx is index into functions/procedures environment
        {
            res=&callStack.back()->GetKW(var->varIx); 
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+
                callStack.back()->GetString(var->varIx));
        }
    | varPtr:VARPTR // DNode.var   is ptr to common block variable
        {
            res=&varPtr->var->Data(); // returns BaseGDL* of var (DVar*) 
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+
                callStack.back()->GetString( *res));
        }
    ;

l_sys_var returns [BaseGDL** res]
    : sysVar:SYSVAR  
        // ProgNodeP->getText() returns name which 
        // has to be searched 
        // in 'sysVarList' (objects.cpp) if ProgNodeP->var is NULL
        {
            if( sysVar->var == NULL) 
            {
                sysVar->var=FindInVarList(sysVarList,sysVar->getText());
                if( sysVar->var == NULL)		    
                throw GDLException( _t, "Not a legal system variable: !"+
                    sysVar->getText());

                // note: this works, because system variables are never 
                //       passed by reference
                SizeT rdOnlySize = sysVarRdOnlyList.size();
                for( SizeT i=0; i<rdOnlySize; ++i)
                  if( sysVarRdOnlyList[ i] == sysVar->var)
                    throw GDLException( _t, 
                    "Attempt to write to a readonly variable: !"+
                    sysVar->getText());
            }
            // system variables are always defined
            res=&sysVar->var->Data();
        }
    ;

// right expressions **********************************************************
// expressions which always only return a value
// expecting to delete any sub-expressions
r_expr returns [BaseGDL* res]
    : e:EXPR
        { res = e->Eval();}
// {
//     BaseGDL* e1;
//     BaseGDL* e2;
// }
//     :	#(NOT_OP e1=expr { res= e1->NotOp();} )
//     |	#(UMINUS e1=expr { res= e1->UMinus();} )
//     |	#(AND_OP e1=expr e2=expr)			// binary operators...
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->AndOp(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->AndOpInv(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->AndOpInv(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->AndOp(e1); // smaller + larger
//         }
//     |	#(OR_OP  e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->OrOp(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->OrOpInv(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->OrOpInv(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->OrOp(e1); // smaller + larger
//         }
//     |	#(XOR_OP e1=expr e2=expr) // xor is only defined for integers
//         {
//             AdjustTypes(e1,e2);
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->XorOp(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->XorOp(e1); // smaller + larger
//         }
//     |	#(EQ_OP e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             res=e1->EqOp(e2);
//         }
//     |	#(NE_OP e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             res=e1->NeOp(e2);
//         }
//     |	#(LE_OP e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             res=e1->LeOp(e2);
//         }
//     |	#(LT_OP e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             res=e1->LtOp(e2);
//         }
//     |	#(GE_OP e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             res=e1->GeOp(e2);
//         }
//     |	#(GT_OP e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             res=e1->GtOp(e2);
//         }
//     |	#(PLUS  e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->AddInv(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->Add(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->Add(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->AddInv(e1); // smaller + larger
//         }
//     |	#(MINUS e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->SubInv(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->Sub(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->Sub(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->SubInv(e1); // smaller + larger
//         }
//     |	#(LTMARK e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->LtMark(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->LtMark(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->LtMark(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->LtMark(e1); // smaller + larger
//         }
//     |	#(GTMARK e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->GtMark(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->GtMark(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->GtMark(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->GtMark(e1); // smaller + larger
//         }
//     |	#(ASTERIX e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->Mult(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->Mult(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->Mult(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->Mult(e1); // smaller + larger
//         }
//     |	#(MATRIX_OP1 e1=expr e2=expr) // #
//         {
//             DType aTy=e1->Type();
//             DType bTy=e2->Type();
//             DType maxTy=(DTypeOrder[aTy] >= DTypeOrder[bTy])? aTy: bTy;

//             DType cTy=maxTy;
//             if( maxTy == BYTE || maxTy == INT)
//             cTy=LONG;
//             else if( maxTy == UINT)
//             cTy=ULONG;

//             if( aTy != cTy) e1=e1->Convert2( cTy);

//             AdjustTypes(e1,e2);
//             res=e1->MatrixOp(e2);
//         }
//     |	#(MATRIX_OP2 e1=expr e2=expr) // ##
//         {
//             DType aTy=e1->Type();
//             DType bTy=e2->Type();
//             DType maxTy=(DTypeOrder[aTy] >= DTypeOrder[bTy])? aTy: bTy;

//             DType cTy=maxTy;
//             if( maxTy == BYTE || maxTy == INT)
//             cTy=LONG;
//             else if( maxTy == UINT)
//             cTy=ULONG;

//             if( aTy != cTy) e1=e1->Convert2( cTy);

//             AdjustTypes(e1,e2);
//             res=e2->MatrixOp(e1);
//         }
//     |	#(SLASH e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->DivInv(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->Div(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->Div(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->DivInv(e1); // smaller + larger
//         }
//     |	#(MOD_OP e1=expr e2=expr)
//         {
//             AdjustTypes(e1,e2);
//             if( e1->Scalar())
//             res= e2->ModInv(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->Mod(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->Mod(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->ModInv(e1); // smaller + larger
//         }
//     |	#(POW e1=expr e2=expr)
//         {
//            // special handling for complex
//             DType aTy=e1->Type();
//             if( aTy == COMPLEX)
//             {
//                 DType bTy=e2->Type();
//                 if( IntType( bTy))
//                 {
//                     e2 = e2->Convert2( FLOAT);
//                     res = e1->Pow( e2);
//                     goto endPOW;
//                 }
//                 else if( bTy == FLOAT)
//                 {
//                     res = e1->Pow( e2);
//                     goto endPOW;
//                 }
//             }
//             else if( aTy == COMPLEXDBL)
//             {
//                 DType bTy=e2->Type();
//                 if( IntType( bTy))
//                 {
//                     e2 = e2->Convert2( DOUBLE);
//                     res = e1->Pow( e2);
//                     goto endPOW;
//                 }
//                 else if( bTy == DOUBLE)
//                 {
//                     res = e1->Pow( e2);
//                     goto endPOW;
//                 }
//             }

//             DType convertBackT; 

//             // convert back
//             if( IntType( e2->Type()) && 
//                 DTypeOrder[e2->Type()] > DTypeOrder[e1->Type()])
//                 convertBackT = e1->Type();
//             else
//                 convertBackT = UNDEF;

//             AdjustTypes(e2,e1); // order crucial here (for converting back)

//             if( e1->Scalar())
//             res= e2->PowInv(e1); // scalar+scalar or array+scalar
//             else
//             if( e2->Scalar())
//             res= e1->Pow(e2); // array+scalar
//             else
//             if( e1->N_Elements() <= e2->N_Elements())
//             res= e1->Pow(e2); // smaller_array + larger_array or same size
//             else
//             res= e2->PowInv(e1); // smaller + larger
//             if( convertBackT != UNDEF)
//             {
//                 res = res->Convert2( convertBackT, BaseGDL::CONVERT);
//             }
//             endPOW:
//         }
    |	#(DEC res=l_decinc_expr[ DEC])
    |	#(INC res=l_decinc_expr[ INC])
    |	#(POSTDEC res=l_decinc_expr[ POSTDEC])
    |	#(POSTINC res=l_decinc_expr[ POSTINC])
//     // logical expressions
//     |   #(LOG_AND e1=expr e2=expr)
//         {
//             if( !e1->LogTrue()) {res = new DByteGDL( 0); break;}
//             if( !e2->LogTrue()) {res = new DByteGDL( 0); break;}
//             res = new DByteGDL( 1);
//         }
//     |   #(LOG_OR e1=expr e2=expr)
//         {
//             if( e1->LogTrue()) {res = new DByteGDL( 1); break;}
//             if( e2->LogTrue()) {res = new DByteGDL( 1); break;}
//             res = new DByteGDL( 0);
//         }
//     |   #(LOG_NEG e1=expr)
//         {
//             res = e1->LogNeg();
//         }
//    | res=constant                  
    | res=array_def
    | res=struct_def
    ;

array_expr returns [BaseGDL* res]
{
    ArrayIndexListT* aL;
    BaseGDL* r;
    ArrayIndexListGuard guard;
    auto_ptr<BaseGDL> r_guard;

    ExprListT        exprList; // for cleanup
    IxExprListT      ixExprList;
    SizeT nExpr;
    BaseGDL* s;
}
    : #(ARRAYEXPR 
            ( r=indexable_expr
            | r=indexable_tmp_expr { r_guard.reset( r);}
            | r=check_expr
                {
                    if( !callStack.back()->Contains( r)) 
                        r_guard.reset( r); // guard if no global data
                }
            )
            //            aL=indexing_list { guard.reset(aL);}
        #(ax:ARRAYIX
                {
                    aL = ax->arrIxList;
                    assert( aL != NULL);
                    
                    guard.reset(aL);
                    
                    nExpr = aL->NParam();

                    if( nExpr == 0)
                    {
                        goto empty;
                    }
                    //                 if( nExpr > 1)
                    //                 {
                    //                     ixExprList.reserve( nExpr);
                    //                     exprList.reserve( nExpr);
                    //                 }
                    //                if( nExpr == 0) goto empty;
                }
                (
                    ( s=indexable_expr
                    | s=check_expr
                        {
                            if( !callStack.back()->Contains( s)) 
                            exprList.push_back( s);
                        }
                    | s=indexable_tmp_expr { exprList.push_back( s);}
                    )
                    {
                        ixExprList.push_back( s);
                        if( ixExprList.size() == nExpr)
                        break; // allows some manual tuning
                    }
                )*
                //            { empty: ;}
            )
            {
empty:
                res = aL->Index( r, ixExprList);
//                 aL->Init( ixExprList);
//                 aL->SetVariable( r);
//                 res=r->Index( aL);
//                ClearTmpList();
            }
        )   
//     | res=expr //indexable_expr
    ;

// for l and r expr
tag_expr [DotAccessDescT* aD] // 2nd...
{
    BaseGDL* e;
}
    : #(EXPR e=expr
            {
                auto_ptr<BaseGDL> e_guard(e);
                
                SizeT tagIx;
                int ret=e->Scalar2index(tagIx);
                if( ret < 1)
                throw GDLException( _t, "Expression must be a scalar"
                    " >= 0 in this context: "+Name(e));
                
                aD->Add( tagIx);
            }
        )                       
    | i:IDENTIFIER
        {
            std::string tagName=i->getText();
            aD->Add( tagName);
        }
    ;

// for l and r expr
tag_array_expr  [DotAccessDescT* aD] // 2nd...
{
    ArrayIndexListT* aL;
}
	: #(ARRAYEXPR tag_expr[ aD] aL=arrayindex_list { aD->AddIx(aL);} )
    | tag_expr[ aD] { aD->AddIx(NULL);} 
    ;

r_dot_indexable_expr [DotAccessDescT* aD] returns [BaseGDL* res] // 1st
{
    BaseGDL** e;
}
    : #(EXPR res=expr { aD->SetOwner( true);}) // ({tag:0}).tag should work 
    | e=l_defined_simple_var { res = *e;}
    |   // we cant use l_sys_var here because of copy protection
        // could use sysvar and SetOwner( true), but this is quicker
        res=sys_var_nocopy // system variables are always defined    
    ;

r_dot_array_expr [DotAccessDescT* aD] // 1st
{
    ArrayIndexListT* aL;
    BaseGDL*         r;
    DStructGDL*      structR;
    ArrayIndexListGuard guard;
    bool isObj = callStack.back()->IsObject();
}
// NOTE: r is owned by aD or a l_... (r must not be deleted here)
    : #(ARRAYEXPR r=r_dot_indexable_expr[ aD] 
            aL=arrayindex_list { guard.reset(aL);} )   
        {
            // check here for object and get struct
            structR=dynamic_cast<DStructGDL*>(r);
            if( structR == NULL)
            {
                if( isObj)
                {
                    DStructGDL* oStruct = ObjectStructCheckAccess( r, _t);
                    
//                    DStructGDL* obj = oStruct->Index( aL);

                    if( aD->IsOwner()) delete r; 
                    aD->SetOwner( false); // object struct, not owned
                    
                    aD->Root( oStruct, guard.release()); 
//                    aD->Root( obj); 

//                     BaseGDL* obj = r->Index( aL);
//                     auto_ptr<BaseGDL> objGuard( obj); // new object -> guard
                    
//                     DStructGDL* oStruct = ObjectStructCheckAccess( obj, _t);

//                     // oStruct cannot be "Assoc_"
//                     if( aD->IsOwner()) delete r; 
//                     aD->SetOwner( false); // object structs are never owned
//                     aD->Root( oStruct); 
                }
                else
                {
                    throw GDLException( _t, "Expression must be a"
                        " STRUCT in this context: "+Name(r));
                }
            }
            else
            {
                if( r->IsAssoc())
                throw GDLException( _t, "File expression not allowed "
                    "in this context: "+Name(r));
                
                aD->Root( structR, guard.release()); 
            }
        }
    | r=r_dot_indexable_expr[ aD]
        {
            // check here for object and get struct
            structR = dynamic_cast<DStructGDL*>(r);
            if( structR == NULL)
            {
                if( isObj) // memeber access to object?
                {
                    DStructGDL* oStruct = ObjectStructCheckAccess( r, _t);

                    // oStruct cannot be "Assoc_"
                    if( aD->IsOwner()) delete r;
                    aD->SetOwner( false); // object structs are never owned
                    aD->Root( oStruct); 
                }
                else
                {
                    throw GDLException( _t, "Expression must be a"
                        " STRUCT in this context: "+Name(r));
                }
            }
            else
            {
                if( r->IsAssoc())
                {
                    throw GDLException( _t, "File expression not allowed "
                        "in this context: "+Name(r));
                }
                
                aD->Root(structR); 
            }
        }
    ;

// Entry function for struct access
//#(DOT array_expr (tag_array_expr)+)                     
dot_expr returns [BaseGDL* res]
    : #(dot:DOT 
            { 
                SizeT nDot=dot->nDot;
                auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
            } 
            r_dot_array_expr[ aD.get()] 
            (tag_array_expr[ aD.get()] /* nDot times*/ )+ 
        )         
        { res= aD->Resolve();}
    ;

// indexable expressions are used to minimize copying of data
// ( via Dup())
// owned by caller
indexable_tmp_expr returns [BaseGDL* res]
{
    BaseGDL*  e1;
}
	: #(q:QUESTION 
            { res = q->Eval(); }
//                 e1=expr
//             { 
//                 auto_ptr<BaseGDL> e1_guard(e1);

//                 if( e1->True())
//                 {   
//                     res=expr(_t);
//                 }
//                 else
//                 {
//                     _t=_t->GetNextSibling(); // jump over 1st expression
//                     res=expr(_t);
//                 }
//             }
        ) // trinary operator
    | res=array_expr
    | res=dot_expr
    | res=assign_expr
    | res=function_call
    | res=r_expr
    | res=lib_function_call_retnew 
    ;

// not owned by caller 
indexable_expr returns [BaseGDL* res]
{
    BaseGDL** e2;
}
    : e2=l_defined_simple_var
        {
            res = *e2;
        }
    | res=sys_var_nocopy
    | res=constant_nocopy
    | e2=l_deref 
        { 
            if( *e2 == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(e2));
            res = *e2;
        }
    ;

// l_expr used as r_expr and true r_expr
expr returns [BaseGDL* res]
    : res=tmp_expr
    | res=check_expr
        {
            if( callStack.back()->Contains( res)) 
                res = res->Dup();
        }
    ;

check_expr returns [BaseGDL* res]
    : res=lib_function_call 
    ;

// l_expr used as r_expr and true r_expr
tmp_expr returns [BaseGDL* res]
{
    BaseGDL*  e1;
    BaseGDL** e2;
}
    : e2=l_deref 
        { 
            if( *e2 == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(e2));
            
            res = (*e2)->Dup();
        }
	| #(q:QUESTION 
            { res = q->Eval();}
//                 e1=expr
//             { 
//                 auto_ptr<BaseGDL> e1_guard(e1);

//                 if( e1->True())
//                 {
//                     res=expr(_t);
//                 }
//                 else
//                 {
//                     _t=_t->GetNextSibling(); // jump over 1st expression
//                     res=expr(_t);
//                 }
//             }
        ) // trinary operator
    | res=array_expr
    | res=dot_expr
    | res=assign_expr
    | res=function_call
    | res=r_expr
    | res=constant
        // *********
    | res=simple_var                         
    | res=sys_var 
    | res=lib_function_call_retnew 
    ;

assign_expr returns [BaseGDL* res]
{
    BaseGDL** l;
    BaseGDL*  r;
}
    : #(ASSIGN 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( res=tmp_expr
                {
                    r_guard.reset( res);
                }
            | res=check_expr
                {
                    if( !callStack.back()->Contains( res)) 
                        r_guard.reset( res);
                }
            )
            l=l_expr[ res]
            { 
                if( r_guard.get() == res) // owner
                    r_guard.release();
                else
                    res = res->Dup();
            } // here res is returned!
        )
    | #(ASSIGN_REPLACE 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( res=tmp_expr
                {
                    r_guard.reset( res);
                }
            | res=check_expr
                {
                    if( !callStack.back()->Contains( res)) 
                        r_guard.reset( res);
                }
            )
            (
              l=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
            | l=l_deref           // DEREF
            | l=l_simple_var      // VAR, VARPTR
            )
            {
                if( res != (*l))
                {
                delete *l;
                *l = res->Dup();     

                if( r_guard.get() == res) // owner
                {
                    r_guard.release(); 
                }
                else
                    res = res->Dup();
                }
            }
        )
    ;

simple_var returns [BaseGDL* res]
    : var:VAR // DNode.varIx is index into functions/procedures environment
        {
            BaseGDL* vData=callStack.back()->GetKW( var->varIx);
            
            if( vData == NULL)
            throw GDLException( _t, "Variable is undefined: "+var->getText());
            
            res=vData->Dup();
        }
    | varPtr:VARPTR // DNode.var   is ptr to common block variable
        {
            BaseGDL* vData=varPtr->var->Data();
            
            if( vData == NULL)
            throw GDLException( _t, "Common block variable is undefined.");
            
            res=vData->Dup();
        }
    ;

sys_var returns [BaseGDL* res]
{
    BaseGDL* sv;
}
    : sv=sys_var_nocopy 
        {
            res=sv->Dup();
        }
    ;

sys_var_nocopy returns [BaseGDL* res]
    : sysVar:SYSVAR 
        // ProgNodeP->getText() returns name which has to be searched 
        // in 'sysVarList' (objects.cpp) if ProgNodeP->var is NULL
        {
            if( sysVar->var == NULL) 
            {
                sysVar->var=FindInVarList(sysVarList,sysVar->getText());
                if( sysVar->var == NULL)		    
                throw GDLException( _t, "Not a legal system variable: !"+
                                    sysVar->getText());
            }

            if( sysVar->getText() == "STIME") SysVar::UpdateSTime();

            // system variables are always defined
            res=sysVar->var->Data(); // no ->Dup()
        }
    ;

constant returns [BaseGDL* res]
    : c:CONSTANT
        {
            res=c->cData->Dup(); 
        }
  ;

constant_nocopy returns [BaseGDL* res]
    : c:CONSTANT
        {
            res=c->cData; // no ->Dup(); 
        }
  ;

lib_function_call returns[ BaseGDL* res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
}
	: #(FCALL_LIB fl:IDENTIFIER
            {
                EnvT* newEnv=new EnvT( fl, fl->libFun);//libFunList[fl->funIx]);
            }
            parameter_def[ newEnv]
            {
                // push id.pro onto call stack
                callStack.push_back(newEnv);
                // make the call
                res=static_cast<DLibFun*>(newEnv->GetPro())->Fun()(newEnv);
                //*** MUST always return a defined expression
            }
        )
    ;    

lib_function_call_retnew returns[ BaseGDL* res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
}
	: #(FCALL_LIB_RETNEW fl:IDENTIFIER
            {
                EnvT* newEnv=new EnvT( fl, fl->libFun);//libFunList[fl->funIx]);
            }
            parameter_def[ newEnv]
            {
                // push id.pro onto call stack
                callStack.push_back(newEnv);
                // make the call
                res=static_cast<DLibFun*>(newEnv->GetPro())->Fun()(newEnv);
                //*** MUST always return a defined expression
            }
        )
    ;    


function_call returns[ BaseGDL* res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
    BaseGDL *self;
    EnvUDT*   newEnv;
}
    :    (
        ( #(MFCALL 
                self=expr mp:IDENTIFIER
                {  
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, mp);

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )
        | #(MFCALL_PARENT 
                self=expr parent:IDENTIFIER p:IDENTIFIER
                {
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, p,
                        parent->getText());

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )
        | #(FCALL f:IDENTIFIER
                {
                    SetFunIx( f);
                    
                    newEnv=new EnvUDT( f, funList[f->funIx]);
                }
                parameter_def[ newEnv]
            )
        )
        {
            // push environment onto call stack
            callStack.push_back(newEnv);
            
            // make the call
            res=call_fun(static_cast<DSubUD*>(newEnv->GetPro())->GetTree());
        } 
        )
	;	

// function call can be l_values (within (#EXPR ...) only)
l_function_call returns[ BaseGDL** res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
    BaseGDL *self;
    BaseGDL *libRes;
    EnvUDT*   newEnv;
}

	: #(FCALL_LIB fl:IDENTIFIER
            {
                EnvT* newEnv=new EnvT( fl, fl->libFun);//libFunList[fl->funIx]);
            }
            parameter_def[ newEnv]
            {
                EnvT* callerEnv = static_cast<EnvT*>(callStack.back());
                // push id.pro onto call stack
                callStack.push_back(newEnv);
                // make the call
                BaseGDL* libRes = 
                static_cast<DLibFun*>(newEnv->GetPro())->Fun()(newEnv);
                
                res = callerEnv->GetPtrTo( libRes);
                if( res == NULL)
                throw GDLException( _t, "Library function must return a "
                    "l-value in this context: "+fl->getText());
            }
        )
    |
        (
        ( #(MFCALL 
                self=expr mp:IDENTIFIER
                {  
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, mp, "", true);

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )
        | #(MFCALL_PARENT 
                self=expr parent:IDENTIFIER p:IDENTIFIER
                {
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, p,
                        parent->getText(), true);

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )

        | #(FCALL f:IDENTIFIER
                {
                    SetFunIx( f);
                    
                    newEnv=new EnvUDT( f, funList[f->funIx], true);
                }
                parameter_def[ newEnv]
            )
        )
        {
            // push environment onto call stack
            callStack.push_back(newEnv);
            
            // make the call
            res=call_lfun(static_cast<DSubUD*>(
                    newEnv->GetPro())->GetTree());
        }   
        )
	;	

ref_parameter returns[ BaseGDL** ret]
    : ret=l_simple_var
    | ret=l_deref
    ;

// the environment must be on the callstack
parameter_def [EnvBaseT* actEnv] 
{
    auto_ptr<EnvBaseT> guard(actEnv); 
    BaseGDL*  kval;
    BaseGDL*  pval;
    BaseGDL** kvalRef;
    BaseGDL** pvalRef;
}
    : (  #(KEYDEF_REF knameR:IDENTIFIER kvalRef=ref_parameter
                {   // pass reference
                    actEnv->SetKeyword( knameR->getText(), kvalRef); 
                }
            )
        | #(KEYDEF_REF_EXPR knameE:IDENTIFIER 
                // execute ++ and assignment
                kval=expr 
                kvalRef=ref_parameter
                {   // pass reference
                    delete kval;
                    actEnv->SetKeyword( knameE->getText(), kvalRef); 
                }
            )
        | #(KEYDEF kname:IDENTIFIER kval=expr
                {   // pass value
                    actEnv->SetKeyword( kname->getText(), kval);
                }
            )
        | #(REF pvalRef=ref_parameter
                {   // pass reference
                    actEnv->SetNextPar(pvalRef); 
                }   
            )
        | #(REF_EXPR 
                // execute ++ and assignment
                pval=expr 
                pvalRef=ref_parameter
                {   // pass reference
                    delete pval;
                    actEnv->SetNextPar(pvalRef); 
                }   
            )
        | pval=expr
            {   // pass value
                actEnv->SetNextPar(pval); 
            }
        | #(KEYDEF_REF_CHECK knameCk:IDENTIFIER 
                kval=check_expr
                {
                    kvalRef = callStack.back()->GetPtrTo( kval);
                    if( kvalRef != NULL)
                    {   // pass reference
                        actEnv->SetKeyword(knameCk->getText(), kvalRef); 
                    }
                    else 
                    {   // pass value
                        actEnv->SetKeyword(knameCk->getText(), kval); 
                    }
                }
            )   
        | #(REF_CHECK
                pval=check_expr
                {
                    pvalRef = callStack.back()->GetPtrTo( pval);
                    if( pvalRef != NULL)
                    {   // pass reference
                        actEnv->SetNextPar( pvalRef); 
                    }
                    else 
                    {   // pass value
                        actEnv->SetNextPar( pval); 
                    }
                }       
            )
        )*             
        {
            actEnv->Extra(); // expand _EXTRA
            guard.release();
        }
	;

arrayindex_list returns [ArrayIndexListT* aL]
{
    ExprListT        exprList; // for cleanup
    IxExprListT      ixExprList;
    SizeT nExpr;
    BaseGDL* s;
}
	: #(ax:ARRAYIX
            {
                aL = ax->arrIxList;
                assert( aL != NULL);

                nExpr = aL->NParam();
                if( nExpr == 0)
                {
                    aL->Init();
                    goto empty;
                }

//                 if( nExpr > 1)
//                 {
//                     ixExprList.reserve( nExpr);
//                     exprList.reserve( nExpr);
//                 }
//                if( nExpr == 0) goto empty;
            }
            (
                ( s=indexable_expr
                | s=check_expr
                    {
                        if( !callStack.back()->Contains( s)) 
                        exprList.push_back( s);
                    }
                | s=indexable_tmp_expr { exprList.push_back( s);}
                )
                {
                    ixExprList.push_back( s);
                    if( ixExprList.size() == nExpr)
                        break; // allows some manual tuning
                }
            )*
//            { empty: ;}
        )
        {
            aL->Init( ixExprList);
            empty:
        }
    ;

// and array_def is a primary expression
array_def returns [BaseGDL* res]
{
    DType  cType=UNDEF; // conversion type
    SizeT maxRank=0;
    BaseGDL* e;
    ExprListT          exprList;
    BaseGDL*           cTypeData;
}
    : #(a:ARRAYDEF 
            (e=expr
                {
                    // add first (this way it will get cleaned up anyway)
                    exprList.push_back(e);
                    
                    DType ty=e->Type();
                    if( ty == UNDEF)
                    {
                        throw GDLException( _t, "Variable is undefined: "+
                            Name(e));
                    }
                    if( cType == UNDEF) 
                    {
                        cType=ty;
                        cTypeData=e;
                    }
                    else 
                    { 
                        if( cType != ty) 
                        {
                            if( DTypeOrder[ty] > 100 || DTypeOrder[cType] > 100) // struct, ptr, object
                            {
                                throw 
                                GDLException( _t, e->TypeStr()+
                                    " is not allowed in this context.");
                            }
                            
                            // update order if larger type (or types are equal)
                            if( DTypeOrder[ty] >= DTypeOrder[cType]) 
                            {
                                cType=ty;
                                cTypeData=e;
                            }
                        }
                        if( ty == STRUCT)
                        {
                            // check for struct compatibility
                            DStructDesc* newS=
                            static_cast<DStructGDL*>(e)->Desc();
                            DStructDesc* oldS=
                            static_cast<DStructGDL*>(cTypeData)->Desc();

                            // *** here (*newS) != (*oldS) must be set when
                            // unnamed structs not in struct list anymore
                            // WRONG! This speeds up things for named structs
                            // unnamed structs all have their own desc
                            // and thus the next is always true for them
                            if( newS != oldS)
                            {
//                                 if( (*newS) == (*oldS))
//                                 {
// Not needed, CatArray puts the right descriptor
//                                     // different structs with same layout
//                                     // replace desc with first one
//                                     if( oldS->IsUnnamed())
//                                         oldS = new DStructDesc( oldS);

//                                     static_cast<DStructGDL*>(e)->SetDesc( oldS);
//                                 }
//                                 else

                                if( (*newS) != (*oldS))
                                    throw GDLException( _t, 
                                        "Conflicting data structures: "+
                                        Name(cTypeData)+", "+Name(e));
                            }
                        }
                    }

                    // memorize maximum Rank
                    SizeT rank=e->Rank();
                    if( rank > maxRank) maxRank=rank;
                }
            )+
        )
        {  
            res=cTypeData->CatArray(exprList,a->arrayDepth,maxRank);
        }
  ;


named_struct_def returns[ BaseGDL* res]
{
    DStructDesc*          nStructDesc;
    auto_ptr<DStructDesc> nStructDescGuard;
    BaseGDL* e;
    BaseGDL* ee;
}
	: #(n:NSTRUC id:IDENTIFIER 
            {
                // definedStruct: no tags present
                if( n->definedStruct == 1) GetStruct( id->getText(), _t);

                // find struct 'id' (for compatibility check)
                DStructDesc* oStructDesc=
                FindInStructList( structList, id->getText());
                
                if( oStructDesc == NULL || oStructDesc->NTags() > 0)
                {
                    // not defined at all yet (-> define now)
                    // or completely defined  (-> define now and check equality)
                    nStructDesc= new DStructDesc( id->getText());
                    
                    // guard it
                    nStructDescGuard.reset( nStructDesc); 
                } 
                else
                {   // NTags() == 0
                    // not completely defined (only name in list)
                    nStructDesc= oStructDesc;
                }
                
                // the instance variable
//                 DStructGDL* instance= new DStructGDL( nStructDesc,
//                                                       dimension(1)); 
                DStructGDL* instance= new DStructGDL( nStructDesc);

                auto_ptr<DStructGDL> instance_guard(instance);
            }

            // PROBLEM: both the descriptor AND the instance must be defined
            ( ee=expr
                {
                    // also adds to descriptor, grabs
                    instance->NewTag( 
                        oStructDesc->TagName( nStructDesc->NTags()),
                        ee);
                }
            | i:IDENTIFIER e=expr // e is a new BaseGDL*
                {
                    // also adds to descriptor, grabs
                    instance->NewTag( i->getText(), e); 
                }
            | INHERITS ii:IDENTIFIER
            // INHERITS triggers read/compile/interpret of 
            // IDENTIFIER__define.pro
            // if the struct named IDENTIFIER is not already known
                {
                    DStructDesc* inherit=GetStruct( ii->getText(), _t);

                    //   nStructDesc->AddParent( inherit);
                    instance->AddParent( inherit);
                }
                
            )+
       
            {
                // inherit refers to nStructDesc, in case of error both have to
                // be freed here
                if( oStructDesc != NULL)
                {
                    if( oStructDesc != nStructDesc)
                    {
                        oStructDesc->AssureIdentical(nStructDesc);
                        instance->DStructGDL::SetDesc(oStructDesc);
                        //delete nStructDesc; // auto_ptr
                    }
                }
                else
                {
                    // release from guard (if not NULL)
                    nStructDescGuard.release();

                    // insert into struct list 
                    structList.push_back(nStructDesc);
                }
                
                instance_guard.release();
                res=instance;
            }
        )
    ;

unnamed_struct_def returns[ BaseGDL* res]
{
    // don't forget the struct in extrat.cpp if you change something here
    // "$" as first char in the name is necessary 
    // as it defines unnnamed structs (see dstructdesc.hpp)
    DStructDesc*   nStructDesc = new DStructDesc( "$truct");

    // instance takes care of nStructDesc since it is unnamed
//     DStructGDL* instance = new DStructGDL( nStructDesc, dimension(1));
    DStructGDL* instance = new DStructGDL( nStructDesc);
    auto_ptr<DStructGDL> instance_guard(instance);

    BaseGDL* e;
}
	: #(STRUC 
            ( si:IDENTIFIER e=expr
                {
                    // also adds to descriptor, grabs
                    instance->NewTag( si->getText(), e); 
                }
            )+
            
            {
//                 DStructDesc* oStructDesc=nStructDesc->FindEqual( structList);
//                 if( oStructDesc != NULL)
//                 {
//                     instance->SetDesc(oStructDesc);
//                     //delete nStructDesc; // auto_ptr
//                 }
//                 else
//                 {
//                     // insert into struct list
//                     structList.push_back( nStructDesc.release());
//                 }
                
                instance_guard.release();
                res=instance;
            }
        )
    ;

// only from named structs
struct_def returns[ BaseGDL* res]
    : res=named_struct_def
    | res=unnamed_struct_def
	| #(NSTRUC_REF idRef:IDENTIFIER 
            {
                // find struct 'id'
                // returns it or throws an exception
                DStructDesc* dStruct=GetStruct( idRef->getText(), _t);
                
                res=new DStructGDL( dStruct, dimension(1));
            }
        )
	;
