#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000, 2001 Free Software Foundation
#
# FILE:
# GConnections.py
#
# DESCRIPTION:
# Class that loads connection definition files and maintains
# database connections.
#
# NOTES:
#
# HISTORY:
#

from GConfigParser import *
import sys, string
import GDebug
import GDataObjects
import GLoginHandler
from gnue.common import openResource, dyn_import

class Error(StandardError):
  # Base error
  pass

class NotFoundError (Error):
  # Raised if a requested connection name does not
  # exist in the Connections Definition File.
  pass

class AdapterNotInstalled (Error):
  # Raised if a provider is requested for which
  # the python libraries are not installed.
  pass

class InvalidFormatError (Error):
  # Raised if the Connections Definition File is
  # in an unreadable format.
  pass


class GConnections:

  def __init__(self, location, loginHandler=None):

    self._loginHandler = loginHandler
    self._parser = GConfigParser()
    self._location = location

    GDebug.printMesg(1,'Conn File: "%s"' % location)

    if len(location):
      fileHandle = openResource(location)

      try:
        self._parser.readfp(fileHandle)
      except DuplicateSectionError:
        raise InvalidFormatError, \
           'The connections file has duplicate source definitions.\n\nFile: %s' \
              % location
      except MissingSectionError:
        raise InvalidFormatError, \
           'The connections file has no source definitions.\n\nFile: %s' \
              % location
      except:
        raise InvalidFormatError, \
           'The connections file cannot be parsed.\n\nFile: %s' % location

    self._openConnections = {}


  def setLoginHandler(self, loginHandler):
    self._loginHandler = loginHandler


  def hasConnectionParameters(self, connection_name):
    return self._parser.has_section(connection_name)


  def getConnectionParameter(self, connection_name, attribute, default=None):
    if self._parser.has_section(connection_name):
      try:
        return self._parser.get(connection_name, attribute)
      except:
        return default
    else:
      raise NotFoundError, \
         'The connections file does not contain a definition \n       for "%s".\n\nFile: %s' \
            % (connection_name, self._location)


  #
  # Returns an dictionary of dictionaries describing all connections:
  #  {connection name: {att name: value}}
  #
  def getAllConnectionParameters(self):
    rv = {}
    for section in self._parser.sections():
       rv[section]={}
       for att in self._parser.options(section):
         rv[section][att] = self._parser.get(section, att)
    return rv

  #
  # Returns a dictionary describing a connection:
  #  {att name: value}
  #
  def getConnectionParameters(self, connection_name):
    rv = {}
    if self._parser.has_section(connection_name):
      for att in self._parser.options(connection_name):
        rv[att] = self._parser.get(connection_name, att)
    else:
      raise NotFoundError, \
         'The connections file does not contain a definition \n       for "%s".\n\nFile: %s' \
            % (connection_name, self._location)

    return rv

  #
  # Add a connection entry (session specific; i.e., doesn't add
  # to the connections.conf file, but to the current instance's
  # list of available connections.
  #
  def addConnectionSpecification (self, name, parameters):
    self._parser.remove_section(string.lower(name))
    self._parser.add_section(string.lower(name))
    for key in parameters.keys():
      self._parser.set(string.lower(name), key, parameters[key])


  #
  # Return a database provider object
  #
  def getDataObject(self, connection_name, connection_type):
    # This will throw a GConnections.NotFoundError if an unknown
    # connection name is specified.  The calling method should
    # catch this exception and handle it properly (exit w/message)
    return _load_dbdriver(self.getConnectionParameter(connection_name, \
        'provider'), connection_type, self)


  #
  # Get a data connection for a specified database
  #
  def requestConnection(self, dataObject, connection):
    connection_name = string.lower(connection)
    if self._openConnections.has_key(connection_name):

      # If a database connetion has already been established, use it
      dataObject.setDataConnection(self._openConnections[connection_name])
      GDebug.printMesg(5,'Reusing data connection to %s' % connection_name)

    else:
      attempts = 4

      GDebug.printMesg(5,'Getting new data connection to %s' % connection_name)

      while attempts:

        try:
          # Ask the UI to prompt for our login data
          loginData = self._loginHandler.getLogin(
              [connection_name, self.getConnectionParameter(connection_name,'comment',''),
               dataObject.getLoginFields()])

          # Add the parameters from the Connections Definition File
          loginData.update(self.getConnectionParameters(connection_name))

          # Ask the data object to connect to the database
          dataObject.connect(loginData)

          # Save the newly opened connection for future datasources
          self._openConnections[connection_name] = \
              dataObject.getDataConnection()

          # We're done!
          attempts = 0
          self._loginHandler.destroyLoginDialog()

        except GDataObjects.LoginError, error:
          # Oops, they must have entered an invalid user/password.
          # Those silly users.
          attempts = attempts - 1

          if not attempts:
            # Four times is plenty...
            self._loginHandler.destroyLoginDialog()
            raise GDataObjects.LoginError, \
               "Unable to log in after 4 attempts.\n\nError: %s" % error

        except GLoginHandler.UserCanceledLogin:
          # Guess they changed their minds. Treat as a login error.
          self._loginHandler.destroyLoginDialog()
          raise GDataObjects.LoginError, "User canceled the login request."




#
# Load the correct DBdriver from gnue/common/dbdrivers/*/
#
def _load_dbdriver(driver, type, connectionManager):
  try:
    dbdriver = dyn_import("gnue.common.dbdrivers.%s" % (driver))
  except ImportError, err:
    GDebug.printMesg(1,'ImportError importing driver %s' % (driver))
    GDebug.printMesg(1,err)
    raise GDataObjects.ProviderNotSupportedError, \
       "No database driver found for provider type '%s'" % driver

  try:
    dd = dbdriver.supportedDataObjects[type]()
    GDebug.printMesg (1,'Attaching to %s (%s)' % (driver,dd.__class__.__name__))
    return dd
  except KeyError:
    raise GDataObjects.ObjectTypeNotAvailableError, \
       "DB Driver '%s' does not support source type '%s'" % (driver, type)


