# GNU Enterprise Common - Database Drivers - Base driver for PostgreSQL
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Connection.py 7007 2005-02-11 16:18:59Z reinhard $

__all__ = ['Connection']

import types

from gnue.common.apps import errors, i18n
from gnue.common.datasources import Exceptions
from gnue.common.datasources.drivers import DBSIG2
from DataObject import *

from gnue.common.datasources.drivers.postgresql.Schema.Discovery.Introspection import Introspection
from gnue.common.datasources.drivers.postgresql.Schema.Creation.Creation import Creation



# =============================================================================
# GConnection object for PostgreSQL-based drivers
# =============================================================================

class Connection (DBSIG2.Connection):

  defaultBehavior = Introspection
  defaultCreator  = Creation
  supportedDataObjects = {
    'object': DataObject_Object,
    'sql':    DataObject_SQL
  }

  _pg_connectString = 'host=%s dbname=%s user=%s password=%s port=%s'


  # ---------------------------------------------------------------------------
  # Connect to the database
  # ---------------------------------------------------------------------------

  def connect (self, connectData):

    if not self._driver:
      self._driver = __import__ (self._driver_module, None, None, '*')

    if not hasattr (self,'_DatabaseError'):
      try:
        self._DatabaseError = self._driver.Error
      except:
        self._DatabaseError = self._driver.DatabaseError

    gDebug (9, "Postgresql database driver initializing")

    try:
      port   = connectData.get ('port', '5432')
      conStr = self._pg_connectString % (connectData ['host'],
                                         connectData ['dbname'],
                                         connectData ['_username'],
                                         connectData ['_password'],
                                         port)

      if isinstance (conStr, types.UnicodeType):
        conStr = conStr.encode (i18n.getencoding ())

      self.native = self._driver.connect (conStr)

    except self._DatabaseError, value:
      gDebug (1, "Connect String: %s" % conStr)
      gDebug (1, "Exception %s" % value)

      raise Exceptions.LoginError, \
          u_("The PostgreSQL driver returned the following error:\n\t%s") \
          % errors.getException () [2]

    self._pg_encoding = pg_encTable.get (self._encoding, '')
    if not pg_encTable.has_key (self._encoding):
      gDebug (1, u_("Encoding '%s' is not supported by postgresql dbdriver. "
                    "Using default encoding.") % self._encoding)

    dtfmt = connectData.get ('datetimeformat', "%Y-%m-%d %H:%M:%S")
    self._dateTimeFormat = "'%s'" % dtfmt
    gDebug (9, "datetime-format set to %s" % self._dateTimeFormat)

    # FIXME: Why isn't this generally done for any new connection?
    self._beginTransaction ()


  # ---------------------------------------------------------------------------
  # Done at the start of each transaction
  # ---------------------------------------------------------------------------

  def _beginTransaction (self):

    # Must set CLIENT_ENCODING per transaction as it is reset on COMMIT or
    # ROLLBACK.

    if self._pg_encoding not in ["", 'DEFAULT']:
      self.sql ("SET CLIENT_ENCODING TO '%s'" % self._pg_encoding)


  # ===========================================================================
  # Extensions to the basic GConnection object
  # ===========================================================================


  # ---------------------------------------------------------------------------
  # Return the current date, according to database
  # ---------------------------------------------------------------------------

  def getTimeStamp (self):

    return self.sql1 ("select current_timestamp")


  # ---------------------------------------------------------------------------
  # Return a sequence number from sequence 'name'
  # ---------------------------------------------------------------------------

  def getSequence (self, name):

    return self.sql1 ("select nextval('%s')" % name)


# =============================================================================
# Encoding-Table 
# =============================================================================

pg_encTable = {
  'ascii'     : 'SQL_ASCII',     # ASCII
  ''          : 'EUC_JP',        # Japanese EUC
  ''          : 'EUC_CN',        # Chinese EUC
  ''          : 'EUC_KR',        # Korean EUC
  ''          : 'JOHAB',         # Korean EUC (Hangle base)
  ''          : 'EUC_TW',        # Taiwan EUC
  'utf-8'     : 'UNICODE',       # Unicode (UTF-8)
  ''          : 'MULE_INTERNAL', # Mule internal code
  'iso8859-1' : 'LATIN1',        # ISO 8859-1 ECMA-94 Latin Alphabet No.1
  'iso8859-2' : 'LATIN2',        # ISO 8859-2 ECMA-94 Latin Alphabet No.2
  'iso8859-3' : 'LATIN3',        # ISO 8859-3 ECMA-94 Latin Alphabet No.3
  'iso8859-4' : 'LATIN4',        # ISO 8859-4 ECMA-94 Latin Alphabet No.4
  'iso8859-9' : 'LATIN5',        # ISO 8859-9 ECMA-128 Latin Alphabet No.5
  'iso8859-10': 'LATIN6',        # ISO 8859-10 ECMA-144 Latin Alphabet No.6
  'iso8859-13': 'LATIN7',        # ISO 8859-13 Latin Alphabet No.7
  'iso8859-14': 'LATIN8',        # ISO 8859-14 Latin Alphabet No.8
  'iso8859-15': 'LATIN9',        # ISO 8859-15 Latin Alphabet No.9
  'iso8859-16': 'LATIN10',       # ISO 8859-16 ASRO SR 14111 Latin Alph. No.10
  'iso8859-5' : 'ISO-8859-5',    # ECMA-113 Latin/Cyrillic
  'iso8859-6' : 'ISO-8859-6',    # ECMA-114 Latin/Arabic
  'iso8859-7' : 'ISO-8859-7',    # ECMA-118 Latin/Greek
  'iso8859-8' : 'ISO-8859-8',    # ECMA-121 Latin/Hebrew
  'koi8-r'    : 'KOI8',          # KOI8-R(U)
  'cp1251'    : 'WIN',           # Windows CP1251
  ''          : 'ALT',           # Windows CP866
  ''          : 'WIN1256',       # Arabic Windows CP1256
  ''          : 'TCVN',          # Vietnamese TCVN-5712 (Windows CP1258)
  ''          : 'WIN874',        # Thai Windows CP874
}
