#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# GObjects.py
#
# DESCRIPTION:
# Base class for GNUe objects which can be represented as XML
#
# NOTES:
#
import sys

from xml.sax import saxutils
from gnue.common.apps import GDebug
import string
import types
from gnue.common.definitions.GParserHelpers import GContent, ParserObj
from gnue.common.formatting import GTypecast
from gnue.common.logic.GTriggerCore import GTriggerCore
import types

#
# Class GObj
#
# Base class for GNUe objects which
# can be represented by XML tags
#
class GObj(GTriggerCore, ParserObj):
  """
  The base class for almost all GNUe objects.

  GObj based objects can be represented by XML tags in a GParser
  based setup.
  """
  def __init__(self, *args, **parms):
    GTriggerCore.__init__(self)
    ParserObj.__init__(self, *args, **parms)
    # This should be ok to remove as glimpse doesn't show it being used anywhere
    # and it's messing up the common namespace property logic
    #self.__properties__ = {}
    self._inits = []

  def buildObject(self, **params):
    """
    A convenience function for applications NOT using GParser to load an object tree.

    """
    self.__dict__.update(params)
    return self._buildObject()


  def buildAndInitObject(self, **params):
    """
    This is a convenience function for applications
    NOT using GParser to load an object tree.
    """
    self.phaseInit(self.buildObject(**params))

  def getParent(self):
    """
    Returns the immediate parent of an object instance in a GObj tree.

    @return: The parent of the object in the GObj tree.
    @rtype: any
    """
    return self._parent

  #
  # phaseInit
  #
  def phaseInit(self, iterations=0):
    """
    Starts GNUe's phased initialization system from this object down.

    Typically called from within a GParser instance.  phaseInit interates
    thru the GObj tree as many times as necessary to fully initialize the
    tree.  It determines the number of interations to perform during it's
    first pass down the tree.

    phaseInit looks for a private list variable called _inits that
    contains a list of functions to execute.  Here is an example from
    gnue-forms GFForm object.

    C{self._inits = [self.primaryInit, None, self.secondaryInit]}

    This list tells phase init to execute the self.primaryInit function
    during the first iteration of the initialization process. During
    the seconds pass through it does nothing on these objects.  During the
    third pass through it executes self.secondaryInit.

    Some may question why we don't do all initialization logic inside the
    python objects __init__ functions.  Frequently you may find that a parent
    object may need specific infomation from some of its children to properly
    initialize itself.  This type of logic cannot be places into an __init__
    as the children may not be loaded yet or may not yet have the needed
    information.
    """
    if iterations == 0:
      iterations = self.maxInits()
    for phase in range(iterations):
      self._phaseInit(phase)

  def _phaseInit(self,phase):

## TODO: Below is a call-less recursive version of
## TODO: phaseInit.  Needs to be profiled both ways.
##    stack = [self]
##    while stack:
##      object = stack.pop()
##      for child in object._children:
##        stack.insert(0,child)
##      try:
##        init = object._inits[phase]
##      except IndexError:
##        continue
##      init()

    inits = self._inits
    if (len(inits) > phase) and inits[phase]:
      GDebug.printMesg(7,"%s: Init Phase %s" % (self._type, phase+1))
      inits[phase]()

    for child in self._children:
      try:
        initter = child._phaseInit
      except AttributeError:
        continue
      initter(phase)


  # This function is called after the parsers have completely
  # constructed. All children should be in place and
  # attributes and content should be set at this point.
  # Return the number of phaseInit iterations your object will
  # need.
  #
  # NOTE: Do not initialize datasources, etc at this point.
  #       This is only so content can be set, etc, after
  #       loading from XML.
  #
  def _buildObject(self):
    return len(self._inits)

  #
  # maxInits functions
  #
  # maxInits returns the maximum size of all the _inits
  # list from this object or it's children
  #
  def maxInits(self):
    self._initCount = 0
    self.walk(self._maxInitsWalker)
    return self._initCount

  def _maxInitsWalker(self, object):
    if hasattr(object,'_inits'):
      self._initCount = max(self._initCount,len(object._inits))


  def getChildrenAsContent(self):
    content = ""
    for child in self._children:
      if isinstance(child, GContent):
        content += child._content
      elif isinstance(child, GBinary):
        content += child.__data__
    return content

  def showTree(self, indent=0):
    """
    A recusive function to print an indented text representation of
    the GObj tree from this object down.

    This is usefull for debugging purposes.

    @param indent: Sets the level of indention.  Used during recursion
    to properly indent the tree.
    @type indent: int
    """
    print ' ' * indent + `self._type`,self

    for child in self._children:
      child.showTree(indent + 2)

  #
  # addChild
  #
  # Adds an object to an instances list of children
  #
  def addChild(self, child):
    self._children.append(child)

  def getXmlTag(self, stripPrefixes=None):
    """
    Returns the xml tag to be used to represent the object.
    
    @param stripPrefixes: A list of prefixes that will automatically
                          be removed from the objects type.  This can be
                          used to remove the gf from the start of all
                          the gnue-forms objects.  If set to None (the default)
                          then the behaviour will be the old style which
                          removes the first two characters from the type.
    @type multipler: list of strings
    @return: The xml tag to use
    @rtype: string
    """
    if stripPrefixes == None:
      return string.lower(string.replace(self._type[2:],'_','-'))
    for prefix in stripPrefixes:
      if prefix == self._type[:len(prefix)]:
        return string.lower(string.replace(self._type[len(prefix):],'_','-'))
    return string.lower(string.replace(self._type,'_','-'))  
    
  def dumpXML(self, lookupDict, treeDump=None, gap="  ", xmlnamespaces={},
              textEncoding='<locale>', stripPrefixes=None):
    """
    Dumps an XML representation of the object
    """
    xmlns = ""
    xmlnsdef = ""
    if textEncoding=='<locale>':
      textEncoding=gConfig('textEncoding')
    try:
      if self._xmlnamespace:
        try:
          xmlns = xmlnamespaces[self._xmlnamespace] + ":"
        except KeyError:
          i = 0
          xmlns = "out"
          while xmlns in xmlnamespaces.values():
            i += 1
            xmlns = "out%s" % i
          xmlnamespaces[self._xmlnamespace] = xmlns
          xmlnsdef = ' xmlns:%s="%s"' % (xmlns, self._xmlnamespace)
          xmlns += ":"
    except AttributeError:
      try:
        if self._xmlchildnamespace:
          if not xmlnamespaces.has_key(self._xmlchildnamespace):
            i = 0
            ns = "out"
            while ns in xmlnamespaces.values():
              i += 1
              ns = "out%s" % i
            xmlnamespaces[self._xmlnamespace] = ns
            xmlnsdef = ' xmlns:%s="%s"' % (ns, self._xmlchildnamespace)
      except AttributeError:
        pass


    try:
      if self._xmlchildnamespaces:
        for abbrev in self._xmlchildnamespaces.keys():
          xmlnsdef += ' xmlns:%s="%s"' % (abbrev,self._xmlchildnamespaces[abbrev])
    except AttributeError:
      pass

    xmlEntity = self.getXmlTag(stripPrefixes)
    xmlString = "%s<%s%s%s" % (gap[:-2],xmlns, xmlEntity, xmlnsdef)

    indent = len(xmlString)
    pos = indent
    attrs = self.__dict__.keys()
    attrs.sort()

    # Make 'name' first
    if 'name' in attrs:
      attrs.pop(attrs.index('name'))
      attrs.insert(0,'name')

    for attribute in attrs:
      # skip keys beginning with _
      if attribute[0] == "_":
        continue
      val = self.__dict__[attribute]
      if (xmlns and attribute in self._listedAttributes) or \
         (not xmlns and lookupDict[xmlEntity].has_key('Attributes') and \
         lookupDict[xmlEntity]['Attributes'].has_key(attribute)):
        if val != None and \
           (xmlns or (not lookupDict[xmlEntity]['Attributes'][attribute].has_key('Default') or \
            (lookupDict[xmlEntity]['Attributes'][attribute]['Default']) != (val))):
          try:
            typecast = lookupDict[xmlEntity]['Attributes'][attribute]['Typecast']
          except:
            typecast = GTypecast.text
          if typecast == GTypecast.boolean:
            if val:
              addl = ' %s="Y"' % (attribute)
            else:
              addl = ' %s="N"' % (attribute)
          elif typecast == GTypecast.names:
            if type(val) == types.StringType:
              #addl = ' %s="%s"' % (attribute, string.join(val.decode(textEncoding),','))
              addl = ' %s="%s"' % (attribute, string.join(unicode(val,textEncoding),','))
            else:
              addl = ' %s="%s"' % (attribute, string.join(val,','))
          else:
            if type(val) == types.StringType:
              #addl = ' %s="%s"' % (attribute, saxutils.escape('%s' % val.decode(textEncoding)))
              addl = ' %s="%s"' % (attribute, saxutils.escape('%s' % unicode(val,textEncoding)))
            else:
              addl = ' %s="%s"' % (attribute, saxutils.escape('%s' % val))
          if len(addl) + pos > 78:
            xmlString += "\n" + " " * indent + addl
            pos = indent
          else:
            xmlString += addl
            pos += len(addl)
      if attribute.find('__') > 0 and attribute.split('__')[0] in xmlnamespaces.keys():
        if val != None:
          if type(val) == types.StringType:
            addl = ' %s="%s"' % (attribute.replace('__',':'), saxutils.escape('%s' % unicode(val,textEncoding)))
          else:
            try:
              if val == int(val):
                val = int(val)
            except:
              pass
            addl = ' %s="%s"' % (attribute.replace('__',':'), saxutils.escape('%s' % val))
          if len(addl) + pos > 78:
            xmlString += "\n" + " " * indent + addl
            pos = indent
          else:
            xmlString += addl
            pos += len(addl)

    if len(self._children):
      hasContent = 0
      for child in self._children:
        hasContent = hasContent or isinstance(child,GContent)
      if hasContent:
        xmlString += ">"
      else:
        xmlString += ">\n"

      if treeDump:
        for child in self._children:
          xmlString += child.dumpXML(lookupDict, 1,gap+"  ",
              textEncoding=textEncoding, xmlnamespaces=xmlnamespaces, stripPrefixes=stripPrefixes)

      if hasContent:
        xmlString += "</%s%s>\n" % (xmlns, xmlEntity)
      else:
        xmlString += "%s</%s%s>\n" % (gap[:-2], xmlns,xmlEntity)
    else:
      xmlString += "/>\n"
    return xmlString


  def walk(self, function, *args, **parms):
    function(self, *args, **parms)
    for child in self._children:
      if isinstance(child, GObj):
        child.walk(function, *args, **parms)

  def findParentOfType(self,type, includeSelf=1):
    """
    Moves upward though the parents of an object till
    it finds the parent of the specified type
    """
    if includeSelf:
      parentObject = self
    else:
      parentObject = self._parent

    while 1:
      if parentObject == None:
        return None
      elif parentObject._type == type:
        return parentObject

      # If passed a type of NONE it finds the top object in the tree
      if not type and not parentObject._parent:
        return parentObject

      parentObject = parentObject._parent

  #
  # findChildOfType
  #
  # Moves downward though the children of an object till
  # it finds the child of the specified type
  #
  def findChildOfType(self, type, includeSelf=1, allowAllChildren=0):

    if includeSelf and self._type == type:
      return self

    for child in self._children:
      if child._type == type:
        return child

    if allowAllChildren:
      for child in self._children:
        o = child.findChildOfType(type,0, 1)
        if o:
          return o

    return None


  #
  # findChildrenOfType
  #
  # find all children of a specific type
  #
  def findChildrenOfType(self, type, includeSelf=1, allowAllChildren=0):

    rs = []

    if includeSelf and self._type == type:
      rs += [self]

    for child in self._children:
      if child._type == type:
        rs += [child]

    if allowAllChildren:
      for child in self._children:
        try:
          rs += child.findChildrenOfType(type,0, 1)
        except AttributeError:
          pass  # not all children will be GObj and support that function

    return rs


  #
  # getDescription
  #
  # Return a useful description of this object
  # Used by designer clients
  #
  def getDescription(self):
    if hasattr(self,'_description'):
      return self._description
    elif hasattr(self,'name'):
      return self.name + " (%s)" % self._type[2:]
    else:
      return self._type[2:] + " (%s)" % self._type[2:]

  # Hooks
  def __getitem__(self, key):
    return self.__dict__[key.replace(':','__')]

  def __setitem__(self, key, value):
    return self._setItemHook(key.replace(':','__'),value)

  def _setItemHook(self, key, value):
    self.__dict__[key] = value


#
# ParserMultiplexor
#
# When a GParser's BaseClass needs to be dependent upon
# an attribute, the tool can use a customized ParserMultiplexor,
# overwriting the getClass method.
#
# e.g., assume we have an xml tag 'goblin', that
# corresponds to a GObj-based class Goblin.  However,
# if <goblin style="boo"> we really need a BooGoblin
# object or if <goblin style="foo"> then we need a
# FooBoblin object, then the a GoblinPlexor would
# define getClass as:
#
# def getClass(self):
#   if self.style == 'boo':
#     return BooGoblin
#   elif self.style == 'foo':
#     return FooGoblin
#   else:
#     return Goblin
#
class ParserMultiplexor(ParserObj):

  def _buildObject(self):
    newObj = self.getClass()(None)
    for attr, value in self.__dict__.items():
      if attr not in ('_buildObject','getClass') and attr[:2] != '__':
        newObj.__dict__[attr] = value

    if self._parent:
      self._parent._children[self._parent._children.find(self)] = newObj
    return newObj._buildObject(self)


  # This should return a GObj-based class
  def getClass(self):
    raise "Virtual method not implemented"


# This is down here to avoid recursive importing
from gnue.common.definitions.GBinary import GBinary
