#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# PropertyEditor.py
#
# DESCRIPTION:
#
# NOTES:
#


import sys, os, string
from wxPython.wx import *
from wxPython.grid import *
from gnue.common.apps import GDebug
from gnue.common.formatting import GTypecast
from gnue.common.logic.GTrigger import GTrigger
from gnue.common.definitions.GParserHelpers import GContent
from gnue.designer.base.ToolBase import *
from gnue.designer.base.uihelpers.GridCellEditors import CharCellEditor

class EventEditor (ToolBase):

  runtime_section = 'EventEditor'
  uses_feedback_bar = 1
  default_dock = 'left-2'

  def init(self):
    self.panel = wxPanel(self,-1,style=wxSIMPLE_BORDER, pos=wxPoint(6,6))
    self.notebook = wxNotebook(self.panel, -1, style=wxNB_BOTTOM)


    self.object = None

    # EventAware provided by ToolBase
    self.registerEventListeners({
                       'ObjectSelected'      : self.onSetCurrentObject,
                       'ObjectCreated'       : self.onCreateObject,
                       'ObjectModified'      : self.onModifyObject,
                       'ObjectDeleted'       : self.onDeleteObject,
                      })

    self.supplemental = []
    EVT_SIZE(self, self.onSize)
    self.main = self.createMainPage()
    self.notebook.AddPage(self.main,'Properties')


  def createMainPage(self):
    return InspectorPanel(self, self.notebook)

  def onSetCurrentObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if object != self.object:
      self.setCurrent(object)

  def setCurrent(self, object):
      self.object = object
      self.reset()
      self.main.setCurrent(object)
      self.notebook.SetPageText(0, self.main.getPageText())
      i = 1
      for page in self.supplemental:
        page.setCurrent(object)
        self.notebook.SetPageText(i, page.getPageText())
        i += 1

  def addPage(self, inspector, label):
    self.supplemental.append(inspector)
    self.notebook.AddPage(inspector, label)

  def reset(self):
    for i in range(len(self.supplemental)):
      self.notebook.DeletePage(1)
    self.supplemental = []


  def onCreateObject (self, event):
    object = event.object
    handler = event.originator
    for insp in [self.main] + self.supplemental:
      insp.onCreateObject(event)


  def onModifyObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if handler != __name__ and self.object == object:
      self.setCurrent(object)
    for insp in [self.main] + self.supplemental:
      insp.onModifyObject(event)


  def onDeleteObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if handler != __name__:
      pass
    for insp in [self.main] + self.supplemental:
      insp.onDeleteObject(event)


  def onSize(self, event):
    w,h = self.GetClientSizeTuple()
    self.panel.SetSize((w-12,h-12))
    self.notebook.SetSize(self.panel.GetClientSizeTuple())


#
#
#
class InspectorPanel(wxPanel):

  NAMESPACE = ""


  def __init__(self, editor, parent):

    wxPanel.__init__(self, parent, -1)
    self.editor = editor

    self.object = None
    self.objectList = []
    self.objectMap = {}
    self.triggerMap = {}
    self.rowList = []

    # Set up our grid
    self.grid = wxGrid(self, -1, pos=wxPoint(0,0))
    self.grid.CreateGrid(1,1)
    self.grid.SetColLabelSize(0)

    self.namedMap = {}

    EVT_GRID_SELECT_CELL(self.grid, self.OnCellSelected)
    EVT_GRID_CELL_CHANGE(self.grid, self.OnCellChange)
    EVT_GRID_CELL_LEFT_DCLICK(self.grid, self.OnLeftDClick)
    EVT_SIZE(self, self.onSize)

    self.editor.rootObject.walk(self.inventoryObject)

  def setCurrent(self, object):
      self._setCurrent(object)

  def _setCurrent(self, object):
    self.object = object
    try:
      self.triggerList = object._validTriggers
    except AttributeError:
      self.triggerList = {}

    # Speed up the process by not refreshing the grid yet
    self.grid.BeginBatch()

    # Delete any old rows from a previous object
    nr = self.grid.GetNumberRows()
    if nr:
      self.grid.DeleteRows(0,nr)


    self.rowList = self.triggerList.keys()
    self.rowList.sort()

    # Create the number of rows we'll need
    self.grid.InsertRows(0,len(self.rowList))

    self.triggerTypes = {'':_('(None)'),
                           '(Custom Trigger)':_('(Custom Trigger)')}
    self.triggerMap = {}

    for name in self.namedMap.keys():
      self.triggerTypes[name] = name + ' (named)'

    mappedTriggers = {}
    for child in object._children:
      if isinstance(child, GTrigger):
        if hasattr(child,'src') and child.src:
          mappedTriggers[child.type.upper()] = child.src
        else:
          mappedTriggers[child.type.upper()] = '(Custom Trigger)'
        self.triggerMap[child.type.upper()] = child

    self.mappedTriggers = mappedTriggers
    i = 0



    for key in self.rowList:
      self.grid.SetRowLabelValue(i,self.triggerList[key])
      self.grid.SetCellEditor(i, 0, EventCellEditor(self.grid,self.triggerTypes))
      try:
        self.grid.SetCellValue(i,0, "%s" % mappedTriggers[key])
      except KeyError:
        self.grid.SetCellValue(i,0, "%s" % '')
      i += 1


    # Redraw the grid
    self.grid.EndBatch()
    try:
      self.grid.ForceRefresh()  # Added in 2.3.1
    except AttributeError:
      pass

    #
    # If the first cell has a trigger, select it
    #
    try:
      trigger=self.triggerMap[self.rowList[0]]
      if trigger.src:
        self.editor.dispatchEvent('TriggerSelected',object=self.namedMap[trigger.src], originator=__name__)
      else:
        self.editor.dispatchEvent('TriggerSelected',object=trigger, originator=__name__)
    except KeyError:
      pass
    except IndexError:
      pass

  def getPageText(self):
    return self.object._type[2:]

  def onCreateObject (self, event):
    self.inventoryObject(event.object)

  def onModifyObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if isinstance(object,GTrigger) and \
       object.type.upper() == 'NAMED' and \
       event.new.has_key('name'):
      try:
        del self.namedMap[event.old['name']]
      except KeyError:
        pass
      self.namedMap[object.name] = object
      self._setCurrent(self.object)

    try:
      if handler != __name__ and self.object == object:
        self._setCurrent(object)
    except wxPyDeadObjectError:
      pass


  def onDeleteObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if handler != __name__:
      pass


  def inventoryObject(self, object):
    if isinstance(object,GTrigger) and object.type.upper() == 'NAMED':
      self.namedMap[object.name] = object


  def onSize(self, event):
    self.grid.SetSize(self.GetClientSizeTuple())
    w,h = self.grid.GetClientSizeTuple()
    self.grid.SetColSize(0, w - self.grid.GetRowLabelSize()-1)
    try:
      self.grid.ForceRefresh()  # Added in 2.3.1
    except AttributeError:
      pass

  # Force a grid cell into Edit mode when Double-Clicked
  def OnLeftDClick(self,evt):
    if self.grid.CanEnableCellControl():
      self.grid.EnableCellEditControl()


  def _generateName(self, object):
    parent = object._parent
    name = '%s.%s (%s-Level Trigger)' % (parent.name, parent._validTriggers[object.type.upper()], parent._type[2:].capitalize())
    return name


  # Fire a TriggerSelected event
  def OnCellSelected(self,evt):
    try:
      trigger=self.triggerMap[self.rowList[evt.GetRow()]]
      if trigger.src:
        self.editor.dispatchEvent('TriggerSelected',object=self.namedMap[trigger.src], originator=__name__)
      else:
        self.editor.dispatchEvent('TriggerSelected',object=trigger, originator=__name__)
    except KeyError:
      pass
    except IndexError:
      pass

    evt.Skip()


  def OnCellChange(self, evt):
    attr = self.rowList[evt.GetRow()]
    value = self.grid.GetCellValue(evt.GetRow(), evt.GetCol())
    try:
      oldvalue = self.mappedTriggers[attr]
    except:
      oldvalue = ""

    if oldvalue == value:
      return

    trigType = self.object._validTriggers[attr]

    try:
      trigger = self.triggerMap[attr]
      new = 0
    except KeyError:
      trigger = GTrigger(self.object, trigType)
      new = 1

    oldStuff = {}
    newStuff = {}

    # If custom trigger, save the text in case the user changes his mind
    if oldvalue == '(Custom Trigger)':
      try:
        self.object.__triggerHistory
      except:
        self.object.__triggerHistory = {}
      self.object.__triggerHistory[attr] = trigger.getChildrenAsContent()
    elif oldvalue:
      oldStuff[src] = oldvalue
      newStuff[src] = ""

    # Clear any old content
    trigger._children = []

    if value:
      if value == '(Custom Trigger)':
        try:
          # Restore a previous life's custom trigger text...
          text = self.object.__triggerHistory[attr]
        except:
          # ... or not, and create a new blank trigger
          text = "##\n## %s\n##\n\nreturn\n\n" % (self._generateName(trigger))
        trigger.src = None
        GContent(trigger,text)
      elif value:
        # named trigger
        trigger.src = value
        if not oldStuff.has_key('src'):
          oldStuff['src'] = ''
        newStuff['src'] = value

      if new:
        self.editor.dispatchEvent('ObjectCreated',object=trigger, originator=__name__)
      else:
        self.editor.dispatchEvent('ObjectModified',object=trigger, originator=__name__,
               old=oldStuff, new=newStuff)

      if trigger.src:
        self.editor.dispatchEvent('TriggerSelected',object=self.namedMap[trigger.src], originator=__name__)
      else:
        self.editor.dispatchEvent('TriggerSelected',object=trigger, originator=__name__)
    else:
      # No trigger
##      self.object._children.pop(self.object._children.index(trigger))
      self.editor.dispatchEvent('ObjectDeleted',object=trigger, originator=__name__)
      self.editor.dispatchEvent('TriggerSelected',object=None, originator=__name__)


##
##
##
class EventCellEditor (CharCellEditor):
  def __init__(self, grid, attributes):
    CharCellEditor.__init__(self, grid, attributes)
    self.valueList = attributes.keys()

    self.valueList.sort()
    self.selectionList = []
    self.valueMap = {}

    i = 0
    for v in self.valueList:
      self.valueMap[v] = i
      self.selectionList.append("%s" % (attributes[v] or v))
      i = i + 1

##    self.valueList.insert(0, '')

  def Create(self, parent, id, evtHandler):
    GDebug.printMesg(10,'Creating CharCellEditor')
    self._tc = wxComboBox(parent, id, "", style=wxCB_READONLY,
                          choices=self.selectionList)
    self.SetControl(self._tc)
    if evtHandler:
      self._tc.PushEventHandler(evtHandler)

  def BeginEdit(self, row, col, grid):
    self.startValue = grid.GetTable().GetValue(row, col)
    self._tc.SetSelection(self.valueMap[self.startValue])
    self._tc.SetFocus()

  def EndEdit(self, row, col, grid):
    changed = false

    val = self.valueList[self._tc.GetSelection()]
    if val != self.startValue:
       changed = true
       grid.GetTable().SetValue(row, col, "%s" % val) # update the table

    self.startValue = ''
    self._tc.SetSelection(0)
    return changed


  def Reset(self):
    self._tc.SetSelection(self.valueMap[self.startValue])
    #self._tc.SetInsertionPointEnd()


  def StartingKey(self, evt):
    key = evt.GetKeyCode()
    ch = None
    if key in [WXK_NUMPAD0, WXK_NUMPAD1, WXK_NUMPAD2, WXK_NUMPAD3, WXK_NUMPAD4,
               WXK_NUMPAD5, WXK_NUMPAD6, WXK_NUMPAD7, WXK_NUMPAD8, WXK_NUMPAD9]:
        ch = ch = chr(ord('0') + key - WXK_NUMPAD0)

    elif key < 256 and key >= 0 and chr(key):
      ch = chr(key)
      if not evt.ShiftDown():
        ch = string.lower(ch)

    if ch is not None:
      # For this example, replace the text.  Normally we would append it.
      #self._tc.AppendText(ch)
      pass
#      self._tc.SetValue(ch)
#      self._tc.SetInsertionPointEnd()
    else:
      evt.Skip()

