#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# $Id: GFParser.py 7108 2005-03-08 08:08:03Z reinhard $

"""
Class that contains a sax based xml processor for GNUe forms

NOTES:
 * Designer uses the 'Positionable' attribute. It is specific to
   forms+designer and is not part of the GParser spec. If set to
   true, then this object is a visible, movable, sizable attribute.
"""

from gnue.common.datasources import GDataSource
from gnue.common.definitions import GParser
from gnue.common.formatting import GTypecast
from gnue.common.logic import GTrigger
from gnue.common import GMenu

import copy, types





########
########  Please keep this file neat !!!
########





##
##
##
def loadFile (buffer, instance, initialize=True, url = None):
  """
  This method loads a form from an XML file and returns
  a GFForm object.  If initialize is 1 (default), then
  the form is initialized and ready to go.

  (initialize=0 is currently not used -- will probably
  be used in the Forms Designer package where we will
  not want the loaded form to connect to databases, etc)
  """
  return GParser.loadXMLObject (buffer, xmlFormsHandler, 'GFForm', 'form',
           initialize, attributes={"_instance": instance,
                               "_parameters": instance._parameters,
                               "_connections": instance.connections },
           url = url)


xmlElements = None


def getXMLelements():

  global xmlElements

  if xmlElements == None:
    from gnue.forms import GFObjects, GFLibrary, GFForm

    xmlElements = {
      'form': {
         'BaseClass': GFForm.GFForm,
         'Required': True,
         'SingleInstance': True,
         'Attributes': {
            'title': {
               'Typecast': GTypecast.text,
               'Default': 'Untitled Form',
               'Label': _('Title'),
               'Description': 'The title of the form.' },
            'readonly': {
               'Typecast': GTypecast.boolean,
               'Default': False,
               'Label': _('Read Only'),
               'Description': 'If set to {Y}, then no modifications to data '
                              'by the end user will be allowed. The form will '
                              'become a query-only form.' },
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
                'Label': _('Name'),
               'Description': 'A unique name or ID for the form.' },
            'style': {
               'Typecast': GTypecast.name,
                'Label': _('Style'),
               'ValueSet': {
                  'normal': {'Label': _('Normal')},
                  'dialog': {'Label': _('Dialog')} },
               'Default': 'normal',
               'Description': 'Display as normal or dialog-style window.'  }, },
         'ParentTags': None,
         'Description': 'Top-level element that encloses all the logic '
                        'and visuals that the user interface will show '
                        'to the user.' },

      'logic': {
         'BaseClass': GFObjects.GFLogic,
         'Required': True,
         'Importable': True,
         'SingleInstance': True,
         'ParentTags': ('form',),
         'Description': 'Separation layer that contains "Business logic": '
                        'blocks, fields, block-level and field-level triggers.'},

      'layout': {
         'BaseClass': GFObjects.GFLayout,
         'Required': True,
         'Importable': True,
         'SingleInstance': True,
         'ParentTags': ('form',),
         'Description': 'Separation layer that contains all the '
                        'visual elements on the form.' ,
         'Attributes': {
           'tabbed': {
             'Typecast': GTypecast.name,
             'Label': _('Tab Location'),
             'ValueSet': {
                'none':   {'Label': _('No tabs')},
                'left':   {'Label': _('Left tabs')},
                'right':  {'Label': _('Right tabs')},
                'bottom': {'Label': _('Botton tabs')},
                'top':    {'Label': _('Top tabs')} },
           'Default': "none",
           'Description': 'Informs the UI subsystem to display a form\'s pages '
                          'as notebook tabs. Allowed values are {left}, '
                          '{right}, {bottom}, {top}.  If the UI driver in use '
                          'does not support the chosen tab position '
                          '(or tabs at all,) then the UI driver may choose '
                          'another tab position.' } } } ,

      'page': {
         'BaseClass': GFObjects.GFPage,
         'Required': True,
         'Importable': True,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'A unique ID for the widget. This is only useful '
                              'when importing pages from a library.' },
            'transparent':{
               'Typecast': GTypecast.boolean,
               'Default': False,
               'Description': 'If set, then you can tab out of the page via next- '
                              'or previous-field events. Makes navigation in '
                              'mutlipage forms easier. If false, focus stays '
                              'within a page until user explicitly moves to '
                              'another page' },
            'style': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'normal': {'Label': _('Normal')},
## TODO ##         'popup': {},
                   },
               'Default': 'normal',
               'Description': 'The type of page.' },
            'caption': {
               'Typecast': GTypecast.text,
               'Description': 'For {tabbed} or {popup} pages, this contains '
                              'the caption to use for the page.' } },
         'ParentTags': ('layout',),
         'Description': 'Encapsulates visual elements to be displayed '
                        'on a page.' },

      'block': {
         'BaseClass': GFObjects.GFBlock,
         'Required': True,
         'Importable': True,
         'Attributes': {
            'name': {
               'Required': True,
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'A unique ID (name) for the widget. '
                              'No blocks can share '
                              'the same name without causing namespace '
                              'collisions in user triggers.' },
            'rows': {
               'Typecast': GTypecast.whole,
               'Description': 'Any widgets inside the block will display this '
                              'number of copies in a verticle column. Simulates '
                              'a grid entry system.'},
            'rowSpacer': {
               'Typecast': GTypecast.whole,
               'Label': _('Row Spacing'),
               'Description': 'Adjusts the vertical gap of this number of rows '
                              'between duplicated widgets. Serves the same '
                              'purpose as some of the gap attributes on '
                              'individual widgets.' },
            'transparent':{
               'Typecast': GTypecast.boolean,
               'Label': _('Transparent Nav'),
               'Default': True,
               'Description': 'If set, then you can tab out of the block via next- '
                              'or previous-field events. Makes navigation in '
                              'multiblock forms easier. If false, focus stays '
                              'within a block until user explicitly moves to '
                              'another block. Note that a block\'s {autoNextRecord}'
                              'setting affects {transparent} behavior' },
            'autoCreate':{
               'Typecast': GTypecast.boolean,
               'Label': _('Auto Create Record'),
               'Default': True,
               'Description': 'If set, then if you attempt to go to the next record '
                              'while at the last record, a new record is created.'},
            'autoNextRecord':{
               'Typecast': GTypecast.boolean,
               'Label': _('Auto Next Record'),
               'Default': False,
               'Description': 'If set, then if you tab at the end of a block, you '
                              'will be taken to the next record. If the current '
                              'record is empty and transparent is true, then '
                              'you will be taken to the next block'},
            'autoCommit':{
               'Typecast': GTypecast.boolean,
               'Label': _('Auto Commit'),
               'Default': False,
               'Description': 'If set, then the datasource will automatically '
                              'commit changes when trying to navigate out of the '
                              'current record.'},
            'autoClear':{
               'Typecast': GTypecast.boolean,
               'Label': _('Auto Clear on Commit'),
               'Default': False,
               'Description': 'If set, then the block is cleared/emptied on '
                              'a commit.'},
            'editable': {
               'Description': 'Can records be edited/created?',
               'Label': _('Allow Editing'),
               'ValueSet': {
                  'Y': {'Label': _('Yes')},
                  'N': {'Label': _('No')},
                  'update': {'Label': _('Update Only')},
                  'new': {'Label': _('New Records Only')} },
               'Typecast': GTypecast.text,
               'Default': 'Y' },
            'queryable': {
               'Description': 'Can records be queried?',
               'Label': _('Allow Querying'),
               'Typecast': GTypecast.boolean,
               'Default': True },
            'deletable': {
               'Description': 'Can records be deleted?',
               'Label': _('Allow Deletes'),
               'Typecast': GTypecast.boolean,
               'Default': True },
            'navigable': {
               'Description': 'Can this block be navigated?',
               'Label': _('Navigable'),
               'Typecast': GTypecast.boolean,
               'Default': True },
            'restrictDelete':{
               'Typecast': GTypecast.boolean,
               'Default': False,
               'Deprecated': 'Use deletable="N"',
               'Label': _('Prevent Deletes'),
               'Description': 'If set then the user will be unable to request '
                               'that a record be deleted via the user interface.' },
            'restrictInsert':{
               'Typecast': GTypecast.boolean,
               'Default': False,
               'Label': _('Prevent Inserts'),
               'Deprecated': 'Use editable="update"',
               'Description': 'If set then the user will be unable to request '
                              'that new records be inserted into the block.' },
            'datasource': {
               'References': 'datasource.name',
               'Typecast': GTypecast.name,
               'Description': 'The name of a datasource (defined in by a '
                              '{<datasource>} tag) that provides this block '
                              'with it\'s data.' } },
         'ParentTags': ('logic',),
         'Description': 'A block contains instructions on how Forms '
                        'should interact with a datasource.' },

      'label': {
         'BaseClass': GFObjects.GFLabel,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'The unique ID of the label.' },
            'text': {
               'Required': True,
               'Typecast': GTypecast.text,
               'Description': 'The text to be displayed.' },
            'alignment': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                  'left': {'Label': _('Left')},
                  'right': {'Label': _('Right')},
                  'center': {'Label': _('Centered')} },
               'Default': "left",
               'Description': 'The justification of the label. Can be one of '
                              'the following: {left}, {right}, or {center}. '
                              'Requires that the {width} attribute be set.'},
            'rows': {
               'Typecast': GTypecast.whole,
               'Description': 'Overrides the rows setting defined at the '
                              'block level. ' },
            'rowSpacer': {
               'Typecast': GTypecast.whole,
               'Label': _('Row Spacing'),
               'Description': 'Overriders the rowSpace setting defined at the '
                              'block level.' } },
         'Positionable': True,
         'ParentTags': ('page',),
         'Description': 'Displays static text' },

      'field': {
         'BaseClass': GFObjects.GFField,
         'Importable': True,
         'Attributes': {
            'name': {
               'Required': True,
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'The unique ID of the entry. Referenced in '
                              'master/detail setups as well as triggers.' },
            'field': {
               'Typecast': GTypecast.name,
               'Label': _('Field (Database)'),
               'Description': 'The name of the field in the datasource to '
                              'which this widget is tied.' },
            'maxLength': {
               'Typecast': GTypecast.whole,
               'Label': _('Max Text Length'),
               'Description': 'The maximum number of characters the user is '
                              'allowed to enter into the entry.' },
            'minLength': {
               'Typecast': GTypecast.whole,
               'Label': _('Min Text Length'),
               'Description': 'The minimum number of characters the user must '
                              'enter into the entry.',
               'Default': 0 },
            'max_length': {
               'Typecast': GTypecast.whole,
               'Deprecated': 'Use maxLength',
               'Description': 'The maximum number of characters the user is '
                              'allowed to enter into the entry.' },
            'min_length': {
               'Typecast': GTypecast.whole,
               'Deprecated': 'Use minLength',
               'Description': 'The minimum number of characters the user must '
                              'enter into the entry.',
               'Default': 0 },
            'readonly': {
               'Typecast': GTypecast.boolean,
               'Label': _('Read Only'),
               'Description': 'It defined the user will be unable to alter '
                              'the contents of this entry. Triggers can still '
                              'alter the value.',
# TODO: I think this should be deprecated, but will see how others use first
#               'Deprecated': 'Use editable="N" instead.',
               'Default': False   },
            'required': {
               'Description': 'This object cannot have an empty value prior '
                              'to a commit.',
               'Typecast': GTypecast.boolean,
               'Default': False   },
            'case': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                  'mixed': {'Label': _('As Entered')},
                  'upper': {'Label': _('Upper case')},
                  'lower': {'Label': _('Lower case')} },
               'Default': 'mixed',
               'Description': 'Convert the value to uppercase/lowercase '
                              'or leave it as it is.'  },
            'typecast': {
               'Typecast': GTypecast.name,
               'Label': _('Data Type'),
               'ValueSet': {
                  'text': {'Label': _('Text')},
                  'number': {'Label': _('Numeric')},
                  'date': {'Label': _('Date/Time')} },
               'Default': 'text',
               'Description': 'The type of data the entry widget will accept. '
                              'Possible values are {text}, {number}, {date}.'},
            'value': {
               'Typecast': GTypecast.text,
               'Deprecated': 'Use default="..." instead',
               'Description': 'Deprecated' },
            'fk_source': {
               'References': 'datasource.name',
               'Label': _('F/K Datasource'),
               'Typecast': GTypecast.name,
               'Description': 'Source table that the foreign key links to.' },
            'fk_key': {
               'Label': _('F/K Bound Field'),
               'Typecast': GTypecast.name,
               'Description': 'The table column (field) in the foreign key '
                              'source table that the foreign key links to.' },
            'fk_description': {
               'Typecast': GTypecast.name,
               'Label': _('F/K Description Field'),
               'Description': 'The description used if a style of dropdown is '
                              'selected. This field\'s value is displayed in '
                              'the dropdown but the foreign_key value is '
                              'actually stored in the field. This allows you '
                              'to display something like the full name of a '
                              'US state but only store it\'s 2 character '
                              'abbreviation.' },
            'fk_refresh': {
               'Typecast': GTypecast.name,
               'Label': _('F/K Refresh Method'),
               'ValueSet': {
                  'startup': {'Label': _('On form startup')},
                  'change': {'Label': _('On field modification')},
                  'commit': {'Label': _('On commit')} },
               'Default': 'startup',
               'Description': 'Decides when the foreign key should be '
                              'refreshed.' },
            'default': {
               'Typecast': GTypecast.text,
               'Label': _('Default (New Records)'),
               'Description': 'The default value for this field when a new '
                              'record is created. '
                              'If the field is visible the user can override '
                              'the value.' },
            'defaultToLast': {
               'Typecast': GTypecast.boolean,
               'Label': _('Default to last entry'),
               'Default': False,
               'Description': 'If {Y}, then new records will default to the '
                              'last value the user entered for this field. If '
                              'no new values have been entered, then defaults '
                              'back to the normal {default} setting.' },
            'queryDefault':{
               'Typecast': GTypecast.text,
               'Label': _('Default (Querying)'),
               'Description': 'The field will be populated with this value '
                              'automatically when a query is requested. If '
                              'the field is visible the user can still '
                              'override the value.' },
            'sloppyQuery': {
               'Typecast': GTypecast.text,
               'Label': _('Sloppy Queries'),
               'Description': 'When set, whatever value the user enters for '
                              'the query mask is rewritten with % between '
                              'each character. Thus {example} would be '
                              'queried as {%e%x%a%m%p%l%e%}' },
            'ignoreCaseOnQuery': {
               'Typecast': GTypecast.boolean,
               'Label': _('Ignore Case on Queries'),
               'Default': False,
               'Description': 'If "Y", the entry widget ignores the case '
                              'of the information entered into the query mask.' },
            'editable': {
               'Description': 'Only allow this object to be edited if it '
                              'is currently empty.',
               'Label': _('Allow Editing'),
               'ValueSet': {
                  'Y': {'Label': _('Yes')},
                  'N': {'Label': _('No')},
                  'null': {'Label': _('Null Only')},
                  'update': {'Label': _('Update Only')},
                  'new': {'Label': _('New Records Only')} },
               'Typecast': GTypecast.text,
               'Default': 'Y' },
            'queryable': {
               'Description': 'Is this object queryable?',
               'Label': _('Allow Query'),
               'Typecast': GTypecast.boolean,
               'Default': True },
            'ltrim': {
               'Label': _('Trim left spaces'),
               'Description': 'Trim extraneous space at '
                              'beginning of user input.',
               'Typecast': GTypecast.boolean,
               'Default': False },
            'rtrim': {
               'Label': _('Trim right spaces'),
               'Description': 'Trim extraneous space at end '
                              'of user input.',
               'Typecast': GTypecast.boolean,
               'Default': True } },
         'ParentTags': ('block',),
         'Description': 'A field represents a column in the database table '
                        'designated by the block.' },

      # If you implement a new entry "style", add to the entryStyles
      # structure after this list
      'entry': {
         'BaseClass': GFObjects.GFEntry,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'The unique ID of the entry.' },
            'label': {
               'Required': False,
               'Typecast': GTypecast.text,
               'Description': 'The optional label displayed next to checkbox.' },
            'field': {
               'Typecast': GTypecast.name,
               'References': 'field.name',
               'Required': True,
               'Description': 'The name of the field that this ties to.' },
            'block': {
               'Typecast': GTypecast.name,
               'Required': True,
               'References': 'block.name',
               'Description': 'The name of the block that this ties to.' },
            'focusorder': {
               'Typecast': GTypecast.whole,
               'Label': _('Focus Order'),
               'Description': 'Defines what order the focus moves through '
                              'entries.'},
            'rows': {
               'Typecast': GTypecast.whole,
               'Description': 'Overrides the rows setting defined at the '
                              'block level.'},
            'rowSpacer': {
               'Typecast': GTypecast.whole,
               'Label': _('Row Spacing'),
               'Description': 'Overrides the rowSpacer setting at the '
                              'block level.' },
            'navigable': {
               'Typecast': GTypecast.boolean,
               'Description': 'If false, the user will be unable to navigate '
                              'to this entry. Triggers can still '
                              'alter the value though.',
               'Default': True   },
            'hidden': {
               'Typecast': GTypecast.boolean,
               'Default': False,
               'Description': 'If "Y", the entry widget will not be '
                              'displayed on the form. This is useful for '
                              'fields the user doesn\'t need to know about '
                              'that you wish to update via triggers.'   },
            'style': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                  'default': {'Label': _('Default')},
                  'password': {'Label': _('Password/Hidden')},
                  'dropdown': {'Label': _('Dropdown/Combo box')},
                  'listbox': {'Label': _('Listbox')},
                  'checkbox': {'Label': _('Checkbox')},
                  'label': {'Label': _('Label (non-editable)')} },
               'Default': 'default',
               'Description': 'The style of entry widget requested. '
                              'Currently either {text}, {label}, {checkbox}, '
                              '{listbox}, or {dropdown}. To use {listbox} or '
                              '{dropdown} you are required to use both the '
                              '{fk_source}, {fk_key}, and {fk_description} '
                              'attributes. The {label} style implies the '
                              '{readonly} attribute.'  },
            'formatmask': {
               'Typecast': GTypecast.text,
               'Label': _('Format Mask'),
               'Description': 'TODO' },
            'inputmask': {
               'Typecast': GTypecast.text,
               'Label': _('Input Mask'),
               'Description': 'Defines how the user will edit a field\'s '
                              'value.' },
            'displaymask': {
               'Label': _('Display Mask'),
               'Typecast': GTypecast.text,
               'Description': 'Defines how the field data will be formatted '
                              'for display.' } },
         'Positionable': True,
         'ParentTags': ('page',),
         'Description': 'An {entry} is the visual counterpart to a {field}, '
                        'it defines how the data in a field will be displayed '
                        'and how it can be edited.'},

      'scrollbar': {
         'BaseClass': GFObjects.GFScrollBar,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'The unique name of the scrollbar.' },
            'block': {
               'Required': True,
               'Typecast': GTypecast.name,
               'References': 'block.name',
               'Description': 'The {block} to which this scrollbar scrolls.' } },
         'Positionable': True,
         'ParentTags': ('page',),
         'Description': 'A scrollbar is a visual element that lets the user '
                        'move vertically layout elements linked to it.' },

      'box': {
         'BaseClass': GFObjects.GFBox,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'The unique name of the box.' },
            'label': {
               'Typecast': GTypecast.text,
               'Description': 'An optional text label that will be displayed '
                              'on the border.' },
            'focusorder': {
               'Typecast': GTypecast.whole,
               'Description': 'Defines what order the focus moves through '
                              'entries.' } },
         'Positionable': True,
         'ParentTags': ('page',),
         'Description': 'A box is a visual element that draws a box around '
                        'other visual elements, thus providing logical '
                        'separation for them.' },
      'image': {
         'BaseClass': GFObjects.GFImage,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'The unique name of the image.' },
            'field': {
               'Typecast': GTypecast.name,
               'References': 'field.name',
               'Required': True,
               'Description': 'The name of the field that this ties to.' },
            'block': {
               'Typecast': GTypecast.name,
               'References': 'block.name',
               'Required': True,
               'Description': 'The name of the block that this ties to.' },
            'type':        {
               'Typecast': GTypecast.name,
               'ValueSet': {
                  'URL': {'Label': _('Field contains the URL of the image')},
                  'PIL': {'Label': _('Field contains a PIL encoding of the image')} },
               'Default': "URL",
               'Description': 'The type of image reference. Can be {URL} '
                              'for a url reference, or {PIL} for an '
                              'embedded image.' },
            'fit':  {
               'Typecast': GTypecast.name,
               'ValueSet': {
                  'none': {'Label': _('Full-size image (no scaling)')},
                  'width': {'Label': _('Scale to width')},
                  'height': {'Label': _('Scale to height')},
                  'both': {'Label': _('Scale width and height (may distort image)')},
                  'auto': {'Label': _('Use a best-fit algorithm')} },
               'Default': "none",
               'Description': 'Defines how the image will fill the space '
                              'provided for it (crop parts outside borders, '
                              'or stretch width/height/both to fit into '
                              'given boundaries).' },
            'editable': {
               'Description': 'Only allow this object to be edited if it '
                              'is currently empty.',
               'Label': _('Allow Editing'),
               'ValueSet': {
                  'Y': {'Label': _('Yes')},
                  'N': {'Label': _('No')},
                  'null': {'Label': _('Null Only')},
                  'update': {'Label': _('Update Only')},
                  'new': {'Label': _('New Records Only')} },
               'Typecast': GTypecast.text,
               'Default': 'Y' },
            'focusorder': {
               'Typecast': GTypecast.whole,
               'Description': 'Defines what order the focus moves through '
                              'entries.'  } },
         'Positionable': True,
         'ParentTags': ('page',),
         'Description': 'Displays an image.' },

      'component': {
         'BaseClass': GFObjects.GFComponent,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'The unique name of the component.' },
            'field': {
               'Typecast': GTypecast.name,
               'References': 'field.name',
               'Required': True,
               'Description': 'The name of the field that this ties to.' },
            'block': {
               'Typecast': GTypecast.name,
               'References': 'block.name',
               'Required': True,
               'Description': 'The name of the block that this ties to.' },
            'mimetype':        {
               'Typecast': GTypecast.name,
               'Required': True,
               'Description': 'TODO' },
            'type':        {
               'Typecast': GTypecast.name,
               'ValueSet': {
                  'URL': {'Label': _('Field contains the URL of the component')},
                  'Base64': {'Label': _("Field contains the data of the "
                                        "component in Base64 encoding")} },
               'Default': "URL",
               'Description': 'TODO' },
            'focusorder': {
               'Typecast': GTypecast.whole,
               'Description': 'Defines what order the focus moves through '
                              'entries.'  } },
         'Positionable': True,
         'ParentTags': ('page',),
         'Description': 'TODO' },

      'button': {
         'BaseClass': GFObjects.GFButton,
         'Importable': True,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'A unique ID for the widget. Useful for '
                              'importable buttons. ' },
            'navigable': {
               'Description': 'Can this button be navigated?',
               'Label': _('Navigable'),
               'Typecast': GTypecast.boolean,
               'Default': True },
            'block': {
               'Typecast': GTypecast.name,
               'References': 'block.name',
               'Description': 'The (optional) name of the block that this '
                              'ties to. If a button is associated with '
                              'a block, then the button honors '
                              'the block\'s rows= value.' },
            'focusorder': {
               'Typecast': GTypecast.whole,
               'Description': 'Defines what order the focus moves through '
                              'entries.'},
            'label': {
               'Typecast': GTypecast.name,
               'Description': 'The text that should appear on the button' } },
         'Positionable': True,
         'ParentTags': ('page',),
         'Description': 'A visual element with text placed on it, that '
                        'the user can push or click, and that event can run '
                        'a bound trigger.' },

     'tree': {
         'BaseClass': GFObjects.GFTree,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'Unique name of the tree.' },
            'rows': {
               'Typecast': GTypecast.whole,
               'Description': 'Overrides the rows setting defined at '
                              'the block level.' },
            'rowSpacer': {
               'Typecast': GTypecast.whole,
               'Label': _('Row Spacing'),
               'Description': 'Overriders the rowSpace setting defined at '
                              'the block level.' },
             'block': {
               'Typecast': GTypecast.name,
               'Required': True,
               'References': 'block.name',
               'Description': 'The name of the block the tree ties to.' },
            'fld_desc': {
               'Label': _('Description field'),
               'Typecast': GTypecast.name,
               'Required': True,
               'Description': 'The visible text of the TreeViewItem' },
            'fld_id': {
               'Label': _('Own ID field'),
               'Typecast': GTypecast.name,
               'Required': True,
               'Description': 'The field containing the TreeViewItem ID' },
            'fld_parentid': {
               'Label': _('Parent ID field'),
               'Typecast': GTypecast.name,
               'Required': True,
               'Description': 'The field containing the TreeViewItem parent' },
            'fld_tblhint': {
               'Label': _('Table hint for leaf tables'),
               'Typecast': GTypecast.name,
               'Required': True,
               'Description': 'The field containing the table-hint for leaves' } },
           'Positionable': True,
           'ParentTags': ('page',),
           'Description': 'Tree/List-View of a table' },

      'leaf': {
         'BaseClass': GFObjects.GFLeaf,
         'Attributes': {
            'name': {
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'Leaf of the Tree Widget' },
            'block': {
               'Typecast': GTypecast.name,
               'Required': True,
               'References': 'block.name',
               'Description': 'Block used for db operations' },
            'fld_id': {
               'Label': _('ID of a field'),   # TODO: we need some kind of restriction
               'Typecast': GTypecast.name,    # TODO: on which entries we pick...
               'Required': True,
               'Description': 'TODO' },
            'fld_desc': {
               'Label': _('Description field'),
               'Typecast': GTypecast.name,
               'Required': True,
               'Description': 'TODO' },
            'tblhint': {
               'Typecast': GTypecast.name,
               'Required': True,
               'Description': 'Table of this leaf' } },
         'Positionable': True,
         'ParentTags': ('tree',),
         'Description': 'Leaf information of a tree view' },

      'column': {
        'BaseClass': GFObjects.GFColumn,
          'Attributes': {
             'name': {
               'Unique': True,
               'Required': True,
               'Typecast': GTypecast.name,
               'Description': 'Column info for leaves/trees' },
             'align': {
               'Typecast': GTypecast.name,
               'Required': True,
               'Description': 'Contents alignment: left/right/center' },
             'field': {  # we'll have to remove this, or replace it with some more general source
               'Typecast': GTypecast.name,
               'Description': 'Field to get column text from' } },
          'Positionable': False,
          'ParentTags': ('tree',),
          'Descrpition': 'Column information for trees' },

      'colsrc': {
        'BaseClass': GFObjects.GFColsrc,
          'Attributes': {
             'covers': {
               'Unique': True,
               'Required': True,
               'Typecast': GTypecast.name,
               'Description': 'Specifies the column this belongs to' },
             'field': {
               'Typecast': GTypecast.name,
               'Description': 'Field to get column text from' } },
          'Positionable': False,
          'ParentTags': ('leaf',),
          'Descrpition': 'Column information for trees' },

      'options': {
         'BaseClass': GFObjects.GFOptions,
         'UsableBySiblings': True,
         'ParentTags': ('form',),
         'Description': 'TODO' },

      'option': {
         'BaseClass': GFObjects.GFOption,
         'Attributes': {
            'name': {
               'Required': True,
               'Typecast': GTypecast.name,
               'Description': 'TODO' },
            'value': {
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'MixedContent': True,
         'ParentTags': ('options',),
         'Description': 'TODO' },

      'title': {
         'BaseClass': GFObjects.GFOption,
         'Attributes': {
            'name': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'title': {} },
               'Default': 'title',
               'Description': 'TODO' },
            'value': {
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'MixedContent': True,
         'SingleInstance': True,
         'Deprecated': 'Use the <form> attribute "title" instead.',
         'ParentTags': ('options',),
         'Description': 'TODO' },

      'name': {
         'BaseClass': GFObjects.GFOption,
         'Attributes': {
            'name': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'name': {} },
               'Default': 'name',
               'Description': 'TODO' },
            'value': {
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'MixedContent': True,
         'SingleInstance': True,
         'ParentTags': ('options',),
         'Description': 'TODO' },

      'author': {
         'BaseClass': GFObjects.GFOption,
         'Attributes': {
            'name': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'author': {} },
               'Default': 'author',
               'Description': 'TODO' },
            'value': {
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'MixedContent': True,
         'SingleInstance': True,
         'ParentTags': ('options',),
         'Description': 'TODO' },

      'description':{
         'BaseClass': GFObjects.GFOption,
         'Attributes': {
            'name': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'description': {} },
               'Default': 'description',
               'Description': 'TODO' },
            'value': {
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'MixedContent': True,
         'SingleInstance': True,
         'ParentTags': ('options',),
         'Description': 'TODO' },

      'version': {
         'BaseClass': GFObjects.GFOption,
         'Attributes': {
            'name': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'version': {} },
               'Default': 'version',
               'Description': 'TODO' },
            'value': {
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'MixedContent': True,
         'SingleInstance': True,
         'ParentTags': ('options',),
         'Description': 'TODO' },

      'tip': {
         'BaseClass': GFObjects.GFOption,
         'Attributes': {
            'name': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'tip': {} },
               'Default': 'tip',
               'Description': 'TODO' },
            'value': {
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'MixedContent': True,
         'SingleInstance': True,
         'ParentTags': ('options',),
         'Description': 'TODO' },
      'parameter':    {
         'BaseClass': GFObjects.GFParameter,
         'Attributes': {
            'name':          {
               'Required': True,
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'Unique name of the parameter.' },
            'required':    {
               'Typecast': GTypecast.boolean,
               'Default': False,
               'Description': 'Is it obligatory to provide this parameter '
                              'to run the form?' },
##            'limited':     {
##               'Typecast': GTypecast.boolean,
##               'Default': False,
##               'Description': 'TODO' },
            'default':     {
               'Typecast': GTypecast.text,
               'Description': 'Default value of the parameter, if the user '
                              'does not provide a value for it.' },
            'description': {
##               'Required': True,
               'Typecast': GTypecast.text,
               'Description': 'Provide some textual information '
                              'to describe the parameter better.' },
##            'source':      {
##               'Typecast': GTypecast.name,
##               'Description': 'TODO' },
            'type':        {
               'Typecast': GTypecast.name,
               'Default': "char",
               'Description': 'TODO' } },
         'ParentTags':  ('form',),
         'Description': 'A form can get parameters from the outer world '
                        'or a calling form, and can pass values back too '
                        'in these parameters.' },
    }


    #
    # Create the dialog alias for the forms
    #
    copy._deepcopy_dispatch[types.FunctionType] = copy._deepcopy_atomic
    copy._deepcopy_dispatch[types.ClassType] = copy._deepcopy_atomic
    copy._deepcopy_dispatch[type(int)] = copy._deepcopy_atomic
    dialog=copy.deepcopy(xmlElements['form'])
    dialog['Required'] = False
    dialog['SingleInstance'] = False
    dialog['Importable'] = True
    dialog['Attributes']['style']['Default']='dialog'
    dialog['ParentTags']= ('form',)
    xmlElements.update({'dialog':dialog})


    #
    # Add DataSource elements
    #
    datasourceElems = GDataSource.getXMLelements(
        updates={'datasource': {
                   'BaseClass': GFObjects.GFDataSource,
                   'ParentTags': ('form',) },
                 'cparam': {
                   'BaseClass': GFObjects.GFCParam }
                 })

    # TODO: This may get moved to GCommon's datasources,
    # TODO: but first have to determine if it will confuse
    # TODO: development for other tools.
    datasourceElems['datasource']['Attributes'].update( {
            'detailmin':  {
               'Label': _('M/D Min Child Rows'),
               'Description': 'If this datasource is the child in a '
                              'master/detail relationship, this property '
                              'specifies the minimum number of child '
                              'records that must be present for a commit '
                              'to be valid. Usually, this is 0 (for '
                              'one-to-many relationships) or 1 '
                              '(for one-to-one relationships).',
               'Default': 0,
               'Typecast': GTypecast.whole },
            'detailmax':  {
               'Label': _('M/D Max Child Rows'),
               'Description': 'If this datasource is the child in a '
                              'master/detail relationship, this property '
                              'specifies the maximum number of child '
                              'records that can be created. Usually, this '
                              'is either omitted (for one-to-many '
                              'relationships) or 1 (for one-to-one '
                              'relationships).',
               'Typecast': GTypecast.whole } })

    xmlElements.update( datasourceElems )

    #
    # Add trigger elements
    #
    xmlElements.update(
      GTrigger.getXMLelements(
        updates={'trigger':{
                   'ParentTags': ('form',)},
                }))

    #
    # Add menu elements
    #
    xmlElements.update(
        GMenu.getXMLelements(
                updates={'menu': {
                   ##'BaseClass': GFObjects.GFMenu,
                   'ParentTags': ('form',) },
                 }))

    xmlElements = GParser.buildImportableTags ('form', xmlElements)
  return xmlElements

#
# Different Types of Entries
#
entryStyles = {'text': 'Text Field',
               'listbox': 'List Box',
               'dropdown': 'Drop Down Box',
               'label': 'Label-like Field',
               'checkbox': 'Check Box' }

#######################################################
#
# xmlFormsHandler
#
#######################################################

class xmlFormsHandler (GParser.xmlHandler):
  """
  This class is called by the XML parser to
  process the .GFD file.
  """
  def __init__(self):


    GParser.xmlHandler.__init__(self)

    # This is a temp thing until we figure out
    # how to better do layout namespaces
    self.xmlNamespaceAttributesAsPrefixes = True

    self.xmlElements = getXMLelements()


