/* protocol.c, part of gnut
            Copyright (c) 2000, Josh Pieper
            Copyright (c) 2000-2001, Robert Munafo */

/* This file is distributed under the GPL, see file COPYING for details. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "connection.h"
#include "conf.h"
#include "http.h"
#include "lib.h"
#include "list.h"
#include "protocol.h"
#include "share.h"
#include "threads.h"

void fre_gpa(gnutella_packet **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

/* int gp.print(gnutella_packet *gpa)
 *
 * writes to standard out a textual version of the gnutella_packet gpa */
int gp_print(gnutella_packet *gpa)
{
  gnutella_qrhdr *grh;
  gnutella_pong *gpong;
  gnutella_qrname *grn;
  gnutella_qrsuf *grs;
  gnutella_push_a *gf_a;
  uchar *buf = 0;
  uint32 dlen;
  uint32 t,t2;
  uint16 port;
  int i;

  dlen = GET_LEU32(gpa->gh + PROT_HDR_DLEN, 26);

  fprintf(stderr, "GUID: ");
  for (i=0; i<16; i++) {
    fprintf(stderr, "%02X", gpa->gh[i]);
  }
  fprintf(stderr,"\n func: %i  ttl: %i  hops: %i  dlen:  %i\n",
	  gpa->gh[PROT_HDR_FUNC], gpa->gh[PROT_HDR_TTL],
	  gpa->gh[PROT_HDR_HOPS], dlen);

  if (dlen>0) {
    switch(gpa->gh[PROT_HDR_FUNC]) {

    case 0:
      break;

    case 0x01:
      fprintf(stderr, "PONG\n");
      if (dlen < PROT_SIZE_PONG) {
	fprintf(stderr, " data is too short!\n");
	return 0;
      }
      gpong = gpa->data;

      t = GET_LEU32((*gpong)+PROT_PONG_FILES, 27);
      t2 = GET_LEU32((*gpong)+PROT_PONG_KBYTES, 28);
      port = GET_LEU16((*gpong)+PROT_PONG_PORT, 29);

      fprintf(stderr," Servent Descriptor: %i.%i.%i.%i:%i files: %i  "
	      "kbytes: %ik\n",
	      (*gpong)[PROT_PONG_IP+0], (*gpong)[PROT_PONG_IP+1], 
	      (*gpong)[PROT_PONG_IP+2], (*gpong)[PROT_PONG_IP+3], port, t, t2);

      if (dlen > PROT_SIZE_PONG) {
	fprintf(stderr, " Extra data past end!\n");
	return 0;
      }
      break;

    case 0x40:
      if (dlen < PROT_SIZE_PUSH) {
	fprintf(stderr, " data is too short!\n");
	return 0;
      }
      gf_a = gpa->data;
      fprintf(stderr," File Request: GUID: ");
      for (i=0; i<16; i++) {
	fprintf(stderr, "%02x", (*gf_a)[i]);
      }

      fprintf(stderr,"\n");

      port = GET_LEU16((*gf_a) + PROT_PUSH_PORT, 30);

      t = GET_LEU32((*gf_a) + PROT_PUSH_REF, 31);

      fprintf(stderr,"  ref: %i  ip: %i.%i.%i.%i:%i\n", t,
	      (*gf_a)[PROT_PUSH_IP], (*gf_a)[PROT_PUSH_IP+1],
	      (*gf_a)[PROT_PUSH_IP+2], (*gf_a)[PROT_PUSH_IP+3], port);

      fprintf(stderr,"\n");
      break;

    case 0x80:
      if  (dlen>2) {
	buf = gpa->data;
	fprintf(stderr," Search query: %s\n", &buf[2]);
      }
      break;

    case 0x81:
      fprintf(stderr, "QREPLY\n");
      if (dlen < PROT_SIZE_QRHDR) {
	fprintf(stderr, " header too short\n");
	return 0;
      }
      grh = gpa->data;
      t = (*grh)[0];
      port = GET_LEU16((*grh) + PROT_QRHDR_PORT, 30);
      fprintf(stderr, " num: %i ip: %i.%i.%i.%i:%i\n",
	      t, (*grh)[PROT_QRHDR_IP], (*grh)[PROT_QRHDR_IP+1],
	      (*grh)[PROT_QRHDR_IP+2], (*grh)[PROT_QRHDR_IP+3], port);
      dlen -= PROT_SIZE_QRHDR;
      grn = (gnutella_qrname *) ((*grh) + PROT_SIZE_QRHDR);
      i = 0;
      while (i < t) {
	uint32 ref;
	uint32 size;

	fprintf(stderr, " item %i: ", i);
	if (dlen < PROT_SIZE_QRNAME) {
	  fprintf(stderr, " incomplete item header\n");
	  return 0;
	}
	ref = GET_LEU32((*grn) + PROT_QRNAME_REF, 31);
	size = GET_LEU32((*grn) + PROT_QRNAME_SIZE, 31);
	fprintf(stderr, "ref %u size %u ", ref, size);
	if (dlen < strlen((*grn) + PROT_QRNAME_NAME)) {
	  fprintf(stderr, " item name runs off end of packet!\n");
	  return 0;
	}
	fprintf(stderr, "name '%s'\n", (*grn) + PROT_QRNAME_NAME);
	dlen -= (PROT_SIZE_QRNAME + strlen((*grn) + PROT_QRNAME_NAME) + 1);
	grn = (gnutella_qrname *)
	  (((char *) grn)
	   + (PROT_SIZE_QRNAME + strlen((*grn) + PROT_QRNAME_NAME) + 1));
	if (dlen <= 0) {
	  fprintf(stderr,
		  " item ends in middle of double null, and no GUID!\n");
	  return 0;
	}
	i++;
      }
      grs = (gnutella_qrsuf *) grn;
      if (dlen < PROT_QRSUF_GUID) {
	fprintf(stderr, " truncated or missing EQHD!\n");
	return 0;
      }
      fprintf(stderr,
	      " EQHD: vendor %c%c%c%c odlen %i flag1 %02x flag2 %02x\n",
	      (*grs)[0], (*grs)[1], (*grs)[2], (*grs)[3],
	      (*grs)[PROT_QRSUF_ODLEN], (*grs)[PROT_QRSUF_FLAG1],
	      (*grs)[PROT_QRSUF_FLAG2]);
      dlen -= PROT_QRSUF_GUID;

      if (dlen < 16) {
	fprintf(stderr, " truncated or missing GUID!\n");
	return 0;
      }
      fprintf(stderr, " Push GUID:");
      for(i=0; i<16; i++) {
	fprintf(stderr, " %02x", (*grs)[PROT_QRSUF_GUID + i]);
      }
      if (memcmp((*grs)+PROT_QRSUF_GUID, conf_get_str("guid"), 16) == 0) {
	fprintf(stderr, " (matches our GUID)");
      } else {
	fprintf(stderr, " (no match)");
      }
      fprintf(stderr, "\n");
      if (dlen > 16) {
	fprintf(stderr, " too much after GUID!\n");
	return 0;
      }
      fprintf(stderr,"\n");
      break;

    default:
      if (dlen>1000) {
	GD_S(3,"capping debug dump at 1000 bytes\n");
	dlen=1000;
      }
      buf = gpa->data;
      fprintf(stderr, " Unknown: func: %i  dump\n", gpa->gh[PROT_HDR_FUNC]);
      for (i=0; i<dlen; i++) {
	fprintf(stderr,"%02x ", buf[i]);
      }
      fprintf(stderr,"\n ");
      for (i=0; i<dlen; i++) {
	if (G_ISPRINT(buf[i])) {
	  fprintf(stderr, "%c", buf[i]);
	}
      }
      fprintf(stderr, "\n");
      break;
    }
  }
  fprintf(stderr, "\n");
  return 0;
}

/* gnutella_packet * gp_reply.make(gchar *guid, GSList *results,
 *      guchar ip[4], guint16 port, guint speed, gchar *mguid, gint ttl)
 *
 * Makes a query reply packet based on the results in the GSList
 * results.  Returns the packet. */
gnutella_packet * gp_reply_make(char *guid, Gnut_List *results,
      uchar *ip, uint16 port, uint32 speed, char *mguid, int ttl)
{
  gnutella_packet * gpa;
  gnutella_qrhdr *gr;
  gnutella_qrname *grn;
  gnutella_qrsuf *grs;
  Gnut_List *ltemp;
  char *name;
  share_item *si;
  uint32 dlen;
  int num;

  GD_S(3, "started\n");

  gpa = (gnutella_packet *) ymaloc(sizeof(gnutella_packet), 277);

  /* first fill in the header info */

  memcpy(gpa->gh, guid, 16);
  gpa->gh[PROT_HDR_FUNC] = 0x81;
  gpa->gh[PROT_HDR_TTL] = ttl;
  gpa->gh[PROT_HDR_HOPS] = 0;

  /* Calculate the length we need to al.locate. This requires counting the
   * length of all the result filenames */
  dlen = PROT_SIZE_QRHDR;
  for (num=0, ltemp=results; ltemp; ltemp=gnut_list_next(ltemp)) {
    con_num_responses++;
    si = ltemp->data;
    dlen += PROT_SIZE_QRNAME;

    name = si->path;
    if (gc_hide_pathname) {
      name = strrchr(si->path, '/');
      if (name) {
	name++;
      } else {
	name = si->path;
      }
    }

    dlen += strlen(name)+1;
    num++;
  }
  dlen += PROT_SIZE_QRSUF;

  gr = (gnutella_qrhdr *) ymaloc(dlen, 278);

  /* Write the header */
  (*gr)[PROT_QRHDR_NUM] = num;
  PUT_LEU16((*gr) + PROT_QRHDR_PORT, port, 39);
  PUT_LEU32((*gr) + PROT_QRHDR_SPEED, speed, 40);
  memcpy((*gr) + PROT_QRHDR_IP, ip, 4);

  /* Add the results */
  grn = (gnutella_qrname *) ((char *)gr + PROT_SIZE_QRHDR);
  for(ltemp=results; ltemp; ltemp=gnut_list_next(ltemp)) {
    si = ltemp->data;

    name = si->path;
    if (gc_hide_pathname) {
      name = strrchr(si->path, '/');
      if (name) {
	name++;
      } else {
	name = si->path;
      }
    }

    PUT_LEU32((*grn) + PROT_QRNAME_REF, si->ref, 41);
    PUT_LEU32((*grn) + PROT_QRNAME_SIZE, si->size, 42);

    strcpy((*grn) + PROT_QRNAME_NAME, name);
    /* Have to add one more null byte */
    (*grn)[PROT_QRNAME_NAME + strlen(name) + 1] = 0;
    grn = (gnutella_qrname *)
      (((char *) grn) + PROT_SIZE_QRNAME + strlen(name) +1);
  }

  /* Add a 7-byte extended query reply descriptor */
  grs = (gnutella_qrsuf *) grn;
  {
    uint8 f1, f2;

    memcpy((*grs)+PROT_QRSUF_VENDOR, "GNUT", 4);
    (*grs)[PROT_QRSUF_ODLEN] = 2;
    /* Compute flag1 and flag2 */
    f1 = 0x1c; /* All our flag bits are meaningful */
    f2 = 0x01; /* All our flag bits are meaningful */
    if (gh_did_receive) {
      f1 |= 0x01; /* Push flag */
    }
    if (urate_measured) {
      f2 |= 0x10; /* UploadSpeed bit */
    }
    if (gh_did_upload) {
      f2 |= 0x08; /* Have-uploaded bit */
    }
    if (num_uploads >= gc_max_uploads) {
      f2 |= 0x04; /* Busy bit */
    }
    (*grs)[PROT_QRSUF_FLAG1] = f1;
    (*grs)[PROT_QRSUF_FLAG2] = f2;
    /* printf("%d GNUT %02x %02x %02x\n", sizeof(gnutella_eqhd), eqhd->odlen,
       eqhd->flag1, eqhd->flag2); */
  }

  /* Finally, stick our host GUID on the end */

  memcpy((*grs)+PROT_QRSUF_GUID, mguid, 16);

  dlen = (((char *) grs) + PROT_SIZE_QRSUF) - ((char *) gr);
  PUT_LEU32(gpa->gh + PROT_HDR_DLEN, dlen, 43);

  gpa->data = gr;

  return gpa;
}

/* gnutella_packet * gp_ping_make(char mac[6], int ttl)
 *
 * Makes a ping packet and returns it */
gnutella_packet * gp_ping_make(char *mac, int ttl)
{
  gnutella_packet * gpa;
  int i;

  gpa = (gnutella_packet *)ycaloc(sizeof(gnutella_packet), 1, 484);

  gpa->data = 0;

  memcpy(gpa->gh, mac, 6);
  for (i=7; i<16; i++) {
    gpa->gh[i] = rand();
  }

  gpa->gh[PROT_HDR_TTL] = ttl;
  gpa->gh[PROT_HDR_HOPS] = 0;

  return gpa;
}

/* gnutella_packet * gp_request_make(char mac[6],char *name, int ttl)
 *
 * Makes a query (0x80) packet and returns it. */
gnutella_packet * gp_request_make(char *mac, char *name, int ttl)
{
  gnutella_packet * gpa;
  char *buf;
  int i;

  gpa = (gnutella_packet *)ymaloc(sizeof(gnutella_packet), 279);

  gpa->gh[PROT_HDR_FUNC] = 0x80;
  gpa->gh[PROT_HDR_TTL] = ttl;
  gpa->gh[PROT_HDR_HOPS] = 0;
  PUT_LEU32(gpa->gh + PROT_HDR_DLEN, strlen(name)+3, 44);

  buf = (char *) ymaloc(strlen(name)+5, 280);
  buf[0] = 0;
  buf[1] = 0;
  strcpy(&buf[2], name);

  gpa->data = buf;

  memcpy(gpa->gh, mac, 6);
  for (i=7; i<16; i++) {
    gpa->gh[i] = rand();
  }

  return gpa;
}

gnutella_packet * gp_push_make(char *mac, int ttl, char *guid,
  uint32 ref, uchar *ip, uint16 port)
{
  gnutella_packet *gpa;
  gnutella_push_a *gp_a;

  gpa = (gnutella_packet *) ymaloc(sizeof(gnutella_packet), 281);
  gpa->gh[PROT_HDR_FUNC] = 0x40;
  gpa->gh[PROT_HDR_TTL] = ttl;
  gpa->gh[PROT_HDR_HOPS] = 0;

  PUT_LEU32(gpa->gh + PROT_HDR_DLEN, PROT_SIZE_PUSH, 45);

  gp_a = (gnutella_push_a *) ymaloc(PROT_SIZE_PUSH, 282);
  gpa->data = gp_a;
  memcpy((*gp_a), guid, 16);
  memcpy((*gp_a) + PROT_PUSH_IP, ip, 4);

  PUT_LEU16((*gp_a) + PROT_PUSH_PORT, port, 46);

  PUT_LEU32((*gp_a) + PROT_PUSH_REF, ref, 47);

  return gpa;
}

/* gnutella_packet *gp.pong.make(uchar ip[4], char *guid, int files,
 *        int bytes, int port, int ttl)
 *
 * Makes a PONG packet, and returns it. */
gnutella_packet * gp_pong_make(char *guid, int files,
        int bytes, uchar *ip, uint16 port, int ttl)
{
  gnutella_packet * gpa;
  gnutella_pong * gpong;

  gpa = (gnutella_packet *) ymaloc(sizeof(gnutella_packet), 283);
  gpong = (gnutella_pong *) ymaloc(PROT_SIZE_PONG, 284);

  memcpy(gpa->gh, guid, 16);
  gpa->gh[PROT_HDR_FUNC] = 0x01;
  gpa->gh[PROT_HDR_TTL] = ttl;
  gpa->gh[PROT_HDR_HOPS] = 0;
  PUT_LEU32(gpa->gh + PROT_HDR_DLEN, PROT_SIZE_PONG, 48);

  gpa->data = gpong;

  PUT_LEU32((*gpong)+PROT_PONG_FILES, files, 49);
  PUT_LEU32((*gpong)+PROT_PONG_KBYTES, bytes, 50);
  memcpy((*gpong)+PROT_PONG_IP, ip, 4);
  PUT_LEU16((*gpong)+PROT_PONG_PORT, port, 51);

  return gpa;
}

/* gnutella_packet * gp.dup(gnutella_packet *gp)
 *
 * duplicates the gnutella_packet gp, and returns a pointer to it */
gnutella_packet *gp_dup(gnutella_packet *gp)
{
  gnutella_packet *ngp;
  uint32 dlen;

  GD_S(3, "gp.dup entering\n");
  ngp = (gnutella_packet *) ymaloc(sizeof(gnutella_packet), 285);

  memcpy(ngp, gp, sizeof(gnutella_packet));

  dlen = GET_LEU32(ngp->gh + PROT_HDR_DLEN, 32);

  /* 0.4.28.c07 Add this case because someone reported their gnut
     tried to allocate space for a 50-megabyte packet (case 0286)
     It turned out to be an endian bug, fixed by 0.4.28,c10 */
  if (dlen > 70000) {
    dlen = 70000;
    PUT_LEU32(ngp->gh + PROT_HDR_DLEN, dlen, 52);
  }

  if (dlen > 0) {
    ngp->data = ymaloc(dlen, 286);
    memcpy(ngp->data, gp->data, dlen);
  } else {
    ngp->data = 0;
  }

  GD_S(3, "gp.dup returning success\n");
  return ngp;
}
