/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  iconbar-popup-menu.c: Popup menus for the shortcut bar.

  Copyright (C)  2001-2003  Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>


  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gconf/gconf-client.h>
#include "satlog.h"
#include "callbacks.h"
#include "satlist-druid.h"
#include "satmap-druid.h"
#include "satmodules.h"
#include "iconbar.h"
#include "iconbar-popup-menu.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif


extern GConfClient *client; /* main.c */


/* popup menu for existing modules (open page on shortcut bar) */
GnomeUIInfo item_popup_menu[] = {
	GNOMEUIINFO_ITEM_NONE (_("Delete module"),
			       _("Delete this module"),
			       NULL),
/* 	GNOMEUIINFO_ITEM_STOCK (_("Delete module"), */
/* 				_("Delete this module"), */
/* 				NULL, */
/* 				GNOME_STOCK_PIXMAP_TRASH), */
			       
	GNOMEUIINFO_ITEM_NONE (_("Configure module"),
			       _("Configure this module"),
			       NULL),
	GNOMEUIINFO_END
};



/* Private function prototypes */
static void iconbar_popup_item_delete           (gchar *, guint);
static void iconbar_popup_item_delete_toggle_cb (GtkToggleButton *, gpointer);


void
iconbar_popup_item_menu (gchar *name, guint modtype, GdkEventButton *button)
{
	/* This function creates and pops up a menu with
	   options for the modules (Open section in the
	   shortcut bar.
	*/
	GtkWidget *menu;
	gint menuitem;

	menu = gnome_popup_menu_new (item_popup_menu);
	menuitem = gnome_popup_menu_do_popup_modal (menu,
						    NULL, NULL,
						    button, NULL);
	switch (menuitem) {
	case 0:
		iconbar_popup_item_delete (name, modtype);
		break;
	case 1:
		switch (modtype) {
		case SATMOD_TYPE_LIST:
			satlist_druid_run (name);
			break;
		case SATMOD_TYPE_MAP:
			satmap_druid_run (name);
			break;
		default:
			not_finished (NULL, NULL);
		}
		break;
	default:
		break;
	}
	gtk_widget_destroy (menu);
}


extern const gchar *modtypes[]; /* satmodules.c */


static void
iconbar_popup_item_delete (gchar *name, guint modtype)
{
	/* This function is called when the user selects the "delete"
	   menu item in the "open" section of the shortcut bar.
	*/
	GtkWidget *dialog,*label,*hbox,*pixmap,*check;
	GtkTooltips *kooltips;
	GnomeVFSResult result;
	gchar *configfile,*tmp,*uri;
	gint button=0;

	/* check file name */
	if (!name) {
		satlog_log (SAT_LOG_CRITICAL,
			    _("iconbar_popup_item_delete called with (NULL) name!"));
		return;
	}
	configfile = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
				  G_DIR_SEPARATOR_S, name, ".", modtypes[modtype],
				  NULL);

	if (!gconf_client_get_bool (client, ICONBAR_POPUP_ITEM_DELETE_CONFIRM_PATH, NULL)) {
		/* get confirmation from user */
		/* create dialog */
		dialog = gnome_dialog_new (_("Confirm delete file"),
					   GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL,
					   NULL);
		gnome_dialog_set_default (GNOME_DIALOG (dialog), 1);
		/* contents of the dialog: hbox containing a pixmap, a label
		   and a check-button.
		*/
		/* pixmap */
		tmp = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				   G_DIR_SEPARATOR_S, "crash.png", NULL);
		pixmap = gnome_pixmap_new_from_file (tmp);
		g_free (tmp);
		/* label */
		tmp = g_strconcat (_("You are about to delete the file:\n"), configfile,
				   _("\nThis action can not be undone!"), NULL);
		label = gtk_label_new (tmp);
		/* check button */
		check = gtk_check_button_new_with_label (_("Don't ask me this question again."));
		gtk_signal_connect (GTK_OBJECT (check), "toggled",
				    GTK_SIGNAL_FUNC (iconbar_popup_item_delete_toggle_cb),
				    NULL);
		kooltips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (kooltips, check,
				      _("Check this box if you do NOT want to be asked for "\
					"corfirmation the next time you try to delete a module"),
				      _("Don't do this unless you are an experienced user!"));
		gtk_widget_show (check);
		hbox = gtk_hbox_new (FALSE, 10);
		gtk_box_pack_start_defaults (GTK_BOX (hbox), pixmap);
		gtk_box_pack_start_defaults (GTK_BOX (hbox), label);
		gtk_widget_show_all (hbox);
		/* add contents to dialog box */
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
				    hbox, TRUE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
				    check, FALSE, FALSE, 5);
		button = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		g_free (tmp);
	}
	if (!button) {
		/* User clicked YES!!! */
		uri = gnome_vfs_get_uri_from_local_path (configfile);
		result = gnome_vfs_unlink (uri);
		if (result != GNOME_VFS_OK) {
			satlog_log (SAT_LOG_CRITICAL, _("ERROR deleting file:"));
			satlog_log (SAT_LOG_CRITICAL, configfile);
			satlog_log (SAT_LOG_CRITICAL, gnome_vfs_result_to_string (result));
		}
		else
			iconbar_remove_module (name, modtype);
		g_free (uri);
	}
	else {
		/* user backs out - make sure check box didn't have any effect! */
		gconf_client_set_bool (client,
				       ICONBAR_POPUP_ITEM_DELETE_CONFIRM_PATH,
				       FALSE,
				       NULL);
		gconf_client_suggest_sync (client, NULL);
	}
	g_free (configfile);
}


static void
iconbar_popup_item_delete_toggle_cb (GtkToggleButton *button, gpointer data)
{
	/* This funcyion is called when the user clicks on the checkbox
	   in the confirmation dialog box.
	*/

	gconf_client_set_bool (client,
			       ICONBAR_POPUP_ITEM_DELETE_CONFIRM_PATH,
			       gtk_toggle_button_get_active (button),
			       NULL);
	gconf_client_suggest_sync (client, NULL);
}
