/*
 * GProFTPD - A GTK+ frontend for the ProFTPD standalone server.
 * Copyright (C) 2001 - 2006 Magnus Loef (Magnus-swe) <magnus-swe@telia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307, USA.
 *
 */




#include "../config.h"
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "widgets.h"
#include "gettext.h"
#include "allocate.h"
#include "commands.h"
#include "show_info.h"
#include "add_user.h"
#include "functions.h"
#include "system_defines.h"

#include "populate_users.h"
#include "populate_user_settings.h"
#include "populate_conf_tab.h"
#include "select_first_user.h"
#include "reread_conf.h"


#ifdef USE_DARWIN
#include "osx_functions.h"
#endif


extern char global_server_address[1024];
extern char global_server_port[1024];
extern char global_server_type[1024];



void add_user(struct w *widgets)
{
    /* Adds a new user to the selected server */
    FILE *fp;
    long conf_size;
    char *old_buffer, *new_buffer;
    char *user_settings, *user_check, *address_buffer, *port_buffer;
    char *home_dir;
    int length=0, limit_access=0, user_added=0;
    int found_server=0, standard_server=0;
    char fixdir[16384]="";
    gchar *utf8=NULL;
    gchar *info, *cmd;

    G_CONST_RETURN gchar *username;
    G_CONST_RETURN gchar *password;
    G_CONST_RETURN gchar *group;
    G_CONST_RETURN gchar *homedir;
    G_CONST_RETURN gchar *comment;
    G_CONST_RETURN gchar *shell;

    username  = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[0]));
    password  = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[1]));
    group     = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[2]));
    comment   = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[3]));
    homedir   = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[4]));
    /* Shell is a gtk_combo_box_entry_new_text */
    shell     = gtk_entry_get_text(GTK_ENTRY(GTK_BIN(widgets->user_set_combo[0])->child));

    /* If the shell is false it will add the users directory as /dev/null and the specified false shell.
     * If the shell is real it will add the users directory as /USERSHOME/username and the specified real shell. 
     * The ftp directory will be located where specified in the application.
     * This is done so that no user gets sshd/etc access unless the admin wants to.
     */

    /* If the username field is empty inform that this cant be done. */
    length = strlen(username);
    if( length == 0 ) 
    {
	info = g_strdup_printf(_("You must specify a username.\n"));
	show_info(info);
	g_free(info);
    	return;
    }

    if( username[0]=='0'||username[0]=='1'||username[0]=='2'||username[0]=='3'||username[0]=='4' 
    ||  username[0]=='5'||username[0]=='6'||username[0]=='7'||username[0]=='8'||username[0]=='9') 
    {
	info = g_strdup_printf(_("Failed adding user: %s\nA user name can not begin with a number.\n"), username);
	show_info(info);
	g_free(info);
    	return;
    }

    if( username[0]=='r' && username[1]=='o' && username[2]=='o' && username[3]=='t' && strlen(username) == 4 ) 
    {
	info = g_strdup_printf(_("Failed adding user: %s\nThe user root can not be added for security reasons.\n"), username);
	show_info(info);
	g_free(info);
    	return;
    }

    if( strstr((char *)username, "<") || strstr((char *)username, ">") ) 
    {
	info = g_strdup_printf(_("Failed adding user: %s\nchars \"<\" and \">\" arent allowed.\n"), username);
	show_info(info);
	g_free(info);
    	return;
    }

    /* If the password field has less then 6 chars we inform that this cant be done */
    length = strlen(password);
    if( length < 6 ) 
    {
	info = g_strdup_printf(_("Failed adding user: %s\nA minimum password length of 6 chars is required.\n"), username);
	show_info(info);
	g_free(info);
    	return;
    }

    /* If the group filed has less then 3 chars we inform that this cant be done */
    length = strlen(group);
    if( length == 0 )
    {
	info = g_strdup_printf(_("Failed adding user: %s\nNo group specified.\n"), username);
	show_info(info);
	g_free(info);
    	return;
    }

    /* If the shell field has less then 3 chars we inform that this cant be done */
    length = strlen(shell);
    if( length < 3 )
    {
	info = g_strdup_printf(_("Failed adding user: %s\nNo shell specified.\n"), username);
	show_info(info);
	g_free(info);
    	return;
    }

    /* If the directory field has 0 chars we inform that this cant be done */
    length = strlen(homedir);
    if( length < 8 )  /* /home/b /var/abc */
    {
	info = g_strdup_printf(_("Failed adding user: %s\nThe specified home directory is too short.\n"), username);
	show_info(info);
	g_free(info);
    	return;
    }

    /* A comment is required */
    length = strlen(comment);
    if( length == 0 ) 
    {
	info = g_strdup_printf(_("A comment is required to add a user.\n"));
	show_info(info);
	g_free(info);
    	return;
    }

    /* Check if the user exists in the selected server */
    if((fp=fopen(PROFTPD_CONF,"r"))==NULL)
    {
	info = g_strdup_printf(_("Failed adding user: %s\nCant open proftpd.conf.\n"), username);
	show_info(info);
	g_free(info);

    	return;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    old_buffer = allocate(conf_size);
    user_check = allocate(4096);

    strcpy(user_check, "User ");
    strcat(user_check, username);
    strcat(user_check, "\n"); /* Similar user confusion */

    address_buffer = allocate(8192+15);
    port_buffer = allocate(8192+3);               

    if( strstr((char *)global_server_type, "Virtualhost") )
      sprintf(address_buffer, "<VirtualHost %s>\n", global_server_address);
    else
      standard_server = 1;
     
    sprintf(port_buffer, "Port %s\n", global_server_port);


    /* Scroll to the correct vhost */
    if( ! standard_server && conf_size > 1 )
    while(fgets(old_buffer, conf_size, fp)!=NULL)
    {
	/* The correct server address is found */
        if( ! found_server && ! strcmp(old_buffer, address_buffer) ) 
	{
	    /* Lets see if its the same port as the selected one */
	    while(fgets(old_buffer, conf_size, fp)!=NULL)
	    {
		if( strstr(old_buffer, "Port") 
		&& ! strcmp(old_buffer, port_buffer) )
		{
		    found_server = 1;
		    break;
		}
		/* End of vhost, break and check the next vhost */
		if( strstr(old_buffer, "</VirtualHost>") )
		  break;
	    }    
	}
	/* This vhost has the correct address and port */
	if( found_server )
	  break;
    }

    /* The selected vhost was not found */
    if( ! standard_server && ! found_server )
    {
	info = g_strdup_printf(_("Failed adding user: %s\nThe selected virtual host was not found.\n"), username);
	show_info(info);
	g_free(info);

	free(old_buffer);
	free(user_check);
	free(address_buffer);
	free(port_buffer);
	fclose(fp);
	return;
    }



    /* We have begun at the top for the standard server 
     * or scrolled to the selected vhost. */
    
    /* Check if the user exists in this selected vhost or standard server */
    if( conf_size > 1 )
    while(fgets(old_buffer, conf_size, fp)!=NULL)
    {
	if( ! strcmp(old_buffer, user_check) 
	&&  ! strstr(old_buffer, "AllowUser")
	&&  ! strstr(old_buffer, "FakeUser") )
	{
	    info = g_strdup_printf(_("Failed adding user: %s\nThe user already exists in this server.\n"), username);
 	    show_info(info);
	    g_free(info);

	    free(user_check);
	    free(old_buffer);
	    free(address_buffer);
	    free(port_buffer);
	    fclose(fp);
	    return;
	}

	/* End the search if we are looking for a standard user 
	   and the end of the standard server is found */
	if( standard_server && strstr(old_buffer, "<VirtualHost") )
	  break;

	/* End the search if we are looking for a vhost user 
	   and the end of the vhost is found */
	if( ! standard_server && strstr(old_buffer, "</VirtualHost") )
	  break;
    }
    free(user_check);
    free(old_buffer);
    fclose(fp);



    /* The selected shell is false, add the group and the user with a false shell and home /dev/null */
    if( strstr(shell, "nologin") || strstr(shell, "false") || strstr(shell, "dev/null") )
    {
	/* All supported systems but darwin */
#ifndef USE_DARWIN

	/* Add the group if it doesnt exist */
	if( ! group_exists(group) )
	{
	    cmd = g_strdup_printf("%s '%s'", ADDGROUP, group);
	    if( ! run_command(cmd) )
	    {
		info = g_strdup_printf(_("Error adding group: %s\n"), group);
 		show_info(info);
		g_free(info);
	    }
	    g_free(cmd);
	}

	/* Add the user to this group if it doesnt exist */
        if( ! user_exists(username) )
        {
	    cmd = g_strdup_printf("%s '%s' -g '%s' -d /dev/null -c '%s' -s %s", ADDUSER, username, group, comment, shell);
	    if( ! run_command(cmd) )
	    {
		info = g_strdup_printf(_("Failed adding user: %s\n"), username);
 		show_info(info);
		g_free(info);
	    }
	    else
	      user_added = 1;

	    g_free(cmd);
	}

#elif USE_DARWIN
	/* Add the false user using darwins niutil commands (Darwin is out of sync) */
	if( ! niutil_user_exists(username) )    
	{
	    if( ! niutil_useradd(username, shell) )
	    {
		info = g_strdup_printf(_("Failed adding user: %s\n"), username);
 		show_info(info);
		g_free(info);
	    }
	    else
	      user_added = 1;
	}
#endif
    }
    else 	
      {

         /* The selected shell is not false add a real user account */
#ifndef USE_DARWIN

	/* Add the group if it doesnt exist */
	if( ! group_exists(group) )
	{
	    cmd = g_strdup_printf("%s '%s'", ADDGROUP, group);
	    if( ! run_command(cmd) )
	    {
		info = g_strdup_printf(_("Failed adding group: %s\n"), group);
 		show_info(info);
		g_free(info);
	    }
	    g_free(cmd);
	}
	
	/* Add the user to this group if it doesnt exist */
	if( ! user_exists(username) )
	{
	    /* Add the user with a real shell to /USERSHOME/ UserName (was: -m -s) */
	    cmd = g_strdup_printf("%s '%s' -g '%s' -d '%s%s' -c '%s' -s %s", ADDUSER, username, group, USERSHOME, username, comment, shell);
	    if( ! run_command(cmd) )
	    {
		info = g_strdup_printf(_("Failed adding user: %s\n"), username);
 		show_info(info);
		g_free(info);
	    }
	    else
	      user_added = 1;

	    g_free(cmd);
	}

#elif USE_DARWIN
	 /* Add a real darwin user using the niutil commands (Darwin is out of sync) */
	 if( ! niutil_user_exists(username) )    
	 {
	     if( ! niutil_useradd(username, shell) )
	     {
		info = g_strdup_printf(_("Failed adding user: %s\n"), username);
 		show_info(info);
		g_free(info);
	     }
	     else
	       user_added = 1;
	 }
#endif
    }


    /* Dont add anything if we couldnt add the system user */
#ifndef USE_DARWIN
    if( ! user_exists(username) )
#elif USE_DARWIN
    if( ! niutil_user_exists(username) )
#endif
    {
	info = g_strdup_printf(_("The system user was not added because uppercase\nor language specific letters are not allowed.\n"));
 	show_info(info);
	g_free(info);

	free(address_buffer);
	free(port_buffer);
	return;
    }



    /* Setup the user profile and directories */


    /* Check that slashes in directory path combinations have slashes in the right place */
    home_dir = allocate(4096);
    strcpy(home_dir, homedir);
    fix_path(home_dir);

    /* Make the users ftp directory in /var/.. and chmod it to 0755 */
    strcpy(fixdir, home_dir);
    make_dir_chmod((gchar *)home_dir, "0755");

    /* The users configuration profile */
    user_settings = allocate(16384);
    strcpy(user_settings, "\n<Anonymous ");
    strcat(user_settings, home_dir);
    strcat(user_settings, ">\n");
    strcat(user_settings, "User ");
    strcat(user_settings, username);
    strcat(user_settings, "\nGroup ");
    strcat(user_settings, group);
    strcat(user_settings, "\n");
    strcat(user_settings, "AnonRequirePassword on\n");
    strcat(user_settings, "MaxClients 5 \"The server is full, hosting %m users\"\n");
    strcat(user_settings, "DisplayLogin welcome.msg\n");
    strcat(user_settings, "DisplayFirstChdir .msg\n");
    strcat(user_settings, "<Limit LOGIN>\n");
    strcat(user_settings, " Allow from all\n");
    strcat(user_settings, " Deny from all\n");
    strcat(user_settings, "</Limit>\n");
    strcat(user_settings, "AllowOverwrite off\n");
    strcat(user_settings, "<Limit RETR LIST NLST MDTM SIZE STAT CWD XCWD PWD XPWD CDUP XCUP>\n");
    strcat(user_settings, " AllowAll\n");
    strcat(user_settings, "</Limit>\n");
    strcat(user_settings, "<Limit DELE APPE STOR STOU SITE_CHMOD SITE_CHGRP RNFR RNTO MKD XMKD RMD XRMD>\n");
    strcat(user_settings, " DenyAll\n");
    strcat(user_settings, "</Limit>\n");
    strcat(user_settings, "</Anonymous>\n");
    free(home_dir);

    /* Add the new user settings and AllowUser to the correct server */
    found_server = 0;

    /* Standard server selected, start adding users directly */
    if( standard_server )
      found_server = 1;

    /* Add AllowUser UserName to the selected server */
    if((fp=fopen(PROFTPD_CONF,"r"))==NULL)
    {
	free(address_buffer);
	free(port_buffer);
	free(user_settings);
    	return;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    old_buffer = allocate(conf_size);
    new_buffer = allocate(conf_size+8192);

    if( conf_size > 1 )
    while(fgets(old_buffer, conf_size, fp)!=NULL)
    {
	strcat(new_buffer, old_buffer);

        if( ! standard_server && ! found_server 
	&&  ! strcmp(old_buffer, address_buffer) )
	{
	    /* Lets see if this is the selected server */
	    while(fgets(old_buffer, conf_size, fp)!=NULL)
	    {
		strcat(new_buffer, old_buffer);

		/* This will expect the servers port on the second line ! 
		 * else itll miss some vaules .. */
		if( strstr(old_buffer, "Port") 
		&& ! strcmp(old_buffer, port_buffer) )
		{
		    found_server = 1;
		    break;
		}
		
		if( strstr(old_buffer, "</Virtualhost>") )
		  break;
	    }    
	}

	
	/* Continue until we find the selected server */
	if( ! found_server )
	  continue;


	/* Add AllowUser Username .. to this server only */
	if( strstr(old_buffer, "<Limit LOGIN") 
	&& found_server && ! limit_access )
	{
	    strcat(new_buffer, "  AllowUser ");
	    strcat(new_buffer, username);
	    strcat(new_buffer, "\n");
	    limit_access = 1; /* just incase so we just change the first occurance */
	    
	    /* Add the user after </Limit> */
	    while(fgets(old_buffer, conf_size, fp)!=NULL)
	    {
	       strcat(new_buffer, old_buffer);
	       if( strstr(old_buffer, "</Limit") && limit_access == 1 )
	       {
		    /* Only add it once */
		    limit_access = 2;
		    strcat(new_buffer, user_settings);
	       }
	    }
	}
	
	/* Add the new user settings if we have another user (once) */
	if( strstr(old_buffer, "</Anonymous") && limit_access == 1 )
	{
	   /* Only add it once */
	   limit_access = 2;
	   strcat(new_buffer, user_settings);
	}
    }
    fclose(fp);
    free(old_buffer);      
    free(address_buffer);
    free(port_buffer);
    free(user_settings);



    /* Password the user if it didnt exist before */
    if( user_added )
    {
#ifndef USE_DARWIN
         password_user(username, password);
#elif USE_DARWIN
	 niutil_password_user(username, password);
#endif
    }
    else
      {
	  info = g_strdup_printf(_("The system user \"%s\" already exists.\nThe user was added to this server but the password was not changed.\n"), username);
 	  show_info(info);
	  g_free(info);
      }

    /* Write the new configuration if the user profile was added. 
     * Since the user could have already existed we use limit_access */
    if( limit_access )
    {
	if((fp=fopen(PROFTPD_CONF,"w+"))==NULL)
	{
	    info = g_strdup_printf(_("Could not write the new user configuration to:\n%s\nRun gproftpd as root\n"), PROFTPD_CONF);
 	    show_info(info);
	    g_free(info);

	    free(new_buffer);
    	    return;
	}
	else
    	  {
	     fputs(new_buffer, fp);
	     fclose(fp);
    	  }
    }

    free(new_buffer);      

    fix_newlines_in_conf();

    /* Update the user list and the user settings */
    populate_users(widgets);
    select_first_user(widgets);

    populate_user_settings(widgets);
    
    populate_conf_tab(widgets);

    /* Update the server */
    reread_conf(widgets);

    if( utf8!=NULL )
      g_free(utf8);
}
