#include "board.h"
#include "httpclient.h"
#include "ignorelist.h"
#ifndef MINIMAL_GPSSHELL
#  include "SReadline.h"
#endif
#include "book.h"

#ifdef USE_TOKYO_CABINET
#  include "gpsshogi/recorddb/facade.h"
#endif
#include "gpsshogi/revision.h"
#include "osl/eval/pieceEval.h"
#include "osl/eval/progressEval.h"
#include "osl/eval/ml/openMidEndingEval.h"
#include "osl/state/numEffectState.h"
#include "osl/record/csa.h"
#include "osl/record/usi.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaIOError.h"
#include "osl/record/kisen.h"
#include "osl/record/ki2.h"
#include "osl/record/kakinoki.h"
#include "osl/record/myshogi.h"
#include "osl/record/opening/openingBook.h"
#include "osl/game_playing/alphaBetaPlayer.h"
#include "osl/game_playing/gameState.h"
#include "osl/misc/random.h"
#include "osl/search/moveGenerator.h"
#include "osl/search/searchState2.h"
#include "osl/search/simpleHashTable.h"
#include "osl/search/quiescenceSearch2.h"
#include "osl/search/quiescenceSearch2.tcc"
#include "osl/rating/featureSet.h"
#include "osl/rating/ratingEnv.h"
#include "osl/move_generator/legalMoves.h"
#include "osl/c/facade.h"
#include "osl/oslConfig.h"

#ifndef MINIMAL_GPSSHELL
#  include <Poco/Net/HTTPStreamFactory.h>
#endif
#include <boost/scoped_ptr.hpp>
#include <boost/function.hpp>
#include <boost/functional.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/program_options.hpp>
#include <boost/tokenizer.hpp>
#include <boost/format.hpp>
#include <boost/foreach.hpp>

#include <iostream>
#include <vector>
#include <deque>
#include <string>
#include <map>
#include <list>
#include <algorithm>
#include <sstream>
#include <fstream>
#include <cstdlib>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

using namespace osl;
#ifndef MINIMAL_GPSSHELL
using namespace swift;
#endif
using namespace boost;
using namespace std;
    
/*========== Global variables ==========*/

boost::filesystem::path cache_dir;

// The following typedefs and the struct are
// for "advanced" completers
typedef function< void ( const std::vector<string>& ) > Func;
typedef pair<string, Func> Element;

static boost::scoped_ptr<gpsshell::Board> board;
/* The stack of the boards the have ever been seen*/
static std::deque<gpsshell::Board> boards;

static boost::scoped_ptr<gpsshell::Book> the_book;
static ::gpsshell::IgnoreList the_ignorelist;
    
struct MyElement : public Element
{
    operator string () const { return first; }
    MyElement(const string& arg1, Func arg2) : Element(arg1, arg2) {}
};

typedef list<MyElement>     MyContainer;
int verbose = 1;

/*========== Classes and Funcitons ==========*/

// A functor to look for a command in the 
// container of completers
class LookupFunctor
{
public:
  // Creates a functor and memorises tokens
  LookupFunctor(const std::vector<string>& params) :
    params(params)
  {}

  // Compares the first token only
  bool operator()(const MyElement&  Element) const
  {
    return strncmp(params.begin()->c_str(),
                   (static_cast<const string&>( Element )).c_str(),
                   params.begin()->size()) == 0;
  }

private:
  const std::vector<string>& params;
};


// The 'help' command processor
class Shelp_executor
{
public:
  // Memorises the list of completers
  Shelp_executor(const MyContainer&  completers) :
    completers(completers)
  {}

  // Prints all the completers on the std out
  void operator() (const std::vector<string>&  /*params*/)
  {
    cout << "List of the command variations:" << endl;
    for (MyContainer::const_iterator k=completers.begin();
          k != completers.end(); ++k)
    {
      cout << static_cast<const string&>(*k) << endl;
    }
    cout << "Press Ctrl+D for gracefull exit" << endl;
  }
private:
  const MyContainer& completers;
};


namespace gpsshell
{
void pushBoard()
{
  if (!board->isInitialState())
  {
    boards.push_back(*board);
  }
}

void popBoard()
{
   if (boards.empty())
     return;
   *board = boards.back();
   boards.pop_back();
}

void rollBack(const std::vector<string>& /*params*/)
{
  popBoard();
}

void openurl(const std::vector<string>& params);
void open(const std::vector<string>& params)
{
  if (params.size() < 2) {
    std::cout << "The open command supposes a csa or kif file name" << std::endl;
    return;
  }
  if (params[1].find("http://") == 0) {
    openurl(params);
    return;
  }
  const boost::filesystem::path path(params[1]);
  const std::string ext = boost::filesystem::extension(path);

  if (boost::algorithm::iequals(".csa", ext))
  {
    const CsaFile file(path.native_file_string());
    const osl::record::Record record = file.getRecord();
    osl::vector<Move> moves;
    osl::vector<int> time;
    osl::vector<std::string> comments;
    osl::vector<osl::record::SearchInfo> search_info;
    record.getMoves(moves, time, comments, search_info);
    board->setMoves(file.getInitialState(), moves);
    board->setTime(time);
    board->setComments(comments);
    board->setSearchInfo(search_info);
    if (params.size() == 3)
      board->next(boost::lexical_cast<size_t>(params[2]));
  }
  else if (boost::algorithm::iequals(".usi", ext))
  {
    const UsiFile file(path.native_file_string());
    const osl::vector<Move> moves = file.getRecord().getMoves();
    board->setMoves(file.getInitialState(), moves);
  }
  else if (boost::algorithm::iequals(".ki2", ext))
  {
    const Ki2File file(path.native_file_string());
    const osl::vector<Move> moves = file.getRecord().getMoves();
    board->setMoves(file.getInitialState(), moves);
  }
  else if (boost::algorithm::iequals(".kif", ext) 
	   && osl::KakinokiFile::isKakinokiFile(path.native_file_string()))
  {
    const KakinokiFile file(path.native_file_string());
    const osl::record::Record record = file.getRecord();
    osl::vector<Move> moves;
    osl::vector<int> time;
    osl::vector<std::string> comments;
    osl::vector<osl::record::SearchInfo> search_info;
    record.getMoves(moves, time, comments, search_info);
    board->setMoves(file.getInitialState(), moves);
    board->setComments(comments);
  }
  else if (boost::algorithm::iequals(".kif", ext))
  {
    size_t index = 0;
    if (params.size() == 3)
      index = boost::lexical_cast<size_t>(params[2]);
    KisenFile file(path.native_file_string());
    if (index >= file.size())
    {
      std::cout << "Error: Out of range " << index << "/" << file.size()-1 <<std::endl;
      return;
    }
    std::cout << "Opening... " << index << "/" << file.size()-1 << std::endl;
    const osl::vector<Move> moves = file.getMoves(index);
    board->setMoves(file.getInitialState(), moves);
  }
  board->showState();
}

void usiposition(const std::vector<string>& params)
{
  std::string all = params[0];
  for (size_t i=1; i<params.size(); ++i)
    all += " " + params[i];
  NumEffectState initial_state;
  osl::vector<Move> moves;
  try {
    record::usi::parse(all.substr(8), initial_state, moves);
  }
  catch (std::exception& e) {
    std::cerr << "usi parse error " << e.what() << "\n";
    return;
  }
  board->setMoves(initial_state, moves);
  board->showState();
}

void openurl(const std::vector<string>& params)
{
  if (params.size() < 2) {
    cout << "the openurl command supposes a url" << endl;
    return;
  }

  const string& url = params[1];
  const boost::filesystem::path temp_file 
    = cache_dir / boost::algorithm::replace_all_copy(url, "/", "_");
  if (gpsshell::getFileOverHttp(url, temp_file.native_file_string()))
    return;
  std::vector<string> tokens_to_pass;
  tokens_to_pass.push_back("open");
  tokens_to_pass.push_back(temp_file.native_file_string());
  if (params.size() == 3)
    tokens_to_pass.push_back(params[2]); // index
  open(tokens_to_pass);
}

// namespace conflict
void next(const std::vector<string>& params)
{
  int distance = 1;
  if (params.size() > 1) {
    std::istringstream is(params[1]);
    is >> distance;
  }
  for (int i=0; i<distance; ++i)
    board->next();
  board->showState();
}

void prev(const std::vector<string>& params)
{
  int distance = 1;
  if (params.size() > 1) {
    std::istringstream is(params[1]);
    is >> distance;
  }
  for (int i=0; i<distance; ++i)
    board->prev();
  board->showState();
}

void first(const std::vector<string>& /*params*/)
{
  board->first();
  board->showState();
}

void last(const std::vector<string>& /*params*/)
{
  board->last();
  board->showState();
}

void history(const std::vector<string>& /*params*/)
{
  board->showHistory();
}

const std::string adjust_move_string(const std::string& src, const NumEffectState& state)
{
  std::string result = src;
  for (size_t i=0; i<result.size(); ++i)
    result[i] = toupper(result[i]);
  if (result[0] != '+' && result[0] != '-')
    result.insert(0, record::csa::show(state.getTurn()));
  return result;
}

void move(const std::vector<string>& params)
{
  if (params.size() < 2) {
    cout << "The move command supposes a move string" << endl;
    return;
  }
  for (size_t i=1; i<params.size(); ++i) {
    const std::string move_string = adjust_move_string(params[i], board->getState());
    osl::Move move;
    try {
      move = osl::record::csa::strToMove(move_string, board->getState());
    }
    catch (std::runtime_error& e) {
      std::cerr << "parse error " << e.what() << "\n";
      return;
    }
    if (!move.isValid() || ! board->getState().isValidMove(move))
    {
      cout << "Invalid move: " << move << endl;
      return;
    }
    if (!board->isEndOfMoves())
    {
      pushBoard();
      board->shrink();
    }
    board->move(move);
    board->showState();
  }
}

void eval(const std::vector<string>& params)
{
  std::string name = "test";
  if (params.size() >= 2)
    name = params[1];
  board->showEval(name);
}

void recorddb(const std::vector<string>& params)
{
#ifdef USE_TOKYO_CABINET
  osl::NumEffectState state = board->getState();
  int win, loss, gps_win, gps_loss, bonanza_win, bonanza_loss;
  gpsshogi::recorddb::query(state, win, loss, gps_win, gps_loss, bonanza_win, bonanza_loss);
  if (win + loss)
    std::cout << "ץ " << win << "" << loss << " " << 100.0*win/(win+loss) << "%" << std::endl;
  if (gps_win + gps_loss)
    std::cout << "GPS v.s. Bonanza " << gps_win << "" << gps_loss << " "
	      << 100.0*gps_win/(gps_win+gps_loss) << "%" << std::endl;
  if (bonanza_win + bonanza_loss)
    std::cout << "Bonanza v.s. GPS " << bonanza_win << "" << bonanza_loss << " "
	      << 100.0*bonanza_win/(bonanza_win+bonanza_loss) << "%" << std::endl;
#endif
}

void search(const std::vector<string>& params)
{
  static const size_t time_default = 30;
  static size_t time = time_default;
  if (params.size() >= 2) 
  {
    try 
    {
      time = boost::lexical_cast<size_t>(params[1]);
    } 
    catch (boost::bad_lexical_cast& blc) 
    {
      std::cerr << blc.what() << std::endl;
      std::cerr << "USAGE: search <time_in_sec>" << std::endl
                << "time is now " << time << " "
                << "(default: " << time_default << ")"
                << std::endl;
      return;
    }
  }

  std::ostringstream os;
  os << board->getState();
  char move[8] =  {'\0'};
  ::search(os.str().c_str(), time, verbose, move);
  std::cout << move << std::endl;
}

/**
 * Show an evaluation value after quiecense search.
 */
void qsearch(const std::vector<string>& params)
{
  using namespace osl;
  using namespace osl::search;
  using namespace osl::misc;
  typedef QuiescenceSearch2<eval::ml::OpenMidEndingEval> qsearch_t;

  static const int depth_default = -2;
  static int depth = depth_default;
  static const bool verbose = false;

  if (params.size() >= 2) {
    try {
      depth = boost::lexical_cast<int>(params[1]);
    } 
    catch (boost::bad_lexical_cast& blc) {
      std::cerr << blc.what() << std::endl;
      std::cerr << "USAGE: qsearch [depth]\n"
                << "  depth: depth [default=" << depth_default << "]\n"
                << "\n"
                << "Queicense-search the current state\n"
                << std::endl;
      return;
    }
  }

  const NumEffectState state(board->getState());
  SimpleHashTable table(1000000, depth, verbose);
  SearchState2Core::checkmate_t  checkmate_searcher;
  SearchState2Core core(state, checkmate_searcher);
  qsearch_t qs(core, table);
  eval::ml::OpenMidEndingEval ev(state);

  const Move last_move = board->getCurrentMove();
  const int val = qs.search(state.getTurn(), ev, last_move, 4);
  const int normalized_val = static_cast<int>(val*200.0/eval::ml::OpenMidEndingEval::captureValue(newPtypeO(WHITE,PAWN)));
  std::cout << normalized_val << std::endl;
}

void checkmate_attack(const std::vector<string>& params)
{
  static const size_t limit_default = 20000000;
  int limit = limit_default;
  if (params.size() >= 2) 
  {
    try 
    {
      limit = boost::lexical_cast<size_t>(params[1]);
    } 
    catch (boost::bad_lexical_cast& blc) 
    {
      std::cerr << blc.what() << std::endl;
      std::cerr << "USAGE: checkmate_attack <limit>" << std::endl
                << "limit is now " << limit << " "
                << "(default: " << limit_default << ")"
                << std::endl;
      return;
    }
  }
  
  std::ostringstream os;
  os << board->getState();
  char move[8] = { '\0' };
  const int ret = ::checkmate_attack(os.str().c_str(), limit, move);
  if (ret)
  {
    std::cout << "Checkmate " << move << std::endl;
    std::cout << "node count " << limit << std::endl;
  }
  else
    std::cout << "No checkmate found" << std::endl;
}

void checkmate_escape(const std::vector<string>& params)
{
  static const size_t limit_default = 20000000;
  static size_t limit = limit_default;
  if (params.size() >= 2) 
  {
    try 
    {
      limit = boost::lexical_cast<size_t>(params[1]);
    } 
    catch (boost::bad_lexical_cast& blc) 
    {
      std::cerr << blc.what() << std::endl;
      std::cerr << "USAGE: checkmate_escape <limit>" << std::endl
                << "limit is now " << limit << " "
                << "(default: " << limit_default << ")"
                << std::endl;
      return;
    }
  }
  
  std::ostringstream os;
  os << board->getState();
  const int ret = ::checkmate_escape(os.str().c_str(), limit);
  if (ret)
    std::cout << "Losing (Checkmated)" << std::endl;
  else
    std::cout << "No checkmate found" << std::endl;
}

void threatmate(const std::vector<string>& params)
{
  static const size_t limit_default = 20000000;
  int limit = limit_default;
  if (params.size() >= 2) 
  {
    try 
    {
      limit = boost::lexical_cast<size_t>(params[1]);
    } 
    catch (boost::bad_lexical_cast& blc) 
    {
      std::cerr << blc.what() << std::endl;
      std::cerr << "USAGE: checkmate_attack <limit>" << std::endl
                << "limit is now " << limit << " "
                << "(default: " << limit_default << ")"
                << std::endl;
      return;
    }
  }
  
  std::ostringstream os;
  NumEffectState state = board->getState();
  state.changeTurn();
  os << state;
  char move[8] = { '\0' };
  const int ret = ::checkmate_attack(os.str().c_str(), limit, move);
  if (ret)
  {
    std::cout << "Threatmate " << move << std::endl;
    std::cout << "node count " << limit << std::endl;
  }
  else
    std::cout << "No checkmate found" << std::endl;
}

void generateMoves(const std::vector<string>& params)
{
  static int limit = 1000;
  if (params.size() > 1) {
    std::istringstream is(params[1]);
    is >> limit;
  }
  static osl::search::SearchState2::checkmate_t checkmate;
  const NumEffectState state = board->getState();
  osl::search::SearchState2 sstate(state, checkmate);
  osl::search::MoveGenerator generator;
  generator.initOnce();
  osl::search::SimpleHashRecord record;
  generator.init(limit, &record, osl::eval::ml::OpenMidEndingEval(state),
		 state, true, Move());
  MoveLogProbVector moves;
  generator.generateAll(state.getTurn(), sstate, moves);

  RatingEnv e;
  e.make(state);
  RatedMoveVector rated_moves;
  rating::StandardFeatureSet::instance().generateRating(state, e, limit, rated_moves);

  std::cout << "limit " << limit << "\n";
  for (size_t i=0; i<moves.size(); ++i) {
    std::cout << record::csa::show(moves[i].getMove()) 
	      << "  " << moves[i].getLogProb();
    const RatedMove *r = rated_moves.find(moves[i].getMove());
    if (r) {
      std::cout << "  " << r->rating() << "  " << r->optimisticRating();
    }
    std::cout << "\n";
  }
}

void generateLegalMoves(const std::vector<string>& params)
{
  const NumEffectState state = board->getState();
  MoveVector moves;
  LegalMoves::generate(state, moves);
  BOOST_FOREACH(Move m, moves)
    std::cout << record::csa::show(m) << "\n";
}

void rating(const std::vector<string>& params)
{
  static int min_rate = -200;
  if (params.size() > 1) {
    std::istringstream is(params[1]);
    is >> min_rate;
  }
  static const osl::rating::StandardFeatureSet& fs=osl::rating::StandardFeatureSet::instance();
  const NumEffectState state = board->getState();
  RatingEnv e;
  e.make(state);
  RatedMoveVector moves;
  fs.generateRating(state, e, 1400, moves);
  for (size_t i=0; i<moves.size(); ++i) {
    if (moves[i].rating() < min_rate)
      break;
    std::cout << record::csa::show(moves[i].move()) 
	      << "  " << moves[i].rating() 
	      << "  " << moves[i].optimisticRating() 
	      << "\n";
  }
}

void annotate(const std::vector<string>& params)
{
  static osl::rating::StandardFeatureSet fs;

  if (params.size() != 2) {
    cout << "The annotate command supposes a move string" << endl;
    return;
  }
  const std::string move_string = adjust_move_string(params[1], board->getState());
  osl::Move move;
  try {
    move = osl::record::csa::strToMove(move_string, board->getState());
  }
  catch (std::runtime_error& e) {
    std::cerr << "parse error " << e.what() << "\n";
    return;
  }
  const NumEffectState state = board->getState();
  if (!move.isValid() || !state.isValidMove(move)) {
    cout << "Invalid move: " << move << endl;
    return;
  }

  RatingEnv env;
  env.make(state);
  RatedMoveVector moves;
  fs.generateRating(state, env, 1400, moves);
  if (const RatedMove *found = moves.find(move)) {
    std::cout << found->rating() << " ";
  }
  std::cout << "[ " << fs.annotate(state, env, move) << " ]\n";
  const int see = PieceEval::computeDiffAfterMoveForRP(state, move);
  std::cout << "see " << see << "\n";
}

void showStates(const std::vector<string>& /*params*/)
{
  while(board->next())
    board->showState();
}

void csaShow(const std::vector<string>& params)
{
  if (params.size() == 1) {
    std::cout << board->getState();
    return;
  }
  std::ofstream os(params[1].c_str());
  os << board->getState();
}

void usiShow(const std::vector<string>& params)
{
  if (params.size() == 1) {
    std::cout << "position " 
	      << record::usi::show(board->getState())
	      << std::endl;
    return;
  }
  std::ofstream os(params[1].c_str());
  os << "position " 
     << record::usi::show(board->getState())
     << std::endl;
}
void usiHistory(const std::vector<string>& /*params*/)
{
  board->showUsiHistory();
}

void myshogi(const std::vector<string>& params)
{
  if (params.size() == 1) {
    std::cout << record::myshogi::show(board->getState());
    return;
  }
  std::ofstream os(params[1].c_str());
  os << record::myshogi::show(board->getState());
}

void setBlackColor(const std::vector<string>& params)
{
  if (params.size() != 2)
    return;

  const std::string& color_name = params[1];
  osl::record::Color color = osl::record::Color::colorFor(color_name);
  board->setBlackColor(color);
  board->showState();
}

void setWhiteColor(const std::vector<string>& params)
{
  if (params.size() != 2)
    return;

  const std::string& color_name = params[1];
  osl::record::Color color = osl::record::Color::colorFor(color_name);
  board->setWhiteColor(color);
  board->showState();
}

void setLastColor(const std::vector<string>& params)
{
  if (params.size() != 2)
    return;

  const std::string& color_name = params[1];
  osl::record::Color color = osl::record::Color::colorFor(color_name);
  board->setLastColor(color);
  board->showState();
}

void setVerbose(const std::vector<string>& params)
{
  if (params.size() != 2)
    return;
  try 
  {
    verbose = boost::lexical_cast<int>(params[1]);
  } 
  catch (boost::bad_lexical_cast& blc) 
  {
    std::cerr << blc.what() << std::endl;
    std::cerr << "USAGE: set_verbose verboseness" << std::endl;
  }
}

/**
 * Set an opening book.
 * @param params a path of a opening book file.
 */
void setOpeningBook(const std::vector<string>& params)
{
  if (params.size() != 2)
    return;
  const std::string& path = params[1];
  the_book.reset(new ::gpsshell::Book(path));
}

void openingShow(const std::vector<string>& params)
{
  if (!the_book)
    return;
  static int opeing_show_count = 3; // default value
  if (params.size() == 2)
    opeing_show_count = boost::lexical_cast<int>(params[1]);
  the_book->showState(board->getState(), board->getMovesToCurrent(), opeing_show_count);
}

void openingNext(const std::vector<string>& params)
{
  if (!the_book)
    return;
  size_t nth = 0; // default value
  if (params.size() == 2)
    nth = boost::lexical_cast<size_t>(params[1]);

  const ::gpsshell::WMoveContainer wmoves = 
    the_book->getMoves(board->getState(), board->getMovesToCurrent());
  if (wmoves.empty())
  {
    std::cout << "There is no next move." << std::endl;
    return;
  }
  if (wmoves.at(0).getWeight() == 0 && params.size() == 1)
  {
    // do not visit a zero-weighted move automatically.
    std::cout << "There is no weighted move. Specify an index (starting with zero) you want to visit.\n";
    return;
  }
  const osl::record::opening::WMove wmove = wmoves.at(std::min(nth, wmoves.size()-1));
  const std::string move_str = osl::record::csa::show(wmove.getMove());
  std::vector<std::string> command;
  command.push_back(params[0]);
  command.push_back(move_str);
  move(command);
}

/**
 * usage: onext_random_level
 */
void openingNextRandomLevel(const std::vector<string>& params)
{
  if (!the_book)
    return;

  ::gpsshell::WMoveContainer wmoves = 
    the_book->getMoves(board->getState(), board->getMovesToCurrent());

  ::gpsshell::deleteLessWeightedMoves(wmoves, 1); // remove zero-weighted moves
  if (wmoves.empty()) {
    std::cout << "There is no next move." << std::endl;
    return;
  }

  const size_t index = osl::misc::time_seeded_random() % wmoves.size();
  std::cout << boost::format("selected %2d of %2d move(s) [1,2,...]\n")
               % (index+1) % wmoves.size();
  const osl::record::opening::WMove wmove = wmoves.at(index);
  const std::string move_str = osl::record::csa::show(wmove.getMove());
  std::vector<std::string> command;
  command.push_back(params[0]);
  command.push_back(move_str);
  move(command);
}

/**
 * usage: onext_random_weight [coef]
 */
void openingNextRandomWeight(const std::vector<string>& params)
{
  if (!the_book)
    return;
  static int coef = 3;
  if (params.size() == 2)
    coef = boost::lexical_cast<int>(params[1]);
  if (coef <= 0) {
    std::cout << "coef should be > 0\n";
    coef = 3;
    return;
  }

  ::gpsshell::WMoveContainer wmoves = 
    the_book->getMoves(board->getState(), board->getMovesToCurrent());

  const int max_weight = ::gpsshell::getMaxWeight(wmoves);
  int criteria = max_weight / coef;
  if (criteria < 1)
    criteria = 1;
  ::gpsshell::deleteLessWeightedMoves(wmoves, criteria);
  if (wmoves.empty()) {
    std::cout << "There is no next move." << std::endl;
    return;
  }

  const int sum = ::gpsshell::getSumOfWeights(wmoves);
  assert(sum>0);
  const int target_weight = osl::misc::time_seeded_random() % sum;
  ::gpsshell::WMoveContainer::iterator it = wmoves.begin();
  for (int total = 0; it != wmoves.end(); ++it) {
    total += it->getWeight();
    if (target_weight < total)
      break;
  }

  std::cout << boost::format("selected %2d of %2d move(s) [1,2,...]\n")
               % (std::distance(wmoves.begin(), it) + 1) % wmoves.size();
  const osl::record::opening::WMove wmove = *it;
  const std::string move_str = osl::record::csa::show(wmove.getMove());
  std::vector<std::string> command;
  command.push_back(params[0]);
  command.push_back(move_str);
  move(command);
}

/**
 *
 * oinclude <file_name> <n-th move>
 */
void openingInclude(const std::vector<string>& params)
{
  const static std::string USAGE = "USAGE: oinclude <file_name> <n-th move>\n";
  if (!the_book)
    return;

  if (params.size() != 3)
  {
    std::cerr << USAGE;
    return;
  }

  const boost::filesystem::path path(params[1]);
  const std::string ext = boost::filesystem::extension(path);
  if (!boost::algorithm::iequals(".csa", ext))
  {
    std::cerr << boost::format("Incorrect file name: %s\n") % params[1];
    std::cerr << USAGE;
    return;
  }

  size_t nth_move = 0;
  try
  {
    nth_move = boost::lexical_cast<size_t>(params[2]);
  }
  catch (boost::bad_lexical_cast& blc) 
  {
    std::cerr << blc.what() << std::endl;
    std::cerr << "USAGE: oinclude <file_name> <n-th move>" << std::endl;
    return;
  }

  try
  {
    const CsaFile file(path.native_file_string());
    const osl::record::Record record = file.getRecord();
    const osl::vector<Move> moves = record.getMoves();

    if (nth_move > moves.size())
    {
      std::cerr << boost::format("Invalid argument. %d is too large for the file %s [%d moves]\n")
                   % nth_move % path.native_file_string() % moves.size();
      std::cerr << USAGE;
      return;
    }

    SimpleState state((HIRATE));
    BOOST_FOREACH(const Move& move, moves)
    {
      ApplyMoveOfTurn::doMove(state, move);
    }

    Player self = nth_move % 2 == 1 ? BLACK : WHITE;
    const boost::shared_ptr<osl::record::opening::WeightedBook> book = 
      the_book->getWeightedBook();
    const int rc =  book->getStateIndex(state, false, self);
    if (rc == -1)
    {
      std::cout << "The state is not found in the book\n";
    }
    else
    {
      std::cout << "The state is included in the book\n";
    }
  }
  catch (osl::record::csa::CsaIOError& e)
  {
    std::cerr << boost::format("Failed to open the csa file: %s\n") % path.native_file_string();
    std::cerr << USAGE;
    return;
  }
}

/**
 * Show the N shortest lines in a book.
 * command: oshow_shortest_lines [top_n]
 */
void openingShowShortestLines(const std::vector<string>& params)
{
  const static std::string USAGE = "USAGE: oshow_shortest_lines [top_n]\n";
  if (!the_book)
    return;

  if (params.size() >= 3) {
    std::cerr << USAGE;
    return;
  }

  static int TOPN = 3; // default value
  if (params.size() == 2)
    TOPN = boost::lexical_cast<int>(params[1]);

  boost::shared_ptr<osl::record::opening::WeightedBook> book = the_book->getWeightedBook();

  std::vector<gpsshell::VisitFlag> history(book->getTotalState(), gpsshell::UNVISITED);
  std::deque<gpsshell::VisitState> fronts;
  fronts.push_back(gpsshell::VisitState(book->getStartState()));
  
  int temp_top = TOPN; 
  while (!fronts.empty()) {
    gpsshell::VisitState vs = fronts.front(); 
    fronts.pop_front();
    history[vs.state_index] = gpsshell::VISITED;

    const record::opening::WeightedBook::WMoveContainer wmoves = book->getMoves(vs.state_index, false);
    if (wmoves.empty()) {
      // found a leaf node!
      std::cout << "\n";
      BOOST_FOREACH(const Move& m, vs.moves) {
        std::cout << osl::record::csa::show(m);
      }
      std::cout << "\n";
      the_book->showState(book->getBoard(vs.state_index), vs.moves, 3);
      if (--temp_top < 1) 
        break; // exit the whole loop
      else
        continue;
    }

    BOOST_FOREACH(const record::opening::WMove& wmove, wmoves) {
      const int next_index = wmove.getStateIndex();
      if (! history[next_index] == gpsshell::UNVISITED)
        continue;

      gpsshell::VisitState next(next_index);
      next.moves = vs.moves; // copy
      next.moves.push_back(wmove.getMove());
      fronts.push_back(next);
      history[next_index] = ENTERED;
    } // for wmoves
  } // while
}

void setIgnoreList(const std::vector<string>& params)
{
  if (params.size() != 2) {
    std::cerr << "set_ignorelists filename\n";
    return;
  }
  const std::string& path = params[1];
  the_ignorelist.openFile(path);
}

void ignoreListShow(const std::vector<string>& params)
{
  if (the_ignorelist.isEmpty()) {
    std::cerr << "set \"set_ignorelists filename\" first\n";
    return;
  }
  const gpsshell::IgnoreList::index_t index = the_ignorelist.getCurrentIndex();
  std::vector<std::string> args;
  args.push_back("open");
  args.push_back(index.get<0>().file_string());
  args.push_back(boost::lexical_cast<std::string>(index.get<1>()));
  const std::string& comment = index.get<2>();
  if (!comment.empty())
  std::cout << boost::format("%s\n") % comment;

  std::cout << boost::format("%s %s\n")
               % args[1] % args[2];
  open(args);
}

void ignoreListNext(const std::vector<std::string>& params)
{
  if (!the_ignorelist.hasNext())
  {
    std::cout << "There is no next game.\n";
    return;
  }

  the_ignorelist.next();
  std::vector<std::string> args;
  args.push_back("ignorelist_show");
  ignoreListShow(args);
}

void ignoreListPrev(const std::vector<std::string>& params)
{
  if (!the_ignorelist.hasPrev())
  {
    std::cout << "There is no previous game.\n";
    return;
  }

  the_ignorelist.prev();
  std::vector<std::string> args;
  args.push_back("ignorelist_show");
  ignoreListShow(args);
}

void ignoreListFirst(const std::vector<std::string>& params)
{
  if (the_ignorelist.isEmpty()) {
    std::cerr << "set \"set_ignorelists filename\" first\n";
    return;
  }
  the_ignorelist.first();
  std::vector<std::string> args;
  args.push_back("ignorelist_show");
  ignoreListShow(args);
}

void ignoreListLast(const std::vector<std::string>& params)
{
  if (the_ignorelist.isEmpty()) {
    std::cerr << "set \"set_ignorelists filename\" first\n";
    return;
  }
  the_ignorelist.last();
  std::vector<std::string> args;
  args.push_back("ignorelist_show");
  ignoreListShow(args);
}

void ignoreListShowAll(const std::vector<std::string>& params)
{
  if (the_ignorelist.isEmpty()) {
    std::cerr << "set \"set_ignorelists filename\" first\n";
    return;
  }
  the_ignorelist.first();
  std::vector<std::string> args;
  args.push_back("ignorelist_show");
  ignoreListShow(args);
  while (the_ignorelist.hasNext())
  {
    args.clear();
    args.push_back("ignorelist_next");
    ignoreListNext(args);

    std::cout << "\n";
  }
}

#ifdef OSL_SMP
void setNumCPUs(const std::vector<string>& params)
{
  if (params.size() != 2)
    return;
  try 
  {
    const int num_cpus = boost::lexical_cast<int>(params[1]);
    osl::OslConfig::setNumCPUs(num_cpus);
  } 
  catch (boost::bad_lexical_cast& blc) 
  {
    std::cerr << blc.what() << std::endl;
    std::cerr << "USAGE: set_num_cpus #cpus" << std::endl;
  }
}
#endif

void version(const std::vector<string>& params)
{
  std::string name = "gpsshogi ";
#ifdef OSL_SMP
  name += "(smp) ";
#endif
  std::cout << name << gpsshogi::gpsshogi_revision << std::endl;
}

} // namespace gpsshell

class MyObserver : public record::RecordVisitorObserver
{
public:
  MyObserver(){}
  ~MyObserver() {}
  void update(record::RecordVisitor* rv);
};

void MyObserver::update(record::RecordVisitor* rv)
{
  const record::MoveRecord *move_record = rv->getLastMove();
  const Move& new_move = move_record->getMove();
  std::vector<std::string> params;
  params.push_back("move");
  params.push_back(record::csa::show(new_move));
  gpsshell::move(params);
  std::cout << "time: " << move_record->getTime() << std::endl;
}


/*========== MAIN ==========*/

void printUsage(std::ostream& out, 
                char **argv,
                const boost::program_options::options_description& command_line_options)
{
  out << 
    "Usage: " << argv[0] << " [options]" << "\n"
      << command_line_options 
      << std::endl;
}

int parseCommandLine(int argc, char **argv, 
                     boost::program_options::variables_map& vm)
{
  boost::program_options::options_description command_line_options;
  command_line_options.add_options()
    ("color",   boost::program_options::value<std::string>(),
                "Specify three comma-separated colors to show colorful pieces for BLACK, WHITE and the last move, respectively. eX. --color blue,brown,red")
    ("stdin",   boost::program_options::value<std::string>(),
                "Read a file from a specified type from the stdin.")
    ("version", "Show version information")
    ("help,h",  "Show help message");
  boost::program_options::positional_options_description p;

  try
  {
    boost::program_options::store(
      boost::program_options::command_line_parser(
	argc, argv).options(command_line_options).positional(p).run(), vm);
    boost::program_options::notify(vm);
    if (vm.count("help"))
    {
      printUsage(cout, argv, command_line_options);
      return 0;
    }
    if (vm.count("version")) 
    {
      std::string name = "gpsshell ";
#ifdef OSL_SMP
      name += "(smp) ";
#endif
      std::cout << name << gpsshogi::gpsshogi_revision << "\n\n"
        << gpsshogi::gpsshogi_copyright << "\n";
      return 0;
    }
  }
  catch (std::exception &e)
  {
    std::cerr << "error in parsing options" << "\n"
	      << e.what() << std::endl;
    printUsage(std::cerr, argv, command_line_options);
    return 1;
  }

  if (vm.count("color"))
  {
    const std::string color = vm["color"].as<std::string>();
    typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
    boost::char_separator<char> sep(",");
    tokenizer tokens(color, sep);
    tokenizer::iterator each = tokens.begin();
    assert(each != tokens.end());
    board->setBlackColor(osl::record::Color::colorFor(*each++));
    assert(each != tokens.end());
    board->setWhiteColor(osl::record::Color::colorFor(*each++));
    assert(each != tokens.end());
    board->setLastColor(osl::record::Color::colorFor(*each++));
    assert(each == tokens.end());
  }
  return 1;
}

void processGpsshellHome(boost::filesystem::path& readline_home)
{
  std::string GPSSHELL_HOME = "/tmp/gpsshell"; // default value
  if (getenv("HOME"))
    GPSSHELL_HOME = std::string(getenv("HOME")) + "/.gpsshell";
  const boost::filesystem::path home(GPSSHELL_HOME);
  if (!boost::filesystem::exists(home))
    boost::filesystem::create_directories(home);

  // re-create download cache 
  cache_dir = boost::filesystem::path(GPSSHELL_HOME) / "download_cache";
  if (boost::filesystem::exists(cache_dir))
    boost::filesystem::remove_all(cache_dir);
  boost::filesystem::create_directories(cache_dir);
  
  readline_home = home / "gpsshell";
}

void setCommandsMain(MyContainer& container)
{
  container.push_back( MyElement( "exit", 0 ) );
  container.push_back( MyElement( "quit", 0 ) );
  container.push_back( MyElement( "version",    gpsshell::version ) );
  container.push_back( MyElement( "open %file", gpsshell::open ) );
  container.push_back( MyElement( "openurl",    gpsshell::openurl ) );
  container.push_back( MyElement( "next",       gpsshell::next ) );
  container.push_back( MyElement( "n",          gpsshell::next ) );
  container.push_back( MyElement( "prev",       gpsshell::prev ) );
  container.push_back( MyElement( "p",          gpsshell::prev ) );
  container.push_back( MyElement( "first",      gpsshell::first ) );
  container.push_back( MyElement( "last",       gpsshell::last ) );
  container.push_back( MyElement( "history",    gpsshell::history ) );
  container.push_back( MyElement( "move",       gpsshell::move ) );
  container.push_back( MyElement( "rollback",   gpsshell::rollBack ) );
  container.push_back( MyElement( "eval",       gpsshell::eval ) );
  container.push_back( MyElement( "recorddb",       gpsshell::recorddb ) );
  container.push_back( MyElement( "search",     gpsshell::search ) );
  container.push_back( MyElement( "qsearch",     gpsshell::qsearch ) );
  container.push_back( MyElement( "checkmate_attack", gpsshell::checkmate_attack ) );
  container.push_back( MyElement( "checkmate_escape", gpsshell::checkmate_escape ) );
  container.push_back( MyElement( "threatmate", gpsshell::threatmate ) );
  container.push_back( MyElement( "generatemoves",    gpsshell::generateMoves ) );
  container.push_back( MyElement( "generatelegalmoves",    gpsshell::generateLegalMoves ) );
  container.push_back( MyElement( "rating",     gpsshell::rating ) );
  container.push_back( MyElement( "annotate",   gpsshell::annotate ) );
  container.push_back( MyElement( "csashow",    gpsshell::csaShow ) );
  container.push_back( MyElement( "usishow",    gpsshell::usiShow ) );
  container.push_back( MyElement( "usihistory", gpsshell::usiHistory ) );
  container.push_back( MyElement( "myshogi",    gpsshell::myshogi ) );
  container.push_back( MyElement( "showstates", gpsshell::showStates ) );
  container.push_back( MyElement( "position",   gpsshell::usiposition ) );
  container.push_back( MyElement( "set_black_color",  gpsshell::setBlackColor ) );
  container.push_back( MyElement( "set_white_color",  gpsshell::setWhiteColor ) );
  container.push_back( MyElement( "set_last_color",   gpsshell::setLastColor ) );
  container.push_back( MyElement( "set_verbose",   gpsshell::setVerbose ) );
  // --- commands rerated with opening books
  container.push_back( MyElement( "set_openingbook %file", gpsshell::setOpeningBook ) );
  container.push_back( MyElement( "oshow",                 gpsshell::openingShow ) );
  container.push_back( MyElement( "onext",                 gpsshell::openingNext ) );
  container.push_back( MyElement( "onext_random_level",    gpsshell::openingNextRandomLevel ) );
  container.push_back( MyElement( "onext_random_weight",   gpsshell::openingNextRandomWeight ) );
  container.push_back( MyElement( "oinclude",              gpsshell::openingInclude ) );
  container.push_back( MyElement( "oshow_shortest_lines",  gpsshell::openingShowShortestLines) );
  // --- commands rerated with ignore lists
  container.push_back( MyElement( "set_ignorelists %file",    gpsshell::setIgnoreList ) );
  container.push_back( MyElement( "ignorelist_show",    gpsshell::ignoreListShow ) );
  container.push_back( MyElement( "ignorelist_showall", gpsshell::ignoreListShowAll ) );
  container.push_back( MyElement( "ignorelist_next",    gpsshell::ignoreListNext ) );
  container.push_back( MyElement( "ignorelist_prev",    gpsshell::ignoreListPrev ) );
  container.push_back( MyElement( "ignorelist_first",   gpsshell::ignoreListFirst ) );
  container.push_back( MyElement( "ignorelist_last",    gpsshell::ignoreListLast ) );
#ifdef OSL_SMP
  container.push_back( MyElement( "set_num_cpus",   gpsshell::setNumCPUs ) );
#endif
}

#ifndef MINIMAL_GPSSHELL
template <class SReadline, class Shelp_executor>
void setCommands(SReadline& reader, 
                 Shelp_executor& help_executor, 
                 MyContainer& container)
{
  // The following is supported:
  // - "identifiers"
  // - special identifier %file - means to perform a file name completion
  container.push_back(
    MyElement("help", 
              boost::bind1st(boost::mem_fun(&Shelp_executor::operator()),
                              &help_executor) 
             ) 
  );
  setCommandsMain(container);
  // Now register the completers.
  // Actually it is possible to re-register another set at any time
  reader.RegisterCompletions( container );
}
#endif

int main(int argc, char **argv)
{
#ifndef MINIMAL_GPSSHELL
  Poco::Net::HTTPStreamFactory::registerFactory();
#endif
  osl::eval::ProgressEval::setUp();
  osl::eval::ml::OpenMidEndingEval::setUp();
  osl::progress::ml::NewProgress::setUp();
  board.reset(new gpsshell::Board());

  boost::program_options::variables_map vm;
  if (!parseCommandLine(argc, argv, vm))
    exit(1);
  
  const static size_t max_stored_commands = 32;
  boost::filesystem::path readline_home;
  processGpsshellHome(readline_home);
#ifndef MINIMAL_GPSSHELL
  SReadline reader(readline_home.native_file_string(), max_stored_commands);
#endif
  // Completers with functions to be called
  MyContainer advanced_completers;
  // Help executor functor
#ifndef MINIMAL_GPSSHELL
  Shelp_executor help_executor(advanced_completers);
  setCommands(reader, help_executor, advanced_completers);
#else
  setCommandsMain(advanced_completers);
#endif

  if (vm.count("stdin"))
  {
    const std::string& file_type = vm["stdin"].as<std::string>();
    if ("csa" == file_type)
    {
      boost::shared_ptr<record::RecordVisitor> rv(new record::RecordVisitor());
      rv->addObserver(new MyObserver()); // ptr_vector
      record::Record record;
      record::csa::InputStream is(std::cin, rv);
      is.load(&record);
      return 0;
    }
    else
    {
      std::cerr << "Not supported file type: " << file_type << std::endl;
      return 1;
    }
  }

  const char *path = osl::OslConfig::openingBook(); // default 
  the_book.reset(new gpsshell::Book(path));
  board->showState();

  bool end_of_input = false; // true when we should exit
  std::vector<string> tokens; // List of the entered tokens
  while(true)
  {
    // We get the list of tokens
#ifdef MINIMAL_GPSSHELL
    std::cerr << "> " << std::flush;
    tokens.clear();
    std::string line, word;
    std::getline(cin, line);
    std::istringstream is(line);
    while (is >> word) 
      tokens.push_back(word);
#else
    reader.GetLine( "> ", tokens, end_of_input );
#endif

    if (end_of_input)
    {
      cout << "End of the session. Exiting." << endl;
      break;
    }
#ifndef MINIMAL_GPSSHELL
    if (tokens.empty())
    {
      // if no command, try the last command.
      std::vector<string> history;
      reader.GetHistory(history);
      std::cout << "history size " << history.size() << std::endl;
      if (!history.empty())
      {
        const string& last_command = history.back();
        SplitTokens(last_command, tokens);
      }
    }
#endif
    if (tokens.front() == "exit" || tokens.front() == "quit")
      break;

    MyContainer::iterator found = 
      find_if(advanced_completers.begin(), 
              advanced_completers.end(), 
              LookupFunctor(tokens));
    if (found == advanced_completers.end())
    {
      cout << 
        "Unknown command. Type 'help' for the list of acceptable commands" 
        << endl;
      continue;
    }
    if (found->second != 0) 
      found->second(tokens); // execute the action
  } // while

  return 0;
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
