/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2003 Bert Vermeulen
    Copyright (C) 2002-2003 Peter Stamfest

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: dt_password.c,v 1.23 2006/04/07 20:33:41 herzi Exp $ */

#include <config.h>

#ifdef HAVE_LIBCRYPTO

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

#if defined(HAVE_LIBCRYPTO)
#include <openssl/rand.h>
#include <openssl/des.h>
#include <openssl/md5.h>
#include <openssl/md4.h>
#include <openssl/sha.h>
#endif /* defined(HAVE_LIBCRYPTO) */

#include "common.h"
#include "util.h"
#include "errorchain.h"
#include "formfill.h"
#include "input.h"
#include "tinput.h"
#include "browse.h"
#include "encode.h"
#include "ldif.h" /* for b64_decode */
#include "dtutil.h"
#include "dt_password.h"

#if defined(HAVE_ICONV_H)
#include <iconv.h>
#endif /* HAVE_ICONV_H */

#if defined(HAVE_LIBCRYPTO)


/* NOTE: The password-hash generating functions may change the data
   in-place. They are explicitly permitted to do so. */
static GByteArray *dt_password_encode_password_crypt(char *data, 
						     unsigned int len);
static GByteArray *dt_password_encode_password_md5(char *data,
						   unsigned int len);
static GByteArray *dt_password_encode_password_sha1(char *data,
						    unsigned int len);
static GByteArray *dt_password_encode_password_ssha1(char *data,
						     unsigned int len);
static GByteArray *dt_password_encode_password_smd5(char *data,
						    unsigned int len);
static GByteArray *dt_password_encode_password_nthash(char *data,
						      unsigned int len);
static GByteArray *dt_password_encode_password_lmhash(char *data,
						      unsigned int len);

typedef GByteArray *(CryptFunc)(char *data, unsigned int len);
#endif

const struct tokenlist cryptmap[] = {
     { 0,                 "Clear", NULL },
#if defined(HAVE_LIBCRYPTO)
     /* currently, we still need unique token codes, should drop this
        requirement */
     { FLAG_ENCODE_CRYPT, "Crypt", dt_password_encode_password_crypt },
     { FLAG_ENCODE_SSHA,  "SSHA",  dt_password_encode_password_ssha1 },
     { FLAG_ENCODE_SMD5,  "SMD5",  dt_password_encode_password_smd5  },
     { FLAG_ENCODE_MD5,   "MD5",   dt_password_encode_password_md5   },
     { FLAG_ENCODE_SHA,   "SHA",   dt_password_encode_password_sha1  },
     { FLAG_ENCODE_NTHASH, "NTHASH", dt_password_encode_password_nthash  },
     { FLAG_ENCODE_LMHASH, "LMHASH", dt_password_encode_password_lmhash  },
#endif
     { 0,		  "",      NULL },
};



static dt_password_handler dt_password_handler_vtab = {
     {
	  "Password",
	  TRUE,
	  TRUE,

	  dt_password_get_widget,
	  dt_password_get_data,
	  dt_password_set_data,
	  bervalLDAPMod			/* reuse method from dt_entry */
     },
#if GTK_MAJOR < 2
     decode_utf8, /* encode method */	/* reuse method from dt_entry */
     encode_utf8, /* decode method */	/* reuse method from dt_entry */
#else
     /* gtk2 uses UTF-8 natively! Yipieeh */
     NULL,
     NULL,
#endif
};

display_type_handler *dt_password_get_handler() {
     return (display_type_handler *) &dt_password_handler_vtab;
}


#if defined(HAVE_LIBCRYPTO)

static GByteArray *dt_password_encode_password_crypt(char *data, 
						     unsigned int len)
{
     GString *salt;
     GByteArray *gb = g_byte_array_new();
     unsigned char *password, rand[16], cryptbuf[32];

     password = (guchar*) g_malloc((len + 1) * sizeof(guchar));
     memcpy(password, data, len);
     password[len] = 0;

     salt = g_string_sized_new(32);
     RAND_pseudo_bytes(rand, 8);
     b64_encode(salt, (gchar*)rand, 8);
     /* crypt(3) says [a-zA-Z0-9./] while base64 has [a-zA-Z0-9+/] */
     if(salt->str[0] == '+') salt->str[0] = '.';
     if(salt->str[1] == '+') salt->str[1] = '.';
     salt->str[2] = 0;

     g_byte_array_append(gb, (guchar*)"{CRYPT}", 7);
     des_fcrypt((gchar*)password, salt->str, (gchar*)cryptbuf);

     g_byte_array_append(gb, cryptbuf, strlen((gchar*)cryptbuf));

     g_string_free(salt, TRUE);
     g_free(password);

     return gb;
}

static GByteArray *dt_password_encode_password_md5(char *data, 
						   unsigned int len)
{
     char md5_out[MD5_DIGEST_LENGTH];
     GString *b64 = g_string_sized_new(MD5_DIGEST_LENGTH * 4 / 3 + 4);

     GByteArray *gb = g_byte_array_new();

     MD5((guchar*)data, len, (guchar*)md5_out);
     b64_encode(b64, md5_out, MD5_DIGEST_LENGTH);

     g_byte_array_append(gb, (guchar*)"{MD5}", 5);
     g_byte_array_append(gb, (guchar*)b64->str, strlen(b64->str));
     
     g_string_free(b64, TRUE);

     return gb;
}

static GByteArray *dt_password_encode_password_sha1(char *data, 
						    unsigned int len)
{
     char sha_out[SHA_DIGEST_LENGTH];
     GString *b64 = g_string_sized_new(SHA_DIGEST_LENGTH * 4 / 3 + 4);

     GByteArray *gb = g_byte_array_new();

     SHA1((guchar*)data, len, (guchar*)sha_out);
     b64_encode(b64, sha_out, SHA_DIGEST_LENGTH);

     g_byte_array_append(gb, (guchar*)"{SHA}", 5);
     g_byte_array_append(gb, (guchar*)b64->str, strlen(b64->str));

     g_string_free(b64, TRUE);

     return gb;
}


static GByteArray *dt_password_encode_password_ssha1(char *data, 
						     unsigned int len)
{
     unsigned char rand[4];
     unsigned char sha_out[SHA_DIGEST_LENGTH + sizeof(rand)];
     SHA_CTX  SHA1context;

     GString *b64 = g_string_sized_new(SHA_DIGEST_LENGTH * 4 / 3 + 4);
     GByteArray *gb = g_byte_array_new();
     RAND_pseudo_bytes(rand, sizeof(rand));

     SHA1_Init(&SHA1context);
     SHA1_Update(&SHA1context, data, len);
     SHA1_Update(&SHA1context, rand, sizeof(rand));
     SHA1_Final(sha_out, &SHA1context);

     memcpy(sha_out + SHA_DIGEST_LENGTH, rand, sizeof(rand));

     b64_encode(b64, (gchar*)sha_out, sizeof(sha_out));

     g_byte_array_append(gb, (guchar*)"{SSHA}", 6);
     g_byte_array_append(gb, (guchar*)b64->str, strlen(b64->str));

     g_string_free(b64, TRUE);

     return gb;
}

static GByteArray *dt_password_encode_password_smd5(char *data,
						    unsigned int len)
{
     unsigned char rand[4];
     unsigned char md5_out[MD5_DIGEST_LENGTH + sizeof(rand)];
     MD5_CTX  MD5context;

     GString *b64 = g_string_sized_new(MD5_DIGEST_LENGTH * 4 / 3 + 4);
     GByteArray *gb = g_byte_array_new();
     RAND_pseudo_bytes(rand, sizeof(rand));

     MD5_Init(&MD5context);
     MD5_Update(&MD5context, data, len);
     MD5_Update(&MD5context, rand, sizeof(rand));
     MD5_Final(md5_out, &MD5context);

     memcpy(md5_out + MD5_DIGEST_LENGTH, rand, sizeof(rand));

     b64_encode(b64, (gchar*)md5_out, sizeof(md5_out));

     g_byte_array_append(gb, (guchar*)"{SMD5}", 6);
     g_byte_array_append(gb, (guchar*)b64->str, strlen(b64->str));

     g_string_free(b64, TRUE);

     return gb;
}

static const char hexdigit[] = "0123456789ABCDEF";
static GByteArray *dt_password_encode_password_nthash(char *data, 
						      unsigned int len)
{
     unsigned char md4_out[MD4_DIGEST_LENGTH];
     unsigned int i;
     GByteArray *unicode = g_byte_array_new();
     GByteArray *gb = g_byte_array_new();
     MD4_CTX MD4context;

#if defined(HAVE_ICONV)
     ICONV_CONST char *in;
     char *out;
     size_t inlen, outlen;
     iconv_t conv;

     conv = iconv_open("UNICODE", gq_codeset);
     if (conv != (iconv_t) -1) {
	  in = (ICONV_CONST char *) data;
	  inlen = len;
	  outlen = len * 2 + 4;
	  g_byte_array_set_size(unicode, outlen);
	  out = (gchar*)unicode->data;
	  
	  while(inlen > 0 && outlen > 0) {
	       if(iconv(conv, &in, &inlen, &out, &outlen) != 0) {
		    in++;		/* *** */
		    inlen--;
	       }
	  }
	  iconv_close(conv);
	  out = (gchar*)unicode->data;

	  /* strip leading endian marker (should be ff fe) */
	  assert(out[0] = 0xff);
	  assert(out[1] = 0xfe);
	  
	  memmove(out, out + 2, len * 2);
	  g_byte_array_set_size(unicode, len * 2);
     } else {
	  for (i = 0 ; i < len ; i++) {
	       g_byte_array_append(unicode, (guchar*)data + i, 1);
	       g_byte_array_append(unicode, (guchar*)"\0", 1);
	  }
     }
#else /* HAVE_ICONV */
     for (i = 0 ; i < len ; i++) {
	  g_byte_array_append(in, data + i, 1);
	  g_byte_array_append(in, "\0", 1);
     }
#endif /* HAVE_ICONV */

     MD4_Init(&MD4context);
     MD4_Update(&MD4context, unicode->data, unicode->len);
     MD4_Final(md4_out, &MD4context);

     for(i = 0 ; i < sizeof(md4_out) ; i++) {
	  char hex[2];
	  hex[0] = hexdigit[md4_out[i] / 16];
	  hex[1] = hexdigit[md4_out[i] % 16];
	  g_byte_array_append(gb, (guchar*)hex, 2);
     }

     return gb;
}


/* FIXME: silently assumes US-ASCII (or a single-byte encoding to be
   handled by toupper) */

static void lm_make_key(const char *pw, des_cblock *key)
{
     int i;
     char *k = (char *) key;
     
     k[0] = 0;
     for ( i = 0 ; i < 7 ; i++ ) {
	  k[i]   |= (pw[i] >> i) & 0xff;
	  k[i+1]  = (pw[i] << (7 - i)) & 0xff;
     }

     des_set_odd_parity(key);
}

static const char *lmhash_key = "KGS!@#$%";
static GByteArray *dt_password_encode_password_lmhash(char *data, 
						      unsigned int len)
{
     unsigned int i;
     char hex[2];
     char plain[15];
     des_key_schedule schedule;
     GByteArray *gb = NULL;
     des_cblock ckey1, ckey2;
     des_cblock bin1,  bin2;

     memset(plain, 0, sizeof(plain));
     
     for (i = 0 ; i < len && i < 14 ; i++) {
	  plain[i] = toupper(data[i]);
     }

     lm_make_key(plain, &ckey1);
     des_set_key_unchecked(&ckey1, schedule);
     des_ecb_encrypt((des_cblock*)lmhash_key, &bin1, schedule, DES_ENCRYPT);

     lm_make_key(plain + 7, &ckey2);
     des_set_key_unchecked(&ckey2, schedule);
     des_ecb_encrypt((des_cblock*)lmhash_key, &bin2, schedule, DES_ENCRYPT);

     gb = g_byte_array_new();

     for(i = 0 ; i < sizeof(bin1) ; i++) {
	  hex[0] = hexdigit[bin1[i] / 16];
	  hex[1] = hexdigit[bin1[i] % 16];
	  g_byte_array_append(gb, (guchar*)hex, 2);
     }
     for(i = 0 ; i < sizeof(bin2) ; i++) {
	  hex[0] = hexdigit[bin2[i] / 16];
	  hex[1] = hexdigit[bin2[i] % 16];
	  g_byte_array_append(gb, (guchar*)hex, 2);
     }
     return gb;
}

#endif /* HAVE_LIBCRYPTO */

GtkWidget *dt_password_get_widget(int error_context,
				  struct formfill *form, GByteArray *data,
				  GtkSignalFunc *activatefunc,
				  gpointer funcdata) 
{
     GtkWidget *hbox;
     GtkWidget *inputbox;
     GtkWidget *combo;
     GtkStyle *style;

     GList *cryptlist = NULL;
     int temp, max_width, i;


     hbox = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox);
    
     inputbox = gtk_entry_new();
     gtk_widget_show(inputbox);


     gtk_box_pack_start(GTK_BOX(hbox), inputbox, TRUE, TRUE, 0);
    
     /* XXX this will need an encoder wedge */
     if(activatefunc)
	  gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
				    GTK_SIGNAL_FUNC(activatefunc),
				    (gpointer) funcdata);
    
     combo = gtk_combo_new();
#ifdef OLD_FOCUS_HANDLING
     GTK_WIDGET_UNSET_FLAGS(GTK_COMBO(combo)->entry, GTK_CAN_FOCUS);
#endif
     gtk_editable_set_editable(GTK_EDITABLE(GTK_COMBO(combo)->entry), FALSE);

     style = gtk_widget_get_style(GTK_COMBO(combo)->entry);

     cryptlist = NULL;
     temp = max_width = 0;

     for(i = 0 ; cryptmap[i].keyword[0] ; i++) {
#if GTK_MAJOR >= 2
	  temp = gdk_string_width(gtk_style_get_font(style), cryptmap[i].keyword);
#else
	  temp = gdk_string_width(style->font, cryptmap[i].keyword);
#endif
	  if (temp > max_width)
	       max_width = temp;
	  cryptlist = g_list_append(cryptlist, (char *) cryptmap[i].keyword);
     }
     gtk_combo_set_popdown_strings(GTK_COMBO(combo), cryptlist);
     gtk_widget_set_usize(GTK_COMBO(combo)->entry, max_width + 20, -1);
     g_list_free(cryptlist);

     gtk_widget_show(combo);
     gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);

     dt_password_set_data(form, data, hbox);

     return hbox;
}


void dt_password_set_data(struct formfill *form, GByteArray *data, 
			  GtkWidget *hbox) 
{
     const char *crypt_type = "Clear";
     GList *boxchildren;
     GtkWidget *entry, *combo;
     int i;

     boxchildren = GTK_BOX(hbox)->children;
     entry = ((struct _GtkBoxChild *) boxchildren->data)->widget;
     combo = ((struct _GtkBoxChild *) boxchildren->next->data)->widget;

     editable_set_text(GTK_EDITABLE(entry), data, 
		       DT_PASSWORD(form->dt_handler)->encode,
		       DT_PASSWORD(form->dt_handler)->decode);
     
     if(data && data->data[0] == '{') {
	  char crypt_prefix[12];
	  for(i = 1; data->data[i] && data->data[i] != '}' && i < 10; i++)
	       crypt_prefix[i - 1] = data->data[i];
	  crypt_prefix[i - 1] = '\0';
	  crypt_type = detokenize(cryptmap, 
				  tokenize(cryptmap, crypt_prefix));
     }

     /* set the combo to the encoding type */
     gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), crypt_type);
}

GByteArray *dt_password_get_data(struct formfill *form, GtkWidget *hbox) 
{
     GByteArray *data;
     GtkWidget *combo;
     GList *boxchildren;
     gchar *crypt_type;

     boxchildren = GTK_BOX(hbox)->children;

     data = editable_get_text(GTK_EDITABLE(((struct _GtkBoxChild *) boxchildren->data)->widget));

     if(boxchildren->next) {
	  int cryptflag;
	  /* advance to crypt combo */
	  boxchildren = boxchildren->next;
	  combo = ((struct _GtkBoxChild *) boxchildren->data)->widget;
	  crypt_type = 
	       gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combo)->entry),
				      0, -1);

	  cryptflag = tokenize(cryptmap, crypt_type);

	  /* check to see if the content was already crypted.... */
	  if (data && data->data[0] != '{') {
	       /* need to crypt the stuff */
	       CryptFunc *cryptfunc = detokenize_data(cryptmap, cryptflag);
	       if (cryptfunc != NULL) {
		    GByteArray *crypted = cryptfunc((gchar*)data->data, data->len);
		    /* overwrite plain-text */
		    memset(data->data, 0, data->len);
		    g_byte_array_free(data, TRUE);
		    data = crypted;
	       }
	  }
	  
	  g_free(crypt_type);
     }

     return data;
}

#endif /* HAVE_LIBCRYPTO */

/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */

