/*
    This file is part of KDE.

    Copyright (c) 2008 Stephen Kelly <steveire@gmail.com>

    This library is free software; you can redistribute it and/or modify it
    under the terms of the GNU Library General Public License as published by
    the Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This library is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
    License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to the
    Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#include <QtTest>
#include <QtCore>
#include <qtestevent.h>
#include <QtCore/QRegExp>
#include <QtGui/QTextCursor>
#include <QtGui/QTextDocument>
#include <QtGui/QTextEdit>

#include "markupdirector.h"
#include "texthtmlbuilder.h"

using namespace Grantlee;

class TestHtmlOutput: public QObject
{
  Q_OBJECT
private Q_SLOTS:

// Test paragraph contents:
  void testSingleFormat();
  void testDoubleFormat();
  void testDoubleStartDifferentFinish();
  void testDoubleStartDifferentFinishReverseOrder();
  void testDifferentStartDoubleFinish();
  void testDifferentStartDoubleFinishReverseOrder();
  void testOverlap();
  void testAnchor();
  void testAnchorWithFormattedContent();
  void testAdjacentAnchors();
  void testNestedFormatting();
  void testSpan();
  void testDoubleSpan();
  void testSpanNesting();
  void testEdgeCaseLeft();
  void testEdgeCaseRight();
  void testImage();
  void testImageResized();
  void testEachFormatTagSingly();
  void testHorizontalRule();
  void testNewlines();
  void testNewlinesThroughQTextEdit();

};

void TestHtmlOutput::testSingleFormat()
{
  QTextDocument *doc = new QTextDocument();

  // One format
  doc->setHtml( QLatin1String( "This <b>text</b> is bold." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();
    QRegExp regex(QLatin1String("^<p>This <strong>text</strong> is bold.</p>\\n$"));

    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testDoubleFormat()
{
    QTextDocument *doc = new QTextDocument();

    // One format
    doc->setHtml( QLatin1String( "Some <b><i>formatted</i></b> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();
    QRegExp regex(QLatin1String("^<p>Some (<strong><em>|<em><strong>)formatted(</em></strong>|</strong></em>) text.</p>\\n$"));

    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testAnchor()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "A <a href=\"http://www.kde.org\">link</a> to KDE." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex(QLatin1String("^<p>A <a href=\"http://www.kde.org\">link</a> to KDE.</p>\\n$"));

    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testAnchorWithFormattedContent()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "A <a href=\"http://www.kde.org\"><b>formatted</b> link</a> to KDE." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex(QLatin1String("^<p>A <a href=\"http://www.kde.org\"><strong>formatted</strong> link</a> to KDE.</p>\\n$"));

    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testAdjacentAnchors()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Two <a href=\"http://www.kde.org\">links</a><a href=\"http://www.google.com\">next</a> to eachother." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex(QLatin1String("^<p>Two <a href=\"http://www.kde.org\">links</a><a href=\"http://www.google.com\">next</a> to eachother.</p>\\n$"));

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testNestedFormatting()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "This <b>text is <i>italic</i> and</b> bold." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex(QLatin1String("^<p>This <strong>text is <em>italic</em> and</strong> bold.</p>\\n$"));

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testSpan()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Some <span style=\"color:#ff0000;\">formatted</span> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Some <span style=\"color:#ff0000;\">formatted</span> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testDoubleSpan()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Some <span style=\"color:#ff0000;background-color:#00ff00;\">formatted</span> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Some <span style=\"(color:#ff0000|background-color:#00ff00);\"><span style=\"(color:#ff0000|background-color:#00ff00);\">formatted</span></span> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testSpanNesting()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Paragraph <span style=\"background-color:#00ff00;\">with some <span style=\"color:#ff0000;\">formatted</span> nested</span> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Paragraph <span style=\"background-color:#00ff00;\">with some <span style=\"color:#ff0000;\">formatted</span> nested</span> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testDoubleStartDifferentFinish()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Paragraph <i><b>with</b> some formatted</i> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Paragraph <em><strong>with</strong> some formatted</em> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testDoubleStartDifferentFinishReverseOrder()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Paragraph <b><i>with</i> some formatted</b> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Paragraph <strong><em>with</em> some formatted</strong> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testDifferentStartDoubleFinish()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Paragraph <i>with some <b>formatted<b></i> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Paragraph <em>with some <strong>formatted</strong></em> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testDifferentStartDoubleFinishReverseOrder()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Paragraph <b>with some <i>formatted</i></b> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Paragraph <strong>with some <em>formatted</em></strong> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testOverlap()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Paragraph <b>with <i>some</i></b><i> formatted</i> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Paragraph <strong>with <em>some</em></strong><em> formatted</em> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testEdgeCaseLeft()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Paragraph <b>with some formatted text.</b>" ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Paragraph <strong>with some formatted text.</strong></p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testEdgeCaseRight()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "<b>Paragraph with some formatted</b> text." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p><strong>Paragraph with some formatted</strong> text.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testImage()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "Paragraph with an inline <img src=\"http://kde.org/img/kde41.png\" /> image." ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Paragraph with an inline <img src=\"http://kde.org/img/kde41.png\" /> image.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testImageResized()
{
    QString result;
    QRegExp regex;
    TextHTMLBuilder *hb;
    MarkupDirector *md;
    QTextDocument *doc = new QTextDocument();

    // width
    doc->setHtml( QLatin1String( "Paragraph with an inline <img src=\"http://kde.org/img/kde41.png\" width=\"10\" /> image." ) );

    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Paragraph with an inline <img src=\"http://kde.org/img/kde41.png\" width=\"10\" /> image.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // height
    doc->setHtml( QLatin1String( "Paragraph with an inline <img src=\"http://kde.org/img/kde41.png\" height=\"10\" /> image." ) );

    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Paragraph with an inline <img src=\"http://kde.org/img/kde41.png\" height=\"10\" /> image.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // height and width
    doc->setHtml( QLatin1String( "Paragraph with an inline <img src=\"http://kde.org/img/kde41.png\" height=\"10\" width=\"10\" /> image." ) );

    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Paragraph with an inline <img src=\"http://kde.org/img/kde41.png\" width=\"10\" height=\"10\" /> image.</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testEachFormatTagSingly()
{
    QString result;
    QRegExp regex;
    TextHTMLBuilder *hb;
    MarkupDirector *md;
    QTextDocument *doc = new QTextDocument();


    // Test bold
    doc->setHtml( QLatin1String( "Some <b>formatted</b> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <strong>formatted</strong> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));


    // Test Italic
    doc->setHtml( QLatin1String( "Some <i>formatted</i> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <em>formatted</em> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // Test Underline
    doc->setHtml( QLatin1String( "Some <u>formatted</u> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <u>formatted</u> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // Test Strikeout
    doc->setHtml( QLatin1String( "Some <s>formatted</s> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <s>formatted</s> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // Test Superscript
    doc->setHtml( QLatin1String( "Some <sup>formatted</sup> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <sup>formatted</sup> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // Test Subscript
    doc->setHtml( QLatin1String( "Some <sub>formatted</sub> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <sub>formatted</sub> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // Test Foreground
    doc->setHtml( QLatin1String( "Some <span style=\"color:#ff0000;\">formatted</span> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <span style=\"color:#ff0000;\">formatted</span> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));


    // Test Background
    doc->setHtml( QLatin1String( "Some <span style=\"background-color:#ff0000;\">formatted</span> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <span style=\"background-color:#ff0000;\">formatted</span> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // Test Font Family
    doc->setHtml( QLatin1String( "Some <span style=\"font-family:courier;\">formatted</span> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <span style=\"font-family:courier;\">formatted</span> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

    // Test Font Size
    doc->setHtml( QLatin1String( "Some <span style=\"font-size:20pt;\">formatted</span> text." ) );
    hb = new TextHTMLBuilder();
    md = new MarkupDirector(hb);
    md->processDocument(doc);
    result = hb->getResult();

    regex = QRegExp( QLatin1String( "^<p>Some <span style=\"font-size:20pt;\">formatted</span> text.</p>\\n$" ) );
    QVERIFY(regex.exactMatch(result));

}

void TestHtmlOutput::testHorizontalRule()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "<p style=\"margin-top:0;margin-bottom:0;\">Foo</p><hr /><p style=\"margin-top:0;margin-bottom:0;\">Bar</p>" ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Foo</p>\\n<hr />\\n<p>Bar</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testNewlines()
{
    QTextDocument *doc = new QTextDocument();
    doc->setHtml( QLatin1String( "<p>Foo<br /><br />\n<p>Bar</p>" ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(doc);
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Foo</p>\\n<p>&nbsp;<p>&nbsp;</p>\\n<p>Bar</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

void TestHtmlOutput::testNewlinesThroughQTextEdit()
{

    QTextEdit *te = new QTextEdit();
    QTextCursor cursor = te->textCursor();
    cursor.movePosition(QTextCursor::Start);
    cursor.insertText( QLatin1String( "Foo" ) );
    cursor.insertText( QLatin1String( "\n" ) );
    cursor.insertText( QLatin1String( "\n" ) );
    cursor.insertText( QLatin1String( "\n" ) );
    cursor.insertText( QLatin1String( "Bar" ) );

    TextHTMLBuilder *hb = new TextHTMLBuilder();
    MarkupDirector *md = new MarkupDirector(hb);
    md->processDocument(te->document());
    QString result = hb->getResult();

    QRegExp regex = QRegExp( QLatin1String( "^<p>Foo</p>\\n<p>&nbsp;<p>&nbsp;<p>Bar</p>\\n$" ) );

    QVERIFY(regex.exactMatch(result));
}

QTEST_MAIN(TestHtmlOutput)
#include "htmlbuildertest.moc"
