/* Groovy CD Rom Player Software

   This program is based upon Thomas McWilliams' WorkBone.  Keep in mind 
   all CD playing routines were originally written by him, and modified 
   to hell by me. :)  Otherwise, please sit back, check your seat belt,
   and enjoy the ride.
   

   Copyright (c) 1994       Thomas McWilliams
   Copyright (c) 1997,1998  Mark Landis

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <sys/utsname.h>
#include <unistd.h>
#include <termios.h>
#include <mntent.h>
#include <getopt.h>
#include <ctype.h>
#include <errno.h>
#include <ncurses.h>


#include "gcd.h"

char *prog_name	= "GroovyCD";
char *prog_ver	= "0.51";

/* These are the cords of the track and time status stuff */
#define	TRACK_Y		1
#define TRACK_X		1
#define TIME_Y		1
#define TIME_X		32

#define DEFAULT_TRACK	0

#define TIME_COLOR	COLOR_PAIR(C_BLUE)
#define TRACK_COLOR	TIME_COLOR|A_BOLD

/* Quick macro to zero out the time status */
#define zero_timeboard()	attrset(COLOR_PAIR(C_BLUE));\
                        	print_number(TIME_Y, TIME_X, " 00:00", TIME_COLOR);

 
static inline void playtime();
static inline void fplaytime();
static inline void playtime1(int);
static void check_drive_mount();
int ask_to_stop();
static void reprint_track_list(int);
static void highlight_track(int);
static void clear_track_highlight(int);
static void update_bottom_status(int);

char *cur_trackname;		/* Take a guess */
int cur_index = 0;		/* Current index mark */
int cur_frame;			/* Current frame number */
struct play *playlist = NULL;
struct cdinfo thiscd, *cd = &thiscd;
int cur_track = 1;		/* Current track number, starting at 1 */
char *cur_artist;		/* Name of current CD's artist */
char cur_avoid;			/* Avoid flag */
char cur_contd;			/* Continued flag */
char *cur_cdname;		/* Album name */
int cur_nsections;		/* Number of sections currently defined */
int exit_on_eject = 0;
int time_type = 0;		/* Time display (forward, time left, ect) */
int repeat = CONTINUE;		/* Repeat track/cd variable */
int last_track = 0;		/* Last track played */
int list_index = 0;

int cur_balance = 10, info_modified;
int cur_pos_abs=0, cur_pos_rel=0, cur_tracklen=0, cur_cdlen=0, cur_cdmode=0,
  cur_ntracks=0, cur_lasttrack=0, cur_firsttrack=0, cur_listno=0;
int helpon = 0, printonly=0; 

/* Miroslav Vasko (vasko@ies.sk) - I want to use 7bit BIG numbers */  
int ascii = 0;


int main (int argc, char *argv[])
{
  int sss, sel_stat;
  int i, done=0, last_mode = -1;
  int scmd = 0, tmppos = 0;
  fd_set rset;
  struct timeval mydelay;

  
  thiscd.trk = NULL;
  thiscd.lists = NULL;
  mydelay.tv_sec = 0;
  mydelay.tv_usec = 500000;	/* initial delay 1/2 sec */
  
  /* Miroslav Vasko (vasko@ies.sk) - argument parsing */
  if (argc == 2)
  {
    if ((strcasecmp(argv[1], "--ascii") == 0) || (strcasecmp(argv[1], "-a") == 0))
    {
      ascii = 1;
    }
    
    if ((strcasecmp(argv[1], "--help") == 0) || (strcasecmp(argv[1], "-h") == 0))
    {
      printf("%s version %s\n", prog_name, prog_ver);
      printf("\nSwitches:\n  --help,     -h     this help\n  --ascii,    -a     ascii chars only (7bit)\n  --stdout,   -c     print cd information to stdout and exit\n  --version,  -v     print version\n\n");
      exit(1);
    }
    
    if ((strcasecmp(argv[1], "--version") == 0) || (strcasecmp(argv[1], "-v") == 0))
    {
      printf("%s version %s\n", prog_name, prog_ver);
      exit(1);
    }
    
    if ((strcasecmp(argv[1], "--stdout") == 0) || (strcasecmp(argv[1], "-c") == 0))
      printonly=1;
  } 

  check_drive_mount();
  ncurses_init();

  srandom(time(NULL)); 

  sss = cd_status ();
  signal (SIGINT, SIG_IGN);
  if (sss == 0 || sss == 4)
    goto done;


  i = cddb_connect(CDDB_HOST, CDDB_PORT);
  if (i)
  {
    if (i > 0)
    {
      if (cddb_login())
      {      
        cddb_query();
        cddb_disconnect();        
      } 
    }
  } else
    default_track_info(6);

  if (printonly)
  {
    exit_smoothly();
    printf("Artist : %s\n", thiscd.artist);
    printf("Album  : %s\n", thiscd.cdname);
    printf("CDDB ID: %08x\n\n", (unsigned int) thiscd.cddb_id);

    printf("Track | Time  | Name\n");
    printf("----------------------------------------------\n");
    
    for (i=0; i<thiscd.ntracks; i++)
       printf("  %02d  | %02d:%02d | %s\n", i+1, 
       thiscd.trk[i].length / 60,
       thiscd.trk[i].length - (thiscd.trk[i].length/60 * 60),
       thiscd.trk[i].songname);
       
    printf("\n\n");
    printf("Total playing time: %02d:%02d\n\n",
       thiscd.length / 60,
       thiscd.length - (thiscd.length/60 * 60));

    exit(0);
  }
  update_bottom_status(0);

  attrset(COLOR_PAIR(C_BLACK)|A_BOLD);
  for (i=0; i<13; i++)
     mvaddch(9+i, 37, ACS_VLINE);

  reprint_track_list(0);
  highlight_track(cur_track);    
  
  if (cur_cdmode != CDPLAY)
    cur_pos_abs = cur_pos_rel = 0;
  fplaytime();   

 
  do {


/*
   use select() to update status 
 */
      FD_ZERO (&rset);
      if (cur_track == last_track)
      {
        FD_SET (STDIN_FILENO, &rset);    
        sel_stat = select (1, &rset, NULL, NULL, &mydelay);
        if (sel_stat == 0)
        {
	 mydelay.tv_sec = 0;
	 mydelay.tv_usec = 200000;
        }
      }

      sss = cd_status ();
      if (sss == 0 || sss == 4 || cur_cdmode == CDEJECT)
      {
	    scmd = '0';
	    goto done;
      }
        
      if (last_mode != cur_cdmode)
      {
        if (cur_cdmode == CDSTOP && last_mode != -1)
        {       
          clear_track_highlight(cur_track-1);          
          last_track = cur_track = list_index+1;
          highlight_track(cur_track);
          
          update_trackboard();
          zero_timeboard();          
        }
        last_mode = cur_cdmode;
      } 

       
      /* if key was pressed, parse it and do function */
      if (FD_ISSET (STDIN_FILENO, &rset))
//    if ((scmd = getch()) != ERR)
      {
//        read(0,&scmd,1);
        scmd = getch();
        
        if (helpon)
        {
          hide_help();
          fplaytime();
          goto temp;
        }
        
        switch (toupper(scmd))
        {
	  case '1':
          case '<':
	    if (cur_cdmode == CDPLAY)
            {
              tmppos = cur_pos_rel - 10;
              play_cd (cur_track, tmppos > 0 ? tmppos : 0, cur_ntracks + 1);
              mydelay.tv_sec = 0;
	    }
	    break;
          case '3':
	  case '>':
	    if (cur_cdmode == CDPLAY)
            {
              tmppos = cur_pos_rel + 10;
              if (tmppos < thiscd.trk[cur_track - 1].length)
              {
                play_cd (cur_track, tmppos, cur_ntracks + 1);
                mydelay.tv_sec = 0;
              }
            }
            break;
          case '2':
          case 'E':
 	    stop_cd ();
	    eject_cd ();
	    break;
          case '4':
          case '-':
	    cur_track--;
	    if (cur_track < 1)
	      cur_track = cur_ntracks;
             if (cur_cdmode == CDPLAY)
               play_cd (cur_track, 0, cur_ntracks + 1);
             else 
               update_trackboard();
	    mydelay.tv_sec = 0;
	    break;
          case '6':
          case '+':
          case '=':
            cur_track++;
 	    if (cur_track > cur_ntracks)
              cur_track = 1;
            if (cur_cdmode == CDPLAY)
   	     play_cd (cur_track, 0, cur_ntracks + 1);
            else
              update_trackboard();
  
 	    mydelay.tv_sec = 0;
	    break;              
          case '5':
	    if (cur_cdmode == CDPLAY)
    	     play_cd (cur_track, 0, cur_ntracks + 1);
	    mydelay.tv_sec = 0;
            break;
          case '7':
          case 'S':            
            if (cur_cdmode == CDPLAY || cur_cdmode == CDPAUZ)           
	     stop_cd();          
	    break;
          case '8':
          case 'Z':
	    if (cur_cdmode == CDPLAY || cur_cdmode == CDPAUZ)
   	      pause_cd ();		
	    break;
          case '9':
          case 'P':
	    if (cur_cdmode == CDSTOP || cur_cdmode == CDNULL)
            {
              if (!cur_track)
                cur_track++;
              play_cd (cur_track, 0, cur_ntracks + 1);
	      mydelay.tv_sec = 0;
            }
	    break;
          case 'R':
            repeat++;
            if (repeat == 4)
              repeat = 0;
            update_bottom_status(1);
            break;
          case 'T':
            time_type++;
            if (time_type > 2)
              time_type = 0;
            break;              
          case 'Q':
            if (cur_cdmode == CDPLAY)
            {
              if (ask_to_stop())
                stop_cd();
              done=1;
            } else
              done = 1;
            break;
          case 'A':
	    /* Miroslav Vasko (vasko@ies.sk) - 7/8 bit toggle */
            ascii = (!ascii);
            if (cur_cdmode != CDPLAY)
              cur_pos_abs = cur_pos_rel = 0;
            fplaytime();
            break;
          case '?':
            show_help();
            break;
          default:
 	    break;
        }
      }

temp:
      if (last_track != cur_track)
      {
        if (last_track)
        {
          switch(repeat)
          {
            case CONTINUE:
              break;
          
            case REPEAT_TRACK:
              if (cur_cdmode == CDPLAY)
              {
                play_cd(last_track, 0, cur_ntracks + 1);
                cur_track = last_track;
              }
              break;
/*          case REPEAT_DISC:
              if (cur_cdmode == CDSTOP && last_mode == CDPLAY &&
                  last_track == cur_ntracks)
              {
                play_cd(last_track, 0, cur_ntracks + 1);
              } else
                last_track = cur_track;
              break; 
*/
	    case RANDOM_TRACK:
	      if (cur_cdmode == CDPLAY)
	      {
	        cur_track = ((int) random() % cur_ntracks) + 1;
	        mvprintw(0, 7, "%02d", cur_track);
	        refresh();
	        play_cd(cur_track, 0, cur_ntracks + 1);
	      }
	      break;
          }
        }            

        clear_track_highlight(last_track-1);
       
        if (cur_track == 1 && last_track == 0)
          last_track = cur_track;
        highlight_track(cur_track);      
       
        last_track = cur_track;
      }
        
       
      /* update display of which track is playing */
    switch (cur_cdmode)
	{
	case 0:		/* CDNULL */
	  cur_track = 1;
	  break;
	case 1:		/* CDPLAY */
	  playtime();
	  break;
	case 3:		/* CDPAUZ */
	  break;
	case 4:		/* CDSTOP */
	  break;
	case 5:		/* CDEJECT */
	  goto done;
	default:
          break;
      }
  } while (!done);
done:
  if (thiscd.trk != NULL)
    free (thiscd.trk);
  signal (SIGINT, SIG_DFL);

  exit_smoothly();  

  return (0);
}

/* This prevents the screen from become borked should we have an error */
void exit_smoothly()
{
   clear();
   refresh();
   endwin();
}

int ask_to_stop()
{
   WINDOW *w;
   int c;
   
   w = newwin(3, 20, 10, 29);
   wbkgd(w, COLOR_PAIR(C_BLUE_HL) | 32);   
   box(w,ACS_VLINE,ACS_HLINE);
   
   wattrset(w, COLOR_PAIR(C_BLUE_YELLOW)|A_BOLD);
   mvwprintw(w, 0, 7, " Quit ");
   
   wattrset(w, COLOR_PAIR(C_BLUE_HL)|A_BOLD);
   mvwprintw(w, 1, 2, "Stop playing CD?");
   while ((c = wgetch(w)) != 'y' && c != 'n')
     ;

   delwin(w);
   refresh();

   return (c == 'y');
}

static void check_drive_mount()
{
  FILE *fp;
  struct mntent *mnt;
 

  return;
  
  /* check if drive is mounted (from Mark Buckaway's cdplayer code) */
  if ((fp = setmntent (MOUNTED, "r")) == NULL)
  {
    fprintf (stderr, "Couldn't open %s: %s\n", MOUNTED, strerror (errno));
    exit (1);
  }
  while ((mnt = getmntent (fp)) != NULL)
  {
    if (strcmp (mnt->mnt_type, "iso9660") == 0)
    {
      fputs ("CDROM already mounted. Operation aborted.\n", stderr);
      endmntent (fp);
      exit (1);
    }
  }
  endmntent (fp);
}

/* Copy into a malloced string. */
void strmcpy (char **t, const char *s)
{
  if (*t != NULL)
    free (*t);

  *t = malloc (strlen (s) + 1);
  if (*t == NULL)
    {
      perror ("strmcpy");
      exit (1);
    }

  (void) strcpy (*t, s);
}

static void reprint_track_list(int start)
{
   int i;
   
   for (i=start; i<thiscd.ntracks && (i-start < 26); i++)
      clear_track_highlight(i);

   /* This will take care of clearing the rest of the display
      if there aren't enough tracks left to fill it */
      
   for (i=i; (i-start)<26; i++)
   {  
      if (i-start < 13)
        mvprintw(9+(i-start), 0, "%-37c", ' ');
      else
        mvprintw(9+(i-start-13), 39, "%-40c", ' ');
   }
   refresh();
   
   if (helpon)
   {
     hide_help();
     show_help();
   }
   
}

static void highlight_track(int track)
{
   track--;

   if (track - list_index == 26)
   {
     list_index = track;
     reprint_track_list(list_index);
   }
   
   if (track - list_index < 0 || track - list_index > 26)
   {
     list_index = 26 * (track / 26);

     reprint_track_list(list_index);
   }

   if (track - list_index < 13)
   {
     attrset(COLOR_PAIR(C_YELLOW)|A_BOLD);
     mvprintw(9+track-list_index, 1, "%-2d", track+1);    
     attrset(COLOR_PAIR(C_CYAN_HL)|A_BOLD);     
     mvprintw(9+track-list_index, 4, "%31.30s ", thiscd.trk[track].songname);     
   } else {
     attrset(COLOR_PAIR(C_CYAN_HL)|A_BOLD);
     mvprintw(9+(track-list_index-13), 39, " %-36.36s",thiscd.trk[track].songname);
     attrset(COLOR_PAIR(C_YELLOW)|A_BOLD);
     mvprintw(9+(track-list_index-13), 77, "%2d", track+1);
   }       
  
/*
   This comment is some code to print the song name and length
   underneath the big numbers.  It was suggested that CDs with
   multiple artists wouldn't fit on the trackboard in the 40 character
   or so limit, so printing the name here would be beneficial as well.
   Well...I'm not sure I like how it looks.  I might make it a toggle
   option later though.
*/
   
   attrset(COLOR_PAIR(C_CYAN));
   mvprintw(6, 4, "%-64.64s", thiscd.trk[track].songname);
   mvprintw(7, 4, "%02d:%02d",
      thiscd.trk[track].length / 60,
      thiscd.trk[track].length - (thiscd.trk[track].length/60 * 60));
      /* Could someone please tell me how that above equation works out?
         I could have sworn  X - (X/Y * Y) gives you 0.  :) 
      */
 
   refresh();
   
   if (helpon)
   {
     hide_help();
     show_help();
   }
}

static void clear_track_highlight(int track)
{    
   if (track < 0)
     return;
  
   if (track - list_index < 13)
   {
     attrset(COLOR_PAIR(C_WHITE));
     mvprintw(9+track-list_index, 1, "%-2d %31.30s ",track+1,thiscd.trk[track].songname);      
   } else {
     attrset(COLOR_PAIR(C_WHITE));
     mvprintw(9+(track-list_index-13),39," %-36.36s %2d",thiscd.trk[track].songname,track+1);
   }        
   refresh();
   
   if (helpon)
   {
     hide_help();
     show_help();
   }
}
      

static inline void playtime()
{
   playtime1(0);
}

static inline void fplaytime()
{
   playtime1(1);
}

static inline void playtime1(int force)
{
  static int mymin; //, emin;
  static int mysec; //, esec;
  static int old_time_type;
  char s[255];
  int tmp = 0;



  if ((cur_pos_rel > 0 && (tmp = cur_pos_rel % 60) == mysec) && (old_time_type == time_type) && !force)
    return;
    
  switch(time_type)
  {
    case 0:
      mysec = tmp;
      mymin = cur_pos_rel / 60;
//    esec = cur_pos_abs % 60;
//    emin = cur_pos_abs / 60;
      break;
    case 1:
      mysec = (cur_tracklen - cur_pos_rel) % 60;
      mymin = (cur_tracklen - cur_pos_rel) / 60;
      break;
    case 2:
      mysec = (cur_cdlen - cur_pos_abs) % 60;
      mymin = (cur_cdlen - cur_pos_abs) / 60;
      break;
  }
      
  old_time_type = time_type;    
   
//  mvprintw(0,0,"%d",times++);
//  refresh();

  update_trackboard();   
 
  sprintf(s,"%c%02d:%02d",(time_type > 0) ? '-':' ', mymin,mysec);
  attrset(COLOR_PAIR(C_BLUE));
  print_number(TIME_Y, TIME_X, s, TIME_COLOR);
   
  return;
}


void update_trackboard() 
{
   char s[4];
  
     
   sprintf(s,"%02d",cur_track);
   attrset(COLOR_PAIR(C_BLUE)|A_BOLD);
   print_number(TRACK_Y, TRACK_X, s, TRACK_COLOR);
}


static void update_bottom_status(int stats)
{
   char *playmodes[] = {
      "Continue    ",
      "Repeat Track",
      "Repeat Disc ",
      "Random Track",
      NULL,
   };

   
   attrset(COLOR_PAIR(C_BLUE_HL)|A_BOLD);
   
   if (!stats)
   {
     mvprintw(23, 1, "%-35.35s", cd->artist);  
     mvprintw(23, 36, "Tracks");
     mvprintw(23, 44, "Length");
   }
   mvprintw(23, 54, "Play Mode");
   
   attrset(COLOR_PAIR(C_BLUE_HL));
   if (!stats)
   {
     mvprintw(24, 1, "%-35.35s", cd->cdname);
     mvprintw(24, 36, "%02d", cd->ntracks);
     mvprintw(24, 44, "%02d:%02d", cd->length/60, cd->length - (cd->length/60 * 60));
   }
   mvprintw(24, 54, "%s", playmodes[repeat]);

   attrset(COLOR_PAIR(C_BLUE_HL)|A_BOLD);
   mvprintw(23, 68, "Need Help");
   attrset(COLOR_PAIR(C_BLUE_HL));
   mvprintw(24, 68, "Press");
   attrset(COLOR_PAIR(C_BLUE_YELLOW)|A_BOLD);
   mvprintw(24, 74, "?");
   
   refresh();
}
