#ifndef _ZSTREAM_H_
#define _ZSTREAM_H_

#include <clog.h>

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <zlib.h>

/* return codes */
#define ZS_OK 0
#define ZS_CALLBACK -1
#define ZS_ERROR -2
#define ZS_STREAMERR -3

/* compression levels */
#define ZS_NO_COMPRESSION      Z_NO_COMPRESSION
#define ZS_BEST_COMPRESSION    Z_BEST_COMPRESSION
#define ZS_FAST_COMPRESSION    Z_BEST_SPEED
#define ZS_DEFAULT_COMPRESSION Z_DEFAULT_COMPRESSION

/* I/O callback functions */
typedef int(*input_func)(int, char *, int, void *);
typedef int(*output_func)(int, const char *, int, void *);

/* DEFLATE */

/* zstream_deflate_init()
 * Returns initialized z_stream object for deflating; NULL on error.
 */
z_stream *zstream_deflate_init( int compr_level );

/* zwrite()
 * The callback function will be called up until all bytes have been written.
 * Let zbuf and zbuflen pertain to a buffer >= 1K; they also must be at
 * least a few bytes (say 48) larger than buf and buflen.
 * Returns:
 *   ZS_OK - when all bytes have been successfuly written
 *   ZS_ERROR - when other compression error occurs
 *   ZS_CALLBACK - when the callback function returned a negative value;
 *                 check output_ret on the value returned
 */
int zwrite( z_stream *z, char *zbuf, int zbuflen,
	int fd, const char *buf, int buflen, void *arg,
	output_func output, int *output_ret );

/* zstream_deflate_end()
 * Releases z_stream object.
 * Returns:
 *   0 - success
 *   -1 - error
 */
int zstream_deflate_end( z_stream *z );

/* INFLATE */

/* zstream_inflate_init()
 * Returns initialized z_stream object for inflating; NULL on error.
 */
z_stream *zstream_inflate_init();

/* zread()
 * The callback function will be called to read in data.  Decompressed data
 * will be returned to buf, there may still be outstanding not-yet-uncompressed
 * data in the zbuf.  A consecutive calls to this function will retrieve the
 * remaining data.
 * For end of file, i suggest that the callback function returns 0, which will
 * cause zread() to return ZS_CALLBACK.  Then, input_ret should equal 0.
 * However, there still may be data in the zbuf buffer.  In order to find out,
 * call zstream_read_available(), and keep getting data using zread() until
 * zstream_read_available() returns false.  No calls to the callback function
 * will happen thereon.
 * Let zbuf and zbuflen pertain to a buffer >= 1K.
 * Returns:
 *   on success, number of bytes read into buf
 *   ZS_STREAMERR - when compressed data was bad or corrupted
 *   ZS_ERROR - when other type of compression error occured
 *   ZS_CALLBACK - when the callback function returned non-positive value.
 *                 check input_ret on the value returned
 */
int zread( z_stream *z, char *zbuf, int zbuflen,
	int fd, char *buf, int buflen, void *arg,
	input_func input, int *input_ret );

/* zstream_read_available()
 * Returns true if there is more data in the temporary compress buffer.
 * This means that the next call to zread will not cause a call the input
 * callback function.
 */
int zstream_read_available( z_stream *z );

/* zstream_inflate_end()
 * Releases z_stream object.
 * It is possible to have more data in the zlib buffer, call
 * zstream_read_available() to check, and get all the data.  You may need to
 * do that more than once.
 * Returns:
 *   0 - success
 *   -1 - error
 */
int zstream_inflate_end( z_stream *z );

#ifdef __cplusplus
}
#endif

#endif  /* _ZSTREAM_H_ */
