#include "input_thread.h"
#include <cstdio>
#include <cstring>
#ifdef GRUB_UNIX
#include <sys/types.h>
#include <sys/socket.h>
#endif

// millisecond sleep
#ifdef GRUB_UNIX
#	define MSLEEP(t) \
	do { \
		if ( (int)t < 1000 ) \
			usleep(1000 * (int)t); \
		else \
			sleep( (int)t / 1000 ); \
	} while(0)
#else
#	define MSLEEP(t) ::Sleep((int)t)
#endif

#define MSEC 1000

#define SET_DEAD(c) \
	do { \
		c->lock(); \
		c->is_threading = false; \
		c->unlock(); \
	} while (0)

#ifndef GRUB_UNIX
void(*comdirtyhack_threadinit_callback)(SocketComm *) = NULL;
void(*comdirtyhack_threaduninit_callback)(SocketComm *) = NULL;
#endif

static SOCKET srv_establish(unsigned short portnum);
static int select_input( SOCKET sock, unsigned int msec );

void *input_thread_main( void *ptr )
{
	SocketComm *scomm = (SocketComm *)ptr;
	SOCKET srv_sock = INVALID_SOCKET, cli_sock = INVALID_SOCKET;
	char *data_block = (char *)0;

#ifndef GRUB_UNIX
	if ( comdirtyhack_threadinit_callback )
		comdirtyhack_threadinit_callback(scomm);
#endif

	scomm->lock();
	scomm->is_threading = true;
	do {
		scomm->unlock();
		MSLEEP( 250 );
		scomm->lock();
	} while ( ! scomm->is_threading_picked );
	scomm->unlock();

	if ( scomm->isServer() )
	{
		srv_sock = scomm->getServerSock();
	}
	else if ( ! scomm->isServer() )
		cli_sock = grub_info.sock;

	while (1) {

		try {
			int fret;

			// fret = 0, 1, -1
			if ( scomm->isServer() && ! scomm->isSockInit() ) {
				if ( ( fret = select_input( srv_sock, MSEC ) ) == -1)
					throw 1;
			}
			else
				if ( ( fret = select_input( cli_sock, MSEC ) ) == -1)
					throw 1;

			scomm->lock();
			if ( scomm->thread_die ) {

				scomm->unlock();
				throw 0;
			}
			scomm->unlock();

			// select() returned timeout -- simply loop again
			if ( fret == 0 )
				continue;

			if ( scomm->isServer() && ! scomm->isSockInit() ) {

				/* we are a server,
				 * no connection established, but a
				 * new client tries to connect */
				cli_sock = accept( srv_sock, NULL, NULL );
				if ( cli_sock == INVALID_SOCKET ) {

					scomm->setError( "accept: Waiting on server "
						"socket error" );
					throw 1;
				}

				scomm->setServerSock( cli_sock );
			}
			else {

				/* something has arrived to be recv */

				unsigned int maxlen = scomm->getMaxDocLen();
				unsigned int len;

				fret = gmsg_recv_available();
				if ( fret < 1 ) {
					char buf[512];

					sprintf( buf, "recv: gmsg_recv_available: %s",
						gmsg_strerror() );
					scomm->setError( buf );
					throw 1;
				}

				if ( (unsigned int)fret > maxlen ) {

					scomm->setError( "Message to be received too long" );
					throw 1;
				}

				len = (unsigned int)fret;

				if ( data_block ) {

					delete [] data_block;
					data_block = 0;
				}

				data_block = new char[len + 1];

				fret = gmsg_recv( data_block );
				if ( fret != GF_SUCCESS ) {
					char buf[512];

					sprintf( buf, "recv: gmsg_recv: %s",
						gmsg_strerror() );
					scomm->setError( buf );
					throw 1;
				}

				SocketReadFunc func = scomm->getReadCallback();
				if ( func ) {

					if ( func( scomm->getReadCallbackArg(),
							data_block, len ) )
					{
						scomm->setError( "User read callback "
							"returned non-zero" );
						throw 2;
					}
				}
			}
		}
		catch (int exp) {

			switch (exp) {
			case 0:
				// exitting due to being ordered to die
				break;
			case 1:
				// error of any kind occured
				break;
			case 2:
				// user function callback returned non-zero
				break;
			}
			goto input_thread_main_end;
		}
	}
input_thread_main_end:

	if ( data_block )
		delete [] data_block;

	SET_DEAD(scomm);

#ifndef GRUB_UNIX
	if ( comdirtyhack_threaduninit_callback )
		comdirtyhack_threaduninit_callback(scomm);
#endif

	return (void *)0;
}

// ret = a socket, INVALID_SOCKET
static SOCKET srv_establish(unsigned short portnum)
{
	char myname[256];
	SOCKET s;
	struct sockaddr_in sa;
	struct hostent *hp;

	memset(&sa, 0, sizeof(struct sockaddr_in));
	gethostname(myname, sizeof(myname));
	hp = gethostbyname(myname);
	if (hp == NULL)
		return(INVALID_SOCKET);
	sa.sin_family = hp->h_addrtype;
	sa.sin_port = htons(portnum);
	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s == INVALID_SOCKET)
		return INVALID_SOCKET;

	/* bind the socket to the internet address */
	if (bind(s, (struct sockaddr *)&sa, sizeof(struct sockaddr_in)) ==
		SOCKET_ERROR)
	{
		closesocket(s);
		return(INVALID_SOCKET);
	}
	listen(s, 3);

	return s;
}

// ret = 1, 0, -1
static int select_input( SOCKET sock, unsigned int msec )
{
	int ret = 0, fret;

	// fret = GF_SUCCESS, GF_NOTREADY, GF_INTRNERR
	fret = gmsg_socket_ready( sock, false, msec );
	switch ( fret ) {
	case GF_SUCCESS:
		ret = 1;
		break;
	case GF_NOTREADY:
		ret = 0;
		break;
	case GF_INTRNERR:
		ret = -1;
		break;
	}  // end switch

	return ret;
}
