/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "typedefs.h"
#include "file-utils.h"
#include "catalog.h"
#include "catalog-list.h"
#include "pixbuf-utils.h"

#include "icons/up_dir.xpm"
#include "icons/catalog_dir.xpm"
#include "icons/catalog_search.xpm"
#include "icons/catalog.xpm"


CatalogList *
catalog_list_new ()
{
	CatalogList * cat_list;
	GtkWidget *clist;
	GtkWidget *scrolled;
	gint height;

	cat_list = g_new (CatalogList, 1);

	/* get base catalogs directory. */
	cat_list->path =  get_catalog_full_path (NULL);
	cat_list->dir_list = NULL;
	cat_list->file_list = NULL;
	cat_list->search_rows = NULL;

	/* Create the widgets. */
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_ALWAYS);
	clist = gtk_clist_new (1);
	gtk_clist_column_titles_passive (GTK_CLIST (clist)); 
	gtk_widget_set_usize (clist, CLIST_WIDTH_DEFAULT, CLIST_HEIGHT_DEFAULT);
	height = (GTK_WIDGET (clist)->style->font->ascent +
		  GTK_WIDGET (clist)->style->font->descent + CLIST_ROW_PAD); 
	gtk_clist_set_row_height (GTK_CLIST (clist), height);	
	gtk_container_add (GTK_CONTAINER (scrolled), clist);
	cat_list->clist = clist;
	
	cat_list->root_widget = scrolled;
	gtk_widget_show_all (cat_list->root_widget);

	return cat_list;
}


void
catalog_list_free (CatalogList *cat_list)
{
	g_return_if_fail (cat_list != NULL);

	g_free (cat_list->path);
	path_list_free (cat_list->dir_list);
	path_list_free (cat_list->file_list);
	g_list_free (cat_list->search_rows);
	g_free (cat_list);
}


gchar *
catalog_list_path_from_row (CatalogList *cat_list,
			    gint row)
{
	gchar *name;

	name = gtk_clist_get_row_data (GTK_CLIST (cat_list->clist), row);
	return g_strdup (name);
}


gboolean
catalog_list_is_search (CatalogList *cat_list,
			gint row)
{
	GList *scan;

	g_return_val_if_fail (cat_list != NULL, FALSE);

	for (scan = cat_list->search_rows; scan; scan = scan->next)
		if (GPOINTER_TO_INT (scan->data) == row)
			return TRUE;

	return FALSE;
}


gint
catalog_list_row_from_path (CatalogList *cat_list,
			    const gchar *path)
{
	GList *scan;
	gint   i;

	g_return_val_if_fail (cat_list != NULL, -1);

	i = 0;
	scan = GTK_CLIST (cat_list->clist)->row_list;
	for (; scan; scan = scan->next) {
		gchar *row_path = GTK_CLIST_ROW (scan)->data;

		if (strcmp (row_path, path) == 0)
			return i;

		i++;
	}

	return -1;
}


gboolean
catalog_list_refresh (CatalogList *cat_list)
{
	GList *scan, *name_scan;
	GList *files, *dirs;
	gint width;
	gint tmp_width;
	gint row;
	GtkWidget *clist;
	GdkPixmap *dir_pixmap, *catalog_pixmap, *search_pixmap, *up_dir_pixmap;
	GdkBitmap *dir_bitmap, *catalog_bitmap, *search_bitmap, *up_dir_bitmap;
	gchar *base;

	/* Set the file and dir lists. */

	if (!path_list_new (cat_list->path, &files, &dirs)) {
		g_print ("ERROR : reading catalogs directory\n");
		return FALSE;
	}

	if (cat_list->dir_list != NULL)
		path_list_free (cat_list->dir_list);
	if (cat_list->file_list != NULL)
		path_list_free (cat_list->file_list);
	if (cat_list->search_rows != NULL) {
		g_list_free (cat_list->search_rows);
		cat_list->search_rows = NULL;
	}

	dirs = g_list_sort (dirs, (GCompareFunc) strcasecmp);
	files = g_list_sort (files, (GCompareFunc) strcasecmp);

	cat_list->dir_list = dirs;
	cat_list->file_list = files;

	/* get the list of dirs names (without path). */
	cat_list->dir_name = NULL;

	for (scan = dirs; scan; scan = scan->next) {
		gchar *name_only = g_strdup (file_name_from_path (scan->data));
		cat_list->dir_name = g_list_prepend (cat_list->dir_name, 
						     name_only);
	}

	cat_list->dir_name = g_list_reverse (cat_list->dir_name);

	/* Add a ".." entry if the current dir is not the base catalog dir. */
	base = get_catalog_full_path (NULL);
	if (strcmp (base, cat_list->path) != 0) {
		gchar *prev_dir = remove_level_from_path (cat_list->path);

		cat_list->dir_list = g_list_prepend (cat_list->dir_list, 
						     prev_dir);
		cat_list->dir_name = g_list_prepend (cat_list->dir_name, 
						     g_strdup (".."));
	}
	g_free (base);

	/* get the list of files names (without path). */
	cat_list->file_name = NULL;
	
	for (scan = files; scan; scan = scan->next) {
		gchar *name_only = remove_extension_from_path (file_name_from_path (scan->data));
		cat_list->file_name = g_list_prepend (cat_list->file_name, 
						      name_only);
	}

	cat_list->file_name = g_list_reverse (cat_list->file_name);

	/* Add items to the clist. */

	clist = cat_list->clist;
	gtk_clist_freeze (GTK_CLIST (clist));
	gtk_clist_clear (GTK_CLIST (clist));

	pixmap_from_xpm ((const char **) catalog_dir_xpm, &dir_pixmap, 
			 &dir_bitmap);
	pixmap_from_xpm ((const char **) catalog_xpm, &catalog_pixmap, 
			 &catalog_bitmap);
	pixmap_from_xpm ((const char **) up_dir_xpm, &up_dir_pixmap, 
			 &up_dir_bitmap);
	pixmap_from_xpm ((const char **) catalog_search_xpm, &search_pixmap, 
			 &search_bitmap);

	width = 0;
	scan = cat_list->dir_list;
	name_scan = cat_list->dir_name;
	while (scan) {
		gchar *buf[2];

		buf[0] = name_scan->data;
		buf[1] = NULL;

		row = gtk_clist_append (GTK_CLIST (clist), buf);

		if (strcmp (name_scan->data, "..") != 0)
			gtk_clist_set_pixtext (GTK_CLIST (clist), row, 0, 
					       name_scan->data, 
					       10, 
					       dir_pixmap, dir_bitmap);
		else
			gtk_clist_set_pixtext (GTK_CLIST (clist), row, 0, 
					       name_scan->data, 
					       10, 
					       up_dir_pixmap, up_dir_bitmap);

		gtk_clist_set_row_data (GTK_CLIST (clist), row, scan->data);
		tmp_width = gdk_string_width (clist->style->font, buf[0]);
		width = MAX (width, tmp_width);

		scan = scan->next;
		name_scan = name_scan->next;
	}

	scan = cat_list->file_list;
	name_scan = cat_list->file_name;
	while (scan) {
		gchar *buf[2];

		buf[0] = name_scan->data;
		buf[1] = NULL;

		row = gtk_clist_append (GTK_CLIST (clist), buf);

		if (file_is_search_result (scan->data)) {
			cat_list->search_rows = g_list_prepend (cat_list->search_rows, GINT_TO_POINTER (row));
			gtk_clist_set_pixtext (GTK_CLIST (clist), row, 0, 
					       name_scan->data, 
					       10, 
					       search_pixmap, search_bitmap);
		} else 
			gtk_clist_set_pixtext (GTK_CLIST (clist), row, 0, 
					       name_scan->data, 
					       10, 
					       catalog_pixmap, catalog_bitmap);

		gtk_clist_set_row_data (GTK_CLIST (clist), row, scan->data);
		tmp_width = gdk_string_width (clist->style->font, buf[0]);
		width = MAX (width, tmp_width);

		scan = scan->next;
		name_scan = name_scan->next;
	}

	gdk_pixmap_unref (catalog_pixmap);
	gdk_bitmap_unref (catalog_bitmap);
	gdk_pixmap_unref (dir_pixmap);
	gdk_bitmap_unref (dir_bitmap);
	gdk_pixmap_unref (up_dir_pixmap);
	gdk_bitmap_unref (up_dir_bitmap);

	gtk_clist_set_column_width (GTK_CLIST (clist), 0, width);
	gtk_clist_thaw (GTK_CLIST (clist));

	return TRUE;
}


void
catalog_list_change_to (CatalogList *cat_list,
			gchar *path)
{
	g_return_if_fail (cat_list != NULL);

	if (path != cat_list->path) {
		g_free (cat_list->path);
		cat_list->path = g_strdup (path);
	}

	catalog_list_refresh (cat_list);
}
