/* gtk-menu - a simple docking menu for launvhing programs
 * Copyright (c) 1998-2001 Decklin Foster <decklin@red-bean.com>
 * Free software! Please see README for details and license.
 *
 * Modified 2001 by Carl Worth <carl@theworths.org>:
 *   Added toplevel docking window
 *   Added reaping of child processes
 *   Added restart on SIGHUP
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <X11/Xatom.h>

#include "menu-misc.h"
#include "strpool.h"

static void display_menu_cb(GtkWidget *widget, gpointer callback_data);
static void hide_menu(GtkWidget *widget, gpointer callback_data);
static void reload_menu(int signum);
static void display_menu(void);
static void destroy_menu(void);
static void create_menu(void);
static void add_menu_item(void *menu, char *label, char *cmd, char *xpm_filename); 
static void *add_submenu(void *menu, char *label);
static void fork_exec_cb(GtkWidget *widget, char *data);
static void fork_exec(char *cmd);
static void reap_children(int signum);
static void install_signal_handlers(void);

GtkWidget *rootmenu = NULL;
strpool_t *strpool = NULL;

GtkWidget *window;

static char *display_target = NULL;
int main(int argc, char **argv)
{
    GtkWidget *button;
    GdkAtom window_type;
    GdkAtom window_type_dock;

    gtk_init(&argc, &argv);

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_container_set_border_width(GTK_CONTAINER (window), 0);
    button = gtk_button_new_with_label("Menu");
    gtk_container_add(GTK_CONTAINER(window), button);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(display_menu_cb), NULL);
    gtk_widget_show(button);

    window_type = gdk_atom_intern("_NET_WM_WINDOW_TYPE", FALSE);
    window_type_dock = gdk_atom_intern("_NET_WM_WINDOW_TYPE_DOCK", FALSE);
    gtk_widget_realize(window);
    gdk_property_change(window->window, window_type, XA_ATOM, 32, GDK_PROP_MODE_REPLACE, (guchar *) &window_type_dock, 1);

    gtk_widget_show(window);

    install_signal_handlers();

    gtk_main();
    return 0;
}

static void install_signal_handlers(void)
{
    struct sigaction action;

    action.sa_flags = 0;
    sigemptyset(&action.sa_mask);
    action.sa_handler = reap_children;
    sigaction(SIGCHLD, &action, NULL);

    action.sa_flags = 0;
    sigemptyset(&action.sa_mask);
    action.sa_handler = reload_menu;
    sigaction(SIGHUP, &action, NULL);
}

static void reap_children(int signum)
{
    pid_t pid;

    do {
	pid = waitpid(-1, NULL, WNOHANG);
    } while (pid > 0);
}

static void reload_menu(int signum)
{
  destroy_menu();
  create_menu();
}

static void create_menu(void)
{
    if (strpool == NULL) {
	strpool = strpool_new();
    }
    if (rootmenu == NULL) {
	rootmenu = gtk_menu_new();
	build_menu("/etc/gtk-menu", rootmenu, add_menu_item, add_submenu);
	gtk_signal_connect_object(GTK_OBJECT(rootmenu), "deactivate",
				  GTK_SIGNAL_FUNC(hide_menu), NULL);
    }
}

static void destroy_menu(void)
{
    if (rootmenu) {
	gtk_widget_destroy(rootmenu);
	free(rootmenu);
	rootmenu = NULL;
    }
    if (strpool) {
	strpool_free(strpool);
	strpool = NULL;
    }
}

static void display_menu_cb(GtkWidget *widget, gpointer callback_data)
{
    create_menu();
    display_menu();
}

static void display_menu(void)
{
    create_menu();
    gtk_menu_popup(GTK_MENU(rootmenu), NULL, NULL, NULL, NULL, 0, 0);
}

static void hide_menu(GtkWidget *widget, gpointer callback_data)
{
    gtk_menu_popdown(GTK_MENU(rootmenu));
}

static void add_menu_item(void *menu, char *label, char *cmd, char *xpm_filename) 
{
    GtkWidget *item, *pixmapwid, *box, *glabel;
    GdkPixmap *pixmap;
    GdkBitmap *mask;
    GtkStyle *style;

    box = gtk_hbox_new(FALSE, 1);
    glabel = gtk_label_new(label);

    if (xpm_filename != NULL)
      {
	 style = gtk_widget_get_style( window );
	 pixmap = gdk_pixmap_create_from_xpm( window->window, &mask,
					      &style->bg[GTK_STATE_NORMAL],
					      xpm_filename );

	pixmapwid =  gtk_pixmap_new( pixmap, mask );
	gtk_box_pack_start (GTK_BOX (box), pixmapwid, FALSE, TRUE, 1);
	gtk_widget_show(pixmapwid);
      }
    gtk_box_pack_start (GTK_BOX (box), glabel, FALSE, TRUE, 1);
    gtk_widget_show(glabel);
    gtk_widget_show(box);
    item = gtk_menu_item_new();
    gtk_container_add(GTK_CONTAINER(item), box);
    gtk_menu_append(GTK_MENU(menu), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
        GTK_SIGNAL_FUNC(fork_exec_cb), strpool_add(strpool, cmd));
    gtk_widget_show(item);
}

static void *add_submenu(void *menu, char *label)
{
  /* no icons here yet ..... */
    GtkWidget *item, *newmenu;

    newmenu = gtk_menu_new();
    item = gtk_menu_item_new_with_label(label);
    gtk_menu_append(GTK_MENU(menu), item);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), newmenu);
    gtk_widget_show(item);

    return newmenu;
}

static void fork_exec_cb(GtkWidget *widget, char *data)
{
    fork_exec(data);
}

static void fork_exec(char *cmd)
{
    pid_t pid;
    if (strspn(cmd,"setenv ")==7)
      {
	if (display_target) free(display_target);
	display_target=strdup(cmd+7);
	return;
      }

    pid = fork();

    switch (pid) {
        case 0:
	    if (display_target) putenv(display_target);
            execlp("/bin/sh", "sh", "-c", cmd, NULL);
            fprintf(stderr, "exec failed, cleaning up child\n");
            exit(1);
        case -1:
            fprintf(stderr, "can't fork\n"); break;
    }
    if (display_target)
      {
	free(display_target);
	display_target = NULL;
      }
}


