/*  GTKtalog.
 *  Copyright (C) 2000 Yves METTIER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#define _USE_BSD
#include <config.h>
#include <gnome.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <signal.h>
#include <errno.h>
#if defined OS_FREEBSD
#  include <sys/param.h>
#  include <sys/mount.h>
#else
#	if (defined(OS_PPC_DARWIN))
#		include <sys/time.h>
#	endif
#  include <sys/resource.h>
#endif


#include "config_common.h"

#include "addisk.h"
#include "folder.h"
#include "compare.h"

struct vfsmime_and_string
{
  MIME_VFS_EXTENSIONS *me;
  gchar *s;
};

static GNode *
vfs_node_exists (GNode * gn, GPtrArray * splitted_name, gint depth,
		 gint enddepth)
{
  GNode *newgn = gn;
  FILE_DATA *fd;
  gchar *name;

  if (depth >= enddepth)
    {
      return (gn);
    }
  else
    {
      name = g_ptr_array_index (splitted_name, depth);
      newgn = g_node_first_child (gn);
      while (newgn)
	{
	  fd = newgn->data;
	  if (strcmp (name, fd->name->str) == 0)
	    break;
	  newgn = g_node_next_sibling (newgn);
	}
      if (newgn)
	newgn = vfs_node_exists (newgn, splitted_name, depth + 1, enddepth);
    }
  return (newgn);
}

static GNode *
create_vfs_node (GNode * vfs_node, GPtrArray * splitted_name, gint depth,
		 guint32 size, time_t date, guint type, gint testvfs)
{
  GNode *parent;
  GNode *me;
  guint32 taille = size;
  FILE_DATA *fd;
  gchar *mime;

  if (depth <= 0)
    return (vfs_node);
  /* Test is the node already exist */
  if ((me = vfs_node_exists (vfs_node, splitted_name, 0, depth)))
    {
      /* the node exists. Update the datas */
      fd = me->data;
      fd->type = IS_DIR;
      fd->taille = 0;
      fd->date = date;
      return (me);
    }
  /* Test if the parent node exists */
  if ((parent = vfs_node_exists (vfs_node, splitted_name, 0, depth - 1)) ==
      NULL)
    {
      /* The parent node does not exists. Create it recursively */
      parent =
	create_vfs_node (vfs_node, splitted_name, depth - 1, 0, date, IS_DIR,
			 testvfs);
    }
  /* Now, the parent exists. Create the new node */
  if (type != IS_FILE)
    taille = 0;

  mime = NULL;

  fd =
    folder_make_data (g_ptr_array_index (splitted_name, depth - 1), NULL,
		      type, mime, taille, date, 0, 0);
  if (testvfs == TEST_VFS_YES)
    fd->a_parent_node_is_vfs = IS_VFS;
  else
    fd->a_parent_node_is_vfs = IS_VFS_EXTENDED;

  me = g_node_append_data (parent, fd);
  fd->node = me;
  fd->ctree_node = NULL;
  if (!G_NODE_IS_ROOT (parent))
    {
      fd = parent->data;
      fd->type = IS_DIR;
    }
  return (me);
}

static GPtrArray *
split_name (gchar * file)
{
  gchar *p1, *p2;
  GPtrArray *result = g_ptr_array_new ();

  p1 = file;
  while (p1[0] == '/')
    p1++;

  while (p1[0])
    {
      p2 = p1;
      while (p2[0] && (p2[0] != '/'))
	p2++;
      g_ptr_array_add (result, g_strndup (p1, p2 - p1));
      p1 = p2;
      while (p1[0] == '/')
	p1++;
    }
  return (result);
}

static gchar *
read_one_line ()
{
  static gchar buffer[READLINE_MAX_LENGTH];
  gint b = 1;
  gint p = 0;

  if (test_abort_scan ())
    return (NULL);
  while ((b = read (0, &buffer[p], sizeof (char))))
    {
      if (test_abort_scan ())
	return (NULL);
      if (b >= 0)
	{
	  g_assert (p < READLINE_MAX_LENGTH);
	  if (buffer[p] == '\n')
	    {
	      buffer[p] = '\0';
	      return (g_strdup (buffer));
	    }
	  p++;
	} else {
	if(errno == EAGAIN) {
	  g_assert (p < READLINE_MAX_LENGTH);
	  if (buffer[p] == '\n')
	    {
	      buffer[p] = '\0';
	      return (g_strdup (buffer));	    }
	} else {return(NULL);}
	}
    }
  return (NULL);
}

static gboolean
insert_node_from_vfsline (GNode * vfs_node, gchar * line,
			  MIME_VFS_EXTENSIONS * ve, gint testvfs)
{
  GArray *fields = g_array_new (FALSE, FALSE, sizeof (gint));
  GPtrArray *result;
  gchar *mydate, *myformat;
  gchar yyyy[5] = { 0, 0, 0, 0, 0 };
  gchar *y;
  gint nb_y = 0;
  gchar mm[3] = { 0, 0, 0 };
  gchar *m;
  gint nb_m = 0;
  gchar dd[3] = { 0, 0, 0 };
  gchar *d;
  gint nb_d = 0;
  gchar ssss[11] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
  gchar *s;
  gint nb_s = 0;
  gchar *myname;
  GPtrArray *splitted_name;
  gchar *myseparator;
  gint i;
  gint last_column;

  gchar *strsize;
  guint32 size;
  time_t date;
  struct tm mytime_t;

  g_array_append_val (fields, ve->size);
  g_array_append_val (fields, ve->date);
  g_array_append_val (fields, ve->name);
  switch (ve->last_column)
    {
    case VFS_LAST_COLUMN_SIZE:
      last_column = ve->size;
      break;
    case VFS_LAST_COLUMN_DATE:
      last_column = ve->date;
      break;
    case VFS_LAST_COLUMN_NAME:
      last_column = ve->name;
      break;
    default:
      last_column = -1;
    }
  myseparator = convert_string_with_slashes (ve->separator->str);
  result = extract_field (line, myseparator, fields, last_column);
  g_free (myseparator);
  g_array_free (fields, TRUE);
  if (result->len >= 2)
    {

/* Get the size */
/****************/
      strsize = g_ptr_array_index (result, 0);
      if (testvfs == TEST_VFS_EXTENDED)
	size = atoi (strsize);
      else
	size = 0;

/* Get the date */
/****************/
      mydate = g_ptr_array_index (result, 1);
      if (testvfs == TEST_VFS_EXTENDED)
	{
	  myformat = ve->date_format->str;
	  y = yyyy;
	  m = mm;
	  d = dd;
	  s = ssss;
/* FIXME: Retreiving the date will not crash. But if the user gave wrong arguments,
 * there should be unpredictable results (maybe null date).
 * Maybe adding a warning should help ?
 */
	  while (myformat[0] && mydate[0])
	    {
	      switch (myformat[0])
		{
		case 'Y':
		  if (nb_y < 4)
		    {
		      y[0] = mydate[0];
		      y++;
		      nb_y++;
		    }
		  break;
		case 'M':
		  if (nb_m < 2)
		    {
		      m[0] = mydate[0];
		      m++;
		      nb_m++;
		    }
		  break;
		case 'D':
		  if (nb_d < 2)
		    {
		      d[0] = mydate[0];
		      d++;
		      nb_d++;
		    }
		  break;
		case 'S':
		  if (nb_s < 10)
		    {
		      s[0] = mydate[0];
		      s++;
		      nb_s++;
		    }
		  break;
		}
	      mydate++;
	      myformat++;
	    }
	  if (nb_s)
	    date = atoi (ssss);
	  else
	    {
	      mytime_t.tm_sec = 0;
	      mytime_t.tm_min = 0;
	      mytime_t.tm_hour = 0;
	      mytime_t.tm_mday = atoi (dd);
	      mytime_t.tm_mon = atoi (mm) - 1;
	      if (nb_y == 2)
		mytime_t.tm_year = atoi (yyyy);
	      else
		mytime_t.tm_year = atoi (yyyy) - 1900;
	      mytime_t.tm_wday = 0;
	      mytime_t.tm_yday = 0;
	      mytime_t.tm_isdst = 0;
	      date = mktime (&mytime_t);
	    }
	}
      else
	date = 0;

/* Work on the name */
/********************/
      myname = g_ptr_array_index (result, 2);
      if (myname)
	{
	  if (!strncmp (myname, "./", 2))
	    splitted_name = split_name (myname + 2);
	  else
	    splitted_name = split_name (myname);
	}
      else
	{
	  splitted_name = split_name (_("(empty file?)"));
	}
      create_vfs_node (vfs_node, splitted_name, splitted_name->len, size,
		       date, IS_FILE, testvfs);

      for (i = 0; i < splitted_name->len; i++)
	g_free (g_ptr_array_index (splitted_name, i));
      for (i = 0; i < result->len; i++)
	g_free (g_ptr_array_index (result, i));
      g_ptr_array_free (splitted_name, TRUE);
      g_ptr_array_free (result, TRUE);
      return (FALSE);
    }
  else
    {
      for (i = 0; i < result->len; i++)
	g_free (g_ptr_array_index (result, i));
      g_ptr_array_free (result, TRUE);
      return (TRUE);
    }
}

static gint
find_extractor (gpointer key, gpointer value, gpointer data)
{
  struct vfsmime_and_string *_find_vmime = data;
  MIME_VFS_EXTENSIONS *me = value;

  if (me->state == FALSE)
    return (FALSE);
  if (strcmp (me->mime->str, _find_vmime->s))
    return (FALSE);
  _find_vmime->me = me;
  return (TRUE);
}


MIME_VFS_EXTENSIONS *
find_vfsextension (const gchar * mime)
{
  static struct vfsmime_and_string _find_vmime;
  _find_vmime.me = NULL;
  _find_vmime.s = g_strdup (mime);
  g_tree_traverse (my_config->mime_vfs_extensions, find_extractor,
		   G_PRE_ORDER, &_find_vmime);
  g_free (_find_vmime.s);
  return (_find_vmime.me);
}

gboolean
_vfs_merge_node_data (GNode * gn, gpointer data)
{
  FOLDER *racine = data;
  FILE_DATA *fd = gn->data;

  if (G_NODE_IS_ROOT (gn) == FALSE)
    {
      g_ptr_array_add (racine->datas, fd);
      gn->data = fd;
    }
  return (FALSE);
}

static void
vfs_merge_trees (GNode * gn, GNode * vfs_node, FOLDER * racine, gint testvfs)
{
  GNode *tmpgn;
  gint i;
  FILE_DATA *fd;

  g_node_traverse (vfs_node, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
		   _vfs_merge_node_data, racine);
  while ((tmpgn = g_node_first_child (vfs_node)))
    {
      g_node_unlink (tmpgn);
      g_node_append (gn, tmpgn);
    }
  g_node_destroy (vfs_node);
  if (testvfs == TEST_VFS_YES)
    change_type (gn, IS_VFS);
  else if (testvfs == TEST_VFS_EXTENDED)
    change_type (gn, IS_VFS_EXTENDED);

  for (i = 0; i < racine->datas->len; i++)
    {
      fd = g_ptr_array_index (racine->datas, i);
      fd->id = i;
    }
}

gboolean
vfs_read (GNode * gn, gchar * vfs_file_name, gint testvfs, FOLDER * racine)
{
  int p[2];
  gchar *buffer = NULL;
  gchar **argvfs;
  int pid;
  MIME_VFS_EXTENSIONS *ve;
  gint i = 0;
  GString *tmpgstr;
  GNode *vfs_node;
  gchar *cmdline;
  gint nb_args;
  gchar **args_ptr;
  const gchar *mime;
  gboolean vfs_scan_nok = FALSE;

  mime = gnome_mime_type_or_default (vfs_file_name, NULL);
  if (!mime)
    return (FALSE);

  if ((ve = find_vfsextension (mime)))
    {

      if (pipe (p) == 0)
	{
	  if ((pid = fork ()) == 0)
	    {
	      /* FIXME: Stderr should be closed too or redirected
	       * in order not to have error messages on the console
	       */

	      cmdline = g_strconcat (ve->prog->str, " ", ve->args->str, NULL);
	      argvfs = g_strsplit (cmdline, " ", 0);
	      nb_args = 0;
	      args_ptr = argvfs;
	      while (args_ptr[0] != NULL)
		{
		  nb_args++;
		  args_ptr++;
		}
	      argvfs =
		(gchar **) g_realloc (argvfs,
				      sizeof (gchar *) * (nb_args + 2));
	      tmpgstr = get_real_path_from_node (gn);
	      argvfs[nb_args] = tmpgstr->str;
	      argvfs[nb_args + 1] = NULL;

	      close (1);
	      dup (p[1]);
	      execvp (argvfs[0], argvfs);
	    }
	  else
	    {
	      int stdin_flags;
	      close (0);
	      dup (p[0]);
	      close (p[1]);
	      stdin_flags = fcntl (0, F_GETFL);
	      fcntl (0, F_SETFL, stdin_flags | O_NONBLOCK);

	      i = 0;
	      vfs_node = g_node_new (NULL);
	      while ((buffer = read_one_line ()))
		{
		  if (insert_node_from_vfsline
		      (vfs_node, buffer, ve, testvfs))
		    vfs_scan_nok = TRUE;
		  g_free (buffer);
		}
	      if (test_abort_scan ())
		{
		  kill (pid, SIGKILL);
		  vfs_scan_nok = TRUE;
		}
	      while (wait4 (pid, NULL, WNOHANG, 0) != pid);

	      close (p[0]);
	      /* FIXME: STDIN should be reopened to the real STDIN. */

	      /* Merge the new vfs into the tree */
	      if (!test_abort_scan ())
		{
		  vfs_merge_trees (gn, vfs_node, racine, testvfs);
		}
	    }
	}
    }
  return (vfs_scan_nok);
}
