/* this file defines a generic way to copy streams  
 * from and into any filedescriptor.  
 * If desired, a process id can simultaneously be watched so's an error  
 * condition will be detected in time */

#include "config.h"

#include <gtk/gtk.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <signal.h>
#include <stdlib.h>
#include <errno.h>
#ifdef HAVE_PTHREADS    
# include <pthread.h>  
#endif                                                                          

#include "datacopydlg.h"
#include "streamcopy.h"
#include "preferences.h"

/* uncomment for debugging */
// #define DEBUG

#define min(x,y) ((x>y)?y:x)

/* this is the plain writing routine. It just pushes the data into the
 * receiver, high-level things are being taken care of by the timeout
 * handler because we shouldn't do high-level things in a separate thread 
 *
 * streamcopy_writetrack returns 1 if the current track is complete
 * or if a fatal error occurred.
 * the state->trackdone remains untouched by this function.
 * it is being updated by the function calling streamcopy_writetrack
 * in a way ensuring that the copy_write thread processing streamcopy_writetrack
 * will no longer access state and exit immediately.
 * 
 * this function is being entered with the state mutex locked.
 * The mutex is freed by the streamcopy_writetrack function several times
 * during it's execution */
int streamcopy_writetrack(streamcopy_state *state)
{
   char readbuf[BUFSIZE];
   int  readcount;
   int  trackdone=0;
   
   /* if in padding mode, don't read any data from the source stream.
    * just calculate the number of 0-bytes to be written in the current
    * loop. readbuf got filled with 0-bytes when the padding flag
    * was set */
   if (!state->padding)
     {
	/* unlock mutex during I/O */
#ifdef HAVE_PTHREADS
	pthread_mutex_unlock(&state->semaphore);
#endif
	readcount=read(state->trackpipe,readbuf,BUFSIZE);
#ifdef HAVE_PTHREADS
	pthread_mutex_lock(&state->semaphore);
#endif
     }
   else
     {
	/* just set the write size in padding mode and make sure it
	 * doesn't get negative */
	readcount=min(BUFSIZE,state->tracksize-state->trackread);
	if (readcount<0)
	  readcount=0;
     };
   
#ifdef DEBUG
   if (readcount==-1)
     perror("streamcopy_writetrack: error reading source data");
#endif
   /* only read tracksize bytes if requested */
   if ((state->options&STREAMCOPY_OPT_CUT)&&
       ((state->trackread+readcount)>state->tracksize)&&
       (readcount>0))
     readcount=state->tracksize-state->trackread;   
   if (state->padding)
     /* fill read buffer with padding bytes 
      * we have to do this everytime this function is being invoked because
      * we don't use a permanent buffer */     
     memset(readbuf,0,BUFSIZE);   
   if (readcount>0)
     {
	/* write only if everything's fine so far */
	if (!state->fatalerror)
	  {
	     int writeresult;
	     /* unlock mutex during I/O */
#ifdef HAVE_PTHREADS
	     pthread_mutex_unlock(&state->semaphore);
#endif
	     writeresult=write(state->inpipe,readbuf,readcount);
#ifdef HAVE_PTHREADS
	     pthread_mutex_lock(&state->semaphore);
#endif
	     /* if we couldn't write. */
	     if (writeresult==-1)
	       {
#ifdef DEBUG
		  printf ("Error No. was %i\n",errno);
		  perror ("streamcopy_writetrack: error writing to destination");
#endif
		  state->fatalerror=1;
		  state->writingstate=STREAMCOPY_STATE_BACKENDEXT;
	       };
	  };
	if (state->fatalerror)
	  trackdone=1;
	else
	  {
	     /* only increase readcount if we could write our data alright */
	     state->trackread+=readcount;
	     state->rsldu+=readcount;
	  };
     }
   else
     /* we have have reached the end of our input stream.
      * We now have to check wether or not to pad and
      * leave the stage in case we don't have to do any padding */
     {
	/* pad track to tracksize bytes if stream didn't deliver
	 * enough data (fill with 0-bytes).
	 * The actual padding isn't done in this function,
	 * we just modify the writing process to do the padding for us */
	if ((state->options&STREAMCOPY_OPT_PAD)&&	  
	    (state->trackread<state->tracksize)&&(!state->fatalerror)&&(!state->padding))
	  {
	     /* print a warning if more than a certain percentage has to be padded */
	     if (((state->tracksize-state->trackread)/state->tracksize)>0)
	       {		  
		  sprintf(readbuf,"padding track by %i%%. Padding by >=1%% might indicate an error!\n",
			  (state->tracksize-state->trackread)/state->tracksize);
		  /* FIXME: we can't print to the recording terminal
		   * in a secondary thread - implement some kind of 
		   * communication pipe between the secondary and the
		   * main thread to accomplish that */
//		  rterm_printcln(Blue,_(readbuf));
	       };	
#ifdef DEBUG
	     printf ("record_writetrack: padding track by %i bytes\n",
		     state->tracksize-state->trackread);
#endif
	     /* just set the padding to 1, zero filling etc. will be done
	      * just before writing the data */
	     state->padding=1;
	  }
	/* we either don't need any padding or padding's already been done */
	else
	  {	 
	     /* has another exit condition already been set ? */
	     if (!trackdone)
	       {		  
		  state->writingstate=STREAMCOPY_STATE_ENDOFTRACK;
		  trackdone=1;
	       };
	  };	     
     };
   return trackdone;
};


/* this is for the multithreading version of Gnometoaster */
#ifdef HAVE_PTHREADS
void *streamcopy_copythread(void *d)  
{
   streamcopy_state *state=(streamcopy_state*)d;
   
# ifdef DEBUG
   printf("streamcopy_copythread: started copying thread\n");
# endif

   /* we don't have anything better to do in this thread.
    * Note that the state mutex will not be blocked all the time.
    * The state structure is freely accessible while Gnometoaster is
    * waiting for data to be received or sent */
   pthread_mutex_lock(&state->semaphore);
   while (!streamcopy_writetrack(state));
   state->trackdone=1;
      
   pthread_mutex_unlock(&state->semaphore);
   
#ifdef DEBUG
   printf ("streamcopy_copythread: ending copying thread\n");
#endif
   
   return NULL;
};
#else
/* this is the fallback function for the non-threading version of 
 * Gnometoaster */
void streamcopy_copyiowatch(gpointer data,
			    gint source,
			    GdkInputCondition condition)
{
   streamcopy_state *state=(streamcopy_state*)data;
   
   /* if writetrack becomes 0, the current track is done or there 
    * has been a fatal error or something. Anyway, we have to check back
    * with the timeout handler */
   state->trackdone=
     streamcopy_writetrack(state);
   
   if (state->trackdone)
     /* deinitialize io handler */
     gtk_input_remove(state->receiverinpipetag);
};
#endif

/* this is highlevel control over the copying thread/iowatch */
gint streamcopy_timeouthandler (gpointer data)
{
   streamcopy_state *state=(streamcopy_state*)data;

#ifdef HAVE_PTHREADS
   /* lock data structure while using it */
   
   /* we won't block the main thread in case we have to wait for this
    * mutex, we just return to gtk_main() and take care of our handler
    * on the next timeout */
   if (!pthread_mutex_trylock(&state->semaphore))
#endif
     {
	/* do we have to check the health of the destination pid 
	 * regularly ? */
	if ((state->receiverpid!=-1)&&
	    (waitpid(state->receiverpid,&state->receiverstatus,WNOHANG)!=0))
	  {
#ifdef DEBUG
	     printf("streamcopy_timeouthandler: backend exited causing a fatal error\n");
#endif
	     /* the loss of the client is fatal */
	     state->fatalerror=1;	     
	     state->writingstate=STREAMCOPY_STATE_BACKENDEXT;
	     /* there's no way we could get out of this 
	      * we don't have to do anything else here.
	      * The writing thread will realize soon enough that there
	      * was something wrong and exit peacefully.
	      * There's no way it can do any more damage because it's
	      * destination fd has already been voided 
	      * 
	      * The state->trackdone flag will be set by the write thread
	      * as soon as it has exited so we can't do anything else here 
	      * at the moment
	      * so the following if (state->trackdone) statement will
	      * not yet be true this time (the mutex is locked -> the
	      * copy thread is waiting to get back control */
	  }
	;
	/* if current track is done.
	 * Do error processing */
	if (state->trackdone)
	  {	     
#ifdef HAVE_PTHREADS
	     /* Make our copy thread exit */
	     pthread_join(*state->copythread,NULL);
	     free(state->copythread);
#endif	     
#ifdef DEBUG
	     if (state->fatalerror)
	       printf ("streamcopy_timeouthandler: aborting due to a fatal error\n");
#endif
	     if (state->options&STREAMCOPY_OPT_CLOSESOURCE)
	       {
#ifdef DEBUG
		  printf("streamcopy_timeouthandler: closing source fd\n");
#endif
		  close(state->trackpipe);
	       };		  
	     
	     if (state->options&STREAMCOPY_OPT_CLOSEDEST)
	       {
#ifdef DEBUG
		  printf("streamcopy_timeouthandler: closing destination fd\n");
#endif		  
		  close(state->inpipe);
	       };
	     if ((state->options&STREAMCOPY_OPT_SIGTERMONUSEREXT)&&
		 (state->receiverpid!=-1)&&
		 (state->writingstate==STREAMCOPY_STATE_USERTERM))
	       {
#ifdef DEBUG
		  printf("streamcopy_timeouthandler: sending SIGTERM to destination process\n");
#endif
		  kill(state->receiverpid,SIGTERM);
	       };
	     if ((state->options&STREAMCOPY_OPT_WAITEXIT)&&
		 (state->receiverpid!=-1)&&
		 (state->writingstate!=STREAMCOPY_STATE_BACKENDEXT))
	       {		  
#ifdef DEBUG
		  printf ("streamcopy_timeouthandler: waiting for destination process to exit\n");
#endif
		  waitpid(state->receiverpid,
			  &state->receiverstatus,0);
#ifdef DEBUG
		  printf ("streamcopy_timeouthandler: destination process exited\n");
#endif
	       };
	     /* remove this handler */
	     gtk_timeout_remove(state->timeouttag);
	     
	     /* call our callback handler with exclusive access to our structure. */
	     ((streamcopy_callback)state->callback)(state,state->data);
	     
#ifdef HAVE_PTHREADS
	     pthread_mutex_unlock(&state->semaphore);
	     /* destroy our mutex */
	     pthread_mutex_destroy(&state->semaphore);
#endif
	     
	     /* free state structure, it's no longer needed and
	      * state->trackdone indicates that it won't be
	      * accessed any more by our secondary thread as well. */
	     free(state);
	  }
	else
	  {	     
	     if (state->rsldu)
	       {	  
		  if (state->dlg!=NULL)
		    datacopydlg_newdatablock(state->dlg,state->rsldu);
		  state->rsldu=0;
	       };
#ifdef HAVE_PTHREADS
	     /* free state structure for further access by the copy thread */
	     pthread_mutex_unlock(&state->semaphore);
#endif
	  };
	/* it is not safe to use the state struct from here on */
     };
   /* end of check wether or not we could lock the state structure */
   return 1;
};
   
streamcopy_state *streamcopy_create(int inpipe,
				    int trackpipe,
				    int receiverpid,
				    int tracksize,
				    datacopydlg_dlginfo *dlg,
				    int options,
				    streamcopy_callback callback,
				    gpointer data)
{
   streamcopy_state *state=(streamcopy_state*)malloc(sizeof(streamcopy_state));
   
   /* parameters */
   state->inpipe=inpipe;
   state->trackpipe=trackpipe;
   state->receiverpid=receiverpid;
   state->tracksize=tracksize;
   state->dlg=dlg;
   state->callback=(gpointer)callback;
   state->data=data;
   state->options=options;
#ifdef DEBUG
   if (
       (!(state->options&STREAMCOPY_OPT_CLOSEDEST))&&
       (state->options&STREAMCOPY_OPT_WAITEXIT)
       )
     printf("WARNING! using STREAMCOPY_OPT_WAITEXIT without STREAMCOPY_OPT_CLOSEDEST might cause a hang (%i) !\n",state->options);
#endif
   
   /* initialization */
   state->receiverstatus=0;
   state->trackread=0;
   state->fatalerror=0;
   state->padding=0;
   state->trackdone=0;
   state->writingstate=0;
   state->rsldu=0;

   /* start up copying system */
   state->timeouttag=
     gtk_timeout_add(100,streamcopy_timeouthandler,(gpointer)state);
#ifdef HAVE_PTHREADS	     
   pthread_mutex_init(&state->semaphore,
		      NULL);
   /* FIXME: free the track info some day */
   state->copythread=(pthread_t*)malloc(sizeof(pthread_t));
   if (pthread_create(state->copythread,NULL,streamcopy_copythread,
		      (void*)state))
     {
	/* the timeout handler is already running at this stage 
	 * so it will take care of this situation */
	state->trackdone=1;
	free(state->copythread);
	state->writingstate=STREAMCOPY_STATE_PTHREADERR;
     };
#else
	/* this is for the non-threaded version */
	state->receiverinpipetag=gdk_input_add (state->inpipe,
						GDK_INPUT_WRITE,
						streamcopy_copyiowatch,
						state);
#endif
   return state;
};

void streamcopy_cancel(streamcopy_state *tag)
{
#ifdef HAVE_PTHREADS
   /* look the streamcopy tag */
   pthread_mutex_lock(&tag->semaphore);
#endif
   /* initiate self-destruct sequence */
   tag->fatalerror=1;  
   tag->writingstate=STREAMCOPY_STATE_USERTERM;
#ifdef HAVE_PTHREADS
   /* that's it */
   pthread_mutex_unlock(&tag->semaphore);
#endif
};

