//$Id: dialog-win-helpers.cc,v 1.2 2003/06/30 14:07:55 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "dialog-win-helpers.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <gnome--/stock.h>
#include "property-ops-resource.h"

using namespace Guikachu;
using namespace Guikachu::DialogWin_Helpers;

ButtonList::ButtonList (Resources::Dialog *res_):
    StringList (res_->buttons, new ButtonChangeOpFactory (res_)),
    res (res_)
{
    Gtk::Button *button_default = Gtk::wrap (GTK_BUTTON (
	gnome_pixmap_button (
	    GTK_WIDGET (manage(new Gnome::Stock::Pixmap (GNOME_STOCK_BUTTON_YES))->gtkobj ()),
	    _("Default"))));
    
    button_default->clicked.connect (SigC::slot (this, &ButtonList::button_default_cb));
    button_default->set_sensitive (false);
    add_button (*manage (button_default));

    update ();

    list_widget.unselect_item (0);
    list_widget.select_item (0);
}

void ButtonList::button_default_cb ()
{
    unsigned int index = get_selected_row ();
    saved_pos = index;

    static ResourceOps::PropChangeOpFactory<unsigned int> op_factory (
	_("Change default button of %s"), res, res->default_button, true);
    
    op_factory.push_change (index);
}

void ButtonList::update ()
{
    update_block = true;
    
    clear_list ();
    
    const value_t &curr_val = property;
    unsigned int index = 0;
    unsigned int default_button = res->default_button;
    for (value_t::const_iterator i = curr_val.begin ();
	 i != curr_val.end (); i++, index++)
    {
	if (index == default_button)
	    add_to_list (index, "*" + *i + "*");
	else
	    add_to_list (index, *i);
    }
    
    update_block = false;

    if (saved_pos >= 0)
    {
	if (curr_val.size ())
	{
	    list_widget.unselect_item (saved_pos);
	    list_widget.select_item (saved_pos);
	} else {
	    selection_cb ();
	}
    }
}



ButtonChangeOpFactory::ButtonChangeOpFactory (Resources::Dialog *res_) :
    StringListOpFactory (_("Add \"%s\" button to %s"),
			 _("Remove \"%s\" button from %s"),
			 _("Change \"%s\" button in %s"),
			 _("Move \"%s\" button in %s"),
			 res_,
			 res_->buttons),
    res (res_)
{
}

void ButtonChangeOpFactory::push_add (index_t new_index, const item_t &new_button)
{
    value_t val = res->buttons;
    index_t old_default = res->default_button;
    index_t new_default = old_default;
    
    if (old_default >= new_index)
	new_default += 1;

    char *label_str = g_strdup_printf (add_template.c_str (),
				       new_button.c_str (),
				       res->id ().c_str ());
    UndoOp *op = new AddOp (label_str, res, new_index, new_button, old_default, new_default);
    g_free (label_str);

    if (new_index >= val.size ())
	val.push_back (new_button);
    else
	val.insert (val.begin () + new_index, new_button);
    
    res->buttons = val;
    res->default_button = new_default;
    undo_manager.push (op);    
}

void ButtonChangeOpFactory::push_remove (index_t index)
{
    value_t val = res->buttons;
    index_t old_default = res->default_button;
    index_t new_default = old_default;
    
    if (old_default != 0 && old_default >= index)
	new_default -= 1;

    char *label_str = g_strdup_printf (remove_template.c_str (),
				       val[index].c_str (),
				       res->id ().c_str ());
    UndoOp *op = new RemoveOp (label_str, res, index, old_default, new_default);
    g_free (label_str);

    val.erase (val.begin () + index);
    
    res->buttons = val;
    res->default_button = new_default;
    undo_manager.push (op);
}

void ButtonChangeOpFactory::push_move (index_t old_index, index_t new_index)
{
    value_t val = res->buttons;
    index_t old_default = res->default_button;
    index_t new_default = old_default;
    
    if (old_index == old_default)
	new_default = new_index;
    else
	if (new_index == old_default)
	    new_default = old_index;
    
    char *label_str = g_strdup_printf (move_template.c_str (),
				       val[old_index].c_str (),
				       res->id ().c_str ());
    UndoOp *op = new MoveOp (label_str, res, old_index, new_index, old_default, new_default);
    g_free (label_str);
    
    value_t::iterator old_i = val.begin () + old_index;
    value_t::iterator new_i = val.begin () + new_index;
    iter_swap (old_i, new_i);
    
    res->buttons = val;
    res->default_button = new_default;
    undo_manager.push (op);
}



ButtonChangeOpFactory::ButtonOp::ButtonOp (const std::string &op_label_,
					   Resources::Dialog *res,
					   index_t            old_default_,
					   index_t            new_default_) :
    op_label (op_label_),
    manager (res->get_manager ()),
    resource_id (res->id),
    old_default (old_default_),
    new_default (new_default_)
{
}

Resources::Dialog * ButtonChangeOpFactory::ButtonOp::get_dialog () const
{
    Resource *res = manager->get_resource (resource_id);
    Resources::Dialog *dialog = static_cast<Resources::Dialog*> (res);
    g_assert (dialog);

    return dialog;
}


ButtonChangeOpFactory::AddOp::AddOp (const std::string &op_label,
				     Resources::Dialog *res,
				     index_t            index_,
				     const item_t      &button_,
				     index_t            old_default,
				     index_t            new_default) :
    ButtonOp (op_label, res, old_default, new_default),
    index (index_),
    button (button_)
{
}

void ButtonChangeOpFactory::AddOp::undo ()
{
    Resources::Dialog *res = get_dialog ();
    value_t buttons = res->buttons;

    buttons.erase (buttons.begin () + index);

    res->buttons = buttons;
    res->default_button = old_default;
}

void ButtonChangeOpFactory::AddOp::redo ()
{
    Resources::Dialog *res = get_dialog ();
    value_t buttons = res->buttons;

    if (index >= buttons.size ())
	buttons.push_back (button);
    else
	buttons.insert (buttons.begin () + index, button);

    res->buttons = buttons;
    res->default_button = new_default;
}


ButtonChangeOpFactory::RemoveOp::RemoveOp (const std::string &op_label,
					   Resources::Dialog *res,
					   index_t            index_,
					   index_t            old_default,
					   index_t            new_default) :
    ButtonOp (op_label, res, old_default, new_default),
    index (index_),
    button (res->buttons()[index])
{
}

void ButtonChangeOpFactory::RemoveOp::undo ()
{
    Resources::Dialog *res = get_dialog ();
    value_t buttons = res->buttons;

    buttons.insert (buttons.begin () + index, button);

    res->buttons = buttons;
    res->default_button = old_default;
}

void ButtonChangeOpFactory::RemoveOp::redo ()
{
    Resources::Dialog *res = get_dialog ();
    value_t buttons = res->buttons;

    buttons.erase (buttons.begin () + index);

    res->buttons = buttons;
    res->default_button = new_default;
}


ButtonChangeOpFactory::MoveOp::MoveOp (const std::string &op_label,
				       Resources::Dialog *res,
				       index_t            old_index,
				       index_t            new_index,
				       index_t            old_default,
				       index_t            new_default) :
    ButtonOp (op_label, res, old_default, new_default)
{
    index_history.push_back (old_index);
    index_history.push_back (new_index);
}

ButtonChangeOpFactory::MoveOp::MoveOp (const std::string  &op_label,
				       Resources::Dialog  *res,
				       const index_list_t &index_history_head,
				       const index_list_t &index_history_tail,
				       index_t             old_default,
				       index_t             new_default) :
    ButtonOp (op_label, res, old_default, new_default),
    index_history (index_history_head)
{
    index_list_t::const_iterator tail_begin = index_history_tail.begin ();
    index_list_t::const_iterator tail_end = index_history_tail.end ();

    g_assert (index_history_tail.front () == index_history_head.back ());
    
    index_history.insert (index_history.end (), ++tail_begin, tail_end);
}



void ButtonChangeOpFactory::MoveOp::undo ()
{
    // Undo/redo may seem patently stupid and over-complicated, but
    // it's the only way to make cascading work

    Resources::Dialog *res = get_dialog ();
    value_t buttons = res->buttons;

    index_list_t::const_reverse_iterator rbegin = index_history.rbegin ();
    index_list_t::const_reverse_iterator rend = index_history.rend ();
    index_list_t::const_reverse_iterator curr, next;
    
    for (curr = rbegin, next = ++rbegin; next != rend; ++curr, ++next)
	std::iter_swap (buttons.begin () + *curr, buttons.begin () + *next);

    res->buttons = buttons;
    res->default_button = old_default;
}

void ButtonChangeOpFactory::MoveOp::redo ()
{
    Resources::Dialog *res = get_dialog ();
    value_t buttons = res->buttons;

    index_list_t::const_iterator begin = index_history.begin ();
    index_list_t::const_iterator end = index_history.end ();
    index_list_t::const_iterator curr, next;
    
    for (curr = begin, next = ++begin; next != end; ++curr, ++next)
	std::iter_swap (buttons.begin () + *curr, buttons.begin () + *next);

    res->buttons = buttons;
    res->default_button = old_default;
}


UndoOp * ButtonChangeOpFactory::MoveOp::combine (UndoOp *other_op) const
{
    ButtonChangeOpFactory::MoveOp *op = dynamic_cast<ButtonChangeOpFactory::MoveOp*> (other_op);
    if (!op)
	return 0;
    
    if (op->resource_id != resource_id)
	return 0;

    if (op->index_history.front () != index_history.back ())
	return 0;
    
    Resources::Dialog *dialog = get_dialog ();
    
    UndoOp *new_op = new MoveOp (get_label (), dialog, index_history, op->index_history,
				 old_default, op->new_default);
    
    return new_op;
}
