/*
 * Program to control ICOM radios
 * 
 * Subroutine library
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "icom.h"

/*
 * Local function prototypes
 */
static void doublefreq(double, int *, int); /* double to frequency */
static double freqdouble(int *);	/* frequency to double */

/*
 * Radio command prototypes
 */
static int s_rband[] = {V_RBAND,FI};	/* read band edge */
static int s_rfreq[] = {V_RFREQ,FI};	/* read frequency */
static int s_rmode[] = {V_RMODE,FI};	/* read mode */
static int s_roffset[] = {V_ROFFS,FI};	/* read tx offset */

/*
 * Initialize
 */
void
init()
{
	initpkt();		/* initialize (device dependent) */
}

/*
 * select_radio(rad, ident) - select/initalize radio structure
 */
struct icom *
select_radio(			/* returns structure pointer or NULL */
	int ident		/* radio identifier */
	)
{
	struct icom *radio;	
	double dtemp1, dtemp2;
	int buf[BMAX], temp1, temp2, i;
	char s1[NAMMAX];

	/*
	 * Find prototype. Bad tables if not found.
	 */
	for (i = 0; name[i].name[0] != '\0'; i++)
		if (name[i].ident == ident)
			break;
	if (name[i].ident == 0)
		return(NULL);

	/*
	 * If already initialized, return structure pointer. Otherwise,
	 * allocate and initialized structure and read band edges.
	 */
	if (name[i].radio != NULL)
		return(name[i].radio);
	radio = (struct icom *)malloc(sizeof(struct icom));
	memset((char *)radio, 0, sizeof(struct icom));
	strcpy(radio->name, name[i].name);
	radio->ident = name[i].ident;
	radio->maxch = name[i].maxch;
	radio->flags = name[i].flags;
	radio->mchan = 1;
	radio->mode = 0xff;
	radio->modetab = name[i].modetab;
	retry = 1;
	temp1 = sndpkt(radio->ident, s_rband, buf);
	if (temp1 < 12 || buf[0] != s_rband[0]) {
		retry = RETRY;
		free(radio);
		return(NULL);
	}
	retry = RETRY;
	radio->lband = freqdouble(&buf[1]) / 1e6;
	radio->uband = freqdouble(&buf[7]) / 1e6;
	if (radio->lband > radio->uband) {
		dtemp1 = radio->lband;
		radio->lband = radio->uband;
		radio->uband = dtemp1;
	}
	radio->lstep = radio->lband;
	radio->ustep = radio->uband;

	/*
	 * If V_VFOM (memory -> vfo) command works, we have to use it
	 * every time the channel is changed. If not, no harm done. We
	 * have to do this first so the subsequent readchan() works.
	 */
	if (setcmd(radio, V_VFOM, FI) == 0)
		radio->flags |= F_VFO;

	/*
	 * If the V_ROFFS command works, the radio can do duplex. We
	 * need to do this before the first readchan() so the duplex bit
	 * gets noticed.
	 */
	if (setcmd(radio, V_ROFFS, FI) == 0)
		radio->flags |= F_OFFSET;

	/*
	 * Determine the frequency resolution. This depends on the radio
	 * and TS button. We first read the frequency and mode of
	 * channel 1, then we write it with a frequency near the lower
	 * band edge. The value is chosen with '1' in each significant
	 * digit ending in the units digit. Then, we read the frequency
	 * back and determine the resolution corresponding to the first
	 * significant digit which contains '1'.
	 */
	if (readchan(radio, 1))
		return(NULL);
	dtemp1 = radio->vfo;
	temp1 = radio->mode;
	if (loadfreq(radio, radio->lband + 11111e-6))
		return(NULL);
	setcmd(radio, V_WRITE, FI);
	readchan(radio, 1);
	sprintf(s1, "%8.6lf", radio->vfo - radio->lband);
	radio->minstep = 6;
	if (s1[7] == '1')
		radio->minstep = 0;
	else if (s1[6] == '1')
		radio->minstep = 3;
	else if (s1[5] == '1')
		radio->minstep = 6;
	else if (s1[4] != '1')
		printf("*** invalid tuning step\n");
	radio->rate = radio->minstep;
	radio->step = logtab[radio->rate];

	/*
	 * Determine whether the frequency needs to be reset after a
	 * mode change. We first write the channel with mode USB, then
	 * read the frequency back and remember it. Then we write the
	 * channel with mode LSB, read back the frequency and see if it
	 * changed. If so, we have to write the frequency every time the
	 * mode is changed. What a drag.
	 */
	if (*modetoa(M_USB, radio->modetab) != '\0' || *modetoa(M_LSB,
	    radio->modetab) != '\0') {
		if (loadmode(radio, M_USB)) {
			return(NULL);
		} else {
			setcmd(radio, V_WRITE, FI);
			readchan(radio, 1);
			dtemp2 = radio->vfo;
			if (loadmode(radio, M_LSB) == 0) {
				setcmd(radio, V_WRITE, FI);
				readchan(radio, 1);
				if (radio->vfo != dtemp2)
					radio->flags |= F_RELD;
				}
		}
	}

	/*
	 * Restore contents of channel 1.
	 */
	loadfreq(radio, dtemp1);
	loadmode(radio, temp1);
	setcmd(radio, V_WRITE, FI);
	name[i].radio = radio;
	return(radio);
}

/*
 * setchan(radio, chan) - set memory channel number
 */
int
setchan(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int chan		/* memory subcommand */
	)
{
	int buf[BMAX], *ptr, temp;
	char chr[5];

	ptr = buf;
	*ptr++ = V_SMEM;
	sprintf(chr, "%04d", chan);
/*
	if (chr[0] != '0' || chr[1] != '0')
*/
		*ptr++ = ((chr[0] & 0x0f) << 4) | (chr[1] & 0x0f);
	*ptr++ = ((chr[2] & 0x0f) << 4) | (chr[3] & 0x0f);
	*ptr++ = FI;
	temp = sndpkt(radio->ident, buf, buf);
	if (temp < 1 || buf[0] != ACK) {
		printf("*** invalid memory channel\n");
		return(1);
	}
	radio->mchan = chan;
	return(0);
}

/*
 * setbank(radio, bank) - set memory bank number (R8500 only)
 */
setbank(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int bank		/* memory bank */
	)
{
	int buf[BMAX], *ptr, temp;
	char chr[5];

	ptr = buf;
	*ptr++ = V_SMEM;
	*ptr++ = 0xa0;
	sprintf(chr, "%02d", bank);
	*ptr++ = ((chr[0] & 0x0f) << 4) | (chr[1] & 0x0f);
	*ptr++ = FI;
	temp = sndpkt(radio->ident, buf, buf);
	if (temp < 1 || buf[0] != ACK) {
		printf("*** invalid memory channel\n");
		return(1);
	}
	radio->bank = bank;
	return(0);
}

/*
 * readchan(radio, chan) - read channel freq, mode and tx offset
 */
int
readchan(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int chan		/* channel number */
	)
{
	int buf[BMAX], temp, i;
	double dtemp;
	char s1[NAMMAX];

	radio->mode = 0xff;
	if (setchan(radio, chan))
		return(1);
	if (radio->flags & F_VFO)
		setcmd(radio, V_VFOM, FI);

	/*
	 * Read mode and frequency
	 */
	temp = sndpkt(radio->ident, s_rmode, buf);
	if (temp < 2 || buf[0] != s_rmode[0]) {
		printf("*** invalid packet format\n");
		return(1);
	}
	radio->mode = buf[1];
	if (temp > 2)
		radio->mode = radio->mode | (buf[2] << 8) | 0x8000;
	if (radio->mode == 0xff)
		return(0);
	if (readfreq(radio))
		return(1);

	/*
	 * Read offset (vhf transceivers only)
	 */
	if (!(radio->flags & F_OFFSET))
		return(0);
	temp = sndpkt(radio->ident, s_roffset, buf);
	if (temp < 4 || buf[0] != s_roffset[0]) {
		printf("*** invalid packet format\n");
		return(1);
	}
	for (i = temp; i < 6; i++)
		buf[i] = 0;
	radio->duplex = freqdouble(&buf[1]) / 10;
	return(0);
}

/*
 * readfreq(radio, freq) - read frequency
 */
int
readfreq(			/* returns 0 (ok), 1 (error) */
	struct icom *radio	/* radio structure */
	)
{
	int buf[BMAX], *ptr, temp;
	double dtemp;

	temp = sndpkt(radio->ident, s_rfreq, buf);
	if (temp < 6 || buf[0] != s_rfreq[0]) {
		printf("*** invalid packet format\n");
		return(1);
	}
	dtemp = (freqdouble(&buf[1]) - radio->bfo[radio->mode & 0x7]) /
	    (1. + radio->vfo_comp);
	radio->vfo = dtemp / 1e6;
	return(0);
}

/*
 * loadfreq(radio, freq) - load frequency
 */
int
loadfreq(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	double freq		/* frequency (MHz) */
	)
{
	int buf[BMAX], *ptr, temp;
	double dtemp, etemp;

	ptr = buf;
	*ptr++ = V_SFREQ;
	dtemp = freq * (1. + radio->vfo_comp) +
	    radio->bfo[radio->mode & 0x7];
/*
	if (dtemp < radio->lband || dtemp > radio->uband) {
		printf("*** invalid frequency\n");
		return(1);
	}
*/
	if (radio->flags & F_735)
		temp = 4;
	else
		temp = 5;
	doublefreq(dtemp * 1e6, ptr, temp);
	temp = sndpkt(radio->ident, buf, buf);
	if (temp < 1 || buf[0] != ACK) {
		printf("*** invalid frequency\n");
		return(1);
	}
	radio->vfo = dtemp;
	return(0);
}

/*
 * loadmode(radio, mode) - load mode
 */
int
loadmode(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int mode		/* mode */
	)
{
	int buf[BMAX], *ptr, temp, i;

	ptr = buf;
	*ptr++ = V_SMODE;
	*ptr++ = mode & 0xff;
	if ((mode & 0x8000) != 0)
		*ptr++ = (mode >> 8) & 0x7f;
	*ptr = FI;
	temp = sndpkt(radio->ident, buf, buf);
	if (temp < 1 || buf[0] != ACK) {
		printf("*** invalid mode\n");
		return(1);
	}
	radio->mode = mode;
	if (radio->flags & F_RELD)
		return(loadfreq(radio, radio->vfo));
	return(0);
}

/*
 * loadoffset(radio, offset) - load tx offset
 */
int
loadoffset(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	double freq		/* tx offset (kHz) */
	)
{
	int buf[BMAX], *ptr, temp;

	ptr = buf;
	*ptr++ = V_SOFFS;
	doublefreq(freq * 10, ptr, 3);
	temp = sndpkt(radio->ident, buf, buf);
	if (temp < 1 || buf[0] != ACK) {
		printf("*** invalid duplex offset\n");
		return(1);
	}
	radio->duplex = freq;
	return(0);
}

/*
 * loadtune(radio, step) - load dial tuning step
 */
int
loadtune(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	double freq		/* dial tuning step (kHz) */
	)
{
	int buf[BMAX], *ptr, temp;

	ptr = buf;
	*ptr++ = V_TUNE;
	doublefreq(freq, ptr, 1);
	temp = sndpkt(radio->ident, buf, buf);
	if (temp < 1 || buf[0] != ACK) {
		printf("*** invalid dial tuning step\n");
		return(1);
	}
	radio->tune = freq;
	return(0);
}


/*
 * setcmd(radio, cmd, subcmd) - utility control
 */
int
setcmd(				/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int cmd,		/* command */
	int subcmd		/* subcommand */
	)
{
	int buf[BMAX], *ptr, temp;

	ptr = buf;
	*ptr++ = cmd;
	*ptr++ = subcmd;
	*ptr++ = FI;
	temp = sndpkt(radio->ident, buf, buf);
	if (temp < 1 || buf[0] == NAK)
		return(1);
	return(0);
}

/*
 * sendcw(radio, string) - send CW message
 */
int
sendcw(				/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	char *string		/* ASCII string */
	)
{
	int cmd[BMAX], rsp[BMAX], *ptr, temp, i;

	ptr = cmd; *ptr++ = V_ASCII;
	for (i = 0; string[i] != '\0'; i++)
		*ptr++ = string[i];
	*ptr++ = FI;
	temp = sndpkt(radio->ident, cmd, rsp);
	if (temp < 1 || rsp[0] == NAK) {
		printf("*** no CW capability\n");
		return(1);
	}
	return(0);
}

/*
 * Conversion routines
 *
 * These routines convert between internal format and radio format.
 * Memory channel, frequency and tx offset data are in 4-bit BCD digits,
 * packed two digits per octet. Memory-channel numbers have two or four
 * digits in big-endian order. Frequency data have ten digits (eight for
 * the 735) in 1-Hz units, little-endian order. Tx offset data have six
 * digits in 100-Hz units, little-endian order.
 */
/*
 * doublefreq(freq, y, len) - convert double to ICOM frequency
 */
void
doublefreq(
	double freq,		/* frequency */
	int *x,			/* radio frequency */
	int len			/* minimum length of padded frequency */
	)
{
	int i;
	char s1[11];
	char *y;

	sprintf(s1, " %10.0lf", freq);
	y = s1 + 10;
	i = 0;
	while (*y != ' ') {
		x[i] = *y-- & 0x0f;
		x[i] = x[i] | ((*y-- & 0x0f) << 4);
		i++;
	}
	for (; i < len; i++)
		x[i] = 0;
	x[i] = FI;
}

/*
 * freqdouble(x) - convert ICOM frequency to double
 */
double
freqdouble(			/* returns frequency */
	int *x			/* radio frequency */
	)
{
	int i;
	char s[11];
	char *y;
	double dtemp;

	y = s + 10;
	*y-- = '\0';
	for (i = 0; i < 5 && x[i] != FI; i++) {
		*y-- = (char)((x[i] & 0x0f) + '0');
		*y-- = (char)(((x[i] >> 4) & 0x0f) + '0');
	}
	sscanf(s, "%lf", &dtemp);
	return(dtemp);
}

/* end program */
