/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
// Implementation of ianimator.h
//

#include "ianimator.h"

#include "ianimatorscript.h"
#include "icommoneventobservers.h"
#include "icontrolmodule.h"
#include "icrosssectionviewsubject.h"
#include "idatareader.h"
#include "idirectory.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iimagecomposer.h"
#include "imath.h"
#include "ishell.h"
#include "ishellfactory.h"
#include "isystem.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "iviewobjectfamily.h"
#include "iwriter.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkImageBlend.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>

#include <icamerapath.h>
#include <vtkCellArray.h>
#include <vtkConeSource.h>

using namespace iParameter;

//
//  Templates
//
#include "iarraytemplate.h"


#define REN this->GetViewModule()->GetRenderer()
#define DATAREADER this->GetViewModule()->GetReader()
#define CAM this->GetViewModule()->GetRenderer()->GetActiveCamera()
#define XSECTION this->GetViewModule()->GetCrossSectionViewSubject()

#define RAN_U (2.0*vtkMath::Random()-1.0)
#define RAN_N (1.4142*tan(2.0*RAN_U/3.1415926))
#define LEN(x) (sqrt(x[0]*x[0]+x[1]*x[1]+x[2]*x[2]))


IOBJECT_DEFINE_TYPE(iAnimator,Animator,a,iObjectType::_Helper);

IOBJECT_DEFINE_KEY(iAnimator,CameraPathClosed,cpc,Bool,1);
IOBJECT_DEFINE_KEY(iAnimator,CameraPathDemo,cpd,Bool,1);
IOBJECT_DEFINE_KEY(iAnimator,CameraPathX,cpx,Float,0);
IOBJECT_DEFINE_KEY(iAnimator,CameraPathY,cpy,Float,0);
IOBJECT_DEFINE_KEY(iAnimator,CameraPathZ,cpz,Float,0);
IOBJECT_DEFINE_KEY(iAnimator,Phi,dp,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,Roll,dr,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,Theta,dt,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,Zoom,dz,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,FocalPathEnabled,fpe,Bool,1);
IOBJECT_DEFINE_KEY(iAnimator,FocalPathToPoint,fpp,Bool,1);
IOBJECT_DEFINE_KEY(iAnimator,FocalPathX,fpx,Float,0);
IOBJECT_DEFINE_KEY(iAnimator,FocalPathY,fpy,Float,0);
IOBJECT_DEFINE_KEY(iAnimator,FocalPathZ,fpz,Float,0);
IOBJECT_DEFINE_KEY(iAnimator,FlybySpeed,fs,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,InheritSettings,is,Bool,1);
IOBJECT_DEFINE_KEY(iAnimator,LoadScriptFile,ls,String,1);
IOBJECT_DEFINE_KEY(iAnimator,LogoFile,lf,String,1);
IOBJECT_DEFINE_KEY(iAnimator,LogoOpacity,lo,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,LogoPosition,lp,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfBlendedFrames,nb,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfFrames,nf,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfCameraPathHandles,nph,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfCameraPathSteps,nps,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfTransitionFrames,nt,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,PositionOnPath,pop,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,RestoreCamera,rc,Bool,1);
IOBJECT_DEFINE_KEY(iAnimator,Style,s,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,ScriptFileName,sfn,String,1);
IOBJECT_DEFINE_KEY(iAnimator,StopOnPath,sop,Bool,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfTitlePageBlendedFrames,tbf,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,TitlePageFile,tf,String,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfTitlePageFrames,tnf,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,UseScript,us,Bool,1);
IOBJECT_DEFINE_KEY(iAnimator,CrossSectionSpeed,xs,Float,1);

IOBJECT_DEFINE_KEY(iAnimator,DebugFlag,-df,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,Debugging,-dd,Bool,1); // need a bool key for GUI shell

//
//  Special keys that contain pointers to internal data. They should never be packed/unpacked,
//  only used for internal communication. Is there a better design?
//
IOBJECT_DEFINE_KEY(iAnimator,LogoImage,-li,Any,0);
IOBJECT_DEFINE_KEY(iAnimator,TitlePageImage,-ti,Any,0);


//
//  iAnimator class
//
iAnimator* iAnimator::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iAnimator(vm); // non-inheritable, so no need to use Object Factory
}


iAnimator::iAnimator(iViewModule *vm) : iObject("Animator"), mViewModule(vm)
{
	mStarted = mStartedRender = mUseScript = mIsPlayingDemo = false;
	mInheritSettings = true;
	mRestoreCamera = true;
	mDebugFlag = 0;

//	scriptFileName = ""; (this is automatic with iString)

	mNewRec = false;
	mCurRec = mPrevRec = -1;
	mCurFrame = mTotFrame = 0;

	mState.mode = 1;
	mState.nframes = 1;
	mState.dphi = 1.0;
	mState.dtheta = 0.0;
	mState.dscale = 1.0;
	mState.droll = 0.0;
	mState.flybySpeed = 0.0;
	mState.slideSpeed = 0.0;
	mState.nBlendedFrames = 0;
	mState.nTransitionFrames = 0;
//	mState.titlePageFile = "";
	mState.titlePageNumFrames = 0;
	mState.titlePageNumBlendedFrames = 0;
//	mState.logoFile = "";
	mState.logoOpacity = 0.5;
	mState.logoPosition = 0;
	mState.stopOnPath = false;
	mState.posOnPath = 0;

	mNumPathSteps = 200;

	mRandStep = 0.03;
	mSeed = 12345;
	vtkMath::RandomSeed(mSeed);

	mDoingTitlePage = false;

	mAnimatorScript = iAnimatorScript::New(this); IERROR_ASSERT(mAnimatorScript);
	mAnimatorObserver = iAnimatorEventObserver::New(this->GetViewModule()); IERROR_ASSERT(mAnimatorObserver);

	mCameraPath = iCameraPath::New(this->GetViewModule()); IERROR_ASSERT(mCameraPath);

	mLeader = 0;
}


iAnimator::~iAnimator()
{
	this->RemoveAllFollowers();
	if(mLeader != 0) mLeader->RemoveFollower(this);

	mAnimatorObserver->Delete();
	mAnimatorScript->Delete();
	while(mBlenderBase.Size() > 0) delete mBlenderBase.RemoveLast();
	mCameraPath->Delete();
}


//
//  This is the main driver
//
void iAnimator::Animate()
{
	this->Start();
	while(this->Continue());
	this->Stop();
}


//
//  Start the animator
//
void iAnimator::Start()
{
	static iString nullScript = "render -1";

	this->GetErrorStatus()->Clear();

	if(mStarted)
	{
		this->GetErrorStatus()->Set("Attempting to start an already started animation.");
		return;
	}
	//
	//  Disactivate Progress Dialog
	//
	iAbortRenderEventObserver::BlockAbortRenderEventObservers(true);
	//
	//  Are we using a script?
	//
	this->SaveState();
	if(mUseScript && !mInheritSettings) this->ResetState();
	
	if(!mUseScript)
	{
		mScriptText = mAnimatorScript->GetText();
		mAnimatorScript->SetText(nullScript);
	}

	this->GetViewModule()->GetWriter()->Start();
	mAnimatorScript->StartExecute();

	mStarted = true;
	this->ClearCache();

	int i;
	for(i=0; i<mFollowers.Size(); i++) mFollowers[i]->Start();
}


//
//  Keep animating until stopped
//
bool iAnimator::Continue()
{
	if(!mStarted)
	{
		this->GetErrorStatus()->Set("Attempting to continue a never started animation.");
		return false;
	}

	this->GetErrorStatus()->Monitor(mAnimatorScript->GetErrorStatus());
	this->ClearCache();
	return mAnimatorScript->ExecuteOneLine();
}


//
//  Stop animation
//
void iAnimator::Stop()
{
	int i;
	for(i=0; i<mFollowers.Size(); i++) mFollowers[i]->Stop();

	if(!mStarted)
	{
		this->GetErrorStatus()->Set("Attempting to finish a never started animation.");
		return;
	}

	this->GetViewModule()->GetWriter()->Finish();
	mAnimatorScript->StopExecute();

	if(!mUseScript)
	{
		mAnimatorScript->SetText(mScriptText);
	}
	//
	//  Restore the original view
	//
	this->Reset();
	//
	//  Restore the original state after execution of a script - must do it after reset()
	//  so that projection is Set properly in case it was changed in a script before the
	//  first render
	//
	this->RestoreState();
	//
	//  Activate Progress Dialog
	//
	iAbortRenderEventObserver::BlockAbortRenderEventObservers(false);

	mStarted = false;
	this->ClearCache();
}


void iAnimator::AddFollower(iAnimator *f)
{
	if(f!=0 && f!=this)
	{
		mFollowers.AddUnique(f);
		f->SetLeader(this);
	}
}


void iAnimator::RemoveFollower(iAnimator *f)
{
	if(f != 0)
	{
		mFollowers.Remove(f);
		f->SetLeader(0);
	}
}


void iAnimator::SetLeader(iAnimator *l)
{
	if(l != this)
	{
		mLeader = l;
	}
}


void iAnimator::RemoveAllFollowers()
{
	int i;
	for(i=0; i<mFollowers.Size(); i++) mFollowers[i]->SetLeader(0);
	mFollowers.Clear();
}


//
//  Internal functions
//
void iAnimator::Reset()
{
	mStartedRender = false;
	mDoingTitlePage = false;
	vtkMath::RandomSeed(mSeed);
	this->GetViewModule()->JustifyLabelLeft(false);
	this->ClearCache();
}


void iAnimator::SetDebugFlag(int s) 
{ 
	mDebugFlag = s;
	this->GetViewModule()->SetDebugMode(s > 1);
	this->ClearCache();
}


void iAnimator::ResetCurrentFile()
{
	mNewRec = true;
	mPrevRec = mCurRec;
	mCurRec = DATAREADER->GetRecordNumber();
	mCurFrame = mTotFrame = 0;
	this->ClearCache();
}


void iAnimator::SaveState()
{
	if(XSECTION != 0) mState.xsecPos = XSECTION->GetLocation(); else mState.xsecPos = 0.0;
	mState.ifBoundingBox = this->GetViewModule()->IsBoundingBoxVisible();
	mState.ifColorBars = this->GetViewModule()->IsColorBarsVisible();
	mState.ifTimeLabel = this->GetViewModule()->IsLabelVisible();

	mState.cameraProjection = CAM->GetParallelProjection();
	CAM->GetPosition(mState.cameraPosition);
	CAM->GetFocalPoint(mState.cameraFocalPoint);
	CAM->GetViewUp(mState.cameraViewUp);
	mState.cameraParallelScale = CAM->GetParallelScale();
	CAM->GetClippingRange(mState.cameraClippingRange);

	if(DATAREADER != 0) mState.currec = this->GetViewModule()->GetReader()->GetRecordNumber();
	mState2 = mState;
	this->ClearCache();
}


void iAnimator::RestoreState()
{
	mState = mState2;

	if(DATAREADER!=0 && mDebugFlag<2)
	{
		DATAREADER->LoadRecord(mState.currec,0,true);
	}

	if(XSECTION != 0) XSECTION->SetLocation(mState.xsecPos);
	this->GetViewModule()->ShowBoundingBox(mState.ifBoundingBox );
	this->GetViewModule()->ShowColorBars(mState.ifColorBars);
	this->GetViewModule()->ShowLabel(mState.ifTimeLabel);

	if(mRestoreCamera)
	{
		CAM->SetParallelProjection(mState.cameraProjection);
		CAM->SetPosition(mState.cameraPosition);
		CAM->SetFocalPoint(mState.cameraFocalPoint);
		CAM->SetViewUp(mState.cameraViewUp);
		CAM->SetParallelScale(mState.cameraParallelScale);
	}
	this->ClearCache();
}


void iAnimator::ResetState()
{
	mState.mode = 0;
	mState.nframes = 1;
	mState.dphi = 0.0;
	mState.dtheta = 0.0;
	mState.dscale = 1.0;
	mState.droll = 0.0;
	mState.flybySpeed = 0.01;
	mState.slideSpeed = 0.0;
	mState.nBlendedFrames = 0;
	mState.nTransitionFrames = 0;
	mState.titlePageFile.Clear();
	mState.titlePageNumFrames = 0;
	mState.titlePageNumBlendedFrames = 0;
	mState.logoFile.Clear();
	mState.logoOpacity = 0.5;
	mState.logoPosition = 0;
	mState.stopOnPath = false;
	mState.posOnPath = 0;

	this->ClearCache();
}


void iAnimator::CopyState(iAnimator *anim)
{
	anim->SaveState();
	mState2 = anim->mState;
	mRestoreCamera = false;  //  we do not want to reset cameras of other vm
	this->RestoreState();
	mRestoreCamera = anim->mRestoreCamera;

	this->ClearCache();
}


void iAnimator::SetStyle(int ma)
{ 
	if(ma>=0 && ma<=4) 
	{
		mState.mode = ma; 
		mCameraPath->SetEnabled(ma == 4);
		this->Reset();
		this->ClearCache();
	}
}


bool iAnimator::SetLogoFile(const iString &s)
{
	this->ClearCache();
	bool ok = mLogoImage.LoadFromFile(s);
	if(ok) mState.logoFile = s; else 
	{ 
		mState.logoFile.Clear();
		mLogoImage.Clear();
	}
	return ok;
}


bool iAnimator::SetTitlePageFile(const iString &s)
{
	this->ClearCache();
	bool ok = mTitlePageImage.LoadFromFile(s);
	if(ok) mState.titlePageFile = s; else 
	{
		mState.titlePageFile.Clear();
		mTitlePageImage.Clear();
	}
	return ok;
}


bool iAnimator::LoadScriptFile(const iString &name)
{
	iString fname(name);
	iDirectory::ExpandFileName(fname,this->GetViewModule()->GetControlModule()->GetShell()->GetEnvironment(Environment::Script));

	iFile f(fname);
	if(!f.Open(iFile::_Read,iFile::_Text))
	{
		this->GetErrorStatus()->Set("Animation script file is not found.");
		return false;
	}

	iString st, tmp;
	while(f.ReadLine(tmp)) st += tmp + "\n";
	if(st.IsEmpty())
	{
		this->GetErrorStatus()->Set("Animation script file is empty or unreadable.");
		return false;
	}

	f.Close();

	mAnimatorScript->SetText(st);
	mUseScript = true;
	this->SetScriptFileName(fname);

	return true;
}


bool iAnimator::RenderImages(bool dumpImage)
{
	static float Pi = 3.1415927;
	double xc[3], x0[3];
	float v0[3], r0[3], r1[3], vA[3], vB[3];
	int i;

	mNewRec = false;
	this->ClearCache();
	this->GetErrorStatus()->Clear();

	if(!mStartedRender)
	{
		if(!DATAREADER->IsFileAnimatable())
		{
			this->GetErrorStatus()->Set("File is not animatable.");
			return false;
		}
		this->GetViewModule()->JustifyLabelLeft(true);
		
		if(mState.mode == 2)
		{
			wData.dphl0 = mState.dphi;
			wData.dthl0 = mState.dtheta;
			wData.r = 0.0;
			wData.ramp = RAN_N;
		}
		
		if(mState.mode == 3)
		{
			for(i=0; i<3; i++)
			{
				wData.xc1[i] = 0.5*RAN_U;
				wData.xc2[i] = 0.5*RAN_U;
			}
			CAM->SetParallelProjection(0);
			CAM->GetPosition(wData.x);
			if(mState.cameraProjection == 1)
			{
				for(i=0; i<3; i++) wData.x[i] = 0.5*wData.x[i];
				CAM->SetPosition(wData.x);
			}
			else
			{
				CAM->GetFocalPoint(wData.xc1);
			}
			CAM->SetFocalPoint(wData.xc1);
			float d = LEN(wData.x);
			wData.v[0] = d*0.5*RAN_U;
			wData.v[1] = d*0.5*RAN_U;
			wData.v[2] = 0.0;
			wData.t = 0.0;
			wData.dt0 = 0.1*Pi;
			wData.dt = wData.dt0;
		}
		
		if(mState.mode == 4)
		{
			mCameraPath->SetEnabled(false);
		}	
		
		if(mState.titlePageFile.IsEmpty() || mDebugFlag!=0)
		{
			mCurFrame = mTotFrame = 0;
			mDoingTitlePage = false;
		}
		else
		{
			mCurFrame = mTotFrame = -mState.titlePageNumFrames;
			mDoingTitlePage = true;
		}
		mPrevRec = -1;
		mCurRec = DATAREADER->GetRecordNumber();
		mStartedRender = true;
	} 

	if(mCurFrame == mState.nframes)
	{
		this->GetErrorStatus()->Monitor(DATAREADER->GetErrorStatus());
		
		iProgressEventObserver *obs = DATAREADER->GetProgressEventObserver(DATAREADER->GetFileSetDataType());
		if(obs != 0) obs->AttachScript(mAnimatorScript);
		DATAREADER->LoadRecord(-1,1,mDebugFlag>1);
		if(obs != 0) obs->AttachScript(0);

		if(this->GetErrorStatus()->IsError())
		{
			if(this->GetErrorStatus()->Level()==10)
			{
				this->GetErrorStatus()->Clear();
				if(mUseScript)
				{
					this->GetErrorStatus()->Set("Not enough data files to complete the script.",-1);
				}
			}
			return false;
		}
		this->GetViewModule()->UpdateLabel();
		mNewRec = true;
		mPrevRec = mCurRec;
		mCurRec = DATAREADER->GetRecordNumber();
		mCurFrame = 0;
	} 
		
	if(this->GetErrorStatus()->IsError()) 
	{
		if(mState.mode == 4)	
		{
			mCameraPath->SetEnabled(true);
		}
		return false;
	}

	mCurFrame++;
	mTotFrame++;

	if(mTotFrame > 0)
	{
		if(fabs(mState.slideSpeed)>1.0e-30 && XSECTION!=0)
		{
			double p = XSECTION->GetLocation();
			p = p + mState.slideSpeed;
			XSECTION->SetLocation(p);
			if(XSECTION->GetOverTheEdgeFlag()) mState.slideSpeed = -mState.slideSpeed;
		}
		//
		//	Add transformations for rotate & tumble
		//	
		if(mState.mode==1 || mState.mode==2)
		{		
			CAM->Azimuth(-mState.dphi);
			CAM->Elevation(-mState.dtheta);
			CAM->Zoom(mState.dscale);
			CAM->Roll(mState.droll);
			CAM->OrthogonalizeViewUp();
			
			if(mState.mode == 2)
			{
				wData.r = wData.r + mRandStep;
				float cr = cos(wData.r*wData.ramp);
				float sr = sin(wData.r*wData.ramp);
				mState.dphi =  wData.dphl0*cr + wData.dthl0*sr;
				mState.dtheta =  wData.dthl0*cr - wData.dphl0*sr;
				if(wData.r > 1.0)
				{
					wData.r = 0.0;
					wData.ramp = RAN_N;
					wData.dphl0 =  mState.dphi;
					wData.dthl0 =  mState.dtheta;
				}
			}
		}
		//
		//  Add transformations for flyby
		//
		if(mState.mode == 3)
		{
			for(i=0; i<3; i++)
			{
				xc[i] = wData.xc1[i] + (wData.xc2[i]-wData.xc1[i])*wData.t;
				x0[i] = wData.x[i];
				v0[i] = wData.v[i];
			}
			
			CAM->SetFocalPoint(xc);
			
			wData.dt *= 2;
			float d0, d1, cot, sot;
			do
			{
				wData.dt *= 0.5;

				for(i=0; i<3; i++)
				{
					r0[i] = x0[i] - xc[i];
					vA[i] = r0[i];
					vB[i] = v0[i];
				}

				cot = cos(wData.dt);
				sot = sin(wData.dt);
				for(i=0; i<3; i++) r1[i] = vA[i]*cot + vB[i]*sot - r0[i];

				d0 = d1 = 0.0;
				for(i=0; i<3; i++)
				{
					d0 = d0 + r0[i]*r0[i];
					d1 = d1 + r1[i]*r1[i];
				}
				d1 = sqrt(d1/d0);
			}
			while(d1>mState.flybySpeed && wData.dt>0.001*wData.dt0);
			
			if(d1 < 0.2*mState.flybySpeed) wData.dt = 1.5*wData.dt;
			
			for(i=0; i<3; i++)
			{
				wData.v[i] = vB[i]*cot-vA[i]*sot;
				wData.x[i] = r0[i] + r1[i] + xc[i];
			}
			
			CAM->SetPosition(wData.x);
			CAM->OrthogonalizeViewUp();
			wData.t = wData.t + wData.dt;
			
			if(wData.t > 1.0)
			{
				wData.t = 0.0;
				for(i=0; i<3; i++)
				{
					wData.xc1[i] = wData.xc2[i];
					wData.xc2[i] = 0.5*RAN_U;
				}
			}
		}

		//
		//  Camera path
		//
		if(mState.mode == 4)
		{
			//
			//  Are we done?
			//
			if(mState.posOnPath == mNumPathSteps)
			{
				if(this->GetCameraPathClosed())
				{
					mState.posOnPath = 0;
				}
				else
				{
					this->Reset();
					mCameraPath->SetEnabled(true);
					return false;
				}
			}
			//
			//  Position the camera
			//
			this->PositionCameraOnThePath();
			//
			//  Update step counter if we are moving
			//
			if(!mState.stopOnPath) mState.posOnPath++;
		}	
		
		this->GetViewModule()->Render(); 
	}
	//
	//  Image data holder
	//
	iStereoImageArray images, tmparr;
	iImage tmp;
	//
	//  Transition effects
	//
	bool doTransitionFrames = dumpImage && mTotFrame>0 && mPrevRec>0 && mCurFrame<=mState.nTransitionFrames && mState.nTransitionFrames>0 && mDebugFlag==0;
	if(doTransitionFrames)
	{
		//
		//  Objects for transition effects (blending with previous record)
		//
		this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
		this->GetViewModule()->CreateImages(images);
		if(this->GetErrorStatus()->NoError())
		{
			this->GetErrorStatus()->Monitor(DATAREADER->GetErrorStatus(),true);
			DATAREADER->LoadRecord(mPrevRec,0,false);
			if(this->GetErrorStatus()->NoError())
			{
				this->GetViewModule()->UpdateLabel();
				this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
				this->GetViewModule()->CreateImages(tmparr);
				if(this->GetErrorStatus()->NoError())
				{
					float ops = (float)mCurFrame/mState.nTransitionFrames;
					for(i=0; i<images.Size(); i++)
					{
						images[i].Blend(tmparr[i],ops);	
					}
				}
			}
		}
		if(this->GetErrorStatus()->NoError())
		{
			this->GetErrorStatus()->Monitor(DATAREADER->GetErrorStatus(),true);
			DATAREADER->LoadRecord(mCurRec,0,false);	
		}
		else return false;
	}

	//
	//  Blending of images
	//
	bool doBlendedFrames = dumpImage && mTotFrame >0 && mState.nBlendedFrames>0 && mDebugFlag==0;
	if(doBlendedFrames)
	{
		if(images.Size() == 0)
		{
			this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
			this->GetViewModule()->CreateImages(images);
		}
		if(this->GetErrorStatus()->NoError())
		{
			int k;
			//
			//  Update the image list
			//
			while(mBlenderBase.Size() < mState.nBlendedFrames)
			{
				iStereoImageArray *ptmparr = new iStereoImageArray; IERROR_ASSERT(ptmparr);
				ptmparr->Copy(images);
				mBlenderBase.Add(ptmparr);
			}
			while(mBlenderBase.Size() > mState.nBlendedFrames)
			{
				delete mBlenderBase.RemoveLast();
			}

			delete mBlenderBase[0];
			for(k=0; k<mBlenderBase.MaxIndex(); k++) mBlenderBase[k]->Copy(*mBlenderBase[k+1]);
			mBlenderBase.Last()->Copy(images);

			//
			//  Make sure that all the arrays are of the same size
			//
			for(k=0; k<mBlenderBase.MaxIndex(); k++)
			{
				while(mBlenderBase[k]->Size() > images.Size()) mBlenderBase[k]->RemoveLast();
				while(mBlenderBase[k]->Size() < images.Size()) mBlenderBase[k]->Add(images[mBlenderBase[k]->Size()]);
			}

			//
			//  Blend the arrays
			//
			int n = 1;
			float ops;
			images.Copy(*mBlenderBase[0]);
			for(k=1; k<mBlenderBase.Size(); k++)
			{
				n += (k+1);
				ops = float(k+1)/n;
				for(i=0; i<images.Size(); i++)
				{
					images[i].Blend((*mBlenderBase[k])[i],ops);
				}
			}
		}
		else return false;
	}
	
	//
	//  Title page
	//
	if(dumpImage && mDoingTitlePage)
	{
		tmp = mTitlePageImage;
		//
		//  We must update composer here because there is no automatic way to call
		//  composer->Update() when vtkRenderWindow changes its size (it does not invoke an event).
		//
		this->GetViewModule()->GetControlModule()->GetImageComposer()->Update();
		tmp.Scale(this->GetViewModule()->GetFullImageWidth(),this->GetViewModule()->GetFullImageHeight());

		if(mTotFrame <= 0)
		{
			//
			//  Plain title page
			//
			for(i=0; i<images.Size(); i++) images[i] = tmp;
		}
		else if(mState.titlePageNumBlendedFrames > 0)
		{
			//
			//  Create the image to blend the title page into, if needed
			//
			if(images.Size() == 0)
			{
				this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
				this->GetViewModule()->CreateImages(images);
			}
			//
			//  Dissolve it; image already contains the correct image
			//
			if(this->GetErrorStatus()->NoError())
			{
				for(i=0; i<images.Size(); i++) images[i].Blend(tmp,(float)mTotFrame/mState.titlePageNumBlendedFrames);
			}
		}
		if(mTotFrame > mState.titlePageNumBlendedFrames) mDoingTitlePage = false;
	}

	//
	//  Logo
	//
	if(dumpImage && mDebugFlag==0 && !mState.logoFile.IsEmpty())
	{
		//
		//  Create the image to place a logo on, if needed
		//
		if(images.Size() == 0)
		{
			this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
			this->GetViewModule()->CreateImages(images);
		}
		if(this->GetErrorStatus()->NoError())  // we do have the image
		{
			//
			//  If the logo is more than 20% of the image, scale it down.
			//  Use tmp as a temp storage
			//
			tmp = mLogoImage;
			if(tmp.Width()>images[0].Width()/5 || tmp.Height()>images[0].Height()/5)
			{
				tmp.Scale(images[0].Width()/5,images[0].Height()/5);
			}

			if(tmp.Width()>=2 && tmp.Height()>=2)
			{
				//
				//  tmp is now the proper logo image
				//
				int ioff, joff;
				//
				//  Where do we place the logo?
				//
				ioff = tmp.Width()/5;
				joff = tmp.Height()/5;
				switch(mState.logoPosition)
				{
				case 1:
					{
						//  upper right corner 
						ioff = images[0].Width() - tmp.Width() - ioff;
						joff = images[0].Height() - tmp.Height() - joff;
						break;
					}
				case 2:
					{
						//  lower left right corner 
						break;
					}
				case 3:
					{
						//  lower right corner 
						ioff = images[0].Width() - tmp.Width() - ioff;
						break;
					}
				default:
					{
						//  upper left corner - the default choice
						joff = images[0].Height() - tmp.Height() - joff;
						break;
					}
				}
				for(i=0; i<images.Size(); i++) images[i].Overlay(ioff,joff,tmp,mState.logoOpacity);
			}
		}
		else return false;
	}

	for(i=0; i<mFollowers.Size(); i++)
	{
		this->GetErrorStatus()->Monitor(mFollowers[i]->GetErrorStatus(),true,"Follower Animator from window #"+iString::FromNumber(1+mFollowers[i]->GetViewModule()->GetWindowNumber()));
		if(!mFollowers[i]->RenderImages(false)) break;
	}

	if(mDebugFlag==0 && dumpImage)
	{
		this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
		this->GetViewModule()->DumpImages(images,ImageType::AnimationFrame);
	}

	return this->GetErrorStatus()->NoError();
}


void iAnimator::PositionCameraOnThePath()
{
	double x1[3], x2[3], dx12[3], xc[3];
	int i, i1, i2, np;
	double d, dl;

	vtkPoints *pts1 = mCameraPath->GetCameraPathPoints();
	vtkPoints *pts2 = mCameraPath->GetFocalPathPoints();
	np = pts1->GetNumberOfPoints();

	int pathStep = mState.posOnPath;

	if(pathStep<0 || pathStep>=mNumPathSteps) return;

	dl = double(pathStep*(np-1))/(mNumPathSteps-1);  // position on the path from 0 to np-1
	i1 = (int)floor(dl);
	if(i1 == np-1)
	{
		//
		//  Last interval frame
		//
		i1 = np - 2;
	}

	pts1->GetPoint(i1,x1);
	i2 = i1 + 1;
	while(i2 < np)
	{
		pts1->GetPoint(i2,x2);
		for(i=0; i<3; i++) dx12[i] = x2[i] - x1[i];
		if(vtkMath::Norm(dx12) > 0) break;
		i2++;
	}
	if(i2 == np) i2--;

	d = (dl-i1)/(i2-i1);
	for(i=0; i<3; i++) xc[i] = x1[i] + d*(x2[i]-x1[i]);
	CAM->SetPosition(xc);

	if(pts2 != 0)
	{
		pts2->GetPoint(i1,x1);
		pts2->GetPoint(i2,x2);
		for(i=0; i<3; i++) xc[i] = x1[i] + d*(x2[i]-x1[i]);
		CAM->SetFocalPoint(xc);
	}
	else
	{
		double dx1[3], dx2[3], x[3];
		vtkMath::Normalize(dx12);

		if(i1 > 0)
		{
			pts1->GetPoint(i1-1,x);
			for(i=0; i<3; i++) dx1[i] = x1[i] - x[i];
			vtkMath::Normalize(dx1);
			for(i=0; i<3; i++) dx1[i] = 0.5*(dx1[i]+dx12[i]);
		}
		else
		{
			for(i=0; i<3; i++) dx1[i] = dx12[i];
		}

		if(i2 < np-1)
		{
			pts1->GetPoint(i2+1,x);
			for(i=0; i<3; i++) dx2[i] = x[i] - x2[i];
			vtkMath::Normalize(dx2);
			for(i=0; i<3; i++) dx2[i] = 0.5*(dx2[i]+dx12[i]);
		}
		else
		{
			for(i=0; i<3; i++) dx2[i] = dx12[i];
		}
		
		for(i=0; i<3; i++) dx12[i] = dx1[i] + d*(dx2[i]-dx1[i]);
		vtkMath::Normalize(dx12);
		double fd = CAM->GetDistance();
		for(i=0; i<3; i++) x[i] = xc[i] + dx12[i]*fd;
		CAM->SetFocalPoint(x);
	}
	CAM->OrthogonalizeViewUp();

	mAnimatorObserver->Execute(0,0UL,0);
}
//
//  Camera path functions
//
int iAnimator::GetNumberOfCameraPathHandles() const
{ 
	return mCameraPath->GetNumberOfHandles(); 
}


bool iAnimator::GetCameraPathClosed() const
{ 
	return mCameraPath->GetClosed(); 
}


bool iAnimator::GetFocalPathEnabled() const
{ 
	return mCameraPath->GetFocalPathEnabled(); 
}


bool iAnimator::GetFocalPathToPoint() const
{ 
	return mCameraPath->GetFocalPathToPoint(); 
}


void iAnimator::SetNumberOfCameraPathHandles(int v)
{
	if(v > 1)
	{
		mCameraPath->SetNumberOfHandles(v);
		this->ClearCache();
	}
}


void iAnimator::SetNumberOfCameraPathSteps(int n)
{
	if(n >= mCameraPath->GetNumberOfHandles()-1)
	{
		mNumPathSteps = n;
		this->ClearCache();
	}
}


void iAnimator::SetCameraPathClosed(bool v)
{
	mCameraPath->SetClosed(v); 
	this->ClearCache();
}


void iAnimator::SetFocalPathEnabled(bool s)
{
	mCameraPath->SetFocalPathEnabled(s);
	this->ClearCache();
}


void iAnimator::SetFocalPathToPoint(bool s)
{ 
	mCameraPath->SetFocalPathToPoint(s);
	this->ClearCache();
}


void iAnimator::SetCameraPathColor(iColor &c)
{
	mCameraPath->SetLineColor(c);
	this->ClearCache();
}


bool iAnimator::IsCameraPathValid() const
{
	vtkPoints *pts1 = mCameraPath->GetCameraPathPoints();
	vtkPoints *pts2 = mCameraPath->GetFocalPathPoints();
	if(pts2 == 0) return true;

	int i, n = pts1->GetNumberOfPoints();
	for(i=0; i<n; i++)
	{
		if(vtkMath::Distance2BetweenPoints(pts1->GetPoint(i),pts2->GetPoint(i)) < 1.0e-8) return false;
	}
	return true;
}


void iAnimator::StopCameraPathDemo()
{
	mIsPlayingDemo = false;
}


void iAnimator::PlayCameraPathDemo()
{
	if(mState.mode == 4)
	{
		int pos = mState.posOnPath;
		float t;
		double camPosition[3], camFocalPoint[3], camViewUp[3];

		//
		//  Save the camera parameters
		//
		CAM->GetPosition(camPosition);
		CAM->GetFocalPoint(camFocalPoint);
		CAM->GetViewUp(camViewUp);

		//
		//  Place in starting position
		//
		mCameraPath->SetHandleOpacity(0.5);

		//
		//  Fly along the main path
		//
		mIsPlayingDemo = true;
		for(; mIsPlayingDemo && mState.posOnPath<mNumPathSteps; mState.posOnPath++)
		{
			this->PositionCameraOnThePath();
			this->GetViewModule()->Render(); 
			//
			//  Measure the render time
			//
			this->GetViewModule()->Render(); 
			t = this->GetViewModule()->GetRenderer()->GetLastRenderTimeInSeconds();
			if(t < 0.1) iSystem::Sleep(round(500.0*(0.1-t)));
			this->ClearCache();
		}
		mIsPlayingDemo = false;

		mCameraPath->SetHandleOpacity(1.0);
		mState.posOnPath = pos;
		PositionCameraOnThePath();

		//
		//  Restore the camera parameters
		//
		CAM->SetPosition(camPosition);
		CAM->SetFocalPoint(camFocalPoint);
		CAM->SetViewUp(camViewUp);

		this->GetViewModule()->Render(); 
	}
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iAnimator::PackStateBody(iString &s) const
{
	this->PackValue(s,KeyDebugFlag(),mDebugFlag);
	this->PackValue(s,KeyDebugging(),mDebugFlag!=0);

	this->PackValue(s,KeyUseScript(),mUseScript);
	this->PackValue(s,KeyScriptFileName(),mScriptFileName);
	this->PackValue(s,KeyFocalPathEnabled(),this->GetFocalPathEnabled());
	this->PackValue(s,KeyFocalPathToPoint(),this->GetFocalPathToPoint());
	this->PackValue(s,KeyCameraPathClosed(),this->GetCameraPathClosed());
	this->PackValue(s,KeyRestoreCamera(),mRestoreCamera);
	this->PackValue(s,KeyInheritSettings(),mInheritSettings);

	this->PackValue(s,KeyStyle(),mState.mode);
	this->PackValue(s,KeyNumberOfFrames(),mState.nframes);
	this->PackValue(s,KeyNumberOfBlendedFrames(),mState.nBlendedFrames);
	this->PackValue(s,KeyNumberOfTransitionFrames(),mState.nTransitionFrames);

	this->PackValue(s,KeyPhi(),mState.dphi);
	this->PackValue(s,KeyTheta(),mState.dtheta);
	this->PackValue(s,KeyRoll(),mState.droll);
	this->PackValue(s,KeyZoom(),mState.dscale);
	this->PackValue(s,KeyCrossSectionSpeed(),mState.slideSpeed);
	this->PackValue(s,KeyFlybySpeed(),mState.flybySpeed);

	this->PackValue(s,KeyTitlePageFile(),mState.titlePageFile);
	this->PackValue(s,KeyNumberOfTitlePageFrames(),mState.titlePageNumFrames);
	this->PackValue(s,KeyNumberOfTitlePageBlendedFrames(),mState.titlePageNumBlendedFrames);
	this->PackValue(s,KeyLogoFile(),mState.logoFile);
	this->PackValue(s,KeyLogoOpacity(),mState.logoOpacity);
	this->PackValue(s,KeyLogoPosition(),mState.logoPosition);

	this->PackValue(s,KeyPositionOnPath(),mState.posOnPath);
	this->PackValue(s,KeyStopOnPath(),mState.stopOnPath);
	this->PackValue(s,KeyNumberOfCameraPathHandles(),this->GetNumberOfCameraPathHandles());
	this->PackValue(s,KeyNumberOfCameraPathSteps(),mNumPathSteps);

	int n = mCameraPath->GetNumberOfHandles();

	float *x1 = new float[n]; IERROR_ASSERT(x1);
	float *x2 = new float[n]; IERROR_ASSERT(x2);
	float *x3 = new float[n]; IERROR_ASSERT(x3);

	mCameraPath->GetCameraPathHandlePositions(x1,x2,x3);
	this->PackValue(s,KeyCameraPathX(),x1,n);
	this->PackValue(s,KeyCameraPathY(),x2,n);
	this->PackValue(s,KeyCameraPathZ(),x3,n);

	if(this->GetFocalPathEnabled())
	{
		mCameraPath->GetFocalPathHandlePositions(x1,x2,x3);
		this->PackValue(s,KeyFocalPathX(),x1,n);
		this->PackValue(s,KeyFocalPathY(),x2,n);
		this->PackValue(s,KeyFocalPathZ(),x3,n);
	}

	delete [] x1;
	delete [] x2;
	delete [] x3;
}


void iAnimator::UnPackStateBody(const iString &s)
{
	int i; bool b; float f; iString ss;

	if(this->UnPackValue(s,KeyDebugFlag(),i)) this->SetDebugFlag(i);
	if(this->UnPackValue(s,KeyUseScript(),b)) this->SetUseScript(b);
	if(this->UnPackValue(s,KeyFocalPathEnabled(),b)) this->SetFocalPathEnabled(b);
	if(this->UnPackValue(s,KeyFocalPathToPoint(),b)) this->SetFocalPathToPoint(b);
	if(this->UnPackValue(s,KeyCameraPathClosed(),b)) this->SetCameraPathClosed(b);
	if(this->UnPackValue(s,KeyRestoreCamera(),b)) this->SetRestoreCamera(b);
	if(this->UnPackValue(s,KeyInheritSettings(),b)) this->SetInheritSettings(b);

	if(this->UnPackValue(s,KeyNumberOfFrames(),i)) this->SetNumberOfFrames(i);
	if(this->UnPackValue(s,KeyNumberOfBlendedFrames(),i)) this->SetNumberOfBlendedFrames(i);
	if(this->UnPackValue(s,KeyNumberOfTransitionFrames(),i)) this->SetNumberOfTransitionFrames(i);

	if(this->UnPackValue(s,KeyPhi(),f)) this->SetPhi(f);
	if(this->UnPackValue(s,KeyTheta(),f)) this->SetTheta(f);
	if(this->UnPackValue(s,KeyRoll(),f)) this->SetRoll(f);
	if(this->UnPackValue(s,KeyZoom(),f)) this->SetZoom(f);
	if(this->UnPackValue(s,KeyCrossSectionSpeed(),f)) this->SetCrossSectionSpeed(f);
	if(this->UnPackValue(s,KeyFlybySpeed(),f)) this->SetFlybySpeed(f);

	if(this->UnPackValue(s,KeyTitlePageFile(),ss)) this->SetTitlePageFile(ss);
	if(this->UnPackValue(s,KeyNumberOfTitlePageFrames(),i)) this->SetNumberOfTitlePageFrames(i);
	if(this->UnPackValue(s,KeyNumberOfTitlePageBlendedFrames(),i)) this->SetNumberOfTitlePageBlendedFrames(i);
	if(this->UnPackValue(s,KeyLogoFile(),ss)) this->SetLogoFile(ss);
	if(this->UnPackValue(s,KeyLogoOpacity(),f)) this->SetLogoOpacity(f);
	if(this->UnPackValue(s,KeyLogoPosition(),i)) this->SetLogoPosition(i);

	if(this->UnPackValue(s,KeyPositionOnPath(),i)) this->SetPositionOnPath(i);
	if(this->UnPackValue(s,KeyStopOnPath(),b)) this->SetStopOnPath(b);
	if(this->UnPackValue(s,KeyNumberOfCameraPathHandles(),i)) this->SetNumberOfCameraPathHandles(i);
	if(this->UnPackValue(s,KeyNumberOfCameraPathSteps(),i)) this->SetNumberOfCameraPathSteps(i);

	int n = mCameraPath->GetNumberOfHandles();
	float *x1 = new float[n]; IERROR_ASSERT(x1);
	float *x2 = new float[n]; IERROR_ASSERT(x2);
	float *x3 = new float[n]; IERROR_ASSERT(x3);

	mCameraPath->GetCameraPathHandlePositions(x1,x2,x3);
	if(this->UnPackValue(s,KeyCameraPathX(),x1,n))
	{
		mCameraPath->SetCameraPathHandlePositions(0,x1);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyCameraPathY(),x2,n))
	{
		mCameraPath->SetCameraPathHandlePositions(1,x2);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyCameraPathZ(),x3,n))
	{
		mCameraPath->SetCameraPathHandlePositions(2,x3);
		this->ClearCache();
	}

	if(this->GetFocalPathEnabled())
	{
		mCameraPath->GetFocalPathHandlePositions(x1,x2,x3);
		if(this->UnPackValue(s,KeyCameraPathX(),x1,n))
		{
			mCameraPath->SetFocalPathHandlePositions(0,x1);
			this->ClearCache();
		}
		if(this->UnPackValue(s,KeyCameraPathY(),x2,n))
		{
			mCameraPath->SetFocalPathHandlePositions(1,x2);
			this->ClearCache();
		}
		if(this->UnPackValue(s,KeyCameraPathZ(),x3,n))
		{
			mCameraPath->SetFocalPathHandlePositions(2,x3);
			this->ClearCache();
		}
	}

	delete [] x1;
	delete [] x2;
	delete [] x3;

	if(this->UnPackValue(s,KeyStyle(),i)) this->SetStyle(i);
	
	//
	//  Action keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states

	if(this->UnPackValue(s,KeyLoadScriptFile(),ss)) this->LoadScriptFile(ss);

	if(this->UnPackValue(s,KeyCameraPathDemo(),b))
	{
		if(b) this->PlayCameraPathDemo(); else this->StopCameraPathDemo();
	}

	iObject::ReportMissingKeys(true);
}


