// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
		\author Dan Erikson (derikson@montana.com)
*/

#include "idocument.h"
#include "istate_change_set.h"
#include "istate_container.h"
#include "istate_recorder.h"
#include "result.h"
#include "state_change_set.h"
#include "utility.h"

#include <algorithm>
#include <functional>
#include <vector>

namespace k3d
{

namespace detail
{

/////////////////////////////////////////////////////////////////////////////
// state_change_set_implementation

/// Provides an implementation of k3dIStateChangeSet
class state_change_set_implementation :
	public istate_change_set
{
public:
	state_change_set_implementation()
	{
	}

	~state_change_set_implementation()
	{
		std::for_each(m_old_states.begin(), m_old_states.end(), delete_object());
		std::for_each(m_new_states.begin(), m_new_states.end(), delete_object());
	}

	void record_old_state(istate_container* const OldState)
	{
		// Sanity checks ...
		return_if_fail(OldState);
		m_old_states.push_back(OldState);
	}

	void record_new_state(istate_container* const NewState)
	{
		// Sanity checks ...
		return_if_fail(NewState);
		m_new_states.push_back(NewState);
	}

	recording_done_signal_t& recording_done_signal()
	{
		return m_recording_done_signal;
	}

	undo_signal_t& undo_signal()
	{
		return m_undo_signal;
	}

	redo_signal_t& redo_signal()
	{
		return m_redo_signal;
	}

	void undo()
	{
		std::for_each(m_old_states.rbegin(), m_old_states.rend(), std::mem_fun(&istate_container::restore_state));
		m_undo_signal.emit();
	}

	void redo()
	{
		std::for_each(m_new_states.begin(), m_new_states.end(), std::mem_fun(&istate_container::restore_state));
		m_redo_signal.emit();
	}

private:
	typedef std::vector<istate_container*> state_collection_t;
	state_collection_t m_old_states;
	state_collection_t m_new_states;

	recording_done_signal_t m_recording_done_signal;
	undo_signal_t m_undo_signal;
	redo_signal_t m_redo_signal;
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// create_state_change_set

std::auto_ptr<istate_change_set> create_state_change_set()
{
	return std::auto_ptr<istate_change_set>(new detail::state_change_set_implementation());
}

/////////////////////////////////////////////////////////////////////////////
// start_state_change_set

void  start_state_change_set(idocument& Document)
{
	Document.state_recorder().start_recording(create_state_change_set());
}

/////////////////////////////////////////////////////////////////////////////
// cancel_state_change_set

void cancel_state_change_set(idocument& Document)
{
	// Tell the document to stop recording ...
	const std::auto_ptr<istate_change_set> changeset(Document.state_recorder().stop_recording());

	// Undo any changes up to this point, and let the leftover data get destroyed ...
	changeset->undo();
}

/////////////////////////////////////////////////////////////////////////////
// finish_state_change_set

void finish_state_change_set(idocument& Document, const std::string& Label)
{
	Document.state_recorder().commit_change_set(Document.state_recorder().stop_recording(), Label);
}

/////////////////////////////////////////////////////////////////////////////
// record_state_change_set

record_state_change_set::record_state_change_set(idocument& Document, const std::string& Label) :
	m_document(Document),
	m_label(Label)
{
	// Sanity checks ...
	assert_warning(m_label.size());

	start_state_change_set(m_document);
}

record_state_change_set::~record_state_change_set()
{
	finish_state_change_set(m_document, m_label);
}

} //namespace k3d


