// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::combo_box classes, which provide a model-view-controller UI for enumerated (fixed set of choices) types
		\author Tim Shead (tshead@k-3d.com)
*/

#include "combo_box.h"
#include "gtkml.h"

#include <k3dsdk/application.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/ioptions.h>
#include <k3dsdk/string_modifiers.h>

#include <sdpgtk/sdpgtkevent.h>
#include <sdpgtk/sdpgtkutility.h>

#include <iterator>

namespace k3d
{

namespace combo_box
{

////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::iunknown* const CommandNodeParent, const std::string CommandNodeName) :
	base(CommandNodeParent, CommandNodeName),
	m_events_frozen(true)
{
	// Create and load our UI template ...
	std::stringstream uitemplate;

	uitemplate <<
		"<gtkml>"
			"<eventbox>"
				"<event signal=\"destroy\" name=\"destroy\"/>"
				"<combo name=\"combo\" homogeneous=\"false\"/>"
			"</eventbox>"
		"</gtkml>\n";

	return_if_fail(load_gtkml(uitemplate, "combo_box builtin template", *this));

	MapEvent("changed", "changed", false, Combo("combo").Entry(), true);
}

control::~control()
{
	// No more events from this point forward ...
	DisconnectAllEvents();

	// Clean-up the GTK+ tree ...
	if(Root())
		RootWidget().Destroy();
}

const std::string control::CustomType() const
{
	return std::string("k3dcombobox");
}

bool control::Create(sdpGtkIObjectContainer* const ObjectContainer, sdpxml::Document& Document, sdpxml::Element& Element)
{
	// Sanity checks ...
	assert_warning(ObjectContainer);
	assert_warning(Element.Name() == "k3dcombobox");

	return true;
}

bool control::attach(std::auto_ptr<idata_proxy> Data, k3d::istate_recorder* const StateRecorder, const std::string StateChangeName)
{
	// Sanity checks ...
	return_val_if_fail(Data.get(), false);

	// Take ownership of the data source ...
	m_data = Data;

	// Complete our own initialization ...
	return_val_if_fail(base::Attach(StateRecorder, StateChangeName), false);

	// Update the display ...
	update();

	// We want to be notified if the data source changes ...
	m_data->changed_signal().connect(SigC::slot(*this, &control::update));

	m_events_frozen = false;

	return true;
}

void control::set_values(const values_t& Values)
{
	bool events_frozen = true;
	std::swap(events_frozen, m_events_frozen);
	Combo("combo").SetList(Values);
	update();
	std::swap(events_frozen, m_events_frozen);
}

void control::update()
{
	// Sanity checks ...
	return_if_fail(m_data.get());

	// Get the current value ...
	const std::string value = m_data->value();

	bool events_frozen = true;
	std::swap(events_frozen, m_events_frozen);
	Combo("combo").SetEntryText(value);
	std::swap(events_frozen, m_events_frozen);
}

void control::OnEvent(sdpGtkEvent* Event)
{
	// Sanity checks ...
	assert_warning(Event);

	if(Event->Name() == "destroy")
		on_destroy();
	else if(Event->Name() == "changed")
		on_changed();
	else
		base::OnEvent(Event);
}

void control::on_destroy()
{
	DisconnectAllEvents();
	Clear();
}

void control::on_changed()
{
	// If events are frozen, we're done ...
	if(m_events_frozen)
		return;

	// Make sure we've got some storage, first!
	return_if_fail(m_data.get());

	// Get the new value ...
	const std::string value = Combo("combo").Entry().GetText();

	// If the value hasn't changed, we're done ...
	if(value == m_data->value())
		return;

	// Record the command for posterity (tutorials) ...
	k3d::application().command_signal().emit(this, k3d::icommand_node::command_t::USER_INTERFACE, "value", value);

	// Turn this into an undo/redo -able event ...
	if(m_StateRecorder)
		m_StateRecorder->start_recording(k3d::create_state_change_set());

	// Store the new value ...
	m_data->set_value(value);

	// Turn this into an undo/redo -able event ...
	if(m_StateRecorder)
		m_StateRecorder->commit_change_set(m_StateRecorder->stop_recording(), m_StateChangeName + ' ' + '"' + value + '"');\
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == "value")
		{
			sdpGtkEditable editable(Combo("combo").Entry());
			editable.InteractiveShow(k3d::application().options().tutorial_speed(), true);
			editable.InteractiveWarpPointer(k3d::application().options().tutorial_speed(), true, false);
			editable.SetText(Arguments.c_str());

			return true;
		}

	return base::execute_command(Command, Arguments);
}

} // namespace combo_box

} // namespace k3d


