// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief An object that converts a scalar to a string using printf() style formatting
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/object.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>

#include <boost/format.hpp>

namespace libk3dcore
{

class format_scalar_implementation : 
	public k3d::persistent<k3d::object> 
{
	typedef k3d::persistent<k3d::object>  base;
public:
	format_scalar_implementation(k3d::idocument& Document) : 
		base(Document),
		m_input(k3d::init_name("input") + k3d::init_description("Input [double]") + k3d::init_value(0.0) + k3d::init_document(Document)),
		m_format(k3d::init_name("format") + k3d::init_description("Format [string]") + k3d::init_value(std::string("%f")) + k3d::init_document(Document)),
		m_output("output", "Output [string]", k3d::method_call(*this, &format_scalar_implementation::get_value))
	{
		register_property(m_input);
		register_property(m_format);
		register_property(m_output);

		enable_serialization(k3d::persistence::proxy(m_input));
		enable_serialization(k3d::persistence::proxy(m_format));

		m_input.changed_signal().connect(m_output.changed_signal().slot());
		m_format.changed_signal().connect(m_output.changed_signal().slot());
	}
	
	std::string get_value()
	{
		try
			{
				return (boost::format(m_format.property_value()) % m_input.property_value()).str();
			}
		catch(std::exception& e)
			{
				std::cerr << error << e.what() << std::endl;
				return std::string();
			}
	}
	
	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<format_scalar_implementation> > factory(
			k3d::uuid(0x3af7d777, 0x691d41b0, 0xaa801d59, 0x7ee4180e),
			"FormatScalar", 
			"Converts a scalar to a string using printf() style formatting", 
			"Objects", 
			k3d::iplugin_factory::STABLE);
		return factory;
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}
	
private:
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_input;
	k3d_data_property(std::string, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_format;
	k3d::computed_property<std::string, k3d::method_call_t<format_scalar_implementation, std::string> > m_output;
};

k3d::iplugin_factory& format_scalar_factory()
{
	return format_scalar_implementation::get_factory();
}

} //namespace libk3dcore
