// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace
{

/// Returns every point within the polyhedron that that is an "interior" point, along with the list of edges that adjoin it
typedef std::map<k3d::point*, k3d::polyhedron::edges_t > point_edges_map_t;
point_edges_map_t interior_points(k3d::polyhedron& Polyhedron)
{
	point_edges_map_t results;

	for(k3d::polyhedron::edges_t::iterator edge = Polyhedron.edges.begin(); edge != Polyhedron.edges.end(); ++edge)
		{
			if(0 == (*edge)->vertex)
				continue;

			if(results.count((*edge)->vertex))
				continue;

			k3d::polyhedron::edges_t edges;
			for(k3d::split_edge* edge2 = *edge; edge2 && edge2->companion; edge2 = edge2->companion->face_clockwise)
				{
					edges.push_back(edge2);

					if(edge2->companion->face_clockwise == *edge)
						{
							results[(*edge)->vertex] = edges;
							break;
						}
				}
		}

	return results;
}

} // namespace

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// smooth_points_implementation

class smooth_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::mouse_event_observer
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	smooth_points_implementation(k3d::idocument& Document) :
		base(Document),
		k3d::mouse_event_observer("LMB drag to smooth points"),
		m_smoothing(k3d::init_name("smoothing") + k3d::init_description("Smoothing [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_smoothing));

		register_property(m_smoothing);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &smooth_points_implementation::on_reset_geometry));
		m_smoothing.changed_signal().connect(SigC::slot(*this, &smooth_points_implementation::on_reshape_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &smooth_points_implementation::on_create_geometry));
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		const double xfactor = (Current[0] - Last[0]);
		const double yfactor = (Current[1] - Last[1]);

		m_smoothing.set_value(m_smoothing.value() + (std::fabs(xfactor) >= std::fabs(yfactor) ? xfactor : yfactor));

		return true;
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.polyhedra.size() == Target.polyhedra.size());
		assert(Source.points.size() == Target.points.size());

		const double smoothing = m_smoothing.property_value();

		point_edges_map_t points_edges;

		for(unsigned int i = 0; i != Source.polyhedra.size(); ++i)
			{
				points_edges = ::interior_points(*Source.polyhedra[i]);
			}

		const bool ignore_selection = !k3d::contains_selection(Source);
		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				Target.points[i]->position = Source.points[i]->position;

				if(!ignore_selection && !Source.points[i]->selected)
					continue;

				point_edges_map_t::iterator p = points_edges.find(Source.points[i]);
				if(p == points_edges.end())
					continue;

				k3d::polyhedron::edges_t& edges = p->second;
				if(edges.empty())
					continue;

				k3d::vector3 position(0, 0, 0);
				for(k3d::polyhedron::edges_t::const_iterator edge = edges.begin(); edge != edges.end(); ++edge)
					position += (*edge)->companion->vertex->position;
				Target.points[i]->position = k3d::mix(Source.points[i]->position, position / edges.size(), smoothing);
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<smooth_points_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x3d6d1c61, 0x706f4968, 0x93462489, 0x9a3134a7),
				"SmoothPoints",
				"Smooths mesh points by averaging their positions with their neighbors'",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_smoothing;
};

/////////////////////////////////////////////////////////////////////////////
// smooth_points_factory

k3d::iplugin_factory& smooth_points_factory()
{
	return smooth_points_implementation::get_factory();
}

} // namespace libk3ddeformation


