// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
	\brief Implements the OpenEXR writer K-3D object
	\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/ibitmap_write_format.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/module.h>
#include <k3dsdk/string_modifiers.h>

#include <ImfOutputFile.h>
#include <ImfChannelList.h>

namespace
{

/////////////////////////////////////////////////////////////////////////////
// exr_writer_implementation

class exr_writer_implementation :
	public k3d::ifile_format,
	public k3d::ibitmap_write_format,
	public k3d::ideletable
{
public:
	exr_writer_implementation()
	{
	}

	unsigned long priority()
	{
		return 10;
	}

	bool query_can_handle(const boost::filesystem::path& File)
	{
		return_val_if_fail(!File.empty(), false);
		return "exr" == k3d::file_extension(File);
	}

	bool write_file(const boost::filesystem::path& File, const k3d::bitmap& Bitmap)
	{
		// Sanity checks ...
		return_val_if_fail(!File.empty(), false);

		std::cerr << info << "Write " << File.native_file_string() << " using OpenEXRWriter" << std::endl;

		try
		{
			Imf::Header header(Bitmap.width(), Bitmap.height());
			header.channels().insert("R", Imf::Channel(Imf::HALF));
			header.channels().insert("G", Imf::Channel(Imf::HALF));
			header.channels().insert("B", Imf::Channel(Imf::HALF));
			header.channels().insert("A", Imf::Channel(Imf::HALF));

			Imf::OutputFile file(File.native_file_string().c_str(), header);
			
			const Imath::Box2i dw = file.header().dataWindow();
			const int width  = dw.max.x - dw.min.x + 1;
			const int height = dw.max.y - dw.min.y + 1;

			const k3d::pixel* pixels = Bitmap.data();
			
			Imf::FrameBuffer frameBuffer;

			frameBuffer.insert("R",
					   Imf::Slice(Imf::HALF,
						      (char *) &pixels[0].red,
						      sizeof(*pixels) * 1,
						      sizeof(*pixels) * width));

			frameBuffer.insert("G",
					   Imf::Slice(Imf::HALF,
						      (char *) &pixels[0].green,
						      sizeof(*pixels) * 1,
						      sizeof(*pixels) * width));

			frameBuffer.insert("B",
					   Imf::Slice(Imf::HALF,
						      (char *) &pixels[0].blue,
						      sizeof(*pixels) * 1,
						      sizeof(*pixels) * width));

			frameBuffer.insert("A",
					   Imf::Slice(Imf::HALF,
						      (char *) &pixels[0].alpha,
						      sizeof(*pixels) * 1,
						      sizeof(*pixels) * width));

			file.setFrameBuffer(frameBuffer);
			file.writePixels(height);	
		}
		catch (const std::exception &exc)
		{
			std::cerr << error << "OpenEXRWriter: Caught exception " <<  exc.what() << std::endl;
			return false;
		}

		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::application_plugin<exr_writer_implementation>,
			k3d::interface_list<k3d::ibitmap_write_format> > factory(
			k3d::uuid(0xf2e8f703, 0xecbf42ba, 0xbf353410, 0x0016b4cf),
			"OpenEXRWriter",
			"OpenEXR (*.exr)",
			"Bitmap File Formats");

		return factory;
	}
};

}//namespace

namespace libk3dopenexr
{

/////////////////////////////////////////////////////////////////////////////
// exr_writer_factory

k3d::iplugin_factory& exr_writer_factory()
{
	return ::exr_writer_implementation::get_factory();
}

} // namespace libk3dopenexr
