/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * autor
 * Micha Podsiadlik
 * michal at gov.one.pl
 *
 * obsluga maildir:
 * Konrad Klimaszewski
 * klimasz at if.pw.edu.pl>
 */

#include <qlineedit.h>
#include <qfiledialog.h>
#include <qvgroupbox.h>
#include <qlistbox.h>

#include "config_file.h"
#include "icons_manager.h"
#include "message_box.h"
#include "misc.h"
#include "../notify/notify.h"

#include <stdlib.h>

#include "mail.h"
#include "accountdialog.h"

#include "debug.h"

extern "C" int mail_init()
{
	mail = new Mail();
	MainConfigurationWindow::registerUiFile(dataPath("kadu/modules/configuration/mail.ui"), mail);
	notification_manager->registerEvent("Mail", "New mail", CallbackNotRequired);

	return 0;
}

extern "C" void mail_close()
{
	MainConfigurationWindow::unregisterUiFile(dataPath("kadu/modules/configuration/mail.ui"), mail);
	notification_manager->unregisterEvent("Mail");
	delete mail;
	mail = 0;
}

Mail::Mail()
{
	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), SLOT(checkmail()));

	import_0_5_0_Configuration();

	for(int i=0;;i++)
	{
		QString config_section;
		config_section.sprintf("Account_%i", i);
		if(config_file.readEntry("Mail", config_section + "_Name").length())
		{
			Pop3Proto* acc=new Pop3Proto(
					config_file.readEntry("Mail", config_section + "_Name"),
					config_file.readEntry("Mail", config_section + "_Server"),
					config_file.readNumEntry("Mail", config_section + "_ServerPort", 110),
					config_file.readEntry("Mail", config_section + "_User"),
					pwHash(config_file.readEntry("Mail", config_section + "_Password", "")),
					(SecureType)config_file.readNumEntry("Mail", config_section + "_Encryption"));
			kdebugm(KDEBUG_INFO,
			"Account %i: Name: %s, Server: %s, ServerPort: %i, User: %s, Encryption: %i\n", i,
					config_file.readEntry("Mail", config_section + "_Name").latin1(),
					config_file.readEntry("Mail", config_section + "_Server").latin1(),
					config_file.readNumEntry("Mail", config_section + "_ServerPort", 110),
					config_file.readEntry("Mail", config_section + "_User").latin1(),
					config_file.readNumEntry("Mail", config_section + "_Encryption"));
			
			connect(acc, SIGNAL(done(int,int,int,QString)),	SLOT(printstat(int,int,int,QString)));
			acc->setLastmails(config_file.readNumEntry("Mail", config_section + "_Last"));
			accounts.append(acc);
		}
		else
			break;
	}

	createDefaultConfiguration();
	timer->start(config_file.readNumEntry("Mail", "Interval", 60)*1000);
	checkmail();
	kdebugm(KDEBUG_INFO, "Mail started\n");
}

Mail::~Mail()
{
	kdebugf();
	// i zapisz wszystko na koniec
	configurationWindowApplied();
	delete timer;
	kdebugm(KDEBUG_INFO, "Mail stopped\n");
}

void Mail::checkmail()
{
	kdebugf();
	if(config_file.readBoolEntry("Mail", "LocalMaildir"))
		maildir();

	Pop3Proto* acc;
	for(acc=accounts.first();acc;acc=accounts.next())
		acc->getStats();
}

void Mail::printstat(int last, int total, int size, QString name)
{
	kdebugf();
	
	if(total>last)
	{
		const UserListElements ules;
		Notification *notification = new Notification("Mail", "Message", ules);
		notification->setText(formatmessage(last, total, size, name));
		notification_manager->notify(notification);
		/*
		 * uruchom klienta poczty
		 */
		if(config_file.readBoolEntry("Mail", "RunClient"))
		{
			openMailClient(QString(""));
		}
	}
}

QString Mail::formatmessage(int last, int total, int size, QString name)
{
	QString msg, msgsize;
	/*
	 * %n liczba nowych wiadomoci
	 * %t liczba wszystkich wiadomoci
	 * %s rozmiar
	 * %a nazwa konta
	 */
	msg=config_file.readEntry("Mail", "Format");
	//ciekawe kto ma tak wielk poczte ??
	if(size>1073741824)
		msgsize.sprintf("%.2f GB", (float)size/1073741824);
	else if(size>1048576)
		msgsize.sprintf("%.2f MB", (float)size/1048576);
	else if(size>1024)
		msgsize.sprintf("%.2f kB", (float)size/1024);
	else
		msgsize.sprintf("%i B", size);

	msg.replace("%n", QString::number(total-last));
	msg.replace("%t", QString::number(total));
	msg.replace("%s", msgsize);
	msg.replace("%a", name);
	return msg;
}

void Mail::maildir()
{
	QString path;
	unsigned int totalsize = 0;
	int last;

	path = config_file.readEntry("Mail", "MaildirPath");
	last = config_file.readNumEntry("Mail", "LastMailDir", 0);

	path += "/new";
	if(path[0] == '~')
		path.replace(0, 1, QDir::homeDirPath());
	path = QDir::cleanDirPath(path);

	QDir dir(path);

	if(!dir.exists())
		MessageBox::msg(tr("Maildir not found!"), true, "Warning");
	else if(!dir.isReadable())		
		MessageBox::msg(tr("Maildir is not readable!"), true, "Warning");
	else 
	{
		const QFileInfoList* filelist = dir.entryInfoList();
		if ( filelist ) 
		{
			QFileInfoListIterator iter( *filelist );
			QFileInfo* file;
				while( (file=iter.current()) != 0 ) 
				{
					++iter;
					if ( file->fileName() == "." || file->fileName() == ".." )
						; // nothing
					else
						totalsize += file->size();
				}
		}
		config_file.writeEntry("Mail", "LastMailDir", last);
		printstat(last, dir.count()-2, totalsize, "MailDir");
	}
}

void Mail::onSelectMaildir()
{
	kdebugf();
	QString dir = QFileDialog::getExistingDirectory();
	if(dir!=QString::null)
		maildirLineEdit->setText(dir);
}

void Mail::onAddButton()
{
	kdebugf();
	Pop3Proto* acc=new Pop3Proto("New", "", 0, "", "");
	AccountDialog* dlg=new AccountDialog(acc);
	if(dlg->exec()==QDialog::Accepted)
	{
		connect(acc, SIGNAL(done(int,int,int,QString)),	SLOT(printstat(int,int,int,QString)));
		acc->setLastmails(0);
		accounts.append(acc);
		updateList();
	}
	else
		delete acc;
}

void Mail::onEditButton()
{
	kdebugf();
	Pop3Proto* acc;

	for(acc=accounts.first();acc;acc=accounts.next())
		if(acc->getName()==accountsListBox->currentText())
		{
			AccountDialog* dlg=new AccountDialog(acc);
			dlg->exec();
			updateList();
		}
}

void Mail::onRemoveButton()
{
	kdebugf();
	Pop3Proto* acc;
	
	for(acc=accounts.first();acc;acc=accounts.next())
		if(acc->getName()==accountsListBox->currentText())
		{
			kdebugm(KDEBUG_INFO, "removing %s\n", acc->getName().latin1());
			acc->disconnect(this);
			accounts.remove(acc);
			updateList();
		}

}

void Mail::updateList()
{
	accountsListBox->clear();

	Pop3Proto* acc;
	for(acc=accounts.first();acc;acc=accounts.next())
		accountsListBox->insertItem(acc->getName());
}

void Mail::import_0_5_0_Configuration()
{
	ConfigFile* oldConfig = new ConfigFile(ggPath("mail.conf"));

	for(int i=0;;i++)
	{
		QString config_section;
		config_section.sprintf("Account_%i", i);
		if(!oldConfig->readEntry(config_section, "Name", "").isEmpty())
		{
			config_file.addVariable("Mail", config_section + "_Name", oldConfig->readEntry(config_section, "Name"));
			oldConfig->removeVariable(config_section, "Name");

			config_file.addVariable("Mail", config_section + "_Server", oldConfig->readEntry(config_section, "Server"));
			oldConfig->removeVariable(config_section, "Server");

			config_file.addVariable("Mail", config_section + "_ServerPort", oldConfig->readEntry(config_section, "ServerPort", "110"));
			oldConfig->removeVariable(config_section, "ServerPort");

			config_file.addVariable("Mail", config_section + "_User", oldConfig->readEntry(config_section, "User"));
			oldConfig->removeVariable(config_section, "User");

			config_file.addVariable("Mail", config_section + "_Password", oldConfig->readEntry(config_section, "Password", ""));
			oldConfig->removeVariable(config_section, "Password");

			config_file.addVariable("Mail", config_section + "_Encryption", oldConfig->readEntry(config_section, "Encryption"));
			oldConfig->removeVariable(config_section, "Encryption");

			config_file.addVariable("Mail", config_section + "_Last", oldConfig->readEntry(config_section, "Last"));
			oldConfig->removeVariable(config_section, "Last");
		}
		else
			break;
	}
	config_file.addVariable("Mail", "MaildirPath", oldConfig->readEntry("Mail", "MaildirPath", ""));
	config_file.addVariable("Mail", "LastMailDir", oldConfig->readEntry("Mail", "LastMailDir", ""));
	config_file.addVariable("Mail", "Format", oldConfig->readEntry("Mail", "Format", ""));
	config_file.addVariable("Mail", "RunClient", oldConfig->readEntry("Mail", "RunClient", ""));

	//domylne ustawienia dymkw z poprzedniej konfiguracji
	config_file.addVariable("Hints", "Event_Mail_fgcolor", oldConfig->readEntry("Mail", "HintColor", ""));
	config_file.addVariable("Hints", "Event_Mail_bgcolor", oldConfig->readEntry("Mail", "HintBackground", ""));
	config_file.addVariable("Hints", "Event_Mail_timeout", oldConfig->readEntry("Mail", "HintTimeout", ""));
	oldConfig->sync();
	delete oldConfig;
}

void Mail::mainConfigurationWindowCreated(MainConfigurationWindow *mainConfigurationWindow)
{
	connect(mainConfigurationWindow, SIGNAL(configurationWindowApplied()), this, SLOT(configurationWindowApplied()));

	ConfigGroupBox *accountsGroupBox = mainConfigurationWindow->configGroupBox("Mail", "General", "Accounts");
	
	QHBox *accountsHBox = new QHBox(accountsGroupBox->widget());
	accountsHBox->setSpacing(5);

	accountsListBox = new QListBox(accountsHBox);

	QWidget *buttons = new QWidget(accountsHBox);
	QVBoxLayout *buttonsLayout = new QVBoxLayout(buttons);
	buttonsLayout->setSpacing(5);

	QPushButton *add = new QPushButton(icons_manager->loadIconSet("AddSelectPathDialogButton"), tr("Add"), buttons);
	QPushButton *remove = new QPushButton(icons_manager->loadIconSet("RemoveSelectPathDialogButton"), tr("Remove"), buttons);
	QPushButton *edit = new QPushButton(icons_manager->loadIconSet("ChangeSelectPathDialogButton"), tr("Edit"), buttons);

	buttonsLayout->addWidget(add);
	buttonsLayout->addWidget(remove);
	buttonsLayout->addWidget(edit);

	connect(add, SIGNAL(clicked()), this, SLOT(onAddButton()));
	connect(remove, SIGNAL(clicked()), this, SLOT(onRemoveButton()));
	connect(edit, SIGNAL(clicked()), this, SLOT(onEditButton()));
	updateList();
	accountsGroupBox->addWidgets(0, accountsHBox);

	ConfigGroupBox *maildirGroupBox = mainConfigurationWindow->configGroupBox("Mail", "General", "Maildir");

	QHBox *maildirHBox = new QHBox(maildirGroupBox->widget());
	QLabel *label = new QLabel(tr("Maildir path: "), maildirHBox);
	maildirLineEdit = new QLineEdit(maildirHBox);
	maildirLineEdit->setText(config_file.readEntry("Mail", "MaildirPath", "~/Maildir"));
	QPushButton *selectFile = new QPushButton(QIconSet(icons_manager->loadIconSet("OpenFile")), "", maildirHBox);
	connect(selectFile, SIGNAL(clicked()), this,SLOT(onSelectMaildir()));
	maildirGroupBox->addWidgets(0, maildirHBox);

	connect(mainConfigurationWindow->widgetById("mail/local_maildir"), SIGNAL(toggled(bool)), maildirLineEdit, SLOT(setEnabled(bool)));

	connect(mainConfigurationWindow->widgetById("mail/local_maildir"), SIGNAL(toggled(bool)), selectFile, SLOT(setEnabled(bool)));
}

void Mail::configurationWindowApplied()
{
	int i=0;
	Pop3Proto* acc;
	QString config_section;

	for(acc=accounts.first();acc;acc=accounts.next())
	{
		config_section.sprintf("Account_%i", i);
		
		config_file.writeEntry("Mail", config_section + "_Name", acc->getName());
		config_file.writeEntry("Mail", config_section + "_Server", acc->getHost());
		config_file.writeEntry("Mail", config_section + "_ServerPort", acc->getPort());
		config_file.writeEntry("Mail", config_section + "_User", acc->getUser());
		config_file.writeEntry("Mail", config_section + "_Password", pwHash(acc->getPassword()));
		config_file.writeEntry("Mail", config_section + "_Last", acc->getLastmails());	
		config_file.writeEntry("Mail", config_section + "_Encryption", acc->getEncryption());

		i++;
	}
	// usuwa (narazie) sie nie da ale puste mona ustawi
	config_section.sprintf("Account_%i", i);
	config_file.removeVariable("Mail", config_section + "_Name");	
}

void Mail::createDefaultConfiguration()
{
	config_file.addVariable("Mail", "LastMailDir", "0");
	config_file.addVariable("Mail", "MaildirPath", "~/Maildir");
	config_file.addVariable("Mail", "LocalMaildir", "false");
	config_file.addVariable("Mail", "Format", tr("You have %n new mail(s) on %a and total you have %t mail(s) witch total size %s"));
	config_file.addVariable("Mail", "RunClient", "false");
	config_file.addVariable("Mail", "Interval", "60");
	//domylne ustawienia dymkw z poprzedniej konfiguracji
	config_file.addVariable("Notify", "Mail_Hints", "true");
	config_file.addVariable("Hints", "Event_Mail_fgcolor", "black");
	config_file.addVariable("Hints", "Event_Mail_bgcolor", "light grey");
	config_file.addVariable("Hints", "Event_Mail_timeout", "10");

}
Mail* mail;
