/*
 * stringParsing.h
 * A handy string parsing function.
 *
 * Copyright (c) 2000, 2001, 2002 The University of Utah and the Flux Group.
 * All rights reserved.
 *
 * @JANOSVM_KAFFE_MISC_LICENSE@
 */

#ifndef __string_parsing_h
#define __string_parsing_h

/*
 * This module provides a function for parsing simple strings.  Its a somewhat
 * unusual function since its arguments are treated like a script to be
 * executed over the input string.  This gives us a tremendous amount of
 * flexibility and the resulting code tends to be much easier to understand
 * compared to the corresponding C code.
 */

/*
 * A parsedString structure is used to store a value that was generated by a
 * parse without modifying the captured string or copying out the data.
 *
 * `data' - Pointer to the start of the captured string.
 * `len' - The length of the string.
 */
typedef struct _parsedString {
	char *data;
	int len;
} parsedString;

/*
 * These are the various commands that the parseString function understands.
 */
enum {
	/* () Mark the end of a block */
	SPO_End,
	/* () Noop */
	SPO_Noop,
	/*
	 * (char *) Scan the input until one of the characters in the argument
	 * string is found
	 */
	SPO_OneOf,
	/* () Start a do/while loop */
	SPO_Do,
	/*
	 * (char *) Start a new block and scan the input looking for the given
	 * string.  If its found then execute the code in the block, otherwise
	 * the cursor is left at its current position and the block is skipped.
	 */
	SPO_Cond,
	/*
	 * () Start a new block and execute any code if the input string isn't
	 * empty.
	 */
	SPO_NotEmpty,
	/*
	 * (char *, char *) Check for the given string in the input and
	 * continue looping otherwise the loop will stop.  NOTE: The loop
	 * doesn't continue until it hits the SPO_End at the end of the block,
	 * any SPO_Handle's should be placed after the while to ensure that all
	 * of the values are captured.
	 */
	SPO_While,
	/* () Continues looping while theres white space. */
	SPO_WhileSpace,
	/*
	 * (int (*)(void *), void *) Call the given function with the given
	 * argument.  Useful for processing the value extracted during a loop.
	 */
	SPO_Handle,
	/*
	 * (char *) Scan the input string until the given string is found.  If
	 * the string isn't found the function will return with an error.
	 */
	SPO_Expect,
	/* () Expect white space */
	SPO_ExpectSpace,
	/* (parsedString *) Capture a string value from the input */
	SPO_String,
	/* (parsedString *) Capture a non-empty string value from the input */
	SPO_NonEmptyString,
	/* (char *) Capture a character */
	SPO_Character,
	/* (char *) Capture a byte */
	SPO_Byte,
	/* (char *) Capture a byte encoded in hex */
	SPO_HexByte,
	/* (short *) Capture a short */
	SPO_Short,
	/* (short *) Capture a short encoded in hex */
	SPO_HexShort,
	/* (int *) Capture an integer */
	SPO_Integer,
	/* (long long *) Capture an integer */
	SPO_LongInteger,
	/* (int *) Capture an integer encoded in hex */
	SPO_HexInteger,
	/* (long long *) Capture an integer encoded in hex */
	SPO_HexLongInteger,
	/* (float *) Capture a float from the input */
	SPO_Float,
	/* (double *) Capture a double from the input */
	SPO_Double,
	/* (int *) Increment the counter on each value found */
	SPO_Count,
	/* () Skip any preceding white space. */
	SPO_SkipSpace
};

#define SPO_MAX_ARG_COUNT 2

typedef struct _stringScript {
	int op;
	void *args[0];
} stringScript;

typedef struct _parseErrorInfo {
	char *position;
	int op;
	void *args[SPO_MAX_ARG_COUNT];
} parseErrorInfo;

#define STRING_SCRIPT_SIZE(arg_count) \
	(sizeof(stringScript) + (sizeof(void *) * (arg_count)))
#define STRING_SCRIPT_NEXT(ss_ptr, arg_count) \
	(stringScript *)(((char *)(ss_ptr)) + STRING_SCRIPT_SIZE(arg_count))

/*
 * This is a somewhat all-purpose function for parsing strings with a
 * relatively simple format.  It arguments are an input string and a list of
 * arguments that determine how the string should be parsed and what values
 * should be captured from the string.  For examples of its usage, look at
 * the test functions in the implementation.
 */
int parseString(char *str, int op, ...);
int parseString_error(parseErrorInfo *pe, char *str, int op, ...);
int parseSubString(parsedString *ps, int op, ...);
int parseString_script(char *str, stringScript *ss, ...);
int parseString_script_values(char *str, stringScript *ss, void **values);

/*
 * Compare a parsedString and a regular C string.
 *
 * `ps' - An initialized out parsedString structure.
 * `str' - The string to compare `ps' with.
 * returns - An integer greater than, equal to, or less than zero, if `ps'
 *  is lexically less than, equal to, or greater than `str'.
 */
int cmpPStrStr(parsedString *ps, char *str);

/*
 * "Promote" a parsed string into a null terminated C string.
 *
 * `ps' - The parsed string to promote.
 * returns - A gc_malloc'ed chunk of memory holding the string.
 */
char *promoteParsedString(parsedString *ps);

/*
 * AddModule: stringParsingModule
 *   Comment: For testing purposes only...
 */

#endif
