/*
    kfile_snes.h - SNES(Super Nintendo Entertainment System) Rom image KFile Plugin

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

     *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#ifndef __KFILE_SNES_H__
#define __KFILE_SNES_H__

/**
 * Note: For further information look into <$KDEDIR/include/kfilemetainfo.h>
 */
#include <kfilemetainfo.h>
#include <qcstring.h> // For QByteArray

class QStringList;
class QFile;
class QDataStream;

/**
 * @brief Super Nintendo ROM File plugin.
 * Retrieve all meta information included in SNES ROM.
 *
 * This plugin is not the most efficient ever but
 * works. SNES ROM file are a pain to support.
 *
 * @author Michaël Larouche <michael.larouche@kdemail.net>
 */
class KSnesFilePlugin : public KFilePlugin
{
	Q_OBJECT
public:
	enum RomMakeupMask { HiRom = 0x01, FastRom = 0x70 };
	
	KSnesFilePlugin( QObject *parent, const char *name, const QStringList& args );
		
	virtual bool readInfo( KFileMetaInfo& info, uint what);
private:
	/**
	 * Retrieve maker(editor) code stored in HEX ASCII.
	 * This is only called when makerCoder == 0x33.
	 *
	 * @param file Reference to a QFile, needed for file access.
	 * @param stream Reference to a QDataStream, needed for file access.
	 *
	 * @return the readed maker(editor) code.
	 */
	int getMakerCode(QFile &file, QDataStream &stream);
	/**
	 * Guess the best header location based on 4 hardcoded
	 * location.
	 *
	 * Needed because SNES ROM file can be HiROM or LoROM and
	 * can be with a 512byte header or not.
	 *
	 * @param file Reference to a QFile, needed for file access.
	 * @param stream Reference to a QDataStream, needed for file access.
	 *
	 * @return the best location.
	 */
	int getBestHeaderLocation(QFile &file, QDataStream &stream);
	/**
	 * Helper function that check the validity of information based on given
	 * location. More the score is higher, more the information is valid.
	 * @return the score.
	 */
	int checkInformationValidity(int location, QFile &file, QDataStream &stream);
	/**
	 * Helper function to check if the SNES Internal name is inside
	 * the printable ASCII range.
	 */
	bool canPrint(const QByteArray &charArray);
};

#endif // __KFILE_SNES_H__ 
