/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#include "VMState.h"
#include "VM.h"
#include "ValueFuns.h"
#include "KayaAPI.h"
#include "stdfuns.h"

#include <assert.h>
#include <ext/hash_map>
#include <cstring>

/// function map is global to all threads.
func* m_funmap[1];
kint m_funmapsize;
kint m_funmaphash;

// Defined by compiling Builtins.k, so obviously any binary of a kaya
// program needs to be linked to at least Builtins.o!
#define INTERNAL_ERROR K_ns__D_Builtins__D_InternalError
extern char* INTERNAL_ERROR;

void initFunMap(kint sz, kint fmhash)
{
    func* funcs = (func*)malloc(sizeof(func)*sz);
    m_funmap[0] = funcs;
    m_funmapsize = sz;
    m_funmaphash = fmhash;
}

void addToFunMap(kint id, func fn)
{
  (m_funmap[0])[id] = fn;
}

func getFn(kint id)
{
  if (id > 0 && id < m_funmapsize) {
    return (m_funmap[0])[id];
  } else {
    // FIXME: do something sensible with this result in the calling context
    return NULL;
  }
}

kint getFnID(func fn)
{
    kint i;
    for (i=1;i<m_funmapsize;++i) {
	if (m_funmap[0][i] == fn) { return i; }
    }
    return -1;
}

kint getFnHash()
{
    return m_funmaphash;
}

VMState::VMState(bool panic):m_return(NULL,KVT_NULL) {
    m_stackalloc=128;
    m_callstackalloc = 1024;
    m_globalloc=128;
    if (!panic) {
	m_valstack = (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*m_stackalloc);
	// Make sure stack is clear.
	memset(m_valstack,0,sizeof(Value*)*m_stackalloc);
	m_maxmem = 0;
	m_backtrace = (CallStackEntry**)GC_MALLOC_UNCOLLECTABLE(sizeof(CallStackEntry*)*m_callstackalloc);
	// Allocate space all at once (max out at <m_stackalloc*10>)
	for(int i=0;i<m_callstackalloc;++i) {
	    m_backtrace[i]=new CallStackEntry;
	}
    }
    else {
	m_valstack = (Value**)malloc(100); // Tiny, if we're panicking.
	m_backtrace = (CallStackEntry**)malloc(100); // Tiny, if we're panicking.
	for(int i=0;i<5;++i) {
	    m_backtrace[i]=new CallStackEntry;
	}
    }
    m_stackptr=m_valstack;
    m_btptr=m_backtrace;

    // value memory allocation pools
    m_vpool = new ValuePool();
    m_spool = new StringPool();
    m_upool = new UnionPool();
    m_apool = new ArrayPool();
    m_rpool = new RealPool();

    lineno(L"START",0);
}

void VMState::finish()
{
    if (!emptyStack()) {
	cerr << "Warning: Stack size is non-zero (" << m_stackptr << "," << m_valstack << ")" << endl;
    }
#ifndef NOCHECK
    if (m_btptr!=m_backtrace) {
	cerr << "Warning: Call stack size is non-zero (" << m_btptr << "," << m_backtrace << ")" << endl;
    }
#endif
}

bool VMState::emptyStack()
{
    return m_stackptr==m_valstack;
}

void VMState::push2(Value* val, Value* val2)
{
    // FIXME: Grow stack if necessary? Might be quite an overhead... maybe 
    // make it an option.
    *(m_stackptr++)=val;
    *(m_stackptr++)=val2;
}

void VMState::push3(Value* val, Value* val2, Value* val3)
{
    // FIXME: Grow stack if necessary? Might be quite an overhead... maybe 
    // make it an option.
    *(m_stackptr++)=val;
    *(m_stackptr++)=val2;
    *(m_stackptr++)=val3;
}

void VMState::push4(Value* val, Value* val2, Value* val3, Value* val4)
{
    // FIXME: Grow stack if necessary? Might be quite an overhead... maybe 
    // make it an option.
    *(m_stackptr++)=val;
    *(m_stackptr++)=val2;
    *(m_stackptr++)=val3;
    *(m_stackptr++)=val4;
}

kint VMState::tag()
{
  valtype t = (*(m_stackptr-1))->getType();
  if (t >= KVT_UNION) {
    return t-KVT_UNION;
  } else if (t == KVT_CUNION) {
    return (*(m_stackptr-1))->getInt();
  } else {
    // no extra cost to doing this check now...
    kaya_rtsError(GETTING_TAG_FROM_NON_UNION);
    // doesn't matter what we return, but it keeps the compiler happy
    return 0;
  }
}

void VMState::doAppend()
{
    Value* str2=doPop();
    Value* str1=doPop();
    String* str=NEWSTRING(str1->getString()->getVal());
    str->append(str2->getString());
    push(MKPVAL(m_vpool,(void*)str,KVT_STRING));
}

void VMState::doAppendChar(const wchar_t x)
{
    // append x onto top stack item
    (*(m_stackptr-1))->getString()->append(x);
    // removed explicit coercion, so need to correct stack
    (*(m_stackptr-2))->setPtr(*(m_stackptr-1));
    discardPop();
}

void VMState::doAppendStr(const wchar_t* x)
{
    // append x onto top stack item
    (*(m_stackptr-1))->getString()->append(x);
}

void VMState::doEqExcept(bool inv)
{
    Value* e2=doPop();
    Value* e1=doPop();
    bool eq = (e2->getExcept()->eq(e1->getExcept()));
    if (inv) { eq = !eq; }
    if (eq) {  push(one); }
    else {  push(zero); }
}

void VMState::doEqString()
{
    Value* e2=doPop();
    Value* e1=doPop();
    bool eq = (e2->getString()->eq(e1->getString()));
    if (eq) {  push(one); }
    else {  push(zero); }
}

void VMState::doNeString()
{
    Value* e2=doPop();
    Value* e1=doPop();
    bool eq = (e2->getString()->eq(e1->getString()));
    if (eq) {  push(zero); }
    else {  push(one); }
}

void VMState::doEqString(const wchar_t* str)
{
    Value* e1=doPop();
    bool eq = (e1->getString()->eq(str));
    if (eq) {  push(one); }
    else {  push(zero); }
}

void VMState::doNeString(const wchar_t* str)
{
    Value* e1=doPop();
    bool eq = (e1->getString()->eq(str));
    if (eq) {  push(zero); }
    else {  push(one); }
}

bool VMState::doTopStrEq(const wchar_t* str)
{
    wchar_t* topstr = (*(m_stackptr-1))->getString()->getVal();
    return (!wcscmp(topstr,str));
}

void VMState::int2str()
{
    // Need to do this on a copy or we break types.
    Value* y = doPop();
    Value* x = (*(m_stackptr-1));
    x->setPtr(y);
    x->int2str(m_spool);
}

void VMState::real2str()
{
    // Need to do this on a copy or we break types.
    Value* y = doPop();
    Value* x = (*(m_stackptr-1));
    x->setPtr(y);
    x->real2str(m_spool);
}

void VMState::str2int()
{
    // Need to do this on a copy or we break types.
    Value* y = doPop();
    Value* x = (*(m_stackptr-1));
    x->setPtr(y);
    x->str2int();
}

void VMState::str2real()
{
    Value* y = doPop();
    Value* x = (*(m_stackptr-1));
    x->setPtr(y);
    x->str2real(m_rpool);
}

void VMState::chr2str()
{
    // Need to do this on a copy or we break types.
    Value* y = doPop();
    Value* x = (*(m_stackptr-1));
    x->setPtr(y);
    x->chr2str(m_spool);
}

void VMState::bool2str()
{
    // Need to do this on a copy or we break types.
    Value* y = doPop();
    Value* x = (*(m_stackptr-1));
    x->setPtr(y);
    x->bool2str(m_spool);
}

void VMState::str2chr()
{
//    valstack[stacksize-1]->str2chr();
    assert(false);
}

void VMState::real2int()
{
    // Need to do this on a copy or we break types.
    Value* y = doPop();
    Value* x = (*(m_stackptr-1));
    x->setPtr(y);
    x->real2int();
}

void VMState::int2real()
{
    // Need to do this on a copy or we break types.
    Value* y = doPop();
    Value* x = (*(m_stackptr-1));
    x->setPtr(y);
    x->int2real(m_rpool);
}

double VMState::topint2real()
{
    // Need to do this on a copy or we break types.
    kint i = doPop()->getInt();
    discardPop(); // normally coercions push an extra stack item
    return (double)i;
}

void VMState::mkArray(kint i)
{
    Array* array = new(m_apool) Array(i*2);
    // leave a bit of room for expansion (costs a little memory)
    if (i > 0){
      while(i>0) {
	i--;
	Value* v= doPop();
	array->push_back(v->cloneTo(m_vpool));
      }
    }

    push(MKPVAL(m_vpool,array,KVT_ARRAY));
}
void VMState::mkArray(Value* v, kint i)
{
    Array* array = new(m_apool) Array(i*2);
    // leave a bit of room for expansion (costs a little memory)
    if (i > 0){
      while(i>0) {
	i--;
	Value* v= doPop();
	array->push_back(v->cloneTo(m_vpool));
      }
    }
    v->setArray(array);
    push(v);
}

void VMState::goToIndex()
{
    --m_stackptr;
    kint idx = (*m_stackptr)->getInt();
    (*(m_stackptr-1)) = (*(m_stackptr-1))->lookup(this,idx);
}

void VMState::pushglobal(Value**& globtable, kint j)
{
    if (globtable == NULL) {
	globtable = (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*m_globalloc);
	for(kint i=0;i<m_globalloc;++i) { //TMP HACK
	  // don't allocate globals from pool as they're uncollectable
	    globtable[i]=MKVAL(0,KVT_INT);
	}
    }

    push(globtable[j]);
}

void VMState::createglobal(wchar_t* modid, kint i)
{
    cerr << "Do not run this function again" << endl;
    exit(1);
}

void VMState::newjmp_buf()
{
    jmp_buf* b = new jmp_buf[1];
    m_except_stack.push(b);
//    stackdata *s = new stackdata();
//    s->data = valstack;
//    s->size = stacksize;
    StackData d;
    d.stackptr = m_stackptr;
    d.csptr = m_btptr;
    m_stack_stack.push(d);

//    cout << "Inside " << except_stack.size() << " tries" << endl;
}

void VMState::kaya_throw(wchar_t* msg,kint code)
{
    push(mkstr(msg));
    push(mkint((void*)code));
    push(MKPVAL(m_vpool,(void*)(new Exception(this)), KVT_EXCEPTION));
    throw_ex();
}

void VMState::kaya_internalError(kint code)
{
    push(mkint((void*)code));
    push(MKPVAL(m_vpool,(void*)(new Exception(this, INTERNAL_ERROR, 1)), 
		   KVT_EXCEPTION));
    throw_ex();
}

void VMState::kaya_rtsError(const char* exceptionID)
{
    push(MKPVAL(m_vpool,(void*)(new Exception(this, exceptionID, 0)), 
		   KVT_EXCEPTION));
    throw_ex();
}

void VMState::throw_ex()
{
    longjmp(*(m_except_stack.top()),2);
}

jmp_buf* VMState::top_ex()
{
    return m_except_stack.top();
}

void VMState::tried()
{
    delete m_except_stack.top(); 
    m_except_stack.pop(); 
    m_stack_stack.pop();
}

void VMState::restore()
{
    m_stackptr = m_stack_stack.top().stackptr;
    m_btptr = m_stack_stack.top().csptr;
}

void VMState::getindex()
{
    kint idx = doPop()->getInt();
    Value* array = doPop();
    Value* el = array->lookup(this,idx);
    push(el);
}

Value* VMState::mkstr(wchar_t* str)
{
    return MKPVAL(m_vpool,(void*)(NEWPSTRING(m_spool,str)),KVT_STRING);
}

Value* VMState::mkint(void* i)
{
    return MKPVAL(m_vpool,i,KVT_INT);
}

void VMState::lineno(const wchar_t* src, int ln)
{
#ifndef NOCHECK
  // copy is necessary again
    m_sourcefile = (wchar_t*)GC_MALLOC_ATOMIC(sizeof(wchar_t)*(wcslen(src)+1));
    wcscpy(m_sourcefile,src);
    m_lineno = ln;
#endif
}

void VMState::pushBT(const wchar_t* fn, const wchar_t* mod, int ln)
{
#ifndef NOCHECK
    CallStackEntry* c = *m_btptr;

    c->fn_name = fn; //(char*)KayaAlloc(strlen(fn)+1);
    c->file = mod; //(char*)KayaAlloc(strlen(mod)+1);
    c->call_file = m_sourcefile; // already gcmalloced
    c->line = ln;
    c->call_line = m_lineno;

    ++m_btptr;
    if ((m_btptr-m_backtrace)>=m_callstackalloc) {
      realloc_callstack();
      //      kaya_rtsError(CALL_STACK_OVERFLOW);
    }
#endif
}

void VMState::popBT()
{
#ifndef NOCHECK
//    cout << "pop backtrace!" << endl;
    if (m_btptr==m_backtrace) {
	cout << "Warning! Call stack underflow!" << endl;
    }
    --m_btptr;
//    *m_btptr = NULL; // set end marker
#endif
}

void VMState::memstat()
{
#ifndef NOCHECK
    int mem = GC_get_heap_size();
    if (mem>m_maxmem) m_maxmem=mem;
#endif
}

void VMState::writestdout()
{
    wchar_t* str=doPop()->getString()->getVal();
    write(1,wctostr(str),strlen(wctostr(str)));
#ifndef WIN32
    fsync(1);
#endif
}

void VMState::realloc_stack() {
  //  cerr << "realloc:" << m_stackalloc << endl;
  Value** newstack = (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*(m_stackalloc<<3));
  memset(newstack,0,sizeof(Value*)*(m_stackalloc<<3));
  for (kint i=0;i<m_stackalloc;i++) {
    newstack[i] = m_valstack[i];
  }
  kint h = m_stackptr-m_valstack;
  Value** oldstack = m_valstack;
  m_valstack = newstack;
  GC_FREE(oldstack);
  m_stackptr = m_valstack+h;
  m_stackalloc = m_stackalloc<<3;
  // generally this only happens with heavily recursive functions
}

void VMState::realloc_callstack() {
  //  cerr << "realloc call:" << m_callstackalloc << endl;
  CallStackEntry** newstack = (CallStackEntry**)GC_MALLOC_UNCOLLECTABLE(sizeof(CallStackEntry*)*(m_callstackalloc<<3));
  for (kint i=0;i<m_callstackalloc;i++) {
    newstack[i] = m_backtrace[i];
  }
  for (kint i=m_callstackalloc;i<(m_callstackalloc<<3);i++) {
    newstack[i]=new CallStackEntry;
  }
  kint h = m_btptr-m_backtrace;
  CallStackEntry** oldstack = m_backtrace;
  m_backtrace = newstack;
  GC_FREE(oldstack);
  m_btptr = m_backtrace+h;
  m_callstackalloc = m_callstackalloc<<3;
  // generally this only happens with heavily recursive functions
}
