/***************************************************************************
 *   Copyright (C) 2005 by Roberto Virga                                   *
 *   rvirga@users.sf.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>

#include <kbsboincmonitor.h>

#include <kbspredictormoleculelog.h>

#include "kbspredictorprojectmonitor.h"

const QString PredictorBurialsOpenName = "burials";
const QString PredictorECovers24OpenName = "ecovers_24";
const QString PredictorProfile3OpenName = "profile3";
const QString PredictorQuasi3OpenName = "quasi3";
const QString PredictorScale3BOpenName = "scale3b";
const QString PredictorS1234OpenName = "s1234";
const QString PredictorS1234HOpenName = "s1234h";
const QString PredictorS1234EOpenName = "s1234e";
const QString PredictorMonssterInitChainOpenName = "monsster.init.chain";
const QString PredictorMonssterInputOpenName = "monsster.input";
const QString PredictorMonssterSeqOpenName = "monsster.seq";
const QString PredictorMonssterRestraintsOpenName = "monsster.restraints";
const QString PredictorMonssterFinalChainOpenName = "monsster.final.chain";
const QString PredictorMonssterFinalPDBOpenName = "monsster.final.pdb";
const QString PredictorMonssterFinalNOEOpenName = "monsster.final.noe";

const QString PredictorParam19InpOpenName = "param19.inp";
const QString PredictorTopH19InpOpenName = "toph19.inp";
const QString PredictorCharmmInpOpenName = "charmm.inp";
const QString PredictorProteinPDBOpenName = "protein.pdb";
const QString PredictorProteinNOEOpenName = "protein.noe";
const QString PredictorSeedStreamOpenName = "seed.stream";
const QString PredictorProteinFinalPDBOpenName = "proteinfinal.pdb";

KBSPredictorProjectMonitor::KBSPredictorProjectMonitor(const QString &project, KBSBOINCMonitor *parent,
                                                       const char *name)
                          : KBSProjectMonitor(project, parent, name)
{
  m_results.setAutoDelete(true);
  
  const BOINCClientState *state = parent->state();
  if(NULL != state) m_start = state->workunit.keys();
  
  connect(parent, SIGNAL(workunitsRemoved(const QStringList &)),
          this, SLOT(removeWorkunits(const QStringList &)));
  
  connect(parent, SIGNAL(resultsCompleted(const QStringList &)),
          this, SLOT(logResults(const QStringList &)));
  
  connect(this, SIGNAL(fileUpdated(const QString &)), this, SLOT(updateFile(const QString &)));
}

const PredictorResult *KBSPredictorProjectMonitor::result(const QString &workunit)
{
  return validWorkunit(workunit) ? m_results.find(workunit) : NULL;
}

void KBSPredictorProjectMonitor::setState(const QString &workunit, const PredictorState &state)
{
  PredictorResult *result = m_results.find(workunit);
  if(NULL == result) {
    result = new PredictorResult();
    result->app_type = state.app_type;
    m_results.insert(workunit, result);
  }
  
  if(result->app_type != state.app_type) return;
  
  if(MFOLD == result->app_type)
    result->mfold.monsster.final.chain = state.monsster_restart.chain;
  
  emit updatedResult(workunit);
}

KBSLogManager *KBSPredictorProjectMonitor::logManager() const
{
  return NULL;
}
    
bool KBSPredictorProjectMonitor::parseable(const QString &openName) const
{
  return(// openName == PredictorBurialsOpenName
   // || openName == PredictorECovers24OpenName
   // || openName == PredictorProfile3OpenName
   // || openName == PredictorQuasi3OpenName
   // || openName == PredictorScale3BOpenName
   // || openName == PredictorS1234OpenName
   // || openName == PredictorS1234HOpenName
   // || openName == PredictorS1234EOpenName
         openName == PredictorMonssterInitChainOpenName
      || openName == PredictorMonssterInputOpenName
      || openName == PredictorMonssterSeqOpenName
      || openName == PredictorMonssterRestraintsOpenName
      || openName == PredictorMonssterFinalChainOpenName
      || openName == PredictorMonssterFinalPDBOpenName
   // || openName == PredictorMonssterFinalNOEOpenName
      || openName == PredictorCharmmInpOpenName
      || openName == PredictorProteinPDBOpenName
      || openName == PredictorProteinNOEOpenName
      || openName == PredictorSeedStreamOpenName
      || openName == PredictorProteinFinalPDBOpenName);
}

bool KBSPredictorProjectMonitor::parseFile(KBSFileInfo *file, const QString &fileName)
{
  qDebug("Parsing file %s...", file->fileName.latin1());
  
  QStringList lines;
  if(!readFile(fileName, lines)) return false;
  
  if(!m_meta.contains(file->fileName)) return false;
  const KBSFileMetaInfo meta = m_meta[file->fileName];
  
  if(PredictorBurialsOpenName == meta.open_name)
  {
    PredictorBurials burials;
    if(!burials.parse(lines)) return false;
    
    setBurials(burials, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorECovers24OpenName == meta.open_name)
  {
    PredictorECovers24 ecovers_24;
    if(!ecovers_24.parse(lines)) return false;
    
    setECovers24(ecovers_24, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorProfile3OpenName == meta.open_name)
  {
    PredictorProfile3 profile3;
    if(!profile3.parse(lines)) return false;
    
    setProfile3(profile3, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorQuasi3OpenName == meta.open_name)
  {
    PredictorQuasi3 quasi3;
    if(!quasi3.parse(lines)) return false;
    
    setQuasi3(quasi3, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorScale3BOpenName == meta.open_name)
  {
    QValueList<PredictorScale3B> scale3b;
    if(!parseScale3B(lines, scale3b)) return false;
    
    setScale3B(scale3b, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorS1234OpenName == meta.open_name)
  {
    PredictorS1234 s1234;
    if(!s1234.parse(lines)) return false;
    
    setS1234(s1234, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorS1234HOpenName == meta.open_name)
  {
    PredictorS1234 s1234h;
    if(!s1234h.parse(lines)) return false;
    
    setS1234H(s1234h, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorS1234EOpenName == meta.open_name)
  {
    PredictorS1234 s1234e;
    if(!s1234e.parse(lines)) return false;
    
    setS1234E(s1234e, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorMonssterInitChainOpenName == meta.open_name)
  {
    QValueList<PredictorMonssterAtom> init_chain;
    if(!parseMonssterChain(lines, init_chain)) return false;
    
    setMonssterInitChain(init_chain, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorMonssterInputOpenName == meta.open_name)
  {
    PredictorMonssterInput input;
    if(!input.parse(lines)) return false;
    
    setMonssterInput(input, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorMonssterSeqOpenName == meta.open_name)
  {
    PredictorMonssterSeq seq;
    if(!seq.parse(lines)) return false;
    
    setMonssterSeq(seq, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorMonssterRestraintsOpenName == meta.open_name)
  {
    QValueList<PredictorMonssterRestraint> restraints;
    if(!parseMonssterRestraints(lines, restraints)) return false;
    
    setMonssterRestraints(restraints, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorMonssterFinalChainOpenName == meta.open_name)
  {
    QValueList<PredictorMonssterAtom> final_chain;
    if(!parseMonssterChain(lines, final_chain)) return false;
    
    setMonssterFinalChain(final_chain, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorMonssterFinalPDBOpenName == meta.open_name)
  {
    PredictorProteinPDB final_pdb;
    if(!final_pdb.parse(lines)) return false;
    
    setMonssterFinalPDB(final_pdb, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorMonssterFinalNOEOpenName == meta.open_name)
  {
    QValueList<PredictorProteinNOE> final_noe;
    if(!parseProteinNOE(lines, final_noe)) return false;
    
    setMonssterFinalNOE(final_noe, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorParam19InpOpenName == meta.open_name)
  {
    setAppType(CHARMM, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorTopH19InpOpenName == meta.open_name)
  {
    setAppType(CHARMM, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorCharmmInpOpenName == meta.open_name)
  {
    PredictorCharmmInp inp;
    if(!inp.parse(lines)) return false;
    
    setCharmmInp(inp, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorProteinPDBOpenName == meta.open_name)
  {
    PredictorProteinPDB pdb;
    if(!pdb.parse(lines)) return false;
    
    setProteinPDB(pdb, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorProteinNOEOpenName == meta.open_name)
  {
    QValueList<PredictorProteinNOE> noe;
    if(!parseProteinNOE(lines, noe)) return false;
    
    setProteinNOE(noe, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorSeedStreamOpenName == meta.open_name)
  {
    unsigned seed;
    if(!parseSeedStream(lines, seed)) return false;
    
    setSeedStream(seed, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else if(PredictorProteinFinalPDBOpenName == meta.open_name)
  {
    PredictorProteinPDB final_pdb;
    if(!final_pdb.parse(lines)) return false;
    
    setProteinFinalPDB(final_pdb, meta.workunits);
    
    qDebug("... parse OK");
    return true;
  }
  else
    return false;
}

PredictorResult *KBSPredictorProjectMonitor::mkResult(const QString &workunit)
{
  PredictorResult *result = m_results.find(workunit);
  
  if(NULL == result) {
    result = new PredictorResult();
    m_results.insert(workunit, result);
  }
  
  return result;
}

bool KBSPredictorProjectMonitor::parseScale3B(const QStringList &lines,
                                              QValueList<PredictorScale3B> &scale3b)
{
  scale3b.clear();
  QStringList::const_iterator line = lines.constBegin();

  if(lines.constEnd() == line) return false;
  const unsigned n = (*line).toUInt(0, 10);
  ++line;
  
  for(unsigned i = 0; i < n; i++)
  {
    if(lines.constEnd() == line) return false;
    
    PredictorScale3B item;
    if(!item.parse(*line)) return false;

    scale3b << item;
    line++;
  }

  return true;
}

bool KBSPredictorProjectMonitor::parseMonssterChain(const QStringList &lines,
                                                    QValueList<PredictorMonssterAtom> &chain)
{
  chain.clear();

  QStringList::const_iterator line = lines.constBegin();
  if(lines.constEnd() == line) return false;

  QStringList fields = QStringList::split(" ", *line);
  if(fields.isEmpty()) return false;
  
  const unsigned n = fields[0].toUInt(0, 10);
  ++line;
  
  for(unsigned i = 0; i < n; i++)
  {
    PredictorMonssterAtom item;
    
    if(lines.constEnd() == line || !item.parse(*line)) return false;
    ++line;

    chain << item;
  }

  return true;
}

bool KBSPredictorProjectMonitor::parseMonssterRestraints(const QStringList &lines,
                                                         QValueList<PredictorMonssterRestraint> &restraints)
{
  restraints.clear();
  QStringList::const_iterator line = lines.constBegin();

  if(lines.constEnd() == line) return false;
  const unsigned n = (*line).toUInt(0, 10);
  ++line;
  
  for(unsigned i = 0; i < n; i++)
  {
    if(lines.constEnd() == line) return false;
    
    PredictorMonssterRestraint item;
    if(!item.parse(*line)) return false;
    
    restraints << item;
    ++line;
  }

  return true;
}

bool KBSPredictorProjectMonitor::parseProteinNOE(const QStringList &lines,
                                                 QValueList<PredictorProteinNOE> &noe)
{
  noe.clear();
  QStringList::const_iterator line = lines.constBegin();
  
  // skip header
  if(lines.constEnd() == line) return false;
  ++line;
  
  while(line != lines.constEnd())
  {
    if((*line).startsWith("END")) break;
    
    PredictorProteinNOE item;
    if(!item.parse(*line)) return false;
    
    noe << item;
    ++line;
  }
    
  return true;
}

bool KBSPredictorProjectMonitor::parseSeedStream(const QStringList &lines, unsigned &seed)
{
  QStringList::const_iterator line = lines.constBegin();
  if(lines.constEnd() == line) return false;
  
  sscanf(*line, "set seed = %u", &seed);
   
  return true;
}

void KBSPredictorProjectMonitor::setAppType(PredictorAppType app_type, const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
    mkResult(*workunit)->app_type = app_type;
}

void KBSPredictorProjectMonitor::setBurials(const PredictorBurials &burials, const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.burials = burials;
  }
}

void KBSPredictorProjectMonitor::setECovers24(const PredictorECovers24 &ecovers_24,
                                              const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.ecovers_24 = ecovers_24;
  }
}

void KBSPredictorProjectMonitor::setProfile3(const PredictorProfile3 &profile3,
                                             const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.profile3 = profile3;
  }
}

void KBSPredictorProjectMonitor::setQuasi3(const PredictorQuasi3 &quasi3, const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.quasi3 = quasi3;
  }
}

void KBSPredictorProjectMonitor::setScale3B(const QValueList<PredictorScale3B> &scale3b,
                                            const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.scale3b = scale3b;
  }
}

void KBSPredictorProjectMonitor::setS1234(const PredictorS1234 &s1234, const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.s1234 = s1234;
  }
}

void KBSPredictorProjectMonitor::setS1234H(const PredictorS1234 &s1234h, const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.s1234h = s1234h;
  }
}

void KBSPredictorProjectMonitor::setS1234E(const PredictorS1234 &s1234e, const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.s1234e = s1234e;
  }
}

void KBSPredictorProjectMonitor::setMonssterInitChain(const QValueList<PredictorMonssterAtom> &init_chain,
                                                      const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.monsster.init_chain = init_chain;
    
    if(!m_start.contains(*workunit) && result->mfold.monsster.seq.groups.count() > 0)
      KBSPredictorMoleculeLog::self()->logWorkunit(*workunit, result);
  }
}

void KBSPredictorProjectMonitor::setMonssterInput(const PredictorMonssterInput &input,
                                                  const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.monsster.input = input;
  }
}

void KBSPredictorProjectMonitor::setMonssterSeq(const PredictorMonssterSeq &seq,
                                                const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.monsster.seq = seq;
    
    if(!m_start.contains(*workunit) && result->mfold.monsster.init_chain.count() > 0)
      KBSPredictorMoleculeLog::self()->logWorkunit(*workunit, result);
  }
}

void KBSPredictorProjectMonitor::setMonssterRestraints(const QValueList<PredictorMonssterRestraint> &restraints,
                                                       const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.monsster.restraints = restraints;
  }
}

void KBSPredictorProjectMonitor::setMonssterFinalChain(const QValueList<PredictorMonssterAtom> &final_chain,
                                                       const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.monsster.final.chain = final_chain;
  }
}

void KBSPredictorProjectMonitor::setMonssterFinalPDB(const PredictorProteinPDB &final_pdb,
                                                       const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.monsster.final.pdb = final_pdb;
  }
}

void KBSPredictorProjectMonitor::setMonssterFinalNOE(const QValueList<PredictorProteinNOE> &final_noe,
                                                       const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = MFOLD;
    result->mfold.monsster.final.noe = final_noe;
  }
}

void KBSPredictorProjectMonitor::setCharmmInp(const PredictorCharmmInp &inp,
                                              const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = CHARMM;
    result->charmm.inp = inp;
    
    if(!m_start.contains(*workunit))
      KBSPredictorMoleculeLog::self()->logWorkunit(*workunit, result);
  }
}

void KBSPredictorProjectMonitor::setProteinPDB(const PredictorProteinPDB &pdb,
                                               const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = CHARMM;
    result->charmm.protein.pdb = pdb;
    
    if(!m_start.contains(*workunit))
      KBSPredictorMoleculeLog::self()->logWorkunit(*workunit, result);
  }
}

void KBSPredictorProjectMonitor::setProteinNOE(const QValueList<PredictorProteinNOE> &noe,
                                               const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = CHARMM;
    result->charmm.protein.noe = noe;
  }
}

void KBSPredictorProjectMonitor::setSeedStream(unsigned seed, const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = CHARMM;
    result->charmm.seed.stream = seed;
  }
}

void KBSPredictorProjectMonitor::setProteinFinalPDB(const PredictorProteinPDB &final_pdb,
                                                    const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
  {
    PredictorResult *result = mkResult(*workunit);
    
    result->app_type = CHARMM;
    result->charmm.protein.final_pdb = final_pdb;
  }
}

void KBSPredictorProjectMonitor::removeWorkunits(const QStringList &workunits)
{
  for(QStringList::const_iterator workunit = workunits.constBegin();
      workunit != workunits.constEnd(); ++workunit)
    m_results.remove(*workunit);
}

void KBSPredictorProjectMonitor::logResults(const QStringList &results)
{
  const BOINCClientState *state = boincMonitor()->state();
  if(NULL == state) return;
  
  for(QStringList::const_iterator result = results.constBegin(); result != results.constEnd(); ++result)
  {
    if(boincMonitor()->project(state->result[*result]) != project()) continue;
    
    PredictorResult *predictorResult = m_results.find(state->result[*result].wu_name);
    if(NULL == predictorResult) continue;
    
    KBSPredictorMoleculeLog::self()->logResult(*result, predictorResult);
  }
}

void KBSPredictorProjectMonitor::updateFile(const QString &fileName)
{
  if(!m_meta.contains(fileName)) return;
  
  QStringList workunits = m_meta[fileName].workunits;
  for(QStringList::const_iterator workunit = workunits.begin();
      workunit != workunits.end(); ++ workunit)
    emit updatedResult(*workunit);
}

#include "kbspredictorprojectmonitor.moc"
