#include <qpainter.h>
#include <qlineedit.h>

#include <kapp.h>
#include <kmessagebox.h>
#include <kglobal.h>
#include <klocale.h>
#include <kglobal.h>
#include <kstddirs.h>
#include <kconfig.h>
#include <kdebug.h>

#include <qpixmap.h>
#include <qtimer.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qbutton.h>
#include <qpushbutton.h>
#include <qlabel.h>

#include <kprocess.h>
#include <stdio.h>
#include <errno.h>

#include "kcd.h"

int Kcd::timeRepr = 0 ;

extern "C"
{
  KPanelApplet* init(QWidget *parent, const QString& configFile)
  {
         KGlobal::locale()->insertCatalogue("kcd");
         return new Kcd(configFile, KPanelApplet::Normal,
                             KPanelApplet::About | KPanelApplet::Help | KPanelApplet::Preferences,
                             parent, "kcd");
  }
}

Kcd::Kcd(const QString& configFile, Type type, int actions,
                             QWidget *parent, const char *name)
  : KPanelApplet(configFile, type, actions, parent, name)
{
    //Try to initialize CD-Rom device ; exit if error
    if((cdHandle = cd_init_device("/dev/cdrom")) < 0){
            if (errno == EBUSY) kdError() << "CD-Rom device already appears to be mounted. \n" ;
                 else kdError() << "Error accessing CD-Rom device(/dev/cdrom). Correct permissions ?\n" ;
                 exit(1) ;
         }

    //Initialize status
    status = STATUS_NOT_READY ;
    timeRepr = TIME_NORMAL ;
    titleinfo = NULL ;
    timeVisible = 0 ;
    currentTrack = 0 ;
    cddbQueried = 0 ;
    ignoreUpdate = 0 ;

    //Main Widget
    QVBox *gui = new QVBox(this) ;

    //info fields panel
    QHBox *info = new QHBox(gui) ;
    trackInfo = new QLabel("--/--", info) ;
    QFont font("System", 12, QFont::Bold) ;
    trackInfo->setFont(font) ;
    timeInfo = new TimeDisplay(info) ;
    timeInfo->setNumDigits(5) ;
    timeInfo->display("00:00") ;
    info->setMaximumSize(100, 25) ;

    //Label to show information (e.g. CDDB)
    title = new QLabel("", gui) ;
    QFont font2("System", 10) ;
    font2.setItalic(true) ;
    title->setFont(font2) ;
    title->setMaximumSize(100, 10) ;

    //command buttons panel
    QHBox *buttons = new QHBox(gui) ;
    QPushButton *prevB = new QPushButton(buttons, "") ;
    prevB->setPixmap(QPixmap(KGlobal::dirs()->findResource("data", "kcd/pics/prev.gif"))) ;
    //It would be better to add a ResourceDir for this purpose, but this doesn't work ...
    connect(prevB, SIGNAL(clicked()), SLOT(previous())) ;
    QPushButton *playB = new QPushButton(buttons, "") ;
    playB->setPixmap(QPixmap(KGlobal::dirs()->findResource("data", "kcd/pics/play.gif"))) ;
    connect(playB, SIGNAL(clicked()), SLOT(play())) ;
    QPushButton *stopB = new QPushButton(buttons, "") ;
    stopB->setPixmap(QPixmap(KGlobal::dirs()->findResource("data", "kcd/pics/stop.gif"))) ;
    connect(stopB, SIGNAL(clicked()), SLOT(stop())) ;
    QPushButton *nextB = new QPushButton(buttons, "") ;
    nextB->setPixmap(QPixmap(KGlobal::dirs()->findResource("data", "kcd/pics/next.gif"))) ;
    connect(nextB, SIGNAL(clicked()), SLOT(next())) ;
    QPushButton *ejectB = new QPushButton(buttons, "") ;
    ejectB->setPixmap(QPixmap(KGlobal::dirs()->findResource("data", "kcd/pics/eject.gif"))) ;
    connect(ejectB, SIGNAL(clicked()), SLOT(eject())) ;
    buttons->setMaximumSize(100, 15) ;

    gui->setMinimumSize(100, 45) ;

    //timer to update time info
    viewTimer = new QTimer(this);
    connect(viewTimer, SIGNAL(timeout()), this, SLOT(updateView()));
    viewTimer->start(500, false);
    statusTimer = new QTimer(this);
    connect(statusTimer, SIGNAL(timeout()), this, SLOT(updateStatus()));
    statusTimer->start(1000, false);
}

void Kcd::about(){
    KMessageBox::information(0, "Kcd - KDE CD Player Applet");
}

void Kcd::help(){
    KMessageBox::information(0, "For Kcd help view README");
}

void Kcd::preferences(){
    KMessageBox::information(0, "Kcd doesn't have any preferences yet.");
}

int Kcd::widthForHeight(int ) const{
    // horizontal
    return 100;
}

int Kcd::heightForWidth(int ) const{
    // vertical
    return 50;
}

Kcd::~Kcd(){
    cd_stop(cdHandle) ;
    viewTimer->stop() ;
    delete viewTimer ;
    statusTimer->stop() ;
    delete statusTimer ;
    cd_finish(cdHandle) ;
}

void Kcd::init(){
    //Init CD-Rom for libcdaudio
    if(cd_stat(cdHandle, &discInfo) < 0){
        kdError() << "Kcd: Unable to open cdrom device ! " << endl ;
        status = STATUS_ERROR ;
    }
    else{
        kdDebug() << "Kcd: initializing cdrom device OK (handle:" << cdHandle << ")" << endl ;
        // This will call cd_stat() a second time, but we need to
        // find out if the CD is already playing, etc.
        updateStatus();
    }

    currentTrack = 0 ;
    timeVisible = 0 ;
    ignoreUpdate = 0 ;

    //CD-player is ready to play ...
    status = STATUS_READY ;
    oldStatus = status ;
}

// Query CDDB to find CD info
void Kcd::cddbQuery(){
    kdDebug() << "Kcd : cddbQuery" << endl ;
    cddbQueried = 1 ;

    struct cddb_server server ;
    strncpy(server.server_name,"www.freedb.org", 256) ;
    server.server_port = 888 ;
    struct cddb_host host ;
    host.host_server = server ;
    strncpy(host.host_addressing,"cgi-bin/cddb.cgi", 256) ;
    host.host_protocol = CDDB_MODE_CDDBP ;
    struct cddb_hello hello ;
    strncpy(hello.hello_program,"Kcd", 256) ;
    strncpy(hello.hello_version,"0.1", 256) ;
    struct cddb_query query ;
    query.query_match = QUERY_EXACT ;

    //opening the socket takes a while
    //TODO: either optimize this using always the same socket or put this in separate thread
    kdDebug() << "Connecting to cddbp://" << host.host_server.server_name << ":" << host.host_server.server_port << endl ;
    int sock = cddb_connect_server(host, NULL, hello) ;
    kdDebug() << "Kcd: CDDB connect result: " << sock << endl ;
    if(sock >= 0){
        int res = cddb_query(cdHandle, sock, CDDB_MODE_CDDBP, &query) ;
        kdDebug() << "Kcd: CDDB query result: " << res << endl ;

        for (int i=0 ; i<query.query_matches ; i++){
            kdDebug() << "Kcd query result: " << query.query_list[i].list_artist << endl ;
        }
        cdData = new struct disc_data ;
        cddb_read_disc_data(cdHandle, cdData) ;

        //close socket (might be optimized to keep the same socket)
        cddb_quit(sock) ;
    }
}

//Update the text of the info field appropriately
void Kcd::updateInfo(){
    kdDebug() << "Kcd : updateInfo" << endl ;

    //updateStatus();

    if(cdData != NULL){
        if(status == STATUS_READY){
            titleinfo = new char[256] ;
            sprintf(titleinfo, "     * %s - %s *", cdData->data_artist, cdData->data_title) ;
        }
        else if(status == STATUS_PLAYING || status == STATUS_PAUSED){
            titleinfo = new char[256] ;
            sprintf(titleinfo, "          %s - %s",cdData->data_artist,cdData->data_track[currentTrack-1].track_name) ;
        }
        else{
            titleinfo = "   " ;
        }
    }
    else{
        titleinfo = "     No info available" ;
    }
    titleindex = 0 ;
}

//Update the view of the applet (cfr. repaint)
//Assumes correct Status, because updateStatus() is often called through the statusTimer
void Kcd::updateView(){
    kdDebug() << "Kcd : updateView" << endl ;

    //Update Time Info
    if(status == STATUS_PLAYING || (status == STATUS_PAUSED && timeVisible >= 1)){
        QString msg ;
        if(timeRepr == TIME_NORMAL)
            msg.sprintf("%02d:%02d", discInfo.disc_track_time.minutes, discInfo.disc_track_time.seconds) ;
        else if(timeRepr == TIME_REVERSE){
            //Easier via QTime ?
            int s = discInfo.disc_track[currentTrack-1].track_length.seconds - discInfo.disc_track_time.seconds ;
            int m = discInfo.disc_track[currentTrack-1].track_length.minutes - discInfo.disc_track_time.minutes ;
            if(s < 0){
                s+= 60 ;
                m-- ;
            }
            msg.sprintf("%02d:%02d", m, s) ;
        }
        timeInfo->display(msg) ;
    }
    else if(status == STATUS_READY) timeInfo->display("00:00") ;
    else if(status == STATUS_EJECTED) timeInfo->display("  :  ") ;
    else if(status == STATUS_PAUSED && timeVisible < 1){
        timeInfo->display("  :  ") ;
    }
    timeVisible = (timeVisible + 1) % 2 ;

    //Update Track Info
    QString msg ;
    if(status == STATUS_EJECTED) msg = "NO CD" ;
    else if(status == STATUS_NOT_READY) msg = "--/--" ;
    else if(status == STATUS_READY) msg.sprintf("00/%02d", discInfo.disc_total_tracks) ;
    else if(status == STATUS_PLAYING || status == STATUS_PAUSED) msg.sprintf("%02d/%02d", currentTrack, discInfo.disc_total_tracks)  ;
    else if(status == STATUS_ERROR) msg = "ERR" ;
    trackInfo->setText(msg) ;

    //Update CDDB Info
    if (titleinfo != NULL){
        titleindex = titleindex >= strlen(titleinfo) ? 0 : titleindex + 1 ;
        title->setText(titleinfo+titleindex) ;
    }
}

void Kcd::changeTimeRepr(){
         timeRepr = (timeRepr % 2) + 1 ; //Does not do total times yet ...
}

void Kcd::play(){
    kdDebug() << "Kcd : play" << endl ;
    //if playing => pause
    if(status == STATUS_PLAYING){
        pause() ;
        status = STATUS_PAUSED ;
        return ;
    }
    //if paused => play
    if(status == STATUS_PAUSED){
        cd_resume(cdHandle) ;
        status = STATUS_PLAYING ;
        return ;
    }
    //if ejected => close and init
    if(status == STATUS_EJECTED){
        eject() ;
    }
    //if not ready => init
    if(status == STATUS_NOT_READY){
        init() ;
    }
    //after init, play first track
    kdDebug() << "Kcd : playing track 1" << endl ;
    currentTrack = 1 ;
    cd_play(cdHandle, currentTrack) ;

    //Actually the next three commands could be deleted, because this is noticed by updateStatus()
    //However, this does it a lot faster ... (this remark also applies to the next commands)
    //status = STATUS_PLAYING ;
    //updateInfo() ;
    //updateView() ;
}

void Kcd::pause(){
    kdDebug() << "Kcd : pause" << endl ;
    //if not ready => NOP
    if (status == STATUS_PLAYING){
        cd_pause(cdHandle) ;

        status = STATUS_PAUSED ;
        updateView() ;
    }
}

void Kcd::stop(){
    kdDebug() << "Kcd : stop" << endl ;
    if (status == STATUS_PLAYING || status == STATUS_PAUSED){
        currentTrack = 0 ;
        cd_stop(cdHandle) ;

        status = STATUS_READY ;
        updateInfo() ;
        updateView() ;
    }
}

void Kcd::next(){
    kdDebug() << "Kcd : next track" << endl ;
    if(discInfo.disc_current_track < discInfo.disc_total_tracks && status == STATUS_PLAYING){
        //order is important ! (otherwise possible synchronization problem with updateStatus)
        currentTrack++ ;
        cd_play(cdHandle, currentTrack) ;

        ignoreUpdate = 1 ;
        updateInfo() ;
        updateView() ;
    }
}

void Kcd::previous(){
    kdDebug() << "Kcd : previous track" << endl ;
    if(discInfo.disc_current_track > 1 && status == STATUS_PLAYING){
        //order is important ! (otherwise possible synchronization problem with updateStatus)
        currentTrack-- ;
        cd_play(cdHandle, currentTrack) ;

        ignoreUpdate = 1 ;
        updateInfo() ;
        updateView() ;
    }
}

//eject or close the tray and init if necessary
void Kcd::eject(){
    kdDebug() << "Kcd : eject" << endl ;
    //if ejected => close and init
    if(status == STATUS_EJECTED){
        cd_close(cdHandle) ;
        init() ;
    }
    else{
        cd_eject(cdHandle) ;
        status = STATUS_EJECTED ;
        updateInfo() ;
        updateView() ;
    }
    currentTrack = 0 ;
    cddbQueried = 0 ;
}


TimeDisplay::TimeDisplay(QWidget *parent, const char *name)
         : QLCDNumber(parent, name)
{
    QColorGroup cg = colorGroup() ;
    cg.setColor(QColorGroup::Foreground, Qt::yellow) ;
    cg.setColor(QColorGroup::Background, Qt::darkBlue) ;
    setPalette(QPalette(cg, cg, cg)) ;
    setSegmentStyle(QLCDNumber::Filled) ;
}

void TimeDisplay::mousePressEvent(QMouseEvent *ev){
    //At present, every mouse button is OK to change the representation
    Kcd::changeTimeRepr() ;
}

// Update the current status of the CD player (e.g. when next song is played, or when
// some other application changed its state).
// Thanks to Ben Burton for his help for this part.
void Kcd::updateStatus(){
    if(cd_stat(cdHandle, &discInfo) < 0)
        status = STATUS_ERROR ;
    else if (! discInfo.disc_present)
        status = STATUS_EJECTED ;
    else switch (discInfo.disc_mode){
        case CDAUDIO_PLAYING:
            status = STATUS_PLAYING ;
            break;
        case CDAUDIO_PAUSED:
            status = STATUS_PAUSED ;
            break;
        case CDAUDIO_COMPLETED:
            status = STATUS_READY ;
            break ;
        case CDAUDIO_NOSTATUS:
            status = STATUS_READY ;
            break;
        default:
            status = STATUS_ERROR ;
    }

    int modified = 0 ;
    if(status != oldStatus){
        oldStatus = status ;
        //if CDDB not queried yet for this cd, do so now ...
        if(status != STATUS_ERROR && status != STATUS_EJECTED && cddbQueried == 0){
            cddbQuery() ;
        }
        modified = 1 ;
    }

    //update currentTrack if necessary
    //(= when changed by another application or when changed automatically at end of previous track)
    if(ignoreUpdate == 1){ignoreUpdate = 0 ;}
    else if(status == STATUS_PLAYING & currentTrack != discInfo.disc_current_track){
	    currentTrack = discInfo.disc_current_track ;
        modified = 1 ;
    }

    //If relevant information changed, update it
    if (modified == 1) updateInfo() ;
}
