/*
 * ******************************************************************
 * $Id: gui_thread.c,v 1.7 2000/12/02 20:40:30 orjana Exp $
 * Filename:      gui_thread.c
 * Description:   A thread that maintains the applets graphics
 * Status:        Experimental, do not distribute.
 * Author:        rjan Nygaard Austvold <austvold@acm.org>
 * Created at:    Mon Nov 26 10:25:57 2000
 * Modified at:   Fri Dec  1 16:02:01 2000
 * Modified by:   rjan Nygaard Austvold <austvold@acm.org>
 *                
 * Copyright (c) 2000 rjan Nygaard Austvold, All Rights Reserved.
 *                
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Revision History:
 *
 *   01-Dec-00 21:56:06   rjan Nygaard Austvold <austvold@acm.org>
 *     Added check towards panel_pixel_size compatibility in
 *     applet-widget.h
 *
 *   01-Dec-00 16:06:48   rjan Nygaard Austvold <austvold@acm.org>
 *     Replaced call to gnome_window_icon_set_default_from_file() with
 *     plain gdk/gtk code to set the window icon.  Note: It doesn't
 *     work as it should, so an extra call to gdk_set_window_icon() is
 *     made for both the about and properties windows when they are
 *     realized.
 *
 * *****************************************************************
 */

#include "gui_thread.h"

extern gint led_pipe[2];
extern struct _settings settings;
extern pthread_t led_th;

static GtkWidget *box, *pixmap[6], *frame[3];
static GtkTooltips *tooltip;
static gint numlock_led, capslock_led, scrollock_led;
static gint applet_box_orientation;
static gint applet_width, applet_height, icon_width, icon_height;
static gint key[3] = {0, 2, 4};
static gint frame_border_size = 1;
static PanelOrientType panel_orientation = ORIENT_DOWN;
#ifdef HAVE_PANEL_PIXEL_SIZE
static gint panel_pixel_size = PIXEL_SIZE_STANDARD;
#else
static gint panel_pixel_size = 48;
#endif

GtkWidget *applet = NULL;
GdkPixmap *icon_pixmap = NULL;
GdkBitmap *icon_mask = NULL;



/*
 * Function gui_thread ()
 *
 *    Main loop of the gui thread. Fetch intial led-state, create an
 *    applet, connect signals and callbacks and enter the
 *    gtk_main-loop.
 *
 */
void gui_thread (void)
{
  gint buffer, old = 0;

  /* the other thread has to be able to cancel this thread asynchronously */
  (void) pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &old);

  /* read in the current lock keys state from the led-pipe and store,
     so we can tell which keys change states later on */
  if (read(led_pipe[0], &buffer, 4) == -1)
    g_error (_("read(): %s"), sys_errlist[MIN(errno, sys_nerr - 1)]);

  numlock_led   = (buffer & NUM_LOCK_MASK) ? 1 : 0;
  capslock_led  = (buffer & CAPS_LOCK_MASK) ? 1 : 0;
  scrollock_led = (buffer & SCROLL_LOCK_MASK) ? 1 : 0;

  /* fill in index information for which pixmap maps to which frame */
  key[0] = (numlock_led) ? NUM_LOCK_ON : NUM_LOCK_OFF;
  key[1] = (capslock_led) ? CAPS_LOCK_ON : CAPS_LOCK_OFF;
  key[2] = (scrollock_led) ? SCROLL_LOCK_ON : SCROLL_LOCK_OFF;
  
  if (!create_applet())
    g_error (_("Couldn't create applet!"));

  set_window_icon();

  /* let the main loop know that we'll be watching for data on the led_pipe */
  gdk_input_add(led_pipe[0], GDK_INPUT_READ, led_pipe_callback, NULL);

  gtk_signal_connect (GTK_OBJECT (applet), "change_orient",
		      GTK_SIGNAL_FUNC (applet_change_orient), NULL);

/* #ifdef HAVE_PANEL_PIXEL_SIZE */

  /* should really have ifdef here, but since it only will create
     a gtk-warning on older libraries I don't care... */
  gtk_signal_connect (GTK_OBJECT (applet), "change_pixel_size",
		      GTK_SIGNAL_FUNC (applet_change_pixel_size), NULL);
/* #endif */

  gtk_signal_connect (GTK_OBJECT (applet), "save_session",
		      GTK_SIGNAL_FUNC(applet_save_session), NULL);
  gtk_signal_connect (GTK_OBJECT (applet), "tooltip_state",
		      GTK_SIGNAL_FUNC(applet_tooltip_state), NULL);
  gtk_signal_connect (GTK_OBJECT (applet), "destroy",
		      GTK_SIGNAL_FUNC(applet_destroy), NULL);

  /* menu items */
  applet_widget_register_stock_callback (APPLET_WIDGET (applet), "properties",
					 GNOME_STOCK_MENU_PROP,
					 _("Properties..."),
					 callback_properties, NULL);
  applet_widget_register_stock_callback (APPLET_WIDGET (applet), "help",
					 GNOME_STOCK_PIXMAP_HELP, _("Help"),
					 callback_help, "index.html");
  applet_widget_register_stock_callback (APPLET_WIDGET (applet), "about",
					 GNOME_STOCK_MENU_ABOUT, _("About..."),
					 callback_about, NULL);

  applet_widget_gtk_main();

  (void) pthread_cancel (led_th);
  pthread_exit (NULL);
}



/*
 * Function create_applet ()
 *
 *    Create the applet-widget and it's sub-widgets.
 *
 */
GtkWidget *create_applet (void)
{
  int i;

  if (!(applet = applet_widget_new (PACKAGE)))
    return NULL;

  load_properties (APPLET_WIDGET(applet)->privcfgpath);

  /* create a hbox since we're going to optimize later, if we're allowed
     to and if it's possible */
  panel_orientation
    = applet_widget_get_panel_orient (APPLET_WIDGET (applet));
#ifdef HAVE_PANEL_PIXEL_SIZE
  panel_pixel_size
    = applet_widget_get_panel_pixel_size (APPLET_WIDGET (applet));
#endif
  applet_box_orientation = HORIZONTAL;
  box = gtk_hbox_new (TRUE, 0);

  gtk_widget_ref (box);
  applet_widget_add (APPLET_WIDGET (applet), box);

  for (i = 0; i < 3; i++) {
    frame[i] = gtk_frame_new (NULL);
    gtk_widget_ref (frame[i]);
    gtk_box_pack_start (GTK_BOX (box), frame[i], TRUE, TRUE, 0);
  }

  gtk_frame_set_shadow_type (GTK_FRAME (frame[NUM_LOCK]),
			     (numlock_led) ? GTK_SHADOW_IN:GTK_SHADOW_OUT);
  gtk_frame_set_shadow_type (GTK_FRAME (frame[CAPS_LOCK]),
			     (capslock_led) ? GTK_SHADOW_IN:GTK_SHADOW_OUT);
  gtk_frame_set_shadow_type (GTK_FRAME (frame[SCROLL_LOCK]),
			     (scrollock_led) ? GTK_SHADOW_IN:GTK_SHADOW_OUT);
  
  tooltip = gtk_tooltips_new();
  update_tooltip();
  reshape_applet(TRUE);

  gtk_widget_show_all (applet);

  return applet;
}



/*
 * Function set_window_icon ()
 *
 *    Set this applications default icon
 *
 */
void set_window_icon (void)
{
  GdkImlibImage *im;

  im = gdk_imlib_load_image (PACKAGE_ICON_DIR"/"WINDOW_ICON);
  gdk_imlib_render (im, im->rgb_width, im->rgb_height);

  icon_pixmap = gdk_imlib_copy_image (im);
  icon_mask   = gdk_imlib_copy_mask  (im);

  gdk_imlib_destroy_image (im);

  gdk_window_set_icon (applet->window, NULL, icon_pixmap, icon_mask);
}



/*
 * Function update_tooltip ()
 *
 *    Whenever the led-state changes, this function is called to
 *    update the tooltip telling the status of leds.
 *
 */
void update_tooltip (void)
{
  char tip[64]; /* :-) */
  sprintf(tip, "%s: %s\n%s: %s\n%s: %s",
	  _("Scroll Lock"), ((scrollock_led) ? _("On") : _("Off")),
	  _("Caps Lock"), ((capslock_led) ? _("On") : _("Off")),
	  _("Num Lock"), ((numlock_led) ? _("On") : _("Off")));
  gtk_tooltips_set_tip (GTK_TOOLTIPS (tooltip), applet, tip, tip);
}



/*
 * Function led_pipe_callback (data, fd, cond)
 *
 *    
 *
 */
void led_pipe_callback (gpointer data, int fd, GdkInputCondition cond)
{
  int buffer;

  if (read(fd, &buffer, 4) == -1)
    g_error (_("read(): %s"), sys_errlist[MIN(errno, sys_nerr - 1)]);

  if (buffer & NUM_LOCK_MASK)
    UPDATE_IMAGE_STATE(NUM_LOCK, numlock_led, NUM_LOCK_ON, NUM_LOCK_OFF);

  if (buffer & CAPS_LOCK_MASK)
    UPDATE_IMAGE_STATE(CAPS_LOCK, capslock_led, CAPS_LOCK_ON, CAPS_LOCK_OFF);

  if (buffer & SCROLL_LOCK_MASK)
    UPDATE_IMAGE_STATE(SCROLL_LOCK, scrollock_led,
		       SCROLL_LOCK_ON, SCROLL_LOCK_OFF);

  update_tooltip();
}



/*
 * Function room_for_optimization (orient)
 *
 *    Determine if there's room for us in the opposite orientation of
 *    the panel. That is; Can a vbox contain the led-icons on a
 *    horizontal panel or vice versa on a vertical panel?
 *
 */
gboolean room_for_optimization (int orient)
{
  int   i, n = 0, w, h, s;
  float r;

  /* count icons to be shown */
  for (i = 0; i < 3; i++)
    if (settings.lock_key_checked[i])
      n++;

  /* calculate minimum width and height of the icons */
  r = settings.aspect_ratio;
  w = (settings.minimum_checked) ? settings.minimum : 1;
  h = (int) rint (((double)w) / ((double)r));

  /* fetch panel pixel size (either width or height) */
  s = panel_pixel_size;

  /* compare with width if the panel is a verical one, height otherwise */
  return (((orient == ORIENT_UP   || orient == ORIENT_DOWN)  && (s >= n*w)) ||
	  ((orient == ORIENT_LEFT || orient == ORIENT_RIGHT) && (s >= n*h))) 
    ? TRUE : FALSE;
}



/*
 * Function rebox_applet (orient)
 *
 *    Create a new container-box for the led-icons
 *
 */
void rebox_applet (int orient)
{
  GtkWidget *new_box;
  int i;

  new_box = (orient == HORIZONTAL)
    ? gtk_hbox_new (TRUE, 0) : gtk_vbox_new (TRUE, 0);
  applet_box_orientation = orient;

  /* re-parent frames to the new box */
  for (i = 0; i < 3; i++) {
    gtk_widget_ref (frame[i]);
    gtk_container_remove (GTK_CONTAINER (box), frame[i]);
    gtk_container_add (GTK_CONTAINER (new_box), frame[i]);
    gtk_widget_unref (frame[i]);
  }

  /* remove the old box-container and assign the new one to the applet */
  gtk_container_remove (GTK_CONTAINER (applet), box);
  gtk_widget_unref (box);
  box = new_box;
  gtk_widget_ref (box);
  applet_widget_add (APPLET_WIDGET (applet), box);
}



/*
 * Function reshape_applet (reload_images)
 *
 *    Whenever the size or orientation of the panel changes or some
 *    property has been changed, this function is called to reshape an
 *    recalculate the geometry of the applet.
 *
 */
void reshape_applet (gboolean reload_images)
{
  int i, aw, ah, iw, ih, nw, s, n = 0;
  float r;
  gboolean flip;

  /* fetch updated info about the panel */
  panel_orientation = applet_widget_get_panel_orient(APPLET_WIDGET(applet));
  flip = settings.optimize_checked && room_for_optimization(panel_orientation);
  
  if (panel_orientation == ORIENT_UP || panel_orientation == ORIENT_DOWN) {
    if (applet_box_orientation == HORIZONTAL) {
      if (flip)
	rebox_applet (VERTICAL);
    } else {
      if (!flip)
	rebox_applet (HORIZONTAL);
    }
  } else {
    if (applet_box_orientation == HORIZONTAL) {
      if (!flip)
	rebox_applet (VERTICAL);
    } else {
      if (flip)
	rebox_applet (HORIZONTAL);
    }
  }

  r = settings.aspect_ratio;
  s = panel_pixel_size;

  /* count number of icons to show */
  for (i = 0; i < 3; i++)
    if (settings.lock_key_checked[i])
      n++;

  /* calculate "best" size for the icon(s) */
  if (panel_orientation == ORIENT_UP || panel_orientation == ORIENT_DOWN) {
    if (!flip) {
      ih = s;
      iw = (int) rint (((double)ih) * ((double)r));
    } else {
      ih = (int) rint (((double)s) / ((double)n));
      iw = (int) rint (((double)ih) * ((double)r));
    }
  } else {
    if (!flip) {
      iw = s;
      ih = (int) rint (((double)iw) / ((double)r));
    } else {
      iw = (int) rint (((double)s) / ((double)n));
      ih = (int) rint (((double)iw) / ((double)r));
    }
  }

  /* limit icon-size by min. max. settings */
  nw = iw;
  if (settings.minimum_checked && nw < settings.minimum)
    nw = settings.minimum;
  if (settings.maximum_checked && nw > settings.maximum)
    nw = settings.maximum;
  if (nw != iw) {
    iw = nw;
    ih = (int) rint (((double)iw) / ((double)r));
  }

  if (panel_orientation == ORIENT_UP || panel_orientation == ORIENT_DOWN)
    if (!flip) {
      ah = ih;
      aw = iw * n;
    } else {
      ah = ih * n;
      aw = iw;
    }
  else
    if (!flip) {
      ah = ih * n;
      aw = iw;
    } else {
      ah = ih;
      aw = iw * n;
    }

  if (applet_width != aw || applet_height != ah) {
    applet_width = aw;
    applet_height = ah;
    gtk_widget_set_usize (applet, aw, ah);
  }
  if (icon_width != iw || icon_height != ih || reload_images) {
    icon_width = iw;
    icon_height = ih;
    scale_pixmaps (iw - 2 * frame_border_size, ih - 2 * frame_border_size);
  }

  gtk_widget_show_all (applet);

  /* hide frame + pixmap for lock-keys that shouldn't be shown */
  for (i = 0; i < 3; i++)
    if (settings.lock_key_checked[i] == FALSE) {
      gtk_widget_hide (frame[i]);
      gtk_widget_hide (pixmap[key[i]]);
    }
}



void scale_pixmaps (int w, int h)
{
  int i;
  struct stat stat_buf;

  /* remove pixmaps from frames */
  for (i = 0; i < 3; i++)
    if (pixmap[key[i]]) {
      gtk_container_remove (GTK_CONTAINER(frame[i]), pixmap[key[i]]);
      pixmap[key[i]] = NULL;
    }

  for (i = 0; i < 6; i++) {
    /* unref unreffed pixmaps */
    if (pixmap[i])
      gtk_widget_unref (pixmap[i]);

    pixmap[i] = gtk_type_new (gnome_pixmap_get_type());
    if (settings.pixmap_file[i] &&
	(stat (settings.pixmap_file[i], &stat_buf) != -1)) {
      gnome_pixmap_load_file_at_size (GNOME_PIXMAP (pixmap[i]),
				      settings.pixmap_file[i], w, h);
    } else
      g_warning (_("Couldn't find pixmap file: '%s'"),settings.pixmap_file[i]);

    gtk_widget_ref (pixmap[i]);
  }
    
  /* reassign pixmaps to frames */
  for (i = 0; i < 3; i++) {
    gtk_widget_show (pixmap[key[i]]);
    gtk_container_add (GTK_CONTAINER (frame[i]), pixmap[key[i]]);
  }
}



void applet_change_orient (AppletWidget *widget,
			   PanelOrientType o, gpointer data)
{
  reshape_applet(FALSE);
}



void applet_change_pixel_size (AppletWidget *widget, int s, gpointer data)
{
  panel_pixel_size = s;
  reshape_applet(FALSE);
}



gint applet_save_session (AppletWidget *widget,
			  const char *privcfgpath, const char *globcfgpath)
{
  save_properties (privcfgpath);

  gnome_config_sync();
  gnome_config_drop_all();

  return FALSE;
}



void applet_tooltip_state (AppletWidget *widget, gint state, gpointer data)
{
  if (state)
    gtk_tooltips_enable (tooltip);
  else
    gtk_tooltips_disable (tooltip);
}



void applet_destroy (AppletWidget *widget, gpointer data)
{
  gtk_widget_destroy(applet);
  (void) pthread_cancel (led_th);
  pthread_exit (NULL);
}



void callback_help (AppletWidget *applet, gpointer data)
{
  GnomeHelpMenuEntry help_entry = { 
    PACKAGE, NULL
  };
  help_entry.path = data;
  gnome_help_display (NULL, &help_entry);
}



void callback_about (AppletWidget *applet, gpointer data)
{
  GtkWidget *about = NULL;
  const gchar *authors[] = { "rjan Nygaard Austvold <austvold@acm.org>",
			     NULL };
  const gchar *logo = PACKAGE_ICON_DIR"/"WINDOW_ICON;

  if (about != NULL) {
    gdk_window_show(about->window);
    gdk_window_raise(about->window);
    return;
  }

  about = gnome_about_new (_("Keyboard Led Applet"),
			   VERSION,
			   "Copyright (C) 2000 rjan Nygaard Austvold",
			   authors,
			   _("Display keyboard led status (num-, caps-, and "
			     "scroll-lock). Useful on keyboards without led "
			     "status (e.g., cordless keyboards)."),
			   logo);

  gtk_signal_connect (GTK_OBJECT (about), "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroyed), &about);
  gtk_widget_show (about);
  gdk_window_set_icon (about->window, NULL, icon_pixmap, icon_mask);
}
