/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 *  $Id: powerd.c,v 1.15 2000/10/10 19:16:41 moyer Exp $
 *
 *  authors: Jeff Moyer <moyer@missioncriticallinux.com>
 *           Tim Burke  <burke@missioncriticallinux.com>
 *
 *  description: Daemon which initializes and periodically checks status
 *           on the power switch.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/syslog.h>
#include <errno.h>
#include <malloc.h>
#include <assert.h>
#include <signal.h>
#include <sys/time.h>

#include <msgsvc.h>
#include <clusterdefs.h>
#include <logger.h>
#include <power.h>
#include <parseconf.h>
#include <clucfg.h>

static const char *version __attribute__ ((unused)) = "$Revision: 1.15 $";

#ifdef DEBUG
#define Dprintf(fmt,args...) printf(fmt,##args)
#define DBG_ASSERT(x)        assert(x)
#else
#define Dprintf(fmt,args...)
#define DBG_ASSERT(x)
#endif

#define POWERD_LOGLEVEL "powerd%logLevel"

/*
 * Globals
 */
static int           pswitch_initialized = 0;
static int           pswitch_failures    = 0;
static time_t        last_good_status    = 0;
static int           last_error          = 0;
static msg_handle_t  msgfd               = -1;

/*
 * XXX - Todo: make this configurable in the cluster database.  Probably
 *       was for the quorum daemon.
 */
static int           pswitch_interval = 10; /* Number of secs between polls. */

/*
 * Local prototypes
 */
static void usage(char *progname);
static int  powerd_read_config(void);
static int  pswitch_init(void);
static int  power_cycle_partner(void);
static int  get_pswitch_status(void);
static void do_power_loop(void);
static void setup_sig_handler(void);
static void sighup_handler(int signum);
static void poll_quorumd(void);
static void process_request(msg_handle_t fd, int cmd, int arg, int auth);
static void do_exit(void);


/*
 * External functions not in header files.
 */
extern void daemon_init(char *prog);


#ifdef DEBUG
/*
 * Our little prototype.
 */
void sigint_handler(int sig);

void
sigint_handler(int sig)
{
    Dprintf("loggerd: trapped signal %d.  attach with gdb.  DO IT NOW!\n",sig);
    for(;;);
}
#endif


static void
setup_sig_handler(void)
{
    sigset_t set;

    sigemptyset(&set);
#ifdef DEBUG
    signal(SIGSEGV,sigint_handler);
    sigaddset(&set, SIGSEGV);
#endif
    sigaddset(&set, SIGHUP);
    sigprocmask(SIG_UNBLOCK, &set, NULL);

    signal(SIGHUP, sighup_handler);

    return;
}


/*
 * Signal handler for hup signal. This causes a re-read of the configuration
 * parameters.  Stole most of this from tim.
 */
static void
sighup_handler(int signum)
{
    CFG_Destroy();	

    /*
     * PWR_configure always returns TRUE.
     */
    (void)PWR_configure(NULL);
    powerd_read_config();

    signal(signum, sighup_handler);
    return;
}


void
poll_quorumd(void)
{
    msg_handle_t          sockfd;
    static DiskMessageSt  *msg=NULL;
    CluCfg     	          *cfg=NULL;
    ssize_t               ret;

    if (cfg == NULL) {
	cfg = get_clu_cfg(NULL);
	if (cfg == NULL) {
	    /*
	     * Ick.  Log an error.
	     */
	    clulog(LOG_ERR, "poll_quorumd: Unable to read cluster config.\n");
	    return;
	}
    }

    sockfd = msg_open(PROCID_QUORUMD, cfg->lid);
    free(cfg);
    cfg = NULL;
    if (sockfd < 0) {
	clulog(LOG_ERR, "poll_quorumd: Unable to open comms w/ quorumd.\n");
	return;
    }

    if (!msg) {
	msg = (DiskMessageSt *)malloc(sizeof(DiskMessageSt));
	if (!msg) {
	    clulog(LOG_ERR, "poll_quorumd: Unable to allocate memory.\n");
	    msg_close(sockfd);
	    return;
	}
	
	msg->hdr.magic = GENERIC_HDR_MAGIC;
	msg->hdr.command = DISK_POWERD_ALIVE;
	msg->hdr.length = sizeof(DiskMessageSt);
	msg->data.daemonPid = getpid();
    }

    ret = msg_send(sockfd, msg, sizeof(DiskMessageSt));
    if (ret < 0) {
	clulog(LOG_WARNING, "poll_quorumd: Unable to send message to quorumd.\n");
    }
    msg_close(sockfd);

    return;
}


static void
usage(char *progname)
{
    fprintf(stderr, "Usage: %s\n", progname);
}


/*
 * CFG_Get returns strings; that's the reason for all of these tmp variables.
 * It's ugly, but it works.
 */
static int
powerd_read_config(void)
{
    char *loglevel_str;
    char lldflt;
    int  currll = 255;

    currll = clu_get_loglevel();
    snprintf(&lldflt, 1, "%d", currll);

    if (CFG_Get((char*)POWERD_LOGLEVEL, &lldflt, &loglevel_str) != CFG_OK) {
	return -1;
    }

    clu_set_loglevel(atoi(loglevel_str));
    return 0;
}


static int
pswitch_init(void)
{
    PWR_result status=0;

    if (PWR_init(&status) == PWR_FALSE) {
	pswitch_initialized = 0;
	if (status & PWR_ERROR)
	    errno = EIO;
	else 
	    errno = ETIMEDOUT;

	return -1;
    }
    pswitch_initialized = 1;
    return 0;
}


static void
pswitch_deinit(void)
{
    (void)PWR_release();
}


static int
get_pswitch_status(void)
{
    PWR_result status = 0;

    /*
     * Below we handle the case where perhaps the power switch
     * controlling the other node has been unplugged, has failed, 
     * or has just been replaced.
     */
    if (!pswitch_initialized) {
	if (pswitch_init() < 0) {
	    return -1;
	}
    }

    status = PWR_status();
    if (!(status & PWR_INIT)) {
	if (pswitch_init() < 0) {
	    pswitch_initialized = 0;
	    return -1;
	}
    }

    DBG_ASSERT(pswitch_initialized);

    if (status & PWR_TIMEOUT) {
	errno = ETIMEDOUT;
	return -1;
    } else if (status & PWR_ERROR) {
	errno = EIO;
	return -1;
    }

    return 0;
}


static int
power_cycle_partner(void)
{
    PWR_result status = 0;

    if (!pswitch_initialized) {
	clulog(LOG_CRIT, "power_cycle_partner: power switch not initialized.\n");
	return -1;
    }

    status = PWR_reboot();
    if (status & PWR_ERROR) {
	clulog(LOG_CRIT, "powerd: power_cycle_partner: error while talking "
                         "to power switch\n");
	errno = EIO;
	return -1;
    } else if (status & PWR_TIMEOUT) {
	clulog(LOG_CRIT, "powerd: power_cycle_partner: timeout while talking "
	                 "to power switch\n");
	errno = ETIMEDOUT;
	return -1;
    }

    return 0;
}


/*
 * void do_power_loop(void)
 *
 * This loop takes care of querying the power switch for status
 * information.  We will notify the quorumd upon changes in state
 * of the power switch.  This essentially entails telling the quorumd
 * that we can't reach the switch anymore.
 */
static void
do_power_loop()
{
    int                retval=-1, nfds=0, auth=0;
    time_t             time_start, 
	               time_now, elapsed;
    struct timeval     tv;
    fd_set             readfds;
    generic_msg_hdr    msghdr;
    msg_handle_t       acceptfd;
    PswitchCmdMsg      msg;

    while (1) {
	    poll_quorumd();
	    time_start = time(NULL);
	    retval = get_pswitch_status();
	    if (retval == 0) { /* success */
		clulog(LOG_DEBUG, "do_power_loop: pswitch_status good.\n");
		last_good_status = time(NULL);
		pswitch_failures = 0;
		last_error = 0;
	    } else {
		last_error = errno;
		pswitch_failures++;
		clulog(LOG_ERR, "do_power_loop: pswitch_status bad: %d.  "
		                  " %d errors\n", errno, pswitch_failures);
	    }
	    /*
	     * Wait for messages.
	     */
	    time_now = time(NULL);
	    elapsed = time_now - time_start;

	    tv.tv_sec = elapsed > pswitch_interval ? 0 : 
		pswitch_interval - elapsed;
	    tv.tv_usec = 0;

	    /*
	     * Message Service stuff goes in here.  Mainly, select on fd.
	     */
	    do {
		FD_ZERO(&readfds);
		FD_SET(msgfd, &readfds);
		msg.arg = 0;

		nfds = select(msgfd+1, &readfds, NULL, NULL, &tv);

		if (nfds == -1)
		    continue;

		if (!FD_ISSET(msgfd, &readfds))
		    continue; /* select returned for perhaps an interrupt? */

		/*
		 * Have to define the messaging protocol between quorumd
		 * and the power daemon.
		 */
		acceptfd = msg_accept(msgfd);

		retval = msg_receive(acceptfd, &msghdr, sizeof(msghdr), &auth);
		if (retval != sizeof(msghdr)) {
		    /*
		     * Bad read?
		     */
		    msg_close(acceptfd);
		    continue;
		}

		/*
		 * Make sure this is a valid message header.
		 */
		if (msghdr.magic != GENERIC_HDR_MAGIC) {
		    msg_close(acceptfd);
		    clulog(LOG_ERR, "do_power_loop: Bad header magic.\n");
		    continue;
		}

		if (msghdr.length) {
		    /*
		     * Currently, we have no commands that need any
		     * arguments or fancy data structures.  For now, 
		     * we just close the socket and process what we got
		     * in the header.
		     */
		    if ((unsigned int)msghdr.length > sizeof(msg)) {
			clulog(LOG_ERR, "do_power_loop: Unsupported message argument\n");
			msg_close(acceptfd);
		    } else {
			msg_receive(acceptfd, &msg, sizeof(msg), &auth);
		    }
		}
		process_request(acceptfd, msghdr.command, 
				msg.arg > 0 ? msg.arg : 0, auth);

	    } while (tv.tv_sec | tv.tv_usec);
    } /* while(1) */
}

/*
 * Notice that we do not send a message header back.  The connecting
 * process knows the return type and size of the data.  The header 
 * would just be redundant information.
 */
static void
process_request(msg_handle_t fd, int cmd, int arg, int auth)
{
    int ret;
    PswitchStatusMsg msg;

    memset(&msg, 0, sizeof(msg));

    switch (cmd) {

    case PSWITCH_QUERY:
	/*
	 * Fill in a PswitchStatusMsg
	 */
	clulog(LOG_DEBUG, "process_request: PSWITCH_QUERY\n");
	msg.status = last_error;
	msg.timestamp = last_good_status;

	ret = msg_send(fd, &msg, sizeof(msg));
	if (ret != sizeof(msg)) {
	    /*
	     * Potentially log something?
	     */
	    break;
	}
	/*
	 * Successfully sent out response.
	 */
	break;

    case PSWITCH_RESET:
	if (!auth) {
	    /*
	     * Some script-kiddie is trying to shutdown the other node.
	     * DENIED!
	     */
	    clulog(LOG_CRIT, "*** process_request: Unauthorized attempt to "
		   "shoot down remote node! ***\n");
	    msg_close(fd);
	    return;
	}
	/*
	 * Authenticated message received.
	 */
	clulog(LOG_DEBUG, "process_request: PSWITCH_RESET\n");
	msg.status = power_cycle_partner();
	msg.timestamp = last_good_status;
	clulog(LOG_DEBUG, "process_request: status %d, last good timestamp "
	       "%d, current timestamp %d\n",
	       msg.status, msg.timestamp, time(NULL));
	ret = msg_send(fd, &msg, sizeof(msg));
	if (ret != sizeof(msg)) {
	    /*
	     * Maybe log an error.
	     */
	    clulog(LOG_CRIT, "process_request: Unable to complete socket write.\n");
	}
	break;

    case PSWITCH_TERM:
	if (!auth) {
	    /*
	     * Some script-kiddie is trying to shutdown powerd.
	     * DENIED!
	     */
	    clulog(LOG_CRIT, "*** process_request: Unauthorized attempt to "
		   "terminate powerd! ***\n");
	    msg_close(fd);
	    return;
	}
	clulog(LOG_DEBUG, "process_request: PSWITCH_TERM\n");
	msg.status = PSWITCH_ACK;
	ret = msg_send(fd, &msg, sizeof(msg));
	msg_close(fd); /* we don't need to check the return value. */
	do_exit(); /* Exits the program. */
	break;

    case PSWITCH_SET_LOGLEVEL:
	if (!auth) {
	    /*
	     * Some script-kiddie is trying to change our loglevel.
	     * DENIED!
	     */
	    clulog(LOG_CRIT, "*** process_request: Unauthorized attempt to "
		   "change logging level! ***\n");
	    msg_close(fd);
	    return;
	}
	(void)clu_set_loglevel(arg);
	msg.status = PSWITCH_ACK;
	ret = msg_send(fd, &msg, sizeof(msg));
	if (ret != sizeof(msg)) {
	    clulog(LOG_CRIT, "process_request: Unable to complete socket write.\n");
	}
	break;

    default:
	clulog(LOG_WARNING, "process_request: Received invalid command.\n");
	break;
    }

    msg_close(fd);
    return;
}


static void
do_exit()
{
    pswitch_deinit();
    msg_close(msgfd);
    clulog_close();
    exit(0);
}


int
main(int argc, char **argv)
{
    if (argc > 1) {
	usage(argv[0]);
	exit(1);
    }

#ifndef DEBUG
    daemon_init(argv[0]);
#else
    clu_set_loglevel(LOG_DEBUG);
#endif
    setup_sig_handler();

    powerd_read_config();
    msgfd = msg_listen(PROCID_POWERD);
    if (msgfd < 0) {
	clulog(LOG_CRIT, "main: Unable to setup listen file descriptor.  Exiting\n");
	Dprintf("powerd: Listen on socket failed.\n");
	exit(1);
    }

    if (pswitch_init() < 0) {
	last_error = errno;
	clulog(LOG_ERR, "main: pswitch_init failed with error %d\n", errno);
    }

    do_power_loop();
    /* NOTREACHED */
    return 0; /* make gcc happy */
}
/*
 * Local variables:
 *  c-basic-offset: 4
 *  c-indent-level: 4
 *  tab-width: 8
 * End:
 */
