/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/

/* clu_config.c

   author: Ron Lawrence <lawrence@missioncriticallinux.com>
*/
#include <parseconf.h>
#include <argp.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <malloc.h>

static const char *version __attribute__ ((unused)) = "$Id: clu_config.c,v 1.11 2000/08/23 22:16:59 lawrence Exp $";

int initAlignedBufStuff(void);
     
/* Program documentation. */
static char doc[] = "clu_config:  read or manipulate cluster parameters\v"

"If no output file is given, writes to the shared database.\n"
"If no input file is given, reads from the shared database.\n"
"If file input, or output file are given as '-', \n"
"\tthen reading or writing is to/from stdin/stdout.\n"
"The default separator is '%', and cannot be modified.\n"
"Note that match takes precedence over expunge, which "
"takes precedence over get and put.\n\n";

/********************************************************************/     
/* A description of the arguments we accept. */
static char args_doc[] = "[KEY] [VALUE]";
     
/* The options we understand. */
static struct argp_option options[] = {
  {"verbose",  'v', 0,          0, "Produce verbose output",              0},
  {"output",   'o', "OUTFILE",  0, "Write to FILE",                       0},
  {"put",      'p', 0,          0, "Set key to value",                    0},
  {"get",      'g', 0,          0, "Get value for key",                   0},
  {"match",    'm', "PATTERN",  0, "Get entries that match pattern",      0},
  {"file",     'f', "INFILE",   0, "The config file to parse",            0},
  {"remove",   'r', 0,          0, "Remove a key",                        0},
  {"expunge",  'x', "PATTERN",  0, "Remove all keys that match pattern",  0},
  {"default",  'd', "DEFAULT",  0, "The default if a value is not found", 0},
  {"version",  'V', 0,          0, "Print version number and exit",       0},
  {"init",     'i', "PARTITION",0, "Copy cluster config file from the quorum partition", 0},
  { 0,0,0,0,0,0 }
};

/* Used by `main' to communicate with `parse_opt'. */
struct arguments
{
    char *args[2];                /* KEY & VALUE */
    int verbose;
    char *output_file;
    char *input_file;
    char *quorum_device;
    char *match;
    char *expunge;
    int get, put, remove;
    char* dflt;
};
     
#define STREQ(s1,s2) (!strcmp((s1),(s2)))

/* Parse a single option. */
static error_t
parse_opt (int key, char *arg, struct argp_state *state)
{
/* Get the INPUT argument from `argp_parse', which we
   know is a pointer to our arguments structure. */
  struct arguments *arguments = state->input;
     
  switch (key)
  {
    case 'V':
      fprintf(stdout, "clu_config version: %s\n", version);
      exit(0);
      break;
    case 'v':
      arguments->verbose = 1;
      break;
    case 'o':
      arguments->output_file = arg;
      break;
    case 'p':
      arguments->put = 1;
      arguments->get = 0;
      arguments->remove = 0;
      break;
    case 'g':
      arguments->get = 1;
      arguments->put = 0;
      arguments->remove = 0;
      break;
    case 'm':
      arguments->match = arg;
      break;
    case 'x':
      arguments->expunge = arg;
      break;
    case 'r':
      arguments->put = 0;
      arguments->get = 0;
      arguments->remove = 1;
      break;
    case 'f':
      arguments->input_file = arg;
      break;
    case 'd':
      arguments->dflt = arg;
      break;
    case 'i':
      arguments->quorum_device = arg;
      break;
    case ARGP_KEY_ARG:
      /* Should it be possible to specify multiple queries or multiple
         key value pairs on the command line?*/
      if(STREQ("",arguments->match) &&
         STREQ("", arguments->expunge)) {
        if (arguments->put && state->arg_num >= 2)
          /* Too many arguments. */
          argp_usage (state);
        if (arguments->get && state->arg_num >= 1)
          /* Too many arguments. */
          argp_usage (state);
      }
      arguments->args[state->arg_num] = arg;
     
      break;
     
    case ARGP_KEY_END:
      break;
     
    default:
      return ARGP_ERR_UNKNOWN;
  }
  return 0;
}
int getDatabaseFromPartition(char* partitionName, 
			     char* buffer, 
			     ssize_t length);

#define BUFFER_SIZE 2048
int copy_shared_database_file(char *partition_name) {
  ssize_t length;
  char *buffer;
  ssize_t res;

  if (initAlignedBufStuff() < 0) {
    fprintf(stderr, __FUNCTION__ 
	    ": Unable to init rawio support.\n");
    return(CFG_FAILED);
  }
  length = BUFFER_SIZE;

  buffer = (char*)malloc(length+1);
  if(NULL == buffer)
    goto failed;
  memset(buffer, 0, length);

  res = getDatabaseFromPartition(partition_name, buffer, length);
  
  if(res >= 0) 
    {
      FILE *f;
      f = fopen(CLU_CONFIG_FILE,"w");
      if(NULL == f)
        goto failed;
      fwrite(buffer,sizeof(char),strlen(buffer),f);
      fclose(f);
      free(buffer);
    }
  else {
    free(buffer);
    goto failed;
  }
  return CFG_OK;
 failed:
  return CFG_FAILED;
}
     
/* Our argp parser. */
static struct argp argp = { options, parse_opt, args_doc, doc,0,0,0 };

static int must_write = 1;
     
int main (int argc, char **argv)
{
  char * result;
  int rc;
  struct arguments arguments;
     
  if (geteuid() != (uid_t)0) {
    fprintf(stderr, "%s must be run as the user root\n", argv[0]);
    exit(1);    
  }

  /* Default values. */
  arguments.args[0] = "";
  arguments.args[1] = "";
  arguments.verbose = 0;
  arguments.output_file = "";
  arguments.input_file = "";
  arguments.quorum_device = "";
  arguments.get = 0;
  arguments.put = 0;
  arguments.match = "";
  arguments.expunge = "";
  arguments.remove = 0;
  arguments.dflt = "not found";

  /* Parse our arguments; every option seen by `parse_opt' will be
     reflected in `arguments'. */
  argp_parse (&argp, argc, argv, 0, 0, &arguments);
     
  if(arguments.verbose) {
    printf("ARG1 = %s\nARG2 = %s\nOUTPUT_FILE = %s\n"
           "VERBOSE = %s\n"
           "INPUT_FILE = %s\nGET = %s\nPUT = %s\n"
           "MATCH = %s\nEXPUNGE = %s\n"
           "REMOVE = %s\nDEFAULT = %s\n",
           arguments.args[0], arguments.args[1],
           arguments.output_file,
           arguments.verbose ? "yes" : "no",
           arguments.input_file,
           arguments.get ? "yes" : "no",
           arguments.put ? "yes" : "no",
           arguments.match,
           arguments.expunge,
           arguments.remove ? "yes" : "no",
           arguments.dflt );
  }

  /*********************************************************************/
  /*  Always read in the database.*/
  /* choose the input channel... */
  if ( !STREQ("", arguments.quorum_device)) {
    /* Get the contents of the shared database on the quorum disk, and
       blast them out to the config file on disk. */
    if(CFG_OK == copy_shared_database_file(arguments.quorum_device)) {

      /* After writing out the local copy of the cluster database, we
	 are finished.*/
      return(0);
    }
    else {
      fprintf(stderr, "Failed to copy specified partition: %s\n",
	      arguments.quorum_device);
      return(1);
    }
  }
  else if(STREQ("-",arguments.input_file)) { /* read from stdin. */
    /* Will need to read stdin into a buffer, until eof. Then pass the
       resulting buffer to CFG_get. */
    rc = CFG_ReadFD(stdin);
    if(CFG_OK != rc) {
      fprintf(stderr,"%s: failed to read stdin\n",argv[0]);
      return rc;
    }
  }
  else if(STREQ("", arguments.input_file)) { /* Read from shared storage. */

    rc = CFG_Read();
    if(CFG_OK != rc) {
      fprintf(stderr,"%s: failed to read from shared database\n",argv[0]);
      return rc;
    }
  }
  else {                        /* Read from file specified. */
    rc = CFG_ReadFile(arguments.input_file);
    if(CFG_OK != rc) {
      fprintf(stderr,"%s: failed to read file %s\n",
              argv[0],arguments.input_file);
      return rc;
    }
  }
  /*********************************************************************/
  /* Database is read in, look up item(s). */

  /* Lookup keys that match the given pattern. */
  if( !STREQ("",arguments.match)) {
    struct CFG_Iter *it;
    char *key, *value;
    CFG_status res;

    CFG_CreateGlobIterator(arguments.match, &it);
    while(CFG_IteratorMore(it)) {
      res = CFG_IteratorNext(it, &key, &value);
      if(res) {
        if(arguments.verbose) 
          printf("key: [%s], value: [%s]\n", key, value);
        else
          printf("%s = %s\n", key,value);
      }
    }
    CFG_DestroyIterator(it);
    /* This is a lookup, so, we don't need to write anything out.*/
    must_write = 0;
  }
  else if( !STREQ("",arguments.expunge)) {
    struct CFG_Iter *it;
    char *key, *value;
    CFG_status res;

    CFG_CreateGlobIterator(arguments.expunge, &it);
    while(CFG_IteratorMore(it)) {
      res = CFG_IteratorNext(it, &key, &value);
      if(res) 
        CFG_Remove(key);
    }
    CFG_DestroyIterator(it);
  }  
  else if(arguments.get) {
    /* Look up one key. */
    rc = CFG_Get(arguments.args[0],arguments.dflt,&result);

    if(CFG_OK == rc) {
      printf("%s\n",result);
      /* On a get we don't need to write out the database. */
      must_write = 0;
    } 
    else {
      switch(rc) {
        case CFG_NO_SUCH_SECTION:
          if(arguments.verbose) {
            fprintf(stderr,"No such path: %s\n", arguments.args[0]);
          }
          return(-rc);
          break; 
        case CFG_DEFAULT:
          if(arguments.verbose) {
            fprintf(stderr,"Using default: %s\n", result);
          } else {
            printf("%s\n",arguments.dflt);
          }
          return(-rc);
          break;
        case CFG_FAILED:
          if(arguments.verbose) {
            fprintf(stderr,"Failed\n");
          }
          return(-rc);
          break;
        case CFG_LINE_TOO_LONG:
          if(arguments.verbose) {
            fprintf(stderr,"Line too long\n");
          }
          return(-rc);
          break;
        case CFG_PARSE_FAILED:
          if(arguments.verbose) {
            fprintf(stderr,"Parse Failed\n");
          }
          return(-rc);
          break;
      }
    }
  } 
  else if(arguments.put) {
    rc = CFG_Set(arguments.args[0], arguments.args[1]);
    if(CFG_OK == rc) {
      if(arguments.verbose)
        printf("%s\n",arguments.args[1]);
    } 
    else {
      switch(rc) {
        case CFG_NO_SUCH_SECTION:
          if(arguments.verbose) {
            fprintf(stderr,"No such path: %s\n", arguments.args[0]);
          }
          return(-rc);
          break; 
        case CFG_DEFAULT:
          if(arguments.verbose) {
            fprintf(stderr,"Using default: %s\n", result);
          } else {
            printf("%s\n",arguments.dflt);
          }
          return(-rc);
          break;
        case CFG_FAILED:
          if(arguments.verbose) {
            fprintf(stderr,"Failed\n");
          }
          return(-rc);
          break;
        case CFG_LINE_TOO_LONG:
          if(arguments.verbose) {
            fprintf(stderr,"Line too long\n");
          }
          return(-rc);
          break;
        case CFG_PARSE_FAILED:
          if(arguments.verbose) {
            fprintf(stderr,"Parse Failed\n");
          }
          return(-rc);
          break;
      }
    }
  }
  else if(arguments.remove) {
    rc = CFG_Remove(arguments.args[0]);
    if(CFG_OK == rc) {
      if(arguments.verbose)
        printf("removed: %s\n",arguments.args[0]);
    } 
    else {
      switch(rc) {
        case CFG_NO_SUCH_SECTION:
          if(arguments.verbose) {
            fprintf(stderr,"No such path: %s\n", arguments.args[0]);
          }
          return(-rc);
          break; 
        case CFG_DEFAULT:
          if(arguments.verbose) {
            fprintf(stderr,"Using default: %s\n", result);
          } else {
            printf("%s\n",arguments.dflt);
          }
          return(-rc);
          break;
        case CFG_FAILED:
          if(arguments.verbose) {
            fprintf(stderr,"Failed\n");
          }
          return(-rc);
          break;
        case CFG_LINE_TOO_LONG:
          if(arguments.verbose) {
            fprintf(stderr,"Line too long\n");
          }
          return(-rc);
          break;
        case CFG_PARSE_FAILED:
          if(arguments.verbose) {
            fprintf(stderr,"Parse Failed\n");
          }
          return(-rc);
          break;
      }
    }
  }
  else {
    /* no action specified, ok? */
  }
  /********************************************************************/
  /* Choose the output channel... */
  if(STREQ("-", arguments.output_file)) { /* Write results to stdout. */
    rc = CFG_WriteFD(stdout);
    if(CFG_OK != rc) {
      fprintf(stderr,"%s: failed to write to stdout\n",argv[0]);
      return rc;
    }
  }
  else if(STREQ("", arguments.output_file)) { /* Write to shared storage. */

    if( must_write ) {          /* We weren't asked to write the
                                   database anywhere, so only write it
                                   if it has been changed. */
      CFG_Write();
      if(CFG_OK != rc) {
        fprintf(stderr,"%s: failed to write to shared database\n",argv[0]);
        return rc;
      }
    } 
    else {
      return 0;
    }
  }
  else {                        /* Write to the specified file. */
    CFG_WriteFile(arguments.output_file);
    if(CFG_OK != rc) {
      fprintf(stderr,"%s: failed to write file %s\n",
              argv[0],arguments.output_file);
      return rc;
    }
  }
  CFG_Destroy();

  exit (0);
}

