/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 *  $Id: diskutil.c,v 1.20 2000/11/20 22:26:09 burke Exp $
 *
 *  Copyright (C) 2000 Mission Critical Linux, LLC
 *
 *  author: Tim Burke <burke@missioncriticallinux.com>
 *  description: "main" entrypoint to disk functions.
 *
 * This program is a test scaffolding used to exercise shared disk state
 * information to be used in a clustered environment.
 */
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <signal.h>
#include <stdlib.h>

#include <clu_lock.h>
#include <diskapis.h>
#include "diskstate.h"
#include "disk_proto.h" 
#include <parseconf.h>
#include <clucfg.h>


static const char *version __attribute__ ((unused)) = "$Id: diskutil.c,v 1.20 2000/11/20 22:26:09 burke Exp $";

extern int delay_shift;
extern int cluConfigChangeNotification(void);
void lock_test(void);
int test_write_database(void);
int test_write_database_corrupt_sync(void);
void lock_test_2(void);
#ifdef CLU_LOCK_MULTITHREADED
void lock_test_mt(void);
#endif
void printPartSize(void);

int 	nodeNumber = -1;
int	initPartition = 0;
char    configFile[MAXPATHLEN];

/*
 * Forward reference declarations
 */
void printHelpMessage(void);
int do_debug_cmd_loop(void);
int testConfigRead(char *filename);
int testConfigWrite(char *filename);
void printNodeStates(void);
int print_quorum_header(void);
static int check_hb_config(void);


int
main(
    int                 argc,
    char                *argv[])
{                         
    char opt;
    int prompt = 0;
    int do_menu = 1;
    int do_access_test = 0;
    int do_initialized_test = 0;
    int do_print_header = 0;
    int retval;

    if (geteuid() != (uid_t)0) {
          fprintf(stderr, "%s must be run as the root user.\n",
                argv[0]);
        exit(1);    
    }

    /*
     * Parameters:
     * -h Displays a usage help message.
     * -i Initializes a partition after prompting first.
     * -I Initializes a partition without prompting.
     * -p Prints out the quorum partition header.
     * -t Partition accessibility validation test.
     * -r Partition accessibility validation test plus initialized test.
     */
    while ((opt = getopt(argc, argv, "hiIprt")) != EOF) {
        switch (opt) {
        case 'h':
	    printHelpMessage();
	    return(0);
            break;
        case 'i': // prompt
            initPartition = 1;
	    do_menu = 0;
            break;
        case 'I': // don't prompt
            initPartition = 2;
	    do_menu = 0;
            break;
        case 'p': 
	    do_menu = 0;
	    do_print_header = 1;
            break;
        case 'r': 
	    do_initialized_test = 1;
	    // fallthrough...
        case 't': 
	    do_menu = 0;
	    do_access_test = 1;
            break;
        default:
            break;
        }
    }           
    /*
     * By default, the CFG libraries will attempt to retrieve configuration
     * information from the shared disk. However when running things like
     * `diskutil -I` you may not have a shared disk established; consequently
     * here we force it to read from the filesystem based configuration
     * file.  While at it, we disable locking because that too looks for
     * config parameters.
     */
    if (CFG_cluster_locking_off() != CFG_OK) {
      fprintf(stderr,"ERROR: Unable to disable cluster locking.\n");
      return(1);
    }
    retval = CFG_ReadFile(CLU_CONFIG_FILE);
    if(CFG_OK != retval) {
      fprintf(stderr,"ERROR: Unable to open config file %s.\n",CLU_CONFIG_FILE);
      return(1);
    }

    // Rudimentary check that heartbeat channels are configured.
    if (check_hb_config() < 0) {
	fprintf(stderr, "ERROR: heartbeat channels are misconfigured.\n");
	return(1);
    }
    // Get diagnostic print level from cluster config file.
    getVerboseLevel();

    /*
     * Perform a sanity check on the partition offsets.
     */
    if (offsetParanoiaCheck()) {
	fprintf(stderr,"ERROR: Aborting due to partition offset errors.\n");
	return(1);
    }

    /*
     * If the user has requested a partition initialization, prompt them
     * for confirmation as this can be a catastrophic operation if used
     * improperly.
     */
    if (initPartition == 1) {
	prompt = 1;
    }

#ifdef DO_RAW_BOUNCEIO
    // Must init raw IO stuff prior to partition initialization.
    if (initAlignedBufStuff() < 0) {
        fprintf(stderr, "diskstate: unable to init rawio support.\n");
        return(-1);
    }
#endif // DO_RAW_BOUNCEIO   
    /*
     * Initialize partition with "magic number" if requested to do so.
     */
    if (initPartition > 0) {
	if (initializePartition(prompt) != 0) {
		fprintf(stderr, "Program Aborted - failed partition initialization.\n");
		return(1);
	}
    }

    /*
     * A request has been made to verify accessibility of the shared disk
     * quorum partitions.
     */
    if (do_access_test) {
	SharedStateHeader hdr;
	/*
	 * Call off to partition init routine.  This tries to retrieve
	 * the partition names from the config file, verify the device
	 * special file and open it up.  The return type for all errors
	 * here is always -1, so its not easy to distinguish, but it
	 * does log the specific error.
	 */
	retval = initSharedFD();
	if (retval != 0) {
	    fprintf(stderr, "Unable to validate quorum partitions.\n");
	    return(1);
	}
	if (do_initialized_test) {
	    retval = readHeaderNoshaddowUnlocked(&hdr);
	    if (retval != 0) {
	        fprintf(stderr, "Quorum partitions are accessible, but not initialized.\n");
	        return(1);
	    }
	}
	printf("Successfully verified that the quorum partitions are accessibile");
	if (do_initialized_test)
	    printf(" and initialized");
	printf(".\n");
	return(0);
    }

    /*
     * A request has been made to read and print out the quorum disk
     * partition header.  Particularly useful to verify that 2 systems are
     * talking to the same physical device.
     */
    if (do_print_header) {
	if (print_quorum_header() < 0) {
	    fprintf(stderr, "Unable to access quorum disk header. Quorum disks\n");
	    fprintf(stderr, "may not be configured or initialized.\n");
	    return(1);
	}
    }

    if (do_menu) {
        /*
         * Put up a command menu allowing the user to issue commands for
         * debugging purposes.
         */
	fprintf(stderr, "\nWARNING - running this utility in this interactive mode\n");
	fprintf(stderr, "is only intended for development and debugging purposes.\n");
	fprintf(stderr, "General usage is not recommended as many of these operations\n");
	fprintf(stderr, "are dangerous and/or incomplete.\n");
	fprintf(stderr, "\nChose the \'x\' option to exit.\n\n");
        do_debug_cmd_loop();
    }

    /*
     * Release resources associated with the service state and disk
     * locking subsystems.
     */
    if (closeServiceSubsys() != 0) {
	fprintf(stderr, "Unable to close disk locking subsystem.\n");
	return(1);
    }
    if (closeLockSubsys() != 0) {
	fprintf(stderr, "Unable to close disk locking subsystem.\n");
	return(1);
    }
    return(0);
}

void printHelpMessage(void) {
	fprintf(stderr, "Program Options:\n");
	fprintf(stderr, "-h\tPrints this help message.\n");
	fprintf(stderr, "-i\tInitialize partition after prompting for confirmation.\n");
	fprintf(stderr, "-I\tInitialize partition without prompting for confirmation.\n");
	fprintf(stderr, "-p\tPrints out the header of the quorum partition.\n");
	fprintf(stderr, "-t\tPerform quorum partition accessibility check.\n");
}

void printCmdMessage(void) {
	fprintf(stderr, "Command Options:\n");
	fprintf(stderr, "-b <num>\tPrints the state of the specified service number.\n");
	fprintf(stderr, "-c\tSends configuration change notification to quorumd.\n");
	fprintf(stderr, "-d <num>\tDeletes the specified service number.\n");
	fprintf(stderr, "-e\tLists the set of services.\n");
	fprintf(stderr, "-c\tCloses the shared state partition.\n");
	fprintf(stderr, "-h\tPrints this list of commands.\n");
	fprintf(stderr, "-i\tInitialize partition after prompting for confirmation.\n");
	fprintf(stderr, "-I\tInitialize partition without prompting for confirmation.\n");
	fprintf(stderr, "-j\tReads in /tmp/db_test_orig and writes to disk configuration database.\n");
	fprintf(stderr, "-k\tCopies the disk configuration database to /tmp/db_test_copy.\n");
	fprintf(stderr, "-l\tSets this node's lock block to TAKEN.\n");
	fprintf(stderr, "-L\tSets this node's lock block to FREE.\n");
	fprintf(stderr, "-m\tPrints out both node's lock blocks.\n");
	fprintf(stderr, "-n\tPrints out both node's state.\n");
	fprintf(stderr, "-o\tOpens the shared state partition.\n");
	fprintf(stderr, "-p\tPrints the partition header.\n");
	fprintf(stderr, "-P\tPrints both node status blocks.\n");
	fprintf(stderr, "-q\tBreaks out of the command prompt loop.\n");
	fprintf(stderr, "-r\tReads in the partition header.\n");
	fprintf(stderr, "-s\tDisplays minimum size of shared state partition.\n");
	fprintf(stderr, "-S\tPerforms a read scan of the entire shared state partition.\n");
	fprintf(stderr, "-u\tSets both lock states to free.\n");
	fprintf(stderr, "-U\tReinitializes configuration database to empty.\n");
	fprintf(stderr, "-w\tWrites out the partition header.\n");
	fprintf(stderr, "-W\tDeletes all services.\n");
	fprintf(stderr, "-x\tBreaks out of the command prompt loop.\n");
}

/*
 * Put up an interactive menu to allow the user to control the sequence
 * of steps for debugging purposes.
 */
void test_trash(void);
void test_trash_loop(void);
int do_debug_cmd_loop(void) {
	char c;
	int proceed = 1;
	SharedStateHeader hdr;
	char inputLine[132];
	int svcNum;
	int scanCount;
	ServiceBlock svcBlk;
	DiskLockBlock lock_block;
	int ret;

	bzero((void *)&hdr, sizeof(SharedStateHeader));
	bzero((void *)&lock_block, sizeof(DiskLockBlock));

	printCmdMessage();
	while (proceed) {
	    printf("CMD> ");
	    if (fscanf(stdin, "%s", (char *)&inputLine) < 1) {
		break;
	    }
	    scanCount = sscanf(inputLine,"%c", &c);
	    if (scanCount< 1) {
		fprintf(stderr, "Error reading command input.\n");
		continue;
	    }
	    /*
	     * Get the service number.
	     * XXX - deplorable command line parsing.
	     */
	    if ((c == 'd') || (c == 'b') || (c == 'B') || (c == 'f')) {
	        if (fscanf(stdin, "%s", (char *)&inputLine) < 1) {
		    break;
	        }
	        scanCount = sscanf(inputLine,"%d", &svcNum);
	        if (scanCount< 1) {
		    fprintf(stderr, "Error reading service number.\n");
		    continue;
	        }
	    }
	    if ((c == 'r') || (c == 'w')) {
	        if (fscanf(stdin, "%s", (char *)&configFile) < 1) {
		    break;
	        }
	    }
	    switch (c) {
		case '\n': printf("\n"); break; // ignore
		case 'b': if (getServiceStatus(svcNum, &svcBlk) == 0)
			    printf("Service %d state is %s.\n", svcNum,
				serviceStateStrings[svcBlk.state]);
			  else printf("getServiceStatus %d failed.\n", svcNum);
			  break;
		case 'c': cluConfigChangeNotification();break;
		case 'd': if (removeService(svcNum) == 0)
			    printf("Removed service %d.\n", svcNum);
			  else printf("removeService %d failed.\n", svcNum);
			  break;
                case 'D': delay_shift--;
                        printf("delay_shift = %d\n", delay_shift);
                        break;

		case 'e': printServiceList(); break;
		case 'E': test_write_database_corrupt_sync(); break;
	        case 'F': test_write_database(); break;
		case '?': 
		case 'h': printCmdMessage(); break;
		case 'i': initializePartition(1); 
			    break;
		case 'I': initializePartition(0); 
			    break;
		case 'j': testConfigWrite(configFile); 
			    break;
		case 'k': testConfigRead(configFile); 
			    break;
#ifdef DEBUG
		case 'l': lock_block.lockData = 1; lockWrite(nodeNumber, &lock_block); break;
		case 'L': lock_block.lockData = 0; lockWrite(nodeNumber, &lock_block); break;
#endif // DEBUG
		case 'm': printBothLockBlocks(1); break;
 	        case 'M': lock_test(); break;
		case 'n': printNodeStates(); break;
 	        case 'O': lock_test_2(); break;
		case 'o': break;
		case 'p': printSharedStateHeader(&hdr); break;
		case 'P': printStatusBlocks(1); break;
		case 'q': 
		case 'Q': proceed = 0; break;
		case 'r': readHeader(&hdr); break;
#ifdef CLU_LOCK_MULTITHREADED
	        case 'R': lock_test_mt(); break;
#endif
		case 's': printPartSize(); break;
		case 'S': scanWholeDisk(); break;
		case 'u': initializePartitionLockBlocks();
				 break;
		case 'U': initializeConfigDatabase();
				 break;
 	        case 'v':
			  ret = clu_try_lock();
			  printf("ret = %d\n", ret);
			  break;
		case 'V':
			  assert_clu_lock_held("test");
			  break;
		case 'w': writeHeader(&hdr); break;
		case 'W': initializePartitionServiceState();
				 break;
		case 'x': proceed = 0; break;
		case 'X': proceed = 0; break;
		case 'y': clu_lock(); break;
		case 'Y': clu_un_lock(); break;
		case 'Z': test_trash(); break;
		case 'z': test_trash_loop(); break;
		default: 
			fprintf(stderr, "Unrecognized command: %c.\n",c);
			printCmdMessage();
	    }
	}

	return(0);
}

/*
 * Testing routine to read in the contents of the file specified by the
 * filename parameter and write it out to the on-disk configuration
 * "database" area.  Basically a file copy from the filesystem to the
 * shared state disk.
 */
int testConfigWrite(char *filename) {
    struct stat stat_st, *stat_ptr;
    off_t length;
    char *buf;
    ssize_t retval;
    int fd;

    // XXX - shameless hack around command line parsing bungle.
    filename = "/tmp/db_test_orig";

    /*
     * First verify that the file exists and its length.
     */
    stat_ptr = &stat_st;
    if (stat(filename, stat_ptr) < 0) {
	fprintf(stderr, "Unable to stat %s.\n", filename);
	perror("stat");
	return(1);
    }
    length = stat_st.st_size;
    fd = open(filename, O_RDONLY);
    if (fd < 0) {
	fprintf(stderr, "Unable to open %s\n", filename);
	perror("open");
	return(1);
    }
    /*
     * Now allocate a memory buffer to read the file data into.
     * This is a kludge to grab an in-memory buffer for the whole file.
     * Really only works for small files, but for testing purposes it will
     * suffice.
     */
    buf = malloc(length);

    retval = read(fd, buf, length);
    if (retval != length) {
	fprintf(stderr, "Unable to read %s.\n", filename);
	perror("read");
	close(fd);
	free(buf);
	return(1);
    }

    clu_lock();

    retval = writeDatabase(buf, length);

    clu_un_lock();

    if (retval != length) {
	fprintf(stderr, "Unable to write %d bytes to disk config DB.\n", (int)length);
	perror("writeDatabase");
	close(fd);
	free(buf);
	return(1);
    }

    fprintf(stderr, "Successfully wrote %d bytes to disk config DB.\n",(int)length);
    close(fd);
    free(buf);
    return(0);
}

/*
 * Testing routine to read in the contents of on-disk configuration
 * "database" area and write it out to the file specified as a parameter.
 * Basically a file copy from the shared state disk to the
 * filesystem.
 */
int testConfigRead(char *filename) {
    ssize_t length;
    char *buf;
    ssize_t retval;
    int fd;

    // XXX - shameless hack around command line parsing bungle.
    filename = "/tmp/db_test_copy";

    clu_lock();

    length = getDatabaseLength();
    if (length < 0) {
	fprintf(stderr, "Unable to determine database length.\n");
	perror("getDatabaseLength");
	return(1);
    }
    fd = creat(filename, 0666);
    if (fd < 0) {
	fprintf(stderr, "Unable to open %s\n", filename);
	perror("open");
	return(1);
    }
    /*
     * Now allocate a memory buffer to read the file data into.
     * This is a kludge to grab an in-memory buffer for the whole file.
     * Really only works for small files, but for testing purposes it will
     * suffice.
     */
    buf = malloc(length);

    retval = readDatabase(buf, length);

    clu_un_lock();   

    if (retval != length) {
	fprintf(stderr, "Unable to read %Zd bytes from disk config DB.\n", length);
	perror("readDatabase");
	close(fd);
        free(buf);
	return(1);
    }

    retval = write(fd, buf, length);
    if (retval != length) {
	fprintf(stderr, "Unable to write %Zd bytes to file %s.\n", length, filename);
	perror("write");
	close(fd);
        free(buf);
	return(1);
    }

    fprintf(stderr, "Successfully wrote %Zd bytes from disk config DB, to %s.\n",
			 length, filename);
    close(fd);
    free(buf);
    return(0);
}

/*
 * Basically just a test routine for the cluGetDiskNodeStates API.
 */
void printNodeStates(void) {
    	SharedDiskNodeStates nodestates;
	int retval;

	retval = cluGetDiskNodeStates(&nodestates);
	if (retval < 0) {
		printf("cluGetDiskNodeStates failed.\n");
		return;
	}

	printf("Node States\n");
	printf("[0] = %d, %s\n", nodestates.states[0], 
		nodeStateStrings[nodestates.states[0]]);
	printf("[1] = %d, %s\n", nodestates.states[1], 
		nodeStateStrings[nodestates.states[1]]);
}
/*
 * Prints out the minimum size of the shared state disk partition.
 */
void printPartSize(void) {
    printf("\nThe minimum size of each shared state disk partition is %d bytes.\n\n", END_OF_DISK);
}

/*
 * Display the contents of the quorum disk header.
 * Basically duplicates the implementation of printSharedStateHeader(), but
 * that one goes to the logger, here it goes to stdout.
 * Returns: 0 - success; -1 failure
 */
int print_quorum_header() {
	SharedStateHeader hdr;

	bzero((void *)&hdr, sizeof(SharedStateHeader));
	if (readHeader(&hdr) != 0) {
		return(-1);
	}
        printf("----- Shared State Header ------\n");
        printf("Magic# = 0x%lx",hdr.magic_number);
        if (hdr.magic_number != SHARED_STATE_MAGIC_NUMBER) { 
             printf(" INVALID");
        }
        printf( "\nVersion = %d\n", hdr.version);
        printf( "Updated on %s", ctime(&hdr.timestamp));
        printf( "Updated by node %d\n", hdr.updateNodenum);
        printf( "--------------------------------\n");    
	return(0);
}

static int check_hb_config() {
	CluCfg *cfg;

	/*
	 * Configuration check to verify that the network interfaces
	 * are mapped properly.
	 */
        cfg = get_clu_cfg(CLU_CONFIG_FILE);
	if (cfg == NULL) {
	    switch (errno) {
		case ENOMEM:
		    fprintf(stderr, "System out of memory.\n");
		    break;
		case ENOENT:
		    fprintf(stderr, "Failed parsing config file.\n");
		    break;
		case EFAULT:
		    fprintf(stderr, "Invalid network configuration.\n");
		    fprintf(stderr, "Unable to associate "
		       "heartbeat channels with configured network adatper.\n");
		    break;
	    }
	    return(-1);
	}
	else {
	    free(cfg);
	}
	return(0); // success
}
