// tbuddylist.h
//
// Kit AIM client
//
// For copyright and license, see accompanying documentation

#ifndef TBUDDYLIST_H
#define TBUDDYLIST_H

class QString;

/**
 * Stores information about a buddy.
 * Represents a node in a doubly linked list of buddies.
 */

typedef struct TBuddy
{
  /// The actual buddy name
	QString name;

  /// User selected alias
	QString alias;

  /// Group buddy is in
	int group;

  /// Buddy's status
	int status;

  /// Type of buddy
	int userClass;

  /// When the buddy signed on
	int signonTime;

  /// Time the buddy has been idle
	int idleTime;

  /// String version of buddy's idle time
	QString idleTimeString;

  /// Level that the buddy has been warned 
	int evil;

  /// When the buddy was last on
	int lastOn;

  /// String version of when buddy was last on
	QString lastOnString;

  /// If they are currently online, offline or away
  int signedOn;
	
  /// Previous buddy in the list
	TBuddy *prev;

  /// Next buddy in the list
	TBuddy *next;
};


/**
 * Information on a group of buddies.
 * Represents a node in a linked list of buddy groups.
 */

typedef struct TBuddyGroup
{
  /// Name of this buddy group
  QString name;

  /// Number of buddies in this group
  int size;

  /// Previous buddy group
  TBuddyGroup *prev;

  /// Next buddy group
  TBuddyGroup *next;

};

class TBuddyList
{
  public:
  /// Create an empty buddy list
    TBuddyList();

    /// Create a new buddy list as a copy of an existing one
    TBuddyList(const TBuddyList &);

    /// Add a buddy list 
    TBuddyList operator+=(const TBuddyList &);

    /// Copy a buddy list
    TBuddyList operator=(const TBuddyList &);

    /// Destroy a buddy list
    ~TBuddyList();

    /// Add a buddy
    int add(const TBuddy *);

    /// Delete the nth buddy from the list
    int del(int num);

    /// Delete a buddy from the list by name
    int del(const QString &name);

    /// Get the name for the nth budd
      QString getName(int num) const;

      /// Get the alias for the nth buddy
      QString getAlias(int num) const;

      /// Get the number of a buddy by name
    int getNum(const QString &name) const;

    /// Get the group of the nth buddy
    int getGroup(int num) const;

    /// Get the status of the nth buddy
    int getStatus(int num) const;

    /// Get buddy information for the nth buddy
    int get(TBuddy *, int num) const;

    /// Get buddy information for a buddy by name
    int get(TBuddy *, const QString &name) const;

    /// Get the number of buddies in the list
    inline int getCount(void) const {return count;};
    
    /// Set the status for the nth buddy
    int setStatus(int num, int status);

    /// Set the user class for the nth buddy
    int setUserClass(int num, int userClass);

    /// Set the sign on time for the nth buddy
    int setSignonTime(int num, int time);

    /// Set the idle time for the nth buddy
		int setIdleTime(int num, int time);

		/// Set the string version of the nth buddy's idle time
		int setIdleTimeString(int num, QString time);

		/// Set the warning level for this buddy
    int setEvil(int num, int evil);

    /// Set the last on time for the nth buddy
    int setLastOn(int num, int time);

    /// Set the string version of the nth buddy's last on time
		int setLastOnString(int num, QString time);

		/// Set the nth buddy's group
    int setGroup(int num, int group);
	 	
    /// Add a group
    int addGroup(const QString &name);

    /// Delete a group
    int delGroup(int num);
		
    /// Delete a group by name
    int delGroup(const QString &name);

    /// Rename a group
    int renameGroup(const QString &oldname, const QString &newname);

    /// Get the name of a group
    QString getNameGroup(int num) const;

    /// Get the number of a group
    int getNumGroup(const QString &name) const;

    /// Get the number of groups
    inline int getCountGroup(void) const {return countG;};


    /// Initializes the buddy list
    void reset(void);

    /// Get a buddy
    TBuddy *getByNum(int) const;

    /// Get a buddy group
    TBuddyGroup *getByNumG(int) const;
  
  private:
    /// The first buddy
		TBuddy *head;

		/// The last buddy
		TBuddy *tail;

		/// Number of buddies in the list
		int count;

		/// The first buddy group
		TBuddyGroup *headG;

		/// The last buddy group
		TBuddyGroup *tailG;

		/// Number of buddy groups in the list
		int countG;

};

#endif

