/*
 * filehandler.cc -- saving DV data into different file formats
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <signal.h>
#include <unistd.h>
#include <fcntl.h>

#include <string>
#include <iostream>
#include <strstream>
#include <iomanip>

using std::strstream;
using std::setw;
using std::setfill;

#include "error.h"
#include "riff.h"
#include "avi.h"
#include "frame.h"
#include "filehandler.h"


FileHandler::FileHandler() : done(false), autoSplit(false), maxFrameCount(999999), framesWritten(0)
{
    /* empty body */
}


FileHandler::~FileHandler()
{
    /* empty body */
}


bool FileHandler::GetAutoSplit() const
{
    return autoSplit;
}


bool FileHandler::GetTimeStamp() const
{
    return timeStamp;
}


string FileHandler::GetBaseName() const
{
    return base;
}


string FileHandler::GetExtension() const
{
    return extension;
}


int FileHandler::GetMaxFrameCount() const
{
    return maxFrameCount;
}


void FileHandler::SetAutoSplit(bool flag)
{
    autoSplit = flag;
}


void FileHandler::SetTimeStamp(bool flag)
{
    timeStamp = flag;
}


void FileHandler::SetBaseName(const string& s)
{
    base = s;
}


void FileHandler::SetMaxFrameCount(int count)
{
    assert(count > 0);
    maxFrameCount = count;
}


void FileHandler::SetEveryNthFrame(int every)
{
    assert (every > 0);

    everyNthFrame = every;
}


void FileHandler::SetSampleFrame(const Frame& sample)
{
    /* empty body */
}


bool FileHandler::Done()
{
    return done;
}


int FileHandler::WriteFrame(const Frame& frame)
{
    /* If the user wants autosplit, start a new file if a
       new recording is detected. */

    if (FileIsOpen() && GetAutoSplit() == true && frame.IsNewRecording()) {
        Close();
    }

    if (FileIsOpen() == false) {

        string          filename;
        strstream       sb;
        static int      counter = 0;

        if (GetTimeStamp() == true) {
            sb << GetBaseName() << frame.GetRecordingDate() << GetExtension();
        } else {
            sb << GetBaseName() << setfill('0') << setw(3) << ++counter << GetExtension();
        }
        sb >> filename;
        fail_if(Create(filename) == false)
        ;
        framesWritten = 0;
        framesToSkip = 0;
    }

    /* write frame */

    if (framesToSkip == 0) {
        Write(frame);
        framesToSkip = everyNthFrame;
        ++framesWritten;
    }
    framesToSkip--;

    /* If the frame count is exceeded, close the current file.
       If the autosplit flag is set, a new file will be created in the next iteration.
       If the flag is not set, we are done. */

    if (framesWritten >= GetMaxFrameCount()) {
        Close();
        if (GetAutoSplit() == false)
            done = true;
    }
    return 0;
}


RawHandler::RawHandler(): fd( -1)
{
    extension = ".dv";
}


RawHandler::~RawHandler()
{
    /* empty body */
}


bool RawHandler::FileIsOpen()
{
    return fd != -1;
}


bool RawHandler::Create(const string& filename)
{
    fd = open(filename.c_str(), O_CREAT | O_TRUNC | O_RDWR, 0644);
    return fd != -1 ? true : false;
}


int RawHandler::Write(const Frame& frame)
{
    return write(fd, frame.data, frame.GetFrameSize());
}


int RawHandler::Close()
{
    if (fd != -1) {
        close(fd);
        fd = -1;
    }
    return 0;
}


AVIHandler::AVIHandler(int format): avi(NULL), aviFormat(format)
{
    extension = ".avi";
}


AVIHandler::~AVIHandler()
{
    /* empty body */
}


void AVIHandler::SetSampleFrame(const Frame& sample)
{
    sample.GetAudioInfo(audioInfo);
    sample.GetVideoInfo(videoInfo);
}


bool AVIHandler::FileIsOpen()
{
    return avi != NULL;
}


bool AVIHandler::Create(const string& filename)
{
    assert(avi == NULL);

    switch (aviFormat) {

    case AVI_DV1_FORMAT:
        fail_null(avi = new AVI1File);
        fail_if(avi->Create(filename.c_str()) == false);
        break;

    case AVI_DV2_FORMAT:
        fail_null(avi = new AVI2File);
        fail_if(avi->Create(filename.c_str()) == false);
        break;

    default:
        assert(aviFormat == AVI_DV1_FORMAT || aviFormat == AVI_DV2_FORMAT);
    }

    avi->Init(videoInfo.isPAL ? AVI_PAL : AVI_NTSC, audioInfo.frequency);

    return avi != NULL ? true : false;
}


int AVIHandler::Write(const Frame& frame)
{
    assert(avi != NULL);

    avi->WriteFrame(frame);
    return 0;
}


int AVIHandler::Close()
{
    if (avi != NULL) {
        avi->WriteRIFF();
        delete avi;
        avi = NULL;
    }
    return 0;
}

