/*
 * frame.h -- utilities for process digital video frames
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _FRAME_H
#define _FRAME_H 1

#include <time.h>
#include <string>
using std::string;

#ifdef WITH_LIBDV
#include <libdv/dv.h>
#endif

typedef struct Pack
{
    /// the five bytes of a packet
    unsigned char data[5];
}
Pack;

typedef struct TimeCode
{
    int hour;
    int min;
    int sec;
    int frame;
}
TimeCode;


typedef struct AudioInfo
{
    int frames;
    int frequency;
    int samples;
}
AudioInfo;


class VideoInfo
{
public:
    int width;
    int height;
    bool isPAL;
    TimeCode timeCode;
    struct tm	recDate;

    VideoInfo();

    //    string GetTimeCodeString();
    //    string GetRecDateString();
}
;


class Frame
{
public:
    /// enough space to hold a PAL frame
    unsigned char data[144000];
    /// the number of bytes written to the frame
    int bytesInFrame;
#ifdef WITH_LIBDV

    static dv_decoder_t *decoder;
#endif

public:
    Frame();
    ~Frame();

    bool GetSSYBPack(int packNum, Pack &pack) const;
    bool GetVAUXPack(int packNum, Pack &pack) const;
    bool GetAAUXPack(int packNum, Pack &pack) const;
    bool GetTimeCode(TimeCode &timeCode) const;
    bool GetRecordingDate(struct tm &recDate) const;
    string GetRecordingDate(void) const;
    bool GetAudioInfo(AudioInfo &info) const;
    bool GetVideoInfo(VideoInfo &info) const;
    int GetFrameSize(void) const;
    bool IsPAL(void) const;
    bool IsNewRecording(void) const;
    bool IsComplete(void) const;
    int ExtractAudio(void *sound) const;
#ifdef WITH_LIBDV
    bool IsPALfromHeader(void) const;
    void ExtractHeader(void);
    int ExtractRGB(void *rgb);
    int ExtractBGR0(void *bgr);
    int ExtractYUV(void *yuv);
    bool IsWide(void) const;
#endif

private:
    static bool header_seen;
    /// flag for initializing the lookup maps once at startup
    static bool maps_initialized;
    /// lookup tables for collecting the shuffled audio data
    static int palmap_ch1[2000];
    static int palmap_ch2[2000];
    static int palmap_2ch1[2000];
    static int palmap_2ch2[2000];
    static int ntscmap_ch1[2000];
    static int ntscmap_ch2[2000];
    static int ntscmap_2ch1[2000];
    static int ntscmap_2ch2[2000];
    static short compmap[4096];
};
#endif
