#include <string.h>

#include "../kpcalendarentry.h"
#include "../kpcalendartime.h"
#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kputil.h"
#include "kplogstore.h"

#define RECORD(d) ((KPLogStoreRecord*)(d))
#define NODE(d) ((GNode*)(d))

/* GObject stuff */
static void         kp_log_store_init            (KPLogStore * pkg_tree);
static void         kp_log_store_class_init      (KPLogStoreClass * klass);
static void         kp_log_store_tree_model_init (GtkTreeModelIface * iface);
static void         kp_log_store_finalize        (GObject * object);

/* GtkTreeModel stuff */
static GtkTreeModelFlags 
                    kp_log_store_get_flags       (GtkTreeModel * tree_model);
static gint         kp_log_store_get_n_columns   (GtkTreeModel * tree_model);
static GType        kp_log_store_get_column_type (GtkTreeModel *tree_model,
                                                  gint index);

static gboolean     kp_log_store_get_iter        (GtkTreeModel *tree_model,
                                                  GtkTreeIter *iter,
                                                  GtkTreePath *path);
static GtkTreePath *kp_log_store_get_path        (GtkTreeModel *tree_model,
	                                          GtkTreeIter *iter);
static void         kp_log_store_get_value       (GtkTreeModel *tree_model,
                                                  GtkTreeIter *iter,
                                                  gint column,
                                                  GValue *value);
static gboolean     kp_log_store_iter_next       (GtkTreeModel *tree_model,
                                                  GtkTreeIter *iter);
static gboolean     kp_log_store_iter_children   (GtkTreeModel *tree_model,
                                                  GtkTreeIter *iter,
                                                  GtkTreeIter *parent);
static gboolean     kp_log_store_iter_has_child  (GtkTreeModel *tree_model,
                                                  GtkTreeIter *iter);
static gint         kp_log_store_iter_n_children (GtkTreeModel *tree_model,
                                                  GtkTreeIter *iter);
static gboolean     kp_log_store_iter_nth_child  (GtkTreeModel *tree_model,
                                                  GtkTreeIter *iter,
                                                  GtkTreeIter *parent,
                                                  gint n);

static gboolean     kp_log_store_iter_parent     (GtkTreeModel *tree_model,
                                                  GtkTreeIter *iter,
                                                  GtkTreeIter *child);

/* Tree functions */
static void         kp_log_store_append_entry    (KPCalendarEntry *entry,
                                                  KPLogStore *kp_log_store);

static GNode       *get_day_node                 (KPLogStore *store,
                                                  guint d,
                                                  guint m,
                                                  guint y,
                                                  GtkTreePath **p);
static GNode       *node_exists                  (GNode *parent,
                                                  KPLogStoreRecordType type,
                                                  guint n,
                                                  gint *pos);
static GNode       *add_month                    (KPLogStore *store,
                                                  GNode *y_node,
                                                  guint m,
                                                  guint y,
                                                  gint *pos);
static GNode       *add_year                     (KPLogStore *store,
                                                  GNode *root,
                                                  guint y,
                                                  gint *pos);
static GNode       *add_day                      (KPLogStore *store,
                                                  GNode *m_node,
                                                  guint d,
                                                  guint m,
                                                  guint y,
                                                  gint *pos);
static GNode       *new_node                     (KPLogStore *store,
                                                  guint d,
                                                  guint m,
                                                  guint y);
static void         destroy_node                 (GNode *parent);


GType
kp_log_store_get_type (void)
{
  static GType kp_log_store_type = 0;

  /* Type */
  if (!kp_log_store_type)
    {
      static const GTypeInfo kp_log_store_info = {
	sizeof (KPLogStoreClass),
	NULL,			/* base_init */
	NULL,			/* base_finalize */
	(GClassInitFunc) kp_log_store_class_init,
	NULL,			/* class finalize */
	NULL,			/* class_data */
	sizeof (KPLogStore),
	0,			/* n_preallocs */
	(GInstanceInitFunc) kp_log_store_init,
	NULL
      };

      kp_log_store_type = g_type_register_static (G_TYPE_OBJECT,
						  "KPLogStore",
						  &kp_log_store_info, 0);
      /* Interface */
      static const GInterfaceInfo tree_model_info = {
	(GInterfaceInitFunc) kp_log_store_tree_model_init,
	NULL,
	NULL
      };

      g_type_add_interface_static (kp_log_store_type,
				   GTK_TYPE_TREE_MODEL,
                                  &tree_model_info);
    }

  return kp_log_store_type;
}

static void
kp_log_store_class_init (KPLogStoreClass * klass)
{
  GObjectClass *object_class;

  /*parent_class = (GObjectClass*) g_type_class_peek_parent (klass); */
  object_class = (GObjectClass *) klass;
  object_class->finalize = kp_log_store_finalize;
}

static void
kp_log_store_tree_model_init (GtkTreeModelIface * iface)
{
  iface->get_flags = kp_log_store_get_flags;
  iface->get_n_columns = kp_log_store_get_n_columns;
  iface->get_column_type = kp_log_store_get_column_type;
  iface->get_iter = kp_log_store_get_iter;
  iface->get_path = kp_log_store_get_path;
  iface->get_value = kp_log_store_get_value;
  iface->iter_next = kp_log_store_iter_next;
  iface->iter_children = kp_log_store_iter_children;
  iface->iter_has_child = kp_log_store_iter_has_child;
  iface->iter_n_children = kp_log_store_iter_n_children;
  iface->iter_nth_child = kp_log_store_iter_nth_child;
  iface->iter_parent = kp_log_store_iter_parent;
}

static void
kp_log_store_init (KPLogStore * model)
{
  GtkTreePath *path;
  GtkTreeIter iter;

  g_assert (KP_LOG_STORE_N_COLUMNS == 1);

  model->column_types[0] = G_TYPE_STRING;	/* KP_LOG_STORE_COL_RECORD    */
  model->n_columns = KP_LOG_STORE_N_COLUMNS;
  model->root = new_node (model, 0, 0, 0);
  RECORD (model->root->data)->record = g_strdup (_("Log"));

  do {
    model->stamp = g_random_int ();
  }
  while (0 == model->stamp);

  path = gtk_tree_path_new ();
  gtk_tree_path_append_index (path, 0);
  kp_log_store_get_iter (GTK_TREE_MODEL (model), &iter, path);
  gtk_tree_model_row_inserted (GTK_TREE_MODEL (model), path, &iter);
  gtk_tree_path_free (path);
}

static void
kp_log_store_finalize (GObject * object)
{
  GObjectClass *parent_class;

  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

KPLogStore *
kp_log_store_new (void)
{
  return KP_LOG_STORE (g_object_new (KP_TYPE_LOG_STORE, NULL));
}

void
kp_log_store_attach_log (KPLogStore *store, KPTrainingLog *log)
{
  kp_training_log_foreach (log, (GFunc) kp_log_store_append_entry, store);
}


static GtkTreeModelFlags
kp_log_store_get_flags (GtkTreeModel * tree_model)
{
  g_return_val_if_fail (KP_IS_LOG_STORE (tree_model), (GtkTreeModelFlags) 0);

  return (GTK_TREE_MODEL_LIST_ONLY | GTK_TREE_MODEL_ITERS_PERSIST);
}

static gint
kp_log_store_get_n_columns (GtkTreeModel * tree_model)
{
  g_return_val_if_fail (KP_IS_LOG_STORE (tree_model), 0);

  return KP_LOG_STORE (tree_model)->n_columns;
}

static GType
kp_log_store_get_column_type (GtkTreeModel * tree_model, gint index)
{
  g_return_val_if_fail (KP_IS_LOG_STORE (tree_model), G_TYPE_INVALID);
  g_return_val_if_fail (index < KP_LOG_STORE (tree_model)->n_columns
			&& index >= 0, G_TYPE_INVALID);

  return KP_LOG_STORE (tree_model)->column_types[index];
}

static gboolean
kp_log_store_get_iter (GtkTreeModel *tree_model, GtkTreeIter *iter,
		       GtkTreePath *path)
{
  KPLogStoreRecord *record;
  KPLogStore *model;
  gint *indices;
  GNode *child;
  GNode *node;
  gint depth;
  guint n;
  gint i;

  g_assert (KP_IS_LOG_STORE (tree_model));
  g_assert (path != NULL);

  model = KP_LOG_STORE (tree_model);

  indices = gtk_tree_path_get_indices (path);
  depth = gtk_tree_path_get_depth (path);

  if (depth == 1)
    {
      iter->stamp = model->stamp;
      iter->user_data = model->root;
      return TRUE;
    }

  for (node = model->root, i = 1; i < depth; i++) {
    n = indices[i];
    child = g_node_nth_child (node, n);

    if (child == NULL)
      return FALSE;
    node = child;
  }
  record = node->data;
  g_assert (record != NULL);

  /* We simply store a pointer to our custom record in the iter */
  iter->stamp = model->stamp;
  iter->user_data = node;

  return TRUE;
}

static GtkTreePath *
kp_log_store_get_path (GtkTreeModel * tree_model, GtkTreeIter * iter)
{
  KPLogStore *model;
  GtkTreePath *path;
  GNode *node;
  gint pos;

  g_return_val_if_fail (KP_IS_LOG_STORE (tree_model), NULL);
  g_return_val_if_fail (iter != NULL, NULL);
  g_return_val_if_fail (iter->user_data != NULL, NULL);

  model = KP_LOG_STORE (tree_model);

  node = iter->user_data;
  path = gtk_tree_path_new ();

  while (node->parent) {
    pos = g_node_child_position (node->parent, node);
    gtk_tree_path_prepend_index (path, pos);
    node = node->parent;
  }
  /* root */
  gtk_tree_path_prepend_index (path, 0);
  
  return path;
}

static void
kp_log_store_get_value (GtkTreeModel *tree_model, GtkTreeIter *iter,
			gint column, GValue *value)
{
  KPLogStoreRecord *record;
  gchar *str;

  g_return_if_fail (KP_IS_LOG_STORE (tree_model));
  g_return_if_fail (iter != NULL);
  g_return_if_fail (column < KP_LOG_STORE (tree_model)->n_columns);

  g_value_init (value, KP_LOG_STORE (tree_model)->column_types[column]);

  record = NODE (iter->user_data)->data;
  g_return_if_fail (record != NULL);
  g_assert (0 == column);

  if (record->calendar_entry)
    g_value_set_string (value, record->record);
  else {
    str = g_strdup_printf ("%s (%u)", record->record, record->n_entries);
    g_value_set_string (value, str);
    g_free (str);
  }
}

static gboolean
kp_log_store_iter_next (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
  KPLogStore *model;
  GNode *node;

  g_return_val_if_fail (KP_IS_LOG_STORE (tree_model), FALSE);

  if (iter == NULL || iter->user_data == NULL)
    return FALSE;

  model = KP_LOG_STORE (tree_model);

  node = iter->user_data;

  if (node->next == NULL)
    return FALSE;

  iter->user_data = node->next;
  iter->stamp = model->stamp;

  return TRUE;
}

static gboolean
kp_log_store_iter_children (GtkTreeModel *model, GtkTreeIter *iter,
                            GtkTreeIter *parent)
{
  GNode *node;

  g_return_val_if_fail (parent == NULL || parent->user_data != NULL, FALSE);
  g_return_val_if_fail (KP_IS_LOG_STORE (model), FALSE);

  node = parent->user_data;

  if (node->children == NULL)
    return FALSE;

  iter->user_data = node->children;
  iter->stamp = KP_LOG_STORE (model)->stamp;

  return TRUE;
}


KPLogStoreRecordType
kp_log_store_get_iter_type (KPLogStore *store, GtkTreeIter *iter)
{
  KPLogStoreRecordType type;
  KPLogStoreRecord *rec;
  gchar *day;

  g_return_val_if_fail (KP_IS_LOG_STORE (store), KP_LOG_STORE_REC_INVALID);
  g_return_val_if_fail (iter != NULL, KP_LOG_STORE_REC_INVALID);

  rec = RECORD (NODE (iter->user_data)->data);
  g_return_val_if_fail (rec != NULL, KP_LOG_STORE_REC_INVALID);

  if (!rec->d && !rec->m && !rec->y)
    return KP_LOG_STORE_REC_ROOT;
 
  if (!rec->d && !rec->m && rec->y)
    return KP_LOG_STORE_REC_YEAR;

  if (!rec->d && rec->m && rec->y)
    return KP_LOG_STORE_REC_MONTH;
  
  if (g_date_valid_dmy (rec->d, rec->m, rec->y)) {
    day = g_strdup_printf ("%d", rec->d);
    
    if (strcmp (rec->record, day) == 0)
      type = KP_LOG_STORE_REC_DAY;
    else
      type = KP_LOG_STORE_REC_ENTRY;
    
    g_free (day);
    return type;
  }
  g_return_val_if_reached (KP_LOG_STORE_REC_INVALID);
}


/*****************************************************************************
 *
 *  kp_log_store_iter_has_child: Returns TRUE or FALSE depending on whether
 *                              the row specified by 'iter' has any children.
 *                              We only have a list and thus no children.
 *
 *****************************************************************************/
static gboolean
kp_log_store_iter_has_child (GtkTreeModel * tree_model, GtkTreeIter * iter)
{
  g_return_val_if_fail (KP_IS_LOG_STORE (tree_model), FALSE);
  g_return_val_if_fail (iter != NULL, FALSE);
  g_return_val_if_fail (iter->user_data != NULL, FALSE);

  return (NODE (iter->user_data)->children != NULL);
}


static gint
kp_log_store_iter_n_children (GtkTreeModel * model, GtkTreeIter * iter)
{
  GNode *node;
  guint n=0;

  g_return_val_if_fail (KP_IS_LOG_STORE (model), -1);
  g_return_val_if_fail (iter->user_data != NULL, -1);

  if (iter == NULL)
    /* We will always have only one toplevel node */
    return 1;
  
  node = iter->user_data;

  if (!node->children)
    return 0;
  else 
    node = node->children;
    
  while (node) {
    n++;
    node = node->next;
  }
  
  return n;
}

static gboolean
kp_log_store_iter_nth_child (GtkTreeModel * tree_model,
			     GtkTreeIter * iter, GtkTreeIter * parent, gint n)
{
  GNode *node;

  g_return_val_if_fail (KP_IS_LOG_STORE (tree_model), FALSE);
  g_return_val_if_fail (iter != NULL, FALSE);

  if (parent == NULL)
    node = KP_LOG_STORE (tree_model)->root;
  else
    node = g_node_nth_child (NODE (parent->user_data), n);

  if (!node)
    return FALSE;

  iter->stamp = KP_LOG_STORE (tree_model)->stamp;
  iter->user_data = node;

  return TRUE;
}

static gboolean
kp_log_store_iter_parent (GtkTreeModel * tree_model, GtkTreeIter * iter,
			  GtkTreeIter * child)
{
  GNode *node;

  g_return_val_if_fail (KP_IS_LOG_STORE (tree_model), FALSE);
  g_return_val_if_fail (child != NULL, FALSE);
  g_return_val_if_fail (child->user_data != NULL, FALSE);
  g_return_val_if_fail (iter != NULL, FALSE);

  node = NODE (child->user_data);

  if (!node)
    return FALSE;

  if (node->parent == NULL)
    return FALSE;

  node = node->parent;

  iter->stamp = KP_LOG_STORE (tree_model)->stamp;
  iter->user_data = node;

  return TRUE;
}

void
kp_log_store_add_mark (KPLogStore *store, guint d, guint m, guint y,
                       const gchar *mark)
{
  GtkTreePath *path;
  GtkTreeIter iter;
  GNode *newnode;
  GNode *node;

  g_return_if_fail (KP_IS_LOG_STORE (store));
  g_return_if_fail (g_date_valid_dmy (d, m, y));
  g_assert (store->root != NULL);
  
  node = get_day_node (store, d, m, y, &path);
  g_assert (node != NULL);
  
  newnode = new_node (store, d, m, y);
  RECORD (newnode->data)->record = g_strdup (mark);
  RECORD (newnode->data)->calendar_entry = TRUE;

  g_node_append (node, newnode);

  gtk_tree_path_append_index (path, g_node_child_position (node, newnode));

  while (node) {
    gtk_tree_path_up (path);
    kp_log_store_get_iter (GTK_TREE_MODEL (store), &iter, path);
    gtk_tree_model_row_changed (GTK_TREE_MODEL (store), path, &iter);
    
    RECORD (node->data)->n_entries++;
    node = node->parent;
  }
  gtk_tree_path_free (path);
}

void
kp_log_store_remove_mark (KPLogStore * store, guint d, guint m, guint y,
                          const gchar *mark)
{
  GtkTreeIter iter;
  GNode *node;
  GNode *day;

  g_return_if_fail (KP_IS_LOG_STORE (store));
  g_return_if_fail (g_date_valid_dmy (d, m, y));

  day = get_day_node (store, d, m, y, NULL);
  g_return_if_fail (day != NULL);
  
  node = day->children;

  while (node) {

    if (strcmp (mark, RECORD (node->data)->record) == 0) {
      iter.user_data = node;
      iter.stamp = store->stamp;

      kp_log_store_remove (store, &iter);
      return;
    }
  
    node = node->next;
  }
  g_warning ("Couldn't find mark to remove in KPLogStore!\n");
}

void
kp_log_store_remove (KPLogStore *store, GtkTreeIter *iter)
{
  GtkTreeIter new_iter;
  GtkTreePath *path;
  GNode *parent;
  GNode *node;
  GNode *next;
  guint n;
  
  g_return_if_fail (KP_IS_LOG_STORE (store));
  g_return_if_fail (iter != NULL);

  node = NODE (iter->user_data);
  next = NODE (iter->user_data)->next;
  parent = node->parent;
 
  /* Don't allow deleting the root node */
  if (parent == NULL)
    return;

  /* Get path before destroying stuff */
  path = kp_log_store_get_path (GTK_TREE_MODEL (store), iter);

  /* Calendar entries don't have child entries but we need to set n = 1
   * to get code below to work */
  if (RECORD (node->data)->calendar_entry)
    n = 1;
  else
    n = RECORD (node->data)->n_entries;
 
 
  g_node_traverse (node,
                   G_POST_ORDER,
                   G_TRAVERSE_ALL,
                  -1,
                  (GNodeTraverseFunc) destroy_node,
                   NULL);

  g_node_destroy (node);
 
  gtk_tree_model_row_deleted (GTK_TREE_MODEL (store), path);

  if (parent->children == NULL) {
    gtk_tree_path_up (path);
    new_iter.stamp = store->stamp;
    new_iter.user_data = parent;
    gtk_tree_model_row_has_child_toggled (GTK_TREE_MODEL (store), path,
                                         &new_iter);
  
    kp_debug ("Child toggled notification: %s\n", 
      gtk_tree_path_to_string (path));
  }

  if (next) {
    iter->stamp = store->stamp;
    iter->user_data = next;
  } else {
    iter->stamp = 0;
    iter->user_data = NULL;
  }
  for (node = parent; node != NULL; node = node->parent) { 
    RECORD (node->data)->n_entries -= n;

    gtk_tree_path_up (path);
    new_iter.stamp = store->stamp;
    new_iter.user_data = node;
    
    gtk_tree_model_row_changed (GTK_TREE_MODEL (store), path, &new_iter);
  
    kp_debug ("Row changed notification: %s ->%s\n", 
      gtk_tree_path_to_string (path), RECORD (node->data)->record);
  }
 
  for (node = parent; node != NULL; node = node->parent)
    if (RECORD (node->data)->n_entries == 0) {
      new_iter.stamp = store->stamp;
      new_iter.user_data = node;
      
      kp_log_store_remove (store, &new_iter);
    }
}

/*
 * CUSTOM FUNCTIONS TO HANDLE LOG_STORE DATA
 */

void
kp_log_store_get_date (KPLogStore *store, GtkTreeIter *iter, guint *d,
                       guint *m, guint *y)
{
  g_return_if_fail (KP_IS_LOG_STORE (store));
  g_return_if_fail (iter != NULL);
  g_return_if_fail (d != NULL && m != NULL && y != NULL);

  *d = RECORD (NODE (iter->user_data)->data)->d;
  *m = RECORD (NODE (iter->user_data)->data)->m;
  *y = RECORD (NODE (iter->user_data)->data)->y;

  kp_debug ("Date: %u.%u.%u", *d, *m, *y);
}
  

static void
kp_log_store_append_entry (KPCalendarEntry *entry, KPLogStore *store)
{
  GDate *date;
  gchar *mark;

  date = kp_calendar_time_get_date (entry->datetime);
  mark = kp_calendar_entry_to_string (entry);

  kp_log_store_add_mark (store,
                         g_date_get_day (date),
                         g_date_get_month (date),
                         g_date_get_year (date),
                         mark);
  g_free (mark);
}

/*
 * SOME TREE BUILDING AND HANDLING CODE.
 */

static GNode *
add_year (KPLogStore * store, GNode * root, guint y, gint * pos)
{
  GNode *node = new_node (store, 0, 0, y);

  node = g_node_append (root, node);

  RECORD (node->data)->record = g_strdup_printf ("%u", y);

  if (pos)
    {
      *pos = g_node_child_position (root, node);
      g_assert (*pos >= 0);
    }
  return node;
}

static GNode *
add_month (KPLogStore * store, GNode * y_node, guint m, guint y, gint * pos)
{
  GNode *node = new_node (store, 0, m, y);

  node = g_node_append (y_node, node);
  RECORD (node->data)->record = g_strdup_printf ("%u", m);
  
  if (pos)
    {
      *pos = g_node_child_position (y_node, node);
      g_assert (*pos >= 0);
    }
  return node;
}

static GNode *
add_day (KPLogStore * store, GNode * m_node, guint d, guint m, guint y, gint * pos)
{
  GNode *node = new_node (store, d, m, y);

  node = g_node_append (m_node, node);
  RECORD (node->data)->record = g_strdup_printf ("%u", d);

  if (pos)
    {
      *pos = g_node_child_position (m_node, node);
      g_assert (*pos >= 0);
    }
  return node;
}

static GNode *
node_exists (GNode * parent, KPLogStoreRecordType type, guint n, gint * pos)
{
  GNode *node;
  gint i = 0;

  g_return_val_if_fail (parent != NULL, NULL);

  node = parent->children;

  while (node)
    {
      switch (type)
	{
	case KP_LOG_STORE_REC_DAY:
	  if (RECORD (node->data)->d == n)
	    {
	      if (pos)
		*pos = i;
	      return node;
	    }
	  break;

	case KP_LOG_STORE_REC_MONTH:
	  if (RECORD (node->data)->m == n)
	    {
	      if (pos)
		*pos = i;
	      return node;
	    }
	  break;

	case KP_LOG_STORE_REC_YEAR:
	  if (RECORD (node->data)->y == n)
	    {
	      if (pos)
		*pos = i;
	      return node;
	    }
	  break;

	case KP_LOG_STORE_REC_ROOT:
	case KP_LOG_STORE_REC_ENTRY:
	case KP_LOG_STORE_REC_INVALID:
	default:
	  g_assert_not_reached ();
	}
      i++;
      node = node->next;
    }
  return NULL;
}

static GNode *
new_node (KPLogStore * store, guint d, guint m, guint y)
{
  KPLogStoreRecord *rec;
  GNode *node;

  rec = g_new0 (KPLogStoreRecord, 1);
  rec->d = d;
  rec->m = m;
  rec->y = y;
  rec->n_entries = 0;
  rec->calendar_entry = FALSE;

  node = g_node_new (rec);

  return node;
}

static void
destroy_node (GNode *node)
{
  KPLogStoreRecord *rec;

  rec = RECORD (node->data);
  
  if (rec->record)
    g_free (rec->record);
  
  g_free (rec);
  rec = NULL; 
}

/*
 * Function that returns the node corresponding the day 'dmy'.
 * If @p is not NULL, path to the location will be stored in it.
 *
 */
static GNode *
get_day_node (KPLogStore *store, guint d, guint m, guint y, GtkTreePath **p)
{
  GtkTreePath *path;
  GtkTreeIter iter;
  GNode *parent;
  GNode *root = store->root;
  GNode *node;			/* y   m   d */
  gint pos[3] = { -1, -1, -1 };

  path = gtk_tree_path_new ();
  gtk_tree_path_append_index (path, 0);
  /*gtk_tree_model_row_inserted (GTK_TREE_MODEL (store), path, &iter);*/

  g_return_val_if_fail (g_date_valid_dmy (d, m, y), NULL);
  g_return_val_if_fail (store != NULL, NULL);
  g_return_val_if_fail (root != NULL, NULL);

  parent = node_exists (root, KP_LOG_STORE_REC_YEAR, y, &pos[0]);

  if (!parent) {
    parent = add_year (store, root, y, &pos[0]);
    gtk_tree_path_append_index (path, pos[0]);
    kp_log_store_get_iter (GTK_TREE_MODEL (store), &iter, path);
    gtk_tree_model_row_inserted (GTK_TREE_MODEL (store), path, &iter);
  }
  else
    gtk_tree_path_append_index (path, pos[0]);

  g_return_val_if_fail (parent != NULL, NULL);
  g_return_val_if_fail (pos[0] >= 0, NULL);

  node = node_exists (parent, KP_LOG_STORE_REC_MONTH, m, &pos[1]);
  
  if (!node) {
    node = add_month (store, parent, m, y, &pos[1]);
    gtk_tree_path_append_index (path, pos[1]);
    kp_log_store_get_iter (GTK_TREE_MODEL (store), &iter, path);
    gtk_tree_model_row_inserted (GTK_TREE_MODEL (store), path, &iter);
  }
  else
    gtk_tree_path_append_index (path, pos[1]);

  g_return_val_if_fail (pos[1] >= 0, NULL);
  parent = node;

  node = node_exists (parent, KP_LOG_STORE_REC_DAY, d, &pos[2]);

  if (!node) {
    node = add_day (store, parent, d, m, y, &pos[2]);
    gtk_tree_path_append_index (path, pos[2]);
    kp_log_store_get_iter (GTK_TREE_MODEL (store), &iter, path);
    gtk_tree_model_row_inserted (GTK_TREE_MODEL (store), path, &iter);
  }
  else
    gtk_tree_path_append_index (path, pos[2]);

  g_return_val_if_fail (node != NULL, NULL);

  if (p)
    *p = path;
  else
    gtk_tree_path_free (path);
  
  return node;
}
