#include <stdio.h>
#include "packet.h"
//#include "frontend.h"

// Returns a pointer in the data block to the size byte of the desired
// tag.
int GetTagOffset(int init_offset, int tagnum, const pkthdr *header, const u_char *data) {
    int cur_tag = 0;
    // Initial offset is 36, that's the first tag
    int cur_offset = init_offset;
    uint8_t len;

    while (1) {
        // Are we over the packet length?
        if (cur_offset > (uint8_t) header->len) {
            return -1;
        }

        // Read the tag we're on and bail out if we're done
        cur_tag = (int) data[cur_offset];
        if (cur_tag == tagnum) {
            cur_offset++;
            break;
        } else if (cur_tag > tagnum) {
            return -1;
        }

        // Move ahead one byte and read the length.
        // I'm not sure why we bitwise-and it, but it seems to work.
        len = (data[cur_offset+1] & 0xFF);

        // Jump the length+length byte, this should put us at the next tag
        // number.
        cur_offset += len+2;
    }

    return cur_offset;
}

// Get the info from a packet
packet_info GetPacketInfo(const pkthdr *header, const u_char *data) {
    packet_info ret;

    // Zero the entire struct
    memset(&ret, 0, sizeof(packet_info));

    // Point to the packet data
    uint8_t *msgbuf = (uint8_t *) data;
    // Temp pointer into the packet guts
    uint8_t temp;

    ret.type = packet_unknown;

    // If we don't even have enough to make up an 802.11 header, bail
    // as a garbage packet
    if (header->len < 2) {
        ret.type = packet_noise;
        return ret;
    }

    // Point the frame control at the beginnng of the data
    frame_control *fc = (frame_control *) data;

    int tag_offset = 0;

    // Raw test to see if it's just noise
    if (msgbuf[0] == 0xff && msgbuf[1] == 0xff) {
        ret.type = packet_noise;
        return ret;
    } else if (fc->type == 0) {
        if (fc->subtype == 8) {
            // beacon frame
            // Extract the SSID

            // Find the offset of tag 0 and fill in the ssid if we got the
            // tag.
            if ((tag_offset = GetTagOffset(36, 0, header, data)) > 0) {
                temp = (msgbuf[tag_offset] & 0xFF) + 1;
                // Byte 38 starts the ssid -- print it into the SSID
                snprintf(ret.ssid, temp, "%s", &msgbuf[tag_offset+1]);
            }

            // Extract the WEP flag
            // Byte 35 contains the capability options
            temp = msgbuf[34];
            // Now we need to pull out bit 4 (least-ordered) to get the
            // WEP capability
            ret.wep = (temp & (1 << 4)) ? 1 : 0;

            // Extract the ADHOC flag
            // We already have byte 35 in temp...
            ret.ap = (temp & (1 << 0));

            // Find the offset of flag 3 and get the channel
            if ((tag_offset = GetTagOffset(36, 3, header, data)) > 0) {
                // Extract the channel from the next byte (GetTagOffset returns
                // us on the size byte)
                temp = msgbuf[tag_offset+1];
                ret.channel = (int) temp;
            }

            // Extract the MAC's
            // Get the dest mac -- offset 4
            memcpy(ret.dest_mac, (const char *) &msgbuf[4], MAC_LEN);
            // Get the source mac -- offset 10
            memcpy(ret.source_mac, (const char *) &msgbuf[10], MAC_LEN);
            // Get the BSSID mac -- offset 16
            memcpy(ret.bssid_mac, (const char *) &msgbuf[16], MAC_LEN);

            // First byte of offsets
            ret.header_offset = 24;

            ret.type = packet_beacon;

        } else if (fc->subtype == 4) {
            // Probe req
            if ((tag_offset = GetTagOffset(24, 0, header, data)) > 0) {
                temp = (msgbuf[tag_offset] & 0xFF) + 1;
                snprintf(ret.ssid, temp, "%s", &msgbuf[tag_offset+1]);
            }

            memcpy(ret.source_mac, (const char *) &msgbuf[10], MAC_LEN);

            ret.type = packet_probe_req;

        } else if (fc->subtype == 5) {
            ret.type = packet_probe_response;
            if ((tag_offset = GetTagOffset(36, 0, header, data)) > 0) {
                temp = (msgbuf[tag_offset] & 0xFF) + 1;
                snprintf(ret.ssid, temp, "%s", &msgbuf[tag_offset+1]);
            }

            memcpy(ret.dest_mac, (const char *) &msgbuf[4], MAC_LEN);
            memcpy(ret.source_mac, (const char *) &msgbuf[10], MAC_LEN);
            memcpy(ret.bssid_mac, (const char *) &msgbuf[16], MAC_LEN);

            // First byte of offsets
            ret.header_offset = 24;

        }

    } else if (fc->type == 2) {
        // Data packets
        ret.type = packet_data;

        // Detect encrypted frames
        if (fc->wep) {
            ret.encrypted = 1;

            // Match the range of cryptographically weak packets and let us
            // know.
            if (msgbuf[24] >= 3 && msgbuf[24] < 16 && msgbuf[25] == 255)
                ret.interesting = 1;
        }

        // Extract ID's
        if (fc->to_ds == 0 && fc->from_ds == 0) {
            // Adhoc's get specially typed and their BSSID is set to
            // their source (I can't think of anything more reasonable
            // to do with them)
            memcpy(ret.dest_mac, (const char *) &msgbuf[4], MAC_LEN);
            memcpy(ret.source_mac, (const char *) &msgbuf[10], MAC_LEN);
            memcpy(ret.bssid_mac, (const char *) &msgbuf[10], MAC_LEN);

            // First byte of offsets
            ret.header_offset = 24;

            ret.type = packet_adhoc;
        } else if (fc->to_ds == 0 && fc->from_ds == 1) {
            memcpy(ret.dest_mac, (const char *) &msgbuf[4], MAC_LEN);
            memcpy(ret.bssid_mac, (const char *) &msgbuf[10], MAC_LEN);
            memcpy(ret.source_mac, (const char *) &msgbuf[16], MAC_LEN);

            // First byte of offsets
            ret.header_offset = 24;

        } else if (fc->to_ds == 1 && fc->from_ds == 0) {
            memcpy(ret.bssid_mac, (const char *) &msgbuf[4], MAC_LEN);
            memcpy(ret.source_mac, (const char *) &msgbuf[10], MAC_LEN);
            memcpy(ret.dest_mac, (const char *) &msgbuf[16], MAC_LEN);

            // First byte of offsets
            ret.header_offset = 24;

        } else if (fc->to_ds == 1 && fc->from_ds == 1) {
            // AP->AP
            // Source is a special offset to the source
            // Dest is the reciever address

            memcpy(ret.source_mac, (const char *) &msgbuf[24], MAC_LEN);
            memcpy(ret.dest_mac, (const char *) &msgbuf[4], MAC_LEN);

            // First byte of offsets
            ret.header_offset = 30;

            ret.type = packet_data_broadcast;
        }

        if (!ret.encrypted && (ret.type == packet_data || ret.type == packet_adhoc || ret.type == packet_data_broadcast))
            ret.proto = GetProtoInfo(&ret, header, data);
    }

    return ret;
}

proto_info GetProtoInfo(const packet_info *in_info, const pkthdr *header, const u_char *in_data) {

    // We cheat a little to protect ourselves.  We define a packet
    // that's double the maximum size, zero it out, and copy our data
    // packet into it.  This should give us a little leeway if a packet
    // is corrupt and we don't detect it -- it's better to read some
    // nulls than it is to fall into a segfault.
    u_char data[MAX_PACKET_LEN * 2];
    memset(data, 0, MAX_PACKET_LEN * 2);
    memcpy(data, in_data, header->len);

    proto_info ret;

    // Zero the entire struct
    memset(&ret, 0, sizeof(proto_info));

    // Point to the data packet
    uint8_t *msgbuf = (uint8_t *) data;

    ret.type = proto_unknown;

    /*
     if (header->len < 56)
        return ret;
        */

    if (memcmp(&data[in_info->header_offset + CISCO_OFFSET], CISCO_SIGNATURE,
               sizeof(CISCO_SIGNATURE)) == 0) {
        // CDP
        unsigned int offset = in_info->header_offset + CISCO_OFFSET + 12;

        while (offset < header->len) {
            // Make sure that whatever we do, we don't wander off the
            // edge of the proverbial world -- segfaulting due to crappy
            // packets is a really bad thing!

            cdp_element *elem = (cdp_element *) &data[offset];

            if (elem->length == 0)
                break;

            if (elem->type == 0x01) {
                // Device id
                snprintf(ret.cdp.dev_id, elem->length-3, "%s", (char *) &elem->data);
            } else if (elem->type == 0x02) {
                // IP range

                cdp_proto_element *proto;
                int8_t *datarr = (int8_t *) &elem->data;

                // We only take the first addr (for now)... And only if
                // it's an IP
                //for (int addr = 0; addr < datarr[3]; addr++) {
                proto = (cdp_proto_element *) &datarr[4];

                if (proto->proto == 0xcc) {
                    memcpy(&ret.cdp.ip, &proto->addr, 4);
                }
                // }
            } else if (elem->type == 0x03) {
                // port id
                snprintf(ret.cdp.interface, elem->length-3, "%s", (char *) &elem->data);
            } else if (elem->type == 0x04) {
                // capabilities
                memcpy(&ret.cdp.cap, &elem->data, elem->length-4);
            } else if (elem->type == 0x05) {
                // software version
                snprintf(ret.cdp.software, elem->length-3, "%s", (char *) &elem->data);
            } else if (elem->type == 0x06) {
                // Platform
                snprintf(ret.cdp.platform, elem->length-3, "%s", (char *) &elem->data);
            }

            offset += elem->length;
        }

        ret.type = proto_cdp;
    } else if (memcmp(&data[in_info->header_offset + UDP_OFFSET], UDP_SIGNATURE,
                      sizeof(UDP_SIGNATURE)) == 0) {
        // UDP
        ret.type = proto_udp;

        memcpy(ret.source_ip, (const uint8_t *) &msgbuf[in_info->header_offset + UDP_OFFSET + 3], 4);
        memcpy(ret.dest_ip, (const uint8_t *) &msgbuf[in_info->header_offset + UDP_OFFSET + 7], 4);

        if (memcmp(&data[in_info->header_offset + DHCPD_OFFSET], DHCPD_SIGNATURE,
                   sizeof(DHCPD_SIGNATURE)) == 0) {

            // DHCP server responding
            ret.type = proto_dhcp_server;

            // Now we go through all the options until we find options 1, 3, and 53
            // netmask.
            unsigned int offset = in_info->header_offset + DHCPD_OFFSET + 252;


            while (offset < header->len) {
                if (data[offset] == 0x01) {
                    /*
                     fprintf(stderr, "OFfset %X %d.%d.%d.%d\n",
                            offset,
                            data[offset+2], data[offset+3], data[offset+4], data[offset+5]);
                            */

                    // Bail out of we're a "boring" dhcp ack
                    if (data[offset+2] == 0x00) {
                        ret.type = proto_udp;
                        break;
                    }
                    memcpy(ret.mask, &data[offset+2], 4);
                } else if (data[offset] == 0x03) {
                    // Bail out of we're a "boring" dhcp ack
                    if (data[offset+2] == 0x00) {
                        ret.type = proto_udp;
                        break;
                    }
                    memcpy(ret.gate_ip, &data[offset+2], 4);
                } else if (data[offset] == 0x35) {
                    if (data[offset+2] != 0x05) {
                        ret.type = proto_udp;
                        break;
                    } else {
                        // Now rip straight to the heart of it and get the offered
                        // IP from the BOOTP segment
                        memcpy(ret.misc_ip, (const uint8_t *) &data[in_info->header_offset + DHCPD_OFFSET + 28], 4);
                    }
                }
                offset += data[offset+1]+2;
            }
        }
    } else if (memcmp(&data[in_info->header_offset + ARP_OFFSET], ARP_SIGNATURE,
               sizeof(ARP_SIGNATURE)) == 0) {
        // ARP
        ret.type = proto_arp;

        memcpy(ret.source_ip, (const uint8_t *) &data[in_info->header_offset + ARP_OFFSET + 16], 4);
        memcpy(ret.misc_ip, (const uint8_t *) &data[in_info->header_offset + ARP_OFFSET + 26], 4);
    }

    return ret;
}

