package java.awt;


import java.util.*;

/**
  * This is the superclass of all AWT components that can hold other
  * components.
  *
  * @author Aaron M. Renn (arenn@urbanophile.com)
  */

public abstract class Container extends Component implements java.io.Serializable
{

/*
 * Instance Variables
 */

// The list of components in this container
private Vector components = new Vector();


/*************************************************************************/

/*
 * Constructors
 */

/**
  * Default constructor for subclasses.
  */
protected
Container()
{
}

/*************************************************************************/

/*
 * Instance Methods
 */

/**
  * Adds the specified component to this container at the end of the
  * component list.
  *
  * @param component The component to add to the container.
  *
  * @return The same component that was added.
  */
public Component
add(Component component)
{
  addImpl(component, null, -1); 
  return(component);
}

/*************************************************************************/

/**
  * Adds the specified component to the container at the end of the
  * component list.  This method should not be used. Instead, use
  * <code>add(Component, Object</code>.
  *
  * @param name FIXME
  * @param component The component to be added.
  *
  * @return The same component that was added.
  */
public Component
add(String name, Component component)
{
  // FIXME: What is the "name" for? Is it the constraint object?
  addImpl(component, name, -1);
  return(component);
}

/*************************************************************************/

/**
  * Adds the specified component to this container at the specified index
  * in the component list.
  *
  * @param component The component to be added.
  * @param index The index in the component list to insert this child
  * at, or -1 to add at the end of the list.
  *
  * @return The same component that was added.
  *
  * @param throws ArrayIndexOutOfBounds If the specified index is invalid.
  */
public Component
add(Component component, int index)
{
  addImpl(component, null, index);
  return(component);
}

/*************************************************************************/

/**
  * Adds the specified component to this container at the end of the
  * component list.  The layout manager will use the specified constraints
  * when laying out this component.
  *
  * @param component The component to be added to this container.
  * @param constraints The layout constraints for this component.
  */
public void
add(Component component, Object constraints)
{
  addImpl(component, constraints, -1);  
}

/*************************************************************************/

/**
  * Adds the specified component to this container at the specified index
  * in the component list.  The layout manager will use the specified
  * constraints when layout out this component.
  *
  * @param component The component to be added.
  * @param constraints The layout constraints for this component.
  * @param index The index in the component list to insert this child
  * at, or -1 to add at the end of the list.
  *
  * @param throws ArrayIndexOutOfBounds If the specified index is invalid.
  */
public void
add(Component component, Object constraints, int index)
{
  addImpl(component, constraints, index);
}

/*************************************************************************/

/**
  * This method is called by all the <code>add()</code> methods to perform
  * the actual adding of the component.  Subclasses who wish to perform
  * their own processing when a component is added should override this
  * method.  Any subclass doing this must call the superclass version of
  * this method in order to ensure proper functioning of the container.
  *
  * @param component The component to be added.
  * @param constraints The layout constraints for this component, or
  * <code>null</code> if there are no constraints.
  * @param index The index in the component list to insert this child
  * at, or -1 to add at the end of the list.
  *
  * @param throws ArrayIndexOutOfBounds If the specified index is invalid.
  */
protected void
addImpl(Component component, Object constraints, int index)
{
  if (index == -1)
    components.addElement(component);
  else
    components.insertElementAt(component, index);
}

/**
  * Removes the specified component from this container.
  *
  * @return component The component to remove from this container.
  */
public void
remove(Component component)
{
  components.removeElement(component);

  // Destroy peer if necessary
}

/*************************************************************************/

/**
  * Removes the component at the specified index from this container.
  *
  * @param index The index of the component to remove.
  */
public void
remove(int index)
{
  remove((Component)components.elementAt(index));
}

/**
  * Removes all components from this container.
  */
public void
removeAll()
{
  // Remove one at a time in order to generate the proper events.

  Enumeration e = components.elements();
  while (e.hasMoreElements())
    remove((Component)e.nextElement());
}


/**
  * Returns the number of components in this container.
  *
  * @return The number of components in this container.
  */
public int
getComponentCount()
{
  return(components.size());
}

/**
  * Returns the number of components in this container.
  *
  * @return The number of components in this container.
  *
  * @deprecated This method is deprecated in favor of 
  * <code>getComponentCount()</code>.
  */
public int
countComponents()
{
  return(getComponentCount());
}

/**
  * Returns the component at the specified index.
  *
  * @param index The index of the component to retrieve.
  *
  * @return The requested component.
  *
  * @exception ArrayIndexOutOfBoundsException If the specified index is not
  * valid.
  */
public Component
getComponent(int index)
{
  return((Component)components.elementAt(index));
}

/**
  * Returns an array of the components in this container.
  *
  * @return The components in this container.
  */
public Component[]
getComponents()
{
  Component[] retval = new Component[getComponentCount()];
  components.copyInto(retval);
  return(retval);
}

/**
  * Returns the component located at the specified point.  This is done
  * by checking whether or not a child component claims to contain this
  * point.  The first child component that does is returned.  If no
  * child component claims the point, the container itself is returned,
  * unless the point does not exist within this container, in which
  * case <code>null</code> is returned.
  *
  * @param x The X coordinate of the point.
  * @param y The Y coordinate of the point.
  *
  * @return The component containing the specified point, or <code>null</code>
  * if there is no such point.
  */
public Component
getComponentAt(int x, int y)
{
  Enumeration e = components.elements();
  while (e.hasMoreElements())
    {
      Component c = (Component)e.nextElement();
      if (c.contains(x, y))
        return(c);
    }

  if (contains(x, y))
    return(this);
  else
    return(null);
}

/**
  * Returns the component located at the specified point.  This is done
  * by checking whether or not a child component claims to contain this
  * point.  The first child component that does is returned.  If no
  * child component claims the point, the container itself is returned,
  * unless the point does not exist within this container, in which
  * case <code>null</code> is returned.
  *
  * @param point The point to return the component at.
  *
  * @return The component containing the specified point, or <code>null</code>
  * if there is no such point.
  */
public Component
getComponetAt(Point point)
{
  return(getComponentAt(point.x, point.y));
}

/**
  * Returns the component located at the specified point.  This is done
  * by checking whether or not a child component claims to contain this
  * point.  The first child component that does is returned.  If no
  * child component claims the point, the container itself is returned,
  * unless the point does not exist within this container, in which
  * case <code>null</code> is returned.
  *
  * @param point The point to return the component at.
  *
  * @return The component containing the specified point, or <code>null</code>
  * if there is no such point.
  *
  * @deprecated This method is deprecated in favor of
  * <code>getComponentAt(int, int)</code>.
  */
public Component
locate(int x, int y)
{
  return(getComponentAt(x, y));
}

}
